
/*
*****************************************************************************
**                                                                          *
**  Copyright 1994, University of Cambridge Computer Laboratory             *
**                                                                          *
**  All Rights Reserved.					            *
**                                                                          *
*****************************************************************************
**
** FACILITY:
**
**      app/test/video/video.c
** 
** FUNCTIONAL DESCRIPTION:
** 
**      Video display program
** 
** ENVIRONMENT: 
**
**      Unpriv domain
** 
** ID : $Id: rvideo.c 1.3 Thu, 20 May 1999 11:44:15 +0100 dr10009 $
**
*/

/*
** This app displays a raw video stream generated by an AVA and send
** via a PVC. The default PVC is VPI=0, VCI=205. The VCI can be can
** be passed in as the first argument or under the binding vci in the 
** root namespace (in this order).
**
** The Video Stream need to be set up with the following parameters
** (or at least are the once we are using and which worked:
**
** Video Format   : Rgb 15-bit Little Endian
** Video Sizes    : Other
** Video Select   : Either
** Sample X Offset: 0
** Sample Y Offset: 8
** Sample Width   : 768
** Sample Height  : 272
** Display Width  : 384
** Display Height : 272
** ...
** PVCs Enabled   : True
** VPI            : 0
** VCI            : at wish
*/


#include <nemesis.h>
#include <time.h>
#include <exceptions.h>
#include <ecs.h>
#include <IDCMacros.h>
#include <IOMacros.h>
#include <ntsc.h>
#include <salloc.h>
#include <stdio.h>

#include <AALPod.ih>
#include <IO.ih>
#include <CRendDisplayMod.ih>
#include <FB.ih>
#include <FBBlit.ih>

#ifdef DEBUG
#define TRC(_x) _x   
#else
#define TRC(_x) 
#endif 


#define RXVPI 0
#define RXVCI 130

#define RXPKTSIZE 8192
#define BATCH_SIZE 64 /* was 32 */
#define NBUFS  (BATCH_SIZE * 2)


typedef struct _video_st {
  IO_clp           aal_io;
  FBBlit_clp       blit;

  CRendDisplay_clp display;         /* Display        */
  CRend_clp        crend;           /* Window         */

  uint32_t         stride;
  uint32_t         depth;

  word_t           win_width, win_height;

} video_st;


void PktThread(video_st *st) {
  IO_Rec rec[BATCH_SIZE];
  word_t value;
  word_t datalen, ntiles, x, y;
  uchar_t  *buf;
  uint32_t i, raw, pducount, type, xtiles, interlace, pack, format, tile;
  word_t frame_x, frame_y, last_y, last_x, last_ntiles;
  word_t win_width, win_height, pending, tiles_this_frame;
  bool_t maybe_resize;

  last_x = 0;
  last_y = 0;
  last_ntiles = 0;
  tiles_this_frame = 0;
  pending = 0;
  win_width = st->win_width;
  win_height = st->win_height;

  TRC(fprintf (stderr, "video: PktThread started.\n"));
  
  while(1) {
    FBBlit_Rec fbrec;
    uint32_t nrecs;
    
    /* Block for incomming AAL 5 frames */
    IO$GetPkt(st->aal_io, 1, &rec[0], FOREVER, &nrecs, &value);
    
    /* Strip of AAL 5 header and decode */
    buf = rec->base;
    datalen = buf[rec->len - 5] | (buf[rec->len - 6]<<8);
    ntiles = datalen / 128;
    
    raw=0;
    for (i=0; i<4; i++) {
      raw=raw<<8;
      raw|=buf[rec->len-13-i];
    }
    pducount=raw&((1<<12)-1);
    raw>>=12;
    type=raw&((1<<4)-1);
    raw>>=4;
    xtiles=raw&((1<<7)-1);
    raw>>=7;
    interlace=raw&1;
    raw>>=1;
    pack=raw&((1<<6)-1);
    raw>>=6;
    format=raw;
    pack=(pack+1);
    
    switch(format) {
    case 0: /* AVA-200 */
      y=buf[rec->len-15];
      x=buf[rec->len-16]; /* XXX XXX probably not */
      break;
    case 2: /* RawVideo */
      /* Sanity checks */
      tile=pack*pducount;
      y=tile/xtiles;
      x=tile%xtiles;
      break;
    default:
      /* Aaaargh! */
      fprintf(stderr,"RawVideo: received non-rawvideo PDU\n");
      break;
    }
    
    maybe_resize = False;
    
    if(y == 0 && x == 0) {
      /* new frame - calculate height */
      frame_y = last_y * 8 + 24;
      maybe_resize = True;
      tiles_this_frame = 0;
    }

    if(y == 0) {
      tiles_this_frame += ntiles;
    }

    if(y != 0 && last_y == 0) {
      /* end of first line - calc width */
      word_t width;
      width = tiles_this_frame - x;
      frame_x = (width / y) * 8;
      maybe_resize = True;
    }

#if 0
    /* This doesn't seem to work at all */
    /* Resize ? */
    if(maybe_resize) {
      if ((frame_y != win_height) || (frame_x != win_width)) {
	if(!pending) pending = 3;
      } else {
	pending = 0;
      }
      
      if(pending) {
	pending --;
	if (!pending) {
	  fprintf(stderr, "Height = %d, width = %d\n", frame_x, frame_y);
	  CRend$ResizeWindow (st->crend, frame_x, frame_y);
	  CRend$Flush (st->crend);   
	  win_width = frame_x;
	  win_height = frame_y;
	}
      }
    }
#endif 

    /* Prepare to blit */
    fbrec.x = x * 8;
    fbrec.y = y * 8;
    
    fbrec.sourceheight = fbrec.destheight = 8;
    fbrec.sourcewidth = ntiles * 8;
    fbrec.data = buf;
    fbrec.destwidth = fbrec.sourcewidth;
    
    last_y = y;
    last_x = x;
    last_ntiles = ntiles;
    
    /* Blit */
    while(FBBlit$Blit(st->blit, &fbrec));
    
    /* Recycle buffs */
    IO$PutPkt(st->aal_io, 1, &rec[0], value, FOREVER);
  }
}



/* See comment below */
void EventThread(video_st *st) 
{
  WS_Event ev;

  TRC(fprintf (stderr, "video: EventThread started.\n"));

  while(1) {
    /* are there pending events on any of our  windows? */
    while(CRendDisplay$EventsPending(st->display) ) {
      /* get an event */
      CRendDisplay$NextEvent(st->display, &ev);
      /* handle the event */
      switch (ev.d.tag) {
	
      case WS_EventType_Expose:
	{
	  WS_Rectangle *r = &ev.d.u.Expose;
	  
	  CRend$ExposeRectangle(st->crend, r->x1, r->y1, 
				r->x2 - r->x1, 
				r->y2 - r->y1);
	  CRend$Flush(st->crend);
	  break;
	}
      default: 
	break;
      }
      Threads$Yield (Pvs(thds));
    }
  }
}

void main (int argc, char **argv)
{

    NOCLOBBER AALPod_clp  aalpod = NULL;
    IOOffer_clp           offer;
    IDCOffer_clp          aal_offer;
    AALPod_Master         master = AALPod_Master_Client;
    int                   width, height;
    char                  wname[256];
    char      * NOCLOBBER bufs;
    video_st *st;

    word_t vci;

    CRendDisplayMod_clp displaymod;
    WS_clp              ws;
    WS_WindowID         wid;
    uint32_t            dw, dh, dxg, dyg, pbits;
    set_t               proto;
    FB_StreamID         sid;
    FBBlit_clp          blit;


    int x_orig, y_orig;
    int i;

    vci = 0;
    TRY {
      string_t str;
      str = NAME_FIND ("vci", string_t);
      sscanf(str,"%d", &vci);
    } CATCH_ALL {
      vci = 0;
    } ENDTRY;

    if (argc >= 2) {
      if ((sscanf(argv[1],"%d", &vci)) != 1)
	vci = RXVCI;
    } else {
      /* default */
      vci = RXVCI;
    }

    fprintf (stderr, "Video: domain entered.\n");
    fprintf (stderr, "==============> VCI %d\n",vci);

    x_orig = 20;
    y_orig = 50;
    width = 46*8;
    height = 32*8;


    if(!(st = Heap$Malloc(Pvs(heap), sizeof(*st)))) {
	fprintf(stderr,"Video: no memory for state.\n");
	Threads$Exit(Pvs(thds));
    }


    /*
    ** Window system set up
    */

    TRC(fprintf(stderr,"Video: create window.\n"));
    displaymod = NAME_FIND("modules>CRendDisplayMod", CRendDisplayMod_clp);
    st->display = CRendDisplayMod$OpenDisplay(displaymod, "");
    st->crend = CRendDisplay$CreateWindow(st->display, x_orig, y_orig,
					  width, height);
    
    TRC(fprintf(stderr,"Video: map window.\n"));
    CRend$Map(st->crend);
    
    TRC(fprintf(stderr,"Video: get window info.\n"));
    ws  = CRendDisplay$GetWS (st->display);
    wid = CRend$GetID        (st->crend);
    dw = WS$Info (ws, &dh, &dxg, &dyg, &st->depth, &pbits, &proto);

    sprintf(wname,"Raw Video - VCI %d", vci);
    
    /* set the title */
    TRC(fprintf(stderr,"Video: set title.\n"));
    WS$SetTitle (ws, wid, wname);
    CRend$Flush (st->crend);

    /* Get a raw connection to the framestore for the video in DFS format */
    TRC(fprintf(stderr,"Video: get update stream.\n"));
    sid = WS$UpdateStream(ws, wid, FB_Protocol_DFS16, 80, True, &offer, &blit);
    TRC(fprintf(stderr,"*** SID %p\n", sid));

    st->blit = blit;
    st->win_width = width;
    st->win_height = height;

#if 0
    /* 
     * If we don't start the event thread, then the CPU usage is
     * more accurate then with the event thread since the event
     * thread will eat as much CPU as it can, do to the tight endless
     * loop. It might be usefull to have it blocking on an event queue 
     * or something.
     * the event loop is not really necessary anyway, since we hope
     * to constantly display new frames anyway and therfor the expose
     * don't need to be handled events.
     */
    TRC(fprintf(stderr,"Video: fork event thread.\n"));
    Threads$Fork(Pvs(thds), EventThread, st, 0);
#endif

    /*
    ** ATM set up
    */
    while (!aalpod)
    {
      TRY {
	aalpod = IDC_OPEN ("dev>atm0", AALPod_clp);
      } CATCH_Context$NotFound (name) {
	fprintf (stderr, "Video: offer (%s) not present, retrying..\n", name);
	PAUSE (SECONDS (5));
      }	ENDTRY;
    }
    TRC(fprintf (stderr, "Video: bound to /dev/atm0 done\n"));


    if(master == AALPod_Master_Client)     
      {
        Stretch_clp           str;
        Stretch_Size          size;
        
        /* Get a stretch for RX buffers. */
        str    = STR_NEW((NBUFS * RXPKTSIZE));
        if (!str) 
	  {
            fprintf(stderr,"Couldn't get stretch");

	  }
        bufs  =  STR_RANGE(str, &size);
        STR_SETGLOBAL(str, SET_ELEM(Stretch_Right_Read) );
        TRC(fprintf(stderr,"Video: Buffer at %p size %lx\n", bufs, size));
      }




    TRC(fprintf(stderr, "Video: AALPod_Open(%p, %d, %d, %d)\n",
		aalpod, AALPod_Dir_RX, RXVPI, vci));
    aal_offer = AALPod$Open(aalpod, 
			    AALPod_Dir_RX,
			    master,
			    AALPod_Mode_AAL5_Raw,
			    RXVPI, vci, NBUFS);
    TRC(fprintf(stderr, "Video: bind to %p\n", aal_offer));

    
    st->aal_io = IO_BIND (aal_offer); 
    TRC(fprintf(stderr, "Video: bound to %p\n", st->aal_io));


    TRC(fprintf (stderr, "video: forking PktThread\n"));
    
    /* Forking the packet thread before priming the FIFO seems
       to prevent a livelock case at startup. SMH writes:

       >Anyway, the way in which IO channels are supposed to work now is 
       >that the number of slots in the FIFO, plus some shared data memory,
       >are negiotiated/allocated within the bind. The way they worked in the 
       >past was that *double* the number of slots passed in were allocated
       >for the FIFO, + no shared data was allocated during the bind. 
       
       >The problem you have observed could be to do with the way in which the
       >Otto driver handles incoming requests: once it is bound, it forks a
       >thread, and then returns. At some point in the future, the worker 
       >thread will be scheduled and do an IOEntry$Rendezvous. If, however, 
       >you have already put down the maximum number of packets by this stage,
       >then it may not actually receive any events (IOEntry is pretty
       >broken). Or it may be something to do with trying to overfill the
       >FIFO. 
       
       >You should look at the depth of the FIFO being used, and check that
       >you are not trying to fill too much of it before trying to get back
       >the info [in fact, looking at the rvideo source quickly, I'd suggest
       >that you fork the PktThread *before* you prime the IO - it'll not
       >received diddly until you pass things down, but at least it's more
       >likely to be ready by the time stuff is there to be had]. 

       
       So, here we go:
       */
    Threads$Fork(Pvs(thds), PktThread, st, 0);

    /* prime fifo*/
    for (i = 0; i < BATCH_SIZE; i++) {
      IO_Rec rec;
      
      rec.base = bufs + i * RXPKTSIZE;
      rec.len  = RXPKTSIZE;
      TRC(fprintf(stderr,"Prime: %5d: %p, %d\n", i, rec.base, rec.len));
      IO$PutPkt (st->aal_io, 1, &rec, 0, FOREVER);
    }


    TRC(fprintf (stderr, "Video: init done\n"));    
    PAUSE(FOREVER);
    TRC(fprintf (stderr, "Video: done\n"));
}

