#ifndef __PORTABLE_DEFNS_H__
#define __PORTABLE_DEFNS_H__

#define MAX_THREADS 128 /* Nobody will ever have more! */

#if defined(SPARC)
#include "sparc_defns.h"
#elif defined(SPARC_V8PLUS)
#include "sparc_v8plus_defns.h"
#elif defined(INTEL)
#include "intel_defns.h"
#elif defined(PPC)
#include "ppc_defns.h"
#elif defined(IA64)
#include "ia64_defns.h"
#elif defined(MIPS)
#include "mips_defns.h"
#elif defined(ALPHA)
#include "alpha_defns.h"
#else
#error "A valid architecture has not been defined"
#endif

#include <string.h>

#ifndef MB_NEAR_CAS
#define RMB_NEAR_CAS() RMB()
#define WMB_NEAR_CAS() WMB()
#define MB_NEAR_CAS()  MB()
#endif

typedef unsigned long int_addr_t;

typedef unsigned int word_t;
typedef word_t volatile *addr_t;

typedef int bool_t;
#define FALSE 0
#define TRUE  1

#define ADD_TO(_v,_x)                                                   \
do {                                                                    \
    int __val = (_v), __newval;                                         \
    while ( (__newval = CASIO(&(_v),__val,__val+(_x))) != __val )       \
        __val = __newval;                                               \
} while ( 0 )

/*
 * Allow us to efficiently align and pad structures so that shared fields
 * don't cause contention on thread-local or read-only fields.
 */
#define CACHE_PAD(_n) char __pad ## _n [CACHE_LINE_SIZE]
#define ALIGNED_ALLOC(_s)                                       \
    ((void *)(((unsigned long)malloc((_s)+CACHE_LINE_SIZE*2) +  \
        CACHE_LINE_SIZE - 1) & ~(CACHE_LINE_SIZE-1)))

/*
 * Interval counting
 */

typedef unsigned int interval_t;
#define get_interval(_i)                                                 \
do {                                                                     \
    interval_t _ni = interval;                                           \
    do { _i = _ni; } while ( (_ni = CASIO(&interval, _i, _i+1)) != _i ); \
} while ( 0 )

/*
 * POINTER MARKING
 */

#define get_marked_ref(_p)      ((void *)(((unsigned long)(_p)) | 1))
#define get_unmarked_ref(_p)    ((void *)(((unsigned long)(_p)) & ~1))
#define is_marked_ref(_p)       (((unsigned long)(_p)) & 1)


/*
 * SUPPORT FOR WEAK ORDERING OF MEMORY ACCESSES
 */

#ifdef WEAK_MEM_ORDER

#define MAYBE_GARBAGE (0)

/* Read field @_f into variable @_x. */
#define READ_FIELD(_x,_f)                                       \
do {                                                            \
    (_x) = (_f);                                                \
    if ( (_x) == MAYBE_GARBAGE ) { RMB(); (_x) = (_f); }        \
 while ( 0 )

#define WEAK_DEP_ORDER_RMB() RMB()
#define WEAK_DEP_ORDER_WMB() WMB()
#define WEAK_DEP_ORDER_MB()  MB()

#else

/* Read field @_f into variable @_x. */
#define READ_FIELD(_x,_f) ((_x) = (_f))

#define WEAK_DEP_ORDER_RMB() ((void)0)
#define WEAK_DEP_ORDER_WMB() ((void)0)
#define WEAK_DEP_ORDER_MB()  ((void)0)

#endif

/*
 * Strong LL/SC operations
 */

static _u32 strong_ll(_u64 *ptr, int p)
{
    _u64 val_read;
    _u64 new_val;
    _u64 flag;

    flag = (1LL << p);

    new_val = *ptr;
    do {
        val_read = new_val;
        new_val = val_read | flag;
    } while ( ((val_read & flag) == 0) &&
              ((new_val = CAS64O(ptr, val_read, new_val)) != val_read) );

    return (_u32) (val_read >> 32);
}

static int strong_vl(_u64 *ptr, int p) 
{
    _u64 val_read;
    _u64 flag;

    flag = (1LL << p);
    val_read = *ptr;

    return (val_read & flag);
}

static int strong_sc(_u64 *ptr, int p, _u32 n) 
{
    _u64 val_read;
    _u64 new_val;
    _u64 flag;

    flag = (1LL << p);
    val_read = *ptr;

    while ( (val_read & flag) != 0 )
    {
        new_val = (((_u64)n) << 32);

        if ( (new_val = CAS64O(ptr, val_read, new_val)) == val_read )
        {
            return 1;
        }

        val_read = new_val;
    }

    return 0;
}

static void s_store(_u64 *ptr, _u32 n) 
{
    _u64 new_val;

    new_val = (((_u64)n) << 32);
    *ptr = new_val;
}

static _u32 s_load(_u64 *ptr) 
{
    _u64 val_read;

    val_read = *ptr;
    return (val_read >> 32);
}


/*
 * MCS lock
 */

typedef struct qnode_t qnode_t;

struct qnode_t {
    qnode_t *next;
    int locked;
};

typedef struct {
    qnode_t *tail;
} mcs_lock_t;

static void mcs_init(mcs_lock_t *lock)
{
    lock->tail = NULL;
}

static void mcs_lock(mcs_lock_t *lock, qnode_t *qn)
{
    qnode_t *pred;

    qn->next = NULL;
    qn->locked = 1;
    WMB_NEAR_CAS();

    pred = FASPO(&lock->tail, qn);
    if ( pred != NULL )
    {
        pred->next = qn;
        while ( qn->locked ) RMB();
    }

    MB();
}

static void mcs_unlock(mcs_lock_t *lock, qnode_t *qn)
{
    qnode_t *t = qn->next;

    MB();

    if ( t == NULL )
    {
        if ( CASPO(&lock->tail, qn, NULL) == qn ) return;
        while ( (t = qn->next) == NULL ) RMB();
        WEAK_DEP_ORDER_MB();
    }

    t->locked = 0;
}


#if defined(FAIR_MRSW_LOCKS)

/*
 * MCS fair MRSW lock.
 */

typedef struct mrsw_qnode_st mrsw_qnode_t;

struct mrsw_qnode_st {
#define CLS_RD 0
#define CLS_WR 1
    int class;
#define ST_NOSUCC   0
#define ST_RDSUCC   1
#define ST_WRSUCC   2
#define ST_SUCCMASK 3
#define ST_BLOCKED  4
    int state;
    mrsw_qnode_t *next;
};

typedef struct {
    mrsw_qnode_t *tail;
    mrsw_qnode_t *next_writer;
    int reader_count;
} mrsw_lock_t;


#define CLEAR_BLOCKED(_qn) ADD_TO((_qn)->state, -ST_BLOCKED)

static void mrsw_init(mrsw_lock_t *lock)
{
    memset(lock, 0, sizeof(*lock));
}

static void rd_lock(mrsw_lock_t *lock, mrsw_qnode_t *qn)
{
    mrsw_qnode_t *pred, *next;

    qn->class = CLS_RD;
    qn->next  = NULL;
    qn->state = ST_NOSUCC | ST_BLOCKED;

    WMB_NEAR_CAS();

    pred = FASPO(&lock->tail, qn);
    
    if ( pred == NULL )
    {
        ADD_TO(lock->reader_count, 1);
        CLEAR_BLOCKED(qn);
    }
    else
    {
        if ( (pred->class == CLS_WR) ||
             (CASIO(&pred->state, ST_BLOCKED|ST_NOSUCC, ST_BLOCKED|ST_RDSUCC)
              == (ST_BLOCKED|ST_NOSUCC)) )
        {
            WEAK_DEP_ORDER_WMB();
            pred->next = qn;
            while ( (qn->state & ST_BLOCKED) ) RMB();
        }
        else
        {
            ADD_TO(lock->reader_count, 1);
            pred->next = qn;
            WEAK_DEP_ORDER_WMB();
            CLEAR_BLOCKED(qn);
        }
    }

    if ( qn->state == ST_RDSUCC )
    {
        while ( (next = qn->next) == NULL ) RMB();
        ADD_TO(lock->reader_count, 1);
        WEAK_DEP_ORDER_WMB();
        CLEAR_BLOCKED(next);
    }

    RMB();
}

static void rd_unlock(mrsw_lock_t *lock, mrsw_qnode_t *qn)
{
    mrsw_qnode_t *next = qn->next;
    int c, oc;

    RMB();

    if ( (next != NULL) || (CASPO(&lock->tail, qn, NULL) != qn) )
    {
        while ( (next = qn->next) == NULL ) RMB();
        if ( (qn->state & ST_SUCCMASK) == ST_WRSUCC )
        {
            lock->next_writer = next;
            WMB_NEAR_CAS(); /* set next_writer before dec'ing refcnt */
        }
    }

    /* Bounded to maximum # readers if no native atomic_decrement */
    c = lock->reader_count;
    while ( (oc = CASIO(&lock->reader_count, c, c-1)) != c ) c = oc;
   
    if ( c == 1 )
    {
        WEAK_DEP_ORDER_MB();
        if ( (next = lock->next_writer) != NULL )
        {
            RMB();
            if ( (lock->reader_count == 0) &&
                 (CASPO(&lock->next_writer, next, NULL) == next) )
            {
                WEAK_DEP_ORDER_WMB();
                CLEAR_BLOCKED(next);
            }
        }
    }
}

static void wr_lock(mrsw_lock_t *lock, mrsw_qnode_t *qn)
{
    mrsw_qnode_t *pred;
    int os, s;

    qn->class = CLS_WR;
    qn->next  = NULL;
    qn->state = ST_NOSUCC | ST_BLOCKED;

    WMB_NEAR_CAS();
    
    pred = FASPO(&lock->tail, qn);

    if ( pred == NULL )
    {
        WEAK_DEP_ORDER_WMB();
        lock->next_writer = qn;
        MB(); /* check reader_count after setting next_writer. */
        if ( (lock->reader_count == 0) &&
             (CASPO(&lock->next_writer, qn, NULL) == qn) )
        {
            CLEAR_BLOCKED(qn);
        }
    }
    else
    {
        s = pred->state;
        /* Bounded while loop: only one other remote update may occur. */
        while ( (os = CASIO(&pred->state, s, s | ST_WRSUCC)) != s ) s = os;
        WMB();
        pred->next = qn;
    }

    while ( (qn->state & ST_BLOCKED) ) RMB();

    MB();
}

static void wr_unlock(mrsw_lock_t *lock, mrsw_qnode_t *qn)
{
    mrsw_qnode_t *next = qn->next;

    MB();

    if ( (next != NULL) || (CASPO(&lock->tail, qn, NULL) != qn) )
    {
        while ( (next = qn->next) == NULL ) RMB();
        WEAK_DEP_ORDER_MB();
        if ( next->class == CLS_RD )
        {
            ADD_TO(lock->reader_count, 1);
            WMB();
        }
        CLEAR_BLOCKED(next);
    }
}





#elif defined(KAF_MRSW_LOCKS)




/*
 * MCS-based MRSW lock.
 * 
 * This lock is Copyright (c) 2002-2003, K A Fraser
 */

typedef struct mrsw_qnode_t mrsw_qnode_t;

struct mrsw_qnode_t {
    mrsw_qnode_t *next;
    int locked;
};

typedef struct {
    _u32           status;   /* contains reader counts, and status flags */
    mrsw_qnode_t  *wr_head;  /* the first writer in sequence */
    mrsw_qnode_t  *wr_tail;  /* the last writer in sequence */
} mrsw_lock_t;

/* Macros for accessing 'status' field. */
#define WRITERS         1  /* Are there any writers active or interested?  */
#define READ_GATE       2  /* Pending readers wait for the gate to change. */
#define GATE_FLIPPED(_l, _s) (((_l)->status ^ (_s)) & READ_GATE)
/* PENDCNT counts the number of readers waiting for READ_GATE to flip. */
#define PENDCNT_MASK    0x0001fffc
#define PENDCNT_SHIFT   2
#define PENDCNT_INCR    (1<<PENDCNT_SHIFT)
/* ACTIVECNT counts the number of readers who currently have the lock. */
#define ACTIVECNT_MASK  0xfffe0000
#define ACTIVECNT_SHIFT 17
#define ACTIVECNT_INCR  (1<<ACTIVECNT_SHIFT)

/* The valid lock states for a queue node. */
#define UNBLOCKED       0
#define UNBLOCK_WRITER  1
#define UNBLOCK_READERS 2

/* UNBLOCK_READERS is passed via the 'wr_tail' field. These hide the gunk. */
#define SET_UNBLOCK_FL(_p)   \
    (mrsw_qnode_t *)((unsigned long)(_p) | UNBLOCK_READERS)
#define CLEAR_UNBLOCK_FL(_p) \
    (mrsw_qnode_t *)((unsigned long)(_p) &~UNBLOCK_READERS)
#define QUERY_UNBLOCK_FL(_p) \
    ((unsigned long)(_p) & UNBLOCK_READERS)

/* Unblock any pending readers, and check & set WRITERS flag as requested. */
static bool_t unblock_readers(mrsw_lock_t *lock, int pre_wp, int post_wp)
{
    _u32 os, s, ns;

    os = lock->status;
    do {
        ns  = s = os;
        if ( (s & WRITERS) != pre_wp ) return FALSE;
        ns += (ns & PENDCNT_MASK) << (ACTIVECNT_SHIFT - PENDCNT_SHIFT);
        ns  = (ns ^ READ_GATE) & ~(PENDCNT_MASK | WRITERS);
        ns |= post_wp;
    } while ( (os = CAS32O(&lock->status, s, ns)) != s );

    return TRUE;
}

static void mrsw_init(mrsw_lock_t *lock)
{
    memset(lock, 0, sizeof(*lock));
}

static void rd_lock(mrsw_lock_t *lock)
{
    _u32 os, s, ns;
    mrsw_qnode_t *oqn, *qn;

    /* Bump up the active count (or pending count if there are writers). */
    s = lock->status;
    while ( (os = CAS32O(&lock->status, s,
             s + ((s & WRITERS) ? PENDCNT_INCR : ACTIVECNT_INCR))) != s )
        s = os;

    /* If no writers are present then we can carry on. */
    if ( !(s & WRITERS) ) goto out;

    if ( (s & PENDCNT_MASK) == 0 )
    {
        /* We're the first pending reader, so tag a writer to unblock us. */
        oqn = lock->wr_tail;
        do {
            /* If there's no writer to tag, try unblocking ourselves. */
            while ( (qn = oqn) == NULL )
            {
                if ( unblock_readers(lock, 0, 0) ) goto out;
                RMB(); oqn = lock->wr_tail;
            }
        }
        while ( (oqn = CASPO(&lock->wr_tail, qn, SET_UNBLOCK_FL(qn))) != qn );
    }

    /* Okay, now wait for a writer to flip the reader gate. */
    while ( !GATE_FLIPPED(lock, s) ) RMB();

 out:
    RMB(); /* Stop critical region accesses from leaking out. */
}

static void rd_unlock(mrsw_lock_t *lock)
{
    _u32 os, s = lock->status;
    mrsw_qnode_t *qn;

    RMB_NEAR_CAS();

    while ( (os = CAS32O(&lock->status, s, s - ACTIVECNT_INCR)) != s ) s = os;

    /* Are we the last reader, with writers pending? Unblock one if so. */
    if ( (s & (ACTIVECNT_MASK|WRITERS)) == (ACTIVECNT_INCR|WRITERS) )
    {
        /* Unblock first writer. Wait for it to set 'wr_head' field first. */
        do { RMB(); } while ( (qn = lock->wr_head) == NULL );
        lock->wr_head = NULL;
        qn->locked = UNBLOCKED;
    }
}

static void wr_lock(mrsw_lock_t *lock, mrsw_qnode_t *qn)
{
    mrsw_qnode_t *pred;
    _u32 os, s;

    qn->next = NULL;
    qn->locked = UNBLOCK_WRITER;
    WMB_NEAR_CAS();

    /* Install our queue node at tail of writer list. */
    pred = FASPO(&lock->wr_tail, qn);

    if ( QUERY_UNBLOCK_FL(pred) )
    {
        pred = CLEAR_UNBLOCK_FL(pred);
        qn->locked = UNBLOCK_READERS;
        WMB();
    }

    if ( pred != NULL )
    {
        /* We have a predecessor who will definitely unblock us. */
        pred->next = qn;
    }
    else
    {
        /*
         * No active predecessor. Wait for last writer to clear writer flag.
         * That may also have side effect of starting up some pending readers.
         */
        do { RMB(); } while ( ((s = lock->status) & WRITERS) );
        /* Register that there is at least one writer pending. */
        while ( (os = CAS32O(&lock->status, s, s | WRITERS)) != s ) s = os;
        /* If there are no active readers then we can carry straight on. */
        if ( (s & ACTIVECNT_MASK) == 0 ) goto out;
        WEAK_DEP_ORDER_WMB();
        lock->wr_head = qn;
    }

    while ( qn->locked != UNBLOCKED ) RMB();
 
 out:
    MB();
}

static void wr_unlock(mrsw_lock_t *lock, mrsw_qnode_t *qn)
{
    mrsw_qnode_t *old_next, *next = qn->next;

    MB();

    if ( next == NULL )
    {
        next = lock->wr_tail;
        while ( CLEAR_UNBLOCK_FL(next) == qn )
        {
            if ( (old_next = CASPO(&lock->wr_tail, next, NULL)) == next )
            {
                if ( QUERY_UNBLOCK_FL(next) )
                    (void)unblock_readers(lock, WRITERS, 0);
                else
                    ADD_TO(lock->status, -WRITERS);
                return;
            }
            next = old_next;
        }
        
        /* A new writer is installing itself. Wait for it to finish setup. */
        while ( (next = qn->next) == NULL ) RMB();
        WEAK_DEP_ORDER_MB();
    }

    if ( next->locked == UNBLOCK_WRITER )
    {
        /* A writer is next in turn, or there are no pending readers. */
        next->locked = 0;
    }
    else
    {
        /* Set up the next writer to be woken by the last pending reader. */
        lock->wr_head = next; /* No need for 'next->locked = UNBLOCK_WRITER' */
        (void)unblock_readers(lock, WRITERS, WRITERS);
    }
}

#define rd_lock(_x, _y)   rd_lock(_x)
#define rd_unlock(_x, _y) rd_unlock(_x)

#endif

#endif /* __PORTABLE_DEFNS_H__ */
