/**********************************************************************
 *
 * The hold-release operations aim to provide a convenient programming
 * abstraction for developing obstruction-free and lock-free algorithms.
 *
 * Each thread can "hold" at most one location at a time.  Similarly,
 * each location can be held by at most one thread at any time. 
 *
 * For example, in our STM implementation, a thread holds the location
 * holding the status of the descriptor that it is currently working
 * on.  This means that -- while it keeps hold of that location -- it
 * knows that it is the only thread working on that descriptor.  This
 * simplifies the logic of many of the STM operations when compared
 * with the design originally presented in our OOPSLA 2003 paper.
 *
 * When a thread holds a location it supplies a 'revocation address'.
 * At any point while holding the location, the system can revoke
 * access and atomically push the ex-holder to its revocation address.
 *
 * Threads can read from locations held by other threads.
 *
 * Threads can write to locations that they hold and to locations that
 * are currently unheld by any thread.  However, threads cannot write
 * to locations that are currently held by another thread.
 *
 * A number of options are possible when one thread ("A") tries to
 * take hold of a location that is already held by "B":
 *
 *   - "A" could be blocked until "B" releases the location
 *
 *   - "A" could cause "B" to be pushed to its revocation address and
 *     "A" could then take over holding the location.
 *
 * There are many other schemes which we hope to discuss in a
 * forthcoming paper.
 *
 * This C API requires the user to decide ahead-of-time which
 * locations may be subject to hold/release operations.  Such
 * locations must always be accessed through hr_read and hr_write
 * operations in place of direct reads and writes.
 *
 * We expose holding using a C macro:
 *
 *     holding(me, addr, {
 *        <statements>
 *     });
 * 
 * This causes the client thread "me" to execute "statements" while
 * holding the locaiton "addr".  The revocation address is set to be
 * the end of the block of statements, meaning that the statements may
 * terminate part-way through rather than running to completion.
 *
 ***********************************************************************/

#ifndef _HOLDREL_H_
#define _HOLDREL_H_

#include "portable_defns.h"

#ifndef SPARC_V8PLUS
#error "Only tested for SPARC V8PLUS"
#endif

#include <setjmp.h>
#include <schedctl.h>
#include <sys/lwp.h>

typedef struct per_thread_t per_thread_t;

/*
 * Register "me" with index "idx" for the hold-release operations
 */

extern void hr_init (per_thread_t *me, int idx);

/*
 * Read and write operations to locations "addr" which may be subject to
 * holding.
 */

static word_t hr_read (per_thread_t *me, addr_t addr); 
static void hr_write (per_thread_t *me, addr_t addr, word_t val);

/*
 * A per_thread_t structure and thread number ("idx") is passed to
 * hr_init for each thread that uses the hold/release interface.
 *
 * "addr" holds the address currently held (NULL => no hold active).  The address
 * holds a tagged pointer to the per-thread structure.
 * 
 * "displaced" holds the displaced contents of the held address.
 *
 * "jb" holds the resumption point if the location is wrested from
 * this thread.  
 */

struct per_thread_t {
  addr_t volatile addr;
  int    volatile displaced;
  int             idx;    
  jmp_buf         jb;

  // Counters to allow lock-free snapshot
  unsigned int volatile holds_started;
  unsigned int volatile holds_completed;

  // Schedctl state for this thread
  schedctl_t *sc;

  // Control fields for wresting ownership from this thread
  lwpid_t     lwp;
  thread_t    thr;
  int         fd_ctl;
};

/*
 * Hold location "addr"
 */

extern void hold (per_thread_t *me, addr_t addr);

/*
 * Release held location
 */

extern void release (per_thread_t *me);

/*
 * Execute statements "c" while "me" holds address "addr".  We use a macro
 * to ensure that the saved jmp_buf remains on an active stack frame.
 */

#define holding(_me,_addr,_c) do {		      \
  if ((_me) -> addr != NULL) release((_me));          \
  if (!setjmp((_me) -> jb)) {                         \
    hold((_me), (_addr));  		              \
    _c;					              \
    release((_me));			              \
  }                                                   \
} while (0)

/*
 * Slow-path code for read/write. 
 */

extern word_t hr_read0 (per_thread_t *me, addr_t addr);
extern void hr_write0 (per_thread_t *me, addr_t addr, word_t val);

/**********************************************************************
 *
 * Fast-path code for read/write to the thread's currently held location.
 */

static void hr_write (per_thread_t *me, addr_t addr, word_t val)
{
  if (me -> addr == addr) {
    me -> displaced = val;
  } else {
    hr_write0 (me, addr, val);
  }
}

static word_t hr_read (per_thread_t *me, addr_t addr) 
{
  if (me -> addr == addr) {
    return me -> displaced;
  } else {
    return hr_read0 (me, addr);
  }
}

#endif /* _HOLDREL_H_ */
