/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios.swing;


import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Logger;

import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.PopupFactory;
import javax.swing.RepaintManager;
import javax.swing.SwingUtilities;

import t3.hrd.state.JOGLHelper;
import t3.portfolios.Person;
import t3.portfolios.PointInputDevice;
import t3.portfolios.PointInputDeviceTypeAndButtonSet;
import t3.portfolios.Portfolio;
import t3.portfolios.PortfolioCommonBehaviour;
import t3.portfolios.PortfolioEvent;
import t3.portfolios.PortfolioServer;

/**
 * A portfolio whose tile appears to be a frame into which you can place Java Swing 
 * components that respond to keyboard and point input device events. 
 * Some limitations apply.
 * <p> 
 * The frame is created automatically when you call the constructor. 
 * The frame's RootPane (ie its contents) has the same width and height 
 * as the tile. 
 * <p>
 * You can get the frame by calling the getFrame method, and then add JComponents
 * in the usual way. 
 * <p>
 * Repaint requests on the frame are trapped by the system. The frame is then painted into
 * the tile's buffered image, which is then sent to the client, where it appears on the
 * tabletop. Synchronization and endOfBurst calls are done automatically (see note on deadlock 
 * below).
 * <p>
 * PortfolioEvents are converted by the system into the corresponding AWT events and dispatched
 * to the appropriate components in the frame. We use subclasses of the AWT event classes
 * that give access to the underlying PortfolioEvent. Synchronization and endOfBurst
 * calls are done automatically.
 * <p>
 * &nbsp;
 * <p>
 * <b>Threading Note 1: Manipulating Swing components in response to a Swing event</b> 
 * <p>
 * If you simply want to create/remove/manipulate Swing components in response to a
 * Swing event (e.g. you want to make a radio button look checked in response to a 
 * MouseEvent) then the code should work just as it would under Swing and you do
 * not need to modify your code at all.
 * <p>
 * &nbsp;
 * <p>
 * <b>Threading Note 2: Manipulating T3 in response to a Swing event</b> 
 * <p>
 * If you wish to manipulate T3 (e.g. create/destroy/move portfolios) in response to a
 * Swing event then you need to take care. The swing event handling code runs in another thread,
 * the swing repaint thread, so you need to follow the standard T3 practice for code that runs
 * outside of T3's event handling routines, by calling myPortfolioServer.peformActionAsynchronously(...);
 * <p>
 * &nbsp;
 * <p>
 * <b>Threading Note 3: Manipulating Swing from T3</b> 
 * <p>
 * <ul>
 * <li>IF you want to manipulate a SwingFramePortfolio (e.g. create a SwingFramePortfolio, or add Swing components to a frame
 * or destroy a SwingFramePortfolio)
 * <lI>AND your manipulation code is not already running from within Swing's repaint thread
 * <ul>
 * 		<li>i.e. you are not responding to a Swing event (e.g. a MouseEvent).
 * </ul> 
 * <li>AND your code is being run from T3
 * <ul>
 * 		<li>e.g. you are within customProcessEventForThisPortfolioNotChildren
 * 		<li>e.g. your code is being run by myPortfolioServer.peformActionAsynchronously(...)
 * </ul> 
 * <li>THEN you must do the manipulation from within Swing's repaint thread, after T3 has finished
 * running your code (and hence dropped its locks) by using the SwingUtilities.invokeLater method.
 * </ul>
 * <p>
 * Here's how to do it. If you're just adding/removing/changing swing components, use:
 * <br>runLaterFromSwingThread( new Runnable() { public void run() {
 * <br>&nbsp;&nbsp;&nbsp;&nbsp;YOUR CODE HERE
 * <br>} } );
 * <p>
 * If you're also creating/destroying/repositioning portfolios, or if you're creating or destroying a swingframeportfolio use:
 * <br>runLaterFromSwingThreadUsesPortfolios( myPortfolioServer, new Runnable() { public void run() {
 * <br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;YOUR CODE HERE
 * <br>&nbsp;&nbsp;&nbsp;&nbsp;}
 * <br>} } );
 * <p> 
 * When using anonymous inner classes like this, don't forget that if you refer to
 * variables that have been defined outside your anonymous class 
 * then you might have to declare the variables final to avoid compile errors. 
 * <p>
 * Full gory details: When you manipulate a SwingFramePortfolio it generates swing repaint
 * messages. These are caught by Swing's repaint thread which then tries to acquire the lock
 * on stateManager before it sends the updates to the T3 portfolio server. If this lock is already
 * held by another thread (A) then the repaint thread just waits until the other thread releases the lock.
 * However, if thread A is blocked waiting for the swing thread to process another runnable on the event queue
 * using invokeAndWait() then obviously the system is now deadlocked. This can happen - JEditorPane uses it. 
 * When this portfolio processes T3 events and dispatches them onto swing frames,
 * it also uses the SwingUtilities.invokeLater method. 
 * @author pjt40
 *
 */
public class SwingFramePortfolio extends Portfolio {

	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios.swing");
	final FrameForPortfolio frame;

	// todo set sun.awt.exception.handler
	static final boolean DEBUG_SWING = true; 
	static final boolean DEBUG_SWING_ONTOP = false; 
    private final boolean autobubble;
	
    public SwingFramePortfolio(
            PortfolioServer pm,
            Portfolio parent,
            PortfolioCommonBehaviour c,
            int width,
            int height,
            int tileFlags,
            int portfolioFlags
        ) {
        this(pm,parent,c,width, height, tileFlags, portfolioFlags, false);
    }
    
	public SwingFramePortfolio(
		PortfolioServer pm,
		Portfolio parent,
		PortfolioCommonBehaviour c,
		int width,
		int height,
		int tileFlags,
		int portfolioFlags,
        boolean bubbleEventsToParentPortfolio
	) {
		super(false, pm, parent, c, true, width, height, tileFlags, portfolioFlags);
        this.autobubble = bubbleEventsToParentPortfolio;
		this.frame = new FrameForPortfolio(this, width, height);
		swingFramePortfoliosThreadSafe.add(this);
	}
    
    public static void runLaterFromSwingThread(Runnable r) {
        SwingUtilities.invokeLater(r);
    }
    
    public static void runLaterFromSwingThreadUsesPortfolios(final PortfolioServer ps, final Runnable r) {
        SwingUtilities.invokeLater(
              new Runnable() { public void run() {
                  ps.performActionAsynchronouslyByCurrentThread(r);
              }}
        );
    }
    	
	public FrameForPortfolio getFrame() {
		return this.frame;
	}
	
	protected boolean customProcessEventForThisPortfolioNotChildren(final PortfolioEvent e, boolean bubbled) { 
		if(
            (e.pointInputDevice==null || e.pointInputDevice.pointInputDeviceType==0) 
            && !bubbled
        ) {
            
			javax.swing.SwingUtilities.invokeLater(new Runnable() {
	            public void run() {
	            	// can also dispatch stright on glasspane
	            	InputEvent[] aes = createAWTEventsFromPortfolioEvent(
	    					SwingFramePortfolio.this, 
	    					e,
	    					SwingFramePortfolio.this.frame
	    				);
	            	for(InputEvent ae: aes) {
		            	if(ae!=null) {
			            	Component source = (Component)ae.getSource();
			            	if(e.eventType!=e.EVENT_PID_MOVE) {
			    				logger.fine("Dispatching event to swing: "+ae);
			            	}
			            	source.dispatchEvent(ae);
			    			ae.consume();
		            	} else {
		            		logger.warning("Couldn't dispatch event to swing: "+ae);
		            		// couldn't make an awt event out of it :(
		            	}
	            	}
	            }
			});	
			//  let's return true or false
			return !autobubble;
		} else {
			// bubble
			return false;
		}
	}
	
	
	protected void customProcessFDOPevent(PortfolioEvent e, double PORTxWhenEnteredFDOPmode, double PORTyWhenEnteredFDOPmode) {
	}
	
	protected void customProcessEndOfFDOPmode(PointInputDevice pen, int button) {
	}
	
	
	public void customRepaintTileForThisPortfolioNotChildren(Rectangle r, BufferedImage update, Graphics2D g) {
		
		// it's a shame we have to allocate new memory all the time
		// but if we just reuse bits of an bigger image then it doesn't 
		// give much speedup since sooner or later we need a copy that's the
		// right size to send.
		
		// check that r is actually within the tile
		// r can go outside tile bounds if we eneter too much text in a text field, apparently!
		// and r can end up having negative width and height, apparently!
					
		
		if(! ( new Rectangle(0,0,this.getTileWidthInTILE(), this.getTileHeightInTILE())).contains(r) ) {
			return;
		}
					
		
		// render the part of the pane into g
		this.frame.getRootPane().paint(g); 
		
		// now we need to render the popups
		for(MyPopup pop:SwingFramePortfolio.mypf.getPopupsForSourceRootPane(this.frame.getRootPane())) {
			
			// get x and y of top left of component to be painted
			Rectangle allPopRectInRootFrame = pop.getRDesiredPopupBoundsInFrameOfRefOfSourceRootPane();
			
			// intersect with redraw rectangle
			Rectangle popRectToDrawInRootFrame  = 
				r.intersection( allPopRectInRootFrame );
			
			Rectangle popRectToDrawInPop = 
				SwingUtilities.convertRectangle(this.frame.getRootPane() , popRectToDrawInRootFrame, pop.contents);
						
			// create graphics gg so that (popRectToDrawInPop.x,popRectToDrawInPop.y) in gg 
			// corresponds to (popRectToDrawInRootFrame.x,popRectToDrawInRootFrame.y) in g
						
			Graphics gg = g.create(
					popRectToDrawInRootFrame.x-popRectToDrawInPop.x,
					popRectToDrawInRootFrame.y-popRectToDrawInPop.y, 
					popRectToDrawInPop.width, 
					popRectToDrawInPop.height
				);
			
			//g.fillRect(popRectToDrawInPop.x,popRectToDrawInPop.y,popRectToDrawInPop.width,popRectToDrawInPop.height);
			
			// now paint the popup into gg
			pop.contents.paint(gg);
			
			gg.dispose();
		}		
		
		g.dispose();

	}
	
	
	

	
	
	
	/* static stuff */
	
	static int nextY = 0;
	static OuterFrame outerFrame;
	static MyPopupFactory mypf;
	static VerboseRepaintManager rpm;
	
	// thread safe 
	static Set<SwingFramePortfolio> swingFramePortfoliosThreadSafe = Collections.synchronizedSet(new HashSet<SwingFramePortfolio>());
	
	private static Map<Person, Component> personToKbf = new HashMap<Person, Component>();
	
	private static InputEvent[] createAWTEventsFromPortfolioEvent(Portfolio p, PortfolioEvent e, FrameForPortfolio frame) {
		//assert false;
		// need to tranform into tile space...
		
		long awtEventWhen = System.currentTimeMillis();
		
		// todo key up, down not just typed
		if(e.eventType == e.EVENT_KEYBOARD_PRESSED || e.eventType == e.EVENT_KEYBOARD_RELEASED || e.eventType==e.EVENT_KEYBOARD_TYPED) {
						
			Component awtEventSource = personToKbf.get(e.person);			
			// used to use this:
			//KeyboardFocusManager.getCurrentKeyboardFocusManager().getFocusOwner();
			
			if(awtEventSource==null) { 
				return new InputEvent[] {}; 
			}
			System.out.println("Person "+e.person+" to component "+awtEventSource.hashCode());
			
			
			return new InputEvent[] { new SwingFramePortfolioKeyEvent(
				e,
				awtEventSource,
				e.eventType == e.EVENT_KEYBOARD_PRESSED 
					? KeyEvent.KEY_PRESSED : (
						e.eventType == e.EVENT_KEYBOARD_RELEASED 
						? KeyEvent.KEY_RELEASED : KeyEvent.KEY_TYPED
					),					
				awtEventWhen,
				e.keyboardAWTModifiers,
				e.keyboardAWTKeyCode,
				e.keyboardAWTChar
				) };
		} else {
			//mouse
			
			int awtEventClickCount;
			
			// get event id
			int awtEventId;
			if(e.eventType == e.EVENT_PID_CLICK) {
				// actually we ignore these messages
				// we need to generate our awt click events
				// straight after our release events
				// otherwise popups can disappear before they've
				// received their events.
				return new InputEvent[] {};
			} else if(e.eventType == e.EVENT_PID_ENTER) {
				awtEventId = MouseEvent.MOUSE_ENTERED;
				awtEventClickCount = 0;	
			} else if(e.eventType == e.EVENT_PID_EXIT) {
				awtEventId = MouseEvent.MOUSE_EXITED;
				awtEventClickCount = 0;	
			} else if(e.eventType == e.EVENT_PID_MOVE) {
				awtEventId = e.pidButtons==0
					? MouseEvent.MOUSE_MOVED
					: MouseEvent.MOUSE_DRAGGED;
				awtEventClickCount =  e.pidButtons==0
					? 0
					: 1;
			} else if(e.eventType == e.EVENT_PID_PRESS) {
				awtEventId = MouseEvent.MOUSE_PRESSED;
				awtEventClickCount = 1;	
			} else if(e.eventType == e.EVENT_PID_RELEASE) {
				awtEventId = MouseEvent.MOUSE_RELEASED;
				awtEventClickCount = 1;	
			} else {
				assert false;
				return new InputEvent[] {};
			}
			
			// get button
			int awtEventButton;
			if(e.pidButton==0) {
				awtEventButton = MouseEvent.BUTTON1;
			} else if(e.pidButton==1) {
				awtEventButton = MouseEvent.BUTTON2;
			} else if(e.pidButton==2) {
				awtEventButton = MouseEvent.BUTTON3;
			} else {
				// we don't do those kinds of events
				return new InputEvent[] {};
			}
            
            // modifiers
			int awtEventModifiers=0;
			if((e.pidButtons & 1<<0) != 0) { awtEventModifiers |= MouseEvent.BUTTON1_DOWN_MASK; }
			if((e.pidButtons & 1<<1) != 0) { awtEventModifiers |= MouseEvent.BUTTON2_DOWN_MASK; }
			if((e.pidButtons & 1<<2) != 0) { awtEventModifiers |= MouseEvent.BUTTON3_DOWN_MASK; }
			// todo modifiers for shift, alt, etc? - prob not
			
			// coords
			double[] unhomogDeventCoordsInTILE = 
				JOGLHelper.unhomogeniseD(
					JOGLHelper.getDFromM(
						p.getmDESKtoTILE().times(
							JOGLHelper.getMFromD(
								e.DESKx,
								e.DESKy,
								1.0
							)
						)
					)				
				);
			
			
			// event source and x and y rel to that source
			
			Point awtEventPointinPortfoliosRootPane = 
				new Point(
						(int)Math.round(unhomogDeventCoordsInTILE[0]),
					(int)Math.round(unhomogDeventCoordsInTILE[1])
				);
			Component awtEventSource=null;
			Point awtPointRelToEventSource;
			
			if(e.eventType == e.EVENT_PID_EXIT || e.eventType == e.EVENT_PID_ENTER) {
				awtEventSource = frame.getGlassPane();
				awtPointRelToEventSource = awtEventPointinPortfoliosRootPane;
			
			} else {
				
				// see if our event lies within a popup...
				for(MyPopup popup:mypf.getPopupsForSourceRootPane(frame.getRootPane())) {
					if(awtEventSource==null && popup.getRDesiredPopupBoundsInFrameOfRefOfSourceRootPane().contains(awtEventPointinPortfoliosRootPane)) {
						
						// event lies in popup!
						// now we have to find the actual component.
						
						//todo!
						Point awtEventPointinPopup = new Point(
								awtEventPointinPortfoliosRootPane.x-popup.getRDesiredPopupBoundsInFrameOfRefOfSourceRootPane().x,
								awtEventPointinPortfoliosRootPane.y-popup.getRDesiredPopupBoundsInFrameOfRefOfSourceRootPane().y
							);
						awtEventSource = SwingUtilities.getDeepestComponentAt(popup.contents,awtEventPointinPopup.x, awtEventPointinPopup.y);
						break;
						
					} else {
						// point was not within this popup
					}
				}
				
				if(awtEventSource==null) {
					// it doesn't lie within a popup so dispatch it to the deepest component in this portfolio's rootpane
					awtEventSource = SwingUtilities.getDeepestComponentAt(frame.getRootPane(),awtEventPointinPortfoliosRootPane.x, awtEventPointinPortfoliosRootPane.y);
				}
				
				if(awtEventSource==null) { 
					// there isn't a component there at all
					return new InputEvent[] {}; 
				} 
				
							
				// convert the coordinates to be relative to the source
				awtPointRelToEventSource = SwingUtilities.convertPoint(frame.getRootPane(),awtEventPointinPortfoliosRootPane,awtEventSource);
			}
			
			if(e.eventType == e.EVENT_PID_PRESS) {
				personToKbf.put(e.person,awtEventSource);
			}
			
			
			if(awtEventId == MouseEvent.MOUSE_RELEASED) {
				// also need to generate mouse clicked to the same component
				// otherwise we can generate a release on a popup and a click  
				// after we've closed the popup...
				
				return new InputEvent[] {
						new SwingFramePortfolioMouseEvent(
							e,
							awtEventSource,
							awtEventId,
							awtEventWhen,
							awtEventModifiers,
							awtPointRelToEventSource.x,
							awtPointRelToEventSource.y,
							awtEventClickCount,
							false,
							awtEventButton
						),
						new SwingFramePortfolioMouseEvent(
							e,
							awtEventSource,
							MouseEvent.MOUSE_CLICKED,
							awtEventWhen,
							awtEventModifiers,
							awtPointRelToEventSource.x,
							awtPointRelToEventSource.y,
							awtEventClickCount,
							false,
							awtEventButton
						)
				};
			} else {
			
				return new InputEvent[] { 
					new SwingFramePortfolioMouseEvent(
						e,
						awtEventSource,
						awtEventId,
						awtEventWhen,
						awtEventModifiers,
						awtPointRelToEventSource.x,
						awtPointRelToEventSource.y,
						awtEventClickCount,
						false,
						awtEventButton
						) };
			}
		}
	}
	
	
	
	
	static void setUpOuterFrameIfNotAlready() {
		if(outerFrame == null) {
			AWTErrorHandler.setHandler();
			outerFrame = new OuterFrame();
			rpm = new VerboseRepaintManager();
	    	RepaintManager.setCurrentManager(rpm);
	    	mypf = new MyPopupFactory(PopupFactory.getSharedInstance());
	    	PopupFactory.setSharedInstance( mypf );
	    	if(System.getProperty("javax.swing.adjustPopupLocationToFit")==null || !System.getProperty("javax.swing.adjustPopupLocationToFit").equals("false")) {
	    		throw new IllegalStateException("You need to run with -Djavax.swing.adjustPopupLocationToFit=false");
	    	}
	    	
		} else {
			// no need to create it!
		}
	}
	
	
	static boolean adjustCoordsToFrameOfRefOfRootPane(Rectangle newRect, JComponent c) {
		JRootPane root = c.getRootPane();
		if(c!=root) {
			do {
				newRect.x+=c.getX();
				newRect.y+=c.getY();
				if(! (c.getParent() instanceof JComponent) ) {
					return false;
				}
				c = (JComponent) c.getParent();				
			} while (c!=root);
		} else {
			// c is rootpane so no need to worry
		}
		return true;
	}
	
	static Point convertPointUpTreeWithoutUsingScreenCoords(Point p, Component frameOfRefOfP, Component desiredFrameOfRef) {
		// desiredFrameOfRef must be an anscestor of frameOfRefOfP
		if(desiredFrameOfRef==frameOfRefOfP) {
			return p;
		} else if(frameOfRefOfP.getParent()==null) {
			throw new IllegalArgumentException("Components are not anscestors.");
		} else {
			return convertPointUpTreeWithoutUsingScreenCoords(
				new Point(p.x+frameOfRefOfP.getX(), p.y+frameOfRefOfP.getY()),
				frameOfRefOfP.getParent(),
				desiredFrameOfRef
			);
		}
	}
	
	static Point convertPointDownTreeWithoutUsingScreenCoords(Point p, Component frameOfRefOfP, Component desiredFrameOfRef) {
		// frameOfRefOfP  must be an anscestor of desiredFrameOfRef
		Point q = convertPointUpTreeWithoutUsingScreenCoords( new Point(0,0), desiredFrameOfRef, frameOfRefOfP);
		return new Point(p.x-q.x, p.y-q.y);
	}

    @Override
    protected void customProcessAboutToBeDestroyed() {
        this.frame.dispose();
        this.swingFramePortfoliosThreadSafe.remove(this);
    }
	
}
