/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios.commonbehaviours;

import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import t3.hrd.state.JOGLHelper;
import t3.hrd.state.ScaRotTraTransformImmutableUniformScale;
import t3.portfolios.PointInputDevice;
import t3.portfolios.PointInputDeviceTypeAndButton;
import t3.portfolios.PointInputDeviceTypeAndButtonSet;
import t3.portfolios.Portfolio;
import t3.portfolios.PortfolioCommonBehaviour;
import t3.portfolios.PortfolioEvent;
import Jama.Matrix;

/**
 * This class allows portfolios to be dragged using a specified button number and pen type.
 * @author pjt40
 *
 */
public class BasicDragMode  implements PortfolioCommonBehaviour {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios.commonbehaviours");
	private Set<Portfolio> portfoliosBeingDragged = new HashSet<Portfolio>();
    
	public final PointInputDeviceTypeAndButtonSet pidtbs;

	public BasicDragMode(PointInputDeviceTypeAndButton pidTypeAndButton) {
		this( new PointInputDeviceTypeAndButtonSet(pidTypeAndButton) );
	}
	

	public BasicDragMode(PointInputDeviceTypeAndButtonSet pidtbs) {
		this.pidtbs = pidtbs;
	}
	
	public boolean customProcessEventForThisPortfolioNotChildren(Portfolio p, PortfolioEvent e, boolean bubbled) {
		if (
				e.pointInputDevice != null
				&& e.positionAndButtonsKnown
				&& this.pidtbs.contains(
					new PointInputDeviceTypeAndButton(e.pointInputDevice.pointInputDeviceType, e.pidButton)
				)
			) {
				if(e.eventType==e.EVENT_PID_PRESS && !this.portfoliosBeingDragged.contains(p)) {
					p.portfolioServer.enterFDOPmodeForPointInputDevice(p, e.pointInputDevice, e.pidButton, e.getPORTx(p), e.getPORTy(p));
                    this.portfoliosBeingDragged.add(p);
					return true;
				} else {
					// we still don't want any events for this button to get through to the underlying
					// portfolio.
					return true;
				}
			} else {
				return false;
			}
	}

	public boolean customProcessFDOPevent(Portfolio p, PortfolioEvent e, double PORTxWhenEnteredFDOPmode, double PORTyWhenEnteredFDOPmode) {
		if(this.pidtbs.contains(
			new PointInputDeviceTypeAndButton(e.pointInputDevice.pointInputDeviceType, e.pidButton)
		)) {
			// firstly let's get a tranform with current tx and ty and our new desired theta and scale 
            ScaRotTraTransformImmutableUniformScale idealNewDESKtoPORT = new ScaRotTraTransformImmutableUniformScale(p.getmDESKtoPORTReadOnly());
			
			// now where do we end up if we use this on our pen coords
			Matrix PORTpoint = idealNewDESKtoPORT.getMatrix2dHomogReadOnly().times( 
					JOGLHelper.getMFromD(e.DESKx, e.DESKy,1.0)
				);
			
			
			// now adjust based on this
			// we can do the translation comparison in portfolio space because 
			// the translation is applied last, after the rotn and scale.
			double tx = ( PORTpoint.get(0,0)/PORTpoint.get(2,0)-PORTxWhenEnteredFDOPmode );
			double ty = ( PORTpoint.get(1,0)/PORTpoint.get(2,0)-PORTyWhenEnteredFDOPmode );
			
			ScaRotTraTransformImmutableUniformScale s = new ScaRotTraTransformImmutableUniformScale(
				idealNewDESKtoPORT.getScale(),
				idealNewDESKtoPORT.getThetaClockwise(),
				idealNewDESKtoPORT.getTx() - tx,
				idealNewDESKtoPORT.getTy() - ty
				);
			p.setDESKtoPORT(s.getMatrix2dHomogReadOnly());
			return true;
		} else {
			return false;
		}
	}

	public boolean customProcessEndOfFDOPmode(Portfolio p, PointInputDevice pid, int button) {
		if(this.pidtbs.contains(
			new PointInputDeviceTypeAndButton(pid.pointInputDeviceType, button)
		)) {
            this.portfoliosBeingDragged.remove(p);
			return true;
		} else {
			return false;
		}
	}
		
	
}


