/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;


import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.net.ServerSocket;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import t3.hrd.input.KeyboardInput;
import t3.hrd.input.PointInputDeviceState;
import t3.hrd.input.ShapeInputDeviceState;
import t3.hrd.state.Cursor;
import t3.hrd.state.OrderedElement;
import t3.hrd.state.StateManager;
import t3.hrd.state.Tile;
import t3.remotehrd.server.RemoteHRDServer;
import t3.remotehrd.server.RemoteHRDServerCallBacks;

/*
 * Locking notes
 *  There is one lock - on the statemanager
 *  that controls modification access on all portfolios and this manager
 *  It is the caller's responsibility to synchronize on it, not our responsibilty
 *  
 *  Except for internal callbacks for which we do synchronization ourselves
 */


/**
 * Represents a Portfolio server, which uses a RemoteHRDServer and provides facilities such
 * as portfolios and event handling.
 * <p>
 * Notes on the event model:
 * <p>
 * Keyboard events are dispatched to whichever portfolio was last pressed on by any PID 
 * of the person owning the keyboard.
 * <p>
 * Point input device (PID) events are generated when PID's state changes. Normally events
 * are sent to the portfolio whose tile is underneath the PID, and may bubble up the portfolio
 * hierarchy from there.
 * <p>
 * However, it is also possible to put the PID temporarily into FDOP (Follow
 * Drag Outside Portfolio mode) by specifying a single portfolio to whom all events from that
 * PID will be directed until a specified button is released.
 * <p>
 * The PID event system is further complicated because each portfolio has a PortfolioCommonBehaviour class
 * which can filter events out of the stream to provide common functionality such as dragging portfolios. 
 * <p>
 * The PID event system proceeds as follows:
 * <ul>
 * <li>If the PID is not in FDOP mode:
 * 		<ul>
 * 		<li>If we know the coordinates and button state of the PID
 * 			<ul><li>We find the portfolio P whose tile is immediately underneath the pid.
 * 			<li>We say the PID is in P and all P's anscestors.
 * 			<lI>We dispatch enter events to any portfolios that the PID is now in but was not previously in.
 * 			<li>We dispatch enter events to any portfolios that the PID was previously in but is now in.
 * 			<li>We dispatch a move event to P if the PID's coordinates have changed.
 * 			<li>If any button state has changed then we dispatch an appropriate event (press, release or click) to P. 
 * 				If P does not handle the event (ie its handler returns false) then the event is bubbled to P's parent.
 * 				If P's parent does not handle the event then it is bubbled to P's grandparent and so on.
 * 			</ul>
 * 		<li>Else we dispatch exit events to any portfolio that the PID was previously in.
 * 		</ul>
 *	<li>Else
 *		<ul><li>If we know the coordinates and button state of the PID and the PID FDOP mode button is still down:
 *			<ul><li>We dispatch an FDOP mode event to the PID's FDOP mode portfolio.</ul>
 * 		<li>Else we exit FDOP mode and dispatch such a message to the PID's FDOP mode portfolio.
 * </ul>
 * <p>
 * When we dispatch an event to a portfolio, we first call a method on the portfolio's PortfolioCommonBehaviour object.
 * If this returns true then the PortfolioCommonBehaviour object is deemed to have handled the event. Only if it returns false
 * do we call a method on the Portfolio object itself.
 * <p>
 * Threading notes: 
 * <p>
 * When you write code in portfolio callback methods, you don't need to worry about
 * multithreading and locking. T3 ensures that the thread running your method will have
 * the correct locks and that at the end of your method, any updates will be 
 * passed through to the clients. 
 * <p>
 * If, however, you wish to call T3 methods outside of portfolio callback methods, 
 * i.e. from your own thread, then you must call performActionAsynchronouslyByCurrentThread, passing your code as a runnable
 * object. This ensures that when your code runs, it runs in the currently executing thread and has the requisite locks,
 * and finishes by instructing T3 to pass changes through to the clients. Furthermore, if you create a thread that exists for
 * a period of time, such as an animator, then you must ensure that this thread runs with low priority and yields regularly. 
 * If not then it can prevent the T3 client threads at the server from responding to client requests for updates. Consequently,
 * the animator thread produces updates at a high rate, but the client threads never run and so these updates never get sent
 * through to the clients. If you wish to use animations, then the AnimatorThread or AnimatorJobsThread will handle all
 * this for you.
 * <p>
 * If you use SwingFramePortfolio then there are further threading constraints. See its
 * javadoc for details. 
 * <p>
 * 
 * @author pjt40
 *
 */
public class PortfolioServer {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios");
	
	
	private final RemoteServerCallBacksForPortfServer ourRemoteServerCallbacksForPortfServer;
	public Map<PointInputDevice, PointInputDeviceData> pidToPidDataReadOnly;
    public final List<Portfolio> allPortfoliosInOrderBackFirstReadOnly;
	private Map<Person, Portfolio > personToKeyboardFocus;
	private Map<Integer, Cursor > clientIdToShapeCursor;
	private final RemoteHRDServer remoteHRDserver;
	
	final IdentifyUnwarpedRects javaHereld; // lock on it before you use it!
	Map<Tile, Portfolio> tileToPortfolio;
	
	public final RootPortfolio rootPortfolio;
	
	
	final StateManager stateManager;
	
	
	/**
	 * Creates a Portfolio server, including a root portfolio and a RemoteHRD server, 
	 * which starts listening on the specified socket. AlwaysServerPush relates to the RemoteHRD server.
	 *  
	 * @param globalPortfolioDragMode	Specifies how Portfolios are dragged around the surface.	
	 * @param s					Socket on which RemoteHRDServer should listen for clients.
	 * @param i					Socket on which RemoteHRDServer should listen for inputsources.
	 * @param alwaysServerPush	See RemoteHRDServer.
	 */
	public PortfolioServer(ServerSocket s, ServerSocket i, boolean alwaysServerPush) {
		// starts listening.
		this.ourRemoteServerCallbacksForPortfServer = new RemoteServerCallBacksForPortfServer();
		this.remoteHRDserver = new RemoteHRDServer(s, i, this.ourRemoteServerCallbacksForPortfServer, alwaysServerPush);
		this.stateManager = this.remoteHRDserver.stateManager;
		this.pidToPidDataReadOnly = new HashMap<PointInputDevice, PointInputDeviceData>();
        this.clientIdToShapeCursor = new HashMap<Integer, Cursor>();
		this.tileToPortfolio = new HashMap<Tile, Portfolio>();
		this.personToKeyboardFocus = new HashMap<Person, Portfolio>();
		this.javaHereld = new IdentifyUnwarpedRects(1000*1000);
		this.rootPortfolio = new RootPortfolio(this);
        this.allPortfoliosInOrderBackFirstReadOnly = new LinkedList<Portfolio>();
	}
    
    
    public void sendLogMessageToAllClients(String message) {
        this.stateManager.opLogMessage(message);
    }
    
    
    public void performActionAsynchronouslyByCurrentThread(Runnable r) {
        synchronized(this.stateManager) {
            r.run();
            this.endOfUpdateFrame();
        }
    }
	
		
	/**
	 * Ensures that any changes to the state can be sent on to the clients. This is called
	 * automatically after any portfolio has processed an input event. However, if you change the
	 * state (e.g. change a portfolio location)  when you are not processing an input event
	 * then you will need to call this yourself. This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 */
	private void endOfUpdateFrame() {
		// no need to synchr because this.remoteHRDserver is threadsafe
		this.remoteHRDserver.endOfUpdateFrame();		
	}
	
    
    
    public void notifyPortfolioWhenPidPressedOutside(Portfolio p, PointInputDevice pointInputDevice) {
        synchronized(this.stateManager) {
            PointInputDeviceData penData = pidToPidDataReadOnly.get(pointInputDevice);
            assert penData !=null;  
            penData.portfoliosToNotifyWhenPressedOutside.add(p);
        }
    }    
    
    
    
	/**
	 * Called by a portfolio when it wishes to enter FDOP mode. This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 * @param p
	 * @param pointInputDevice
	 * @param button
	 * @param PORTxWhenFDOPmodeEntered
	 * @param PORTyWhenFDOPmodeEntered
	 */
	public void enterFDOPmodeForPointInputDevice(Portfolio p, PointInputDevice pointInputDevice, int button, double PORTxWhenFDOPmodeEntered, double PORTyWhenFDOPmodeEntered) {
		synchronized(this.stateManager) {
			PointInputDeviceData penData = pidToPidDataReadOnly.get(pointInputDevice);
			assert penData !=null;	
			if(penData.fdopModePortfolio !=null) {
				throw new IllegalStateException("Pen is already in FDOP mode: "+pointInputDevice);
			}
			penData.fdopModePortfolio = p;
			penData.fdopModeButton = button;
			penData.DESKxWhenLastFDOPeventCreated = penData.DESKxWhenLastEventCreated;
			penData.DESKyWhenLastFDOPeventCreated = penData.DESKyWhenLastEventCreated;
			penData.buttonsWhenLastFDOPeventCreated = penData.buttonsWhenLastEventCreated;
            penData.extraWhenLastFDOPeventCreated = penData.extraWhenLastEventCreated;
			penData.PORTxWhenFDOPmodeEntered = PORTxWhenFDOPmodeEntered;
			penData.PORTyWhenFDOPmodeEntered = PORTyWhenFDOPmodeEntered;
		}
	}
	
	/**
	 * Called by a portfolio when it wishes to leave FDOP mode. This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 * @param pid
	 */
	public void leaveFDOPmodeForPointInputDevice(PointInputDevice pid) {
		synchronized(this.stateManager) {
			PointInputDeviceData penData = pidToPidDataReadOnly.get(pid);
			assert penData !=null;		
			if(penData.fdopModePortfolio ==null) {
				throw new IllegalStateException("Pen was not in FDOP mode: "+pid);
			}
			if(
				!penData.fdopModePortfolio.commonBehaviour.customProcessEndOfFDOPmode(penData.fdopModePortfolio, pid, penData.fdopModeButton)
			) {
				penData.fdopModePortfolio.customProcessEndOfFDOPmode(pid, penData.fdopModeButton);
			} else {
				// event was processed by the common behaviour so no need to pass it to the portfolio
			}
			penData.fdopModePortfolio = null;
			
			// now reprocess normally. we're not in fdop mode!
			this.ourRemoteServerCallbacksForPortfServer.remoteServerCallBack_pointInputDeviceStateChanged(
					new PointInputDeviceState(
						pid.person.clientId,
						pid.pointInputDeviceType,
						pid.person.personId,
						true,
						penData.DESKxWhenLastFDOPeventCreated,
						penData.DESKyWhenLastFDOPeventCreated,
						penData.buttonsWhenLastFDOPeventCreated,
                        penData.extraWhenLastFDOPeventCreated
					)
			);
		}
	}
	
	
	/**
	 * Sets the portfolio to which keyboard input events from a specified person
	 * will be sent. Any portfolio that processes a pid press event automatically
	 * becomes the keyboard focus for the pid's person. 
	 * This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 * @param p
	 * @param person
	 */
	public void setKeyboardFocusForPerson(Portfolio p, Person person) {
		synchronized(this.stateManager) {
			this.personToKeyboardFocus.put(person,p);
		}
	}
	
	/**
	 * Returns the portfolio that currently receives keyboard input events from
	 * a specified person's keyboard. This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 * @param person
	 * @return
	 */
	public Portfolio getKeyboardFocusForPerson(Person person) {
		synchronized(this.stateManager) {
			return this.personToKeyboardFocus.get(person);
		}
	}
	
	/**
	 * Returns the PointInputDeviceData for a specified PointInputDevice. The 
	 * PointInputDeviceData contains housekeeping data and you should not
	 * normally need to use this method. This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 * @param pid
	 * @return
	 */
	public PointInputDeviceData getPidDataForPid(PointInputDevice pid) {
		synchronized(this.stateManager) {
			return this.pidToPidDataReadOnly.get(pid);
		}
	}
	

	
	
	

	
	
	private boolean dispatchEventToPortfolioAndBubble(Portfolio p, PortfolioEvent e, boolean alreadyBubbled ) {
		assert (e.eventType==e.EVENT_PID_CLICK || e.eventType==e.EVENT_PID_PRESS  || e.eventType==e.EVENT_PID_PRESSOUTSIDE || e.eventType==e.EVENT_PID_RELEASE );
		return 
			dispatchEventToPortfolioWithoutBubble(p, e, alreadyBubbled ) 
			|| (!p.isRoot() && dispatchEventToPortfolioAndBubble(p.getParent(), e, true));
	}
	
	private boolean dispatchEventToPortfolioWithoutBubble(Portfolio p, PortfolioEvent e, boolean alreadyBubbled ) {
		// fdop mode events don't go this way. all other events do.
		assert (e.eventType!=e.EVENT_PID_FDOPMODE);
		boolean processed = 
			p.commonBehaviour.customProcessEventForThisPortfolioNotChildren(p,e,alreadyBubbled)
			|| p.customProcessEventForThisPortfolioNotChildren(e, alreadyBubbled);
		if(processed && e.eventType==e.EVENT_PID_PRESS) {
			this.setKeyboardFocusForPerson(p,e.person);
		} else {
			// no need to set keyboard focus
		}
		return processed;
	}
	
	
	
	/**
	 * Call this if a fatal error occurs and you wish to kill the server.
	 * This method is thread safe; it can be called from
	 * any thread without having to synchronize.
	 * @param e
	 */
	public void fatalError(Throwable e) {
		synchronized(this.stateManager) {
			this.stateManager.fatalError(e);
		}
	}
	
	private static int hasButtonStateChanged(boolean oldButtonsKnown, int oldButtons, int newButtons, int buttonNumber) {
		// 0: no change
		// 1: button released
		// 2: button pressed
		boolean oldButtonDown = oldButtonsKnown && isButtonDown(oldButtons, buttonNumber);
		boolean newButtonDown = isButtonDown(newButtons, buttonNumber);
		return 
			oldButtonDown==newButtonDown ? 0 :
			(oldButtonDown ? 1 : 2);
		
	}
	
	/**
	 * Returns true iff bit buttonNumber is set in buttons.
	 * @param buttons
	 * @param buttonNumber
	 * @return
	 */
	public static boolean isButtonDown(int buttons, int buttonNumber) {
		return (buttons & (1<<buttonNumber) ) != 0;
	}
	
	void childOrderChanged() {
		synchronized(this.stateManager) {
			List<OrderedElement> l = new LinkedList<OrderedElement>();
            allPortfoliosInOrderBackFirstReadOnly.clear();
			this.rootPortfolio.addOrderedElsAndPortfoliosToListsInOrderForThisAndChildren(l,allPortfoliosInOrderBackFirstReadOnly);
			try {
				this.stateManager.opReorderTilesAndLinks(l);	
			} catch(Throwable e) {
				this.fatalError(e);
			}
		}
	}

	
	
	private class RemoteServerCallBacksForPortfServer implements RemoteHRDServerCallBacks {
		
		
		public int remoteServerCallBack_decideShouldCompressInitial(BufferedImage b) {
			return 0;
		}
		
		public void remoteServerCallBack_receivedOtherMessageFromClientOrInputSource(Object msg) {
			assert false;
		}
		
		public final void remoteServerCallBack_clientJoined(int clientId) {
		}
		
		public final void remoteServerCallBack_clientLeft(int clientId) {
			/*
			 * In days gone by we would have removed the pens and keyboard focus associated
			 * with the client.
			 */
		}
	
	
		public void remoteServerCallBack_repaintUnbackedTile(Tile tile, Rectangle r, BufferedImage update, Graphics2D g) {
			PortfolioServer.this.tileToPortfolio.get(tile).customRepaintTileForThisPortfolioNotChildren(r, update, g);
		}	
		
		
		public void remoteServerCallBack_keyboardInput(KeyboardInput keyboardInputMsg) {
			synchronized(PortfolioServer.this.stateManager) {
				Person person = new Person(keyboardInputMsg.clientId, keyboardInputMsg.personId);				
				Portfolio kbFocus = PortfolioServer.this.getKeyboardFocusForPerson(person);
				if(kbFocus==null) { kbFocus = PortfolioServer.this.rootPortfolio; }
				int eventType = 
					keyboardInputMsg.messageType==KeyboardInput.MESSAGE_TYPE_PRESSED 
						? PortfolioEvent.EVENT_KEYBOARD_PRESSED : (
							keyboardInputMsg.messageType==KeyboardInput.MESSAGE_TYPE_RELEASED 
							? PortfolioEvent.EVENT_KEYBOARD_RELEASED : PortfolioEvent.EVENT_KEYBOARD_TYPED );
				dispatchEventToPortfolioWithoutBubble(
						kbFocus,
						new PortfolioEvent(
							0,
							0,
							person,
							null,
							0.0,
							0.0,
							0.0,
							0.0,
                            null,
							eventType,
							false,
							false,
							keyboardInputMsg.awtKeyCode,
							keyboardInputMsg.awtKeyChar,
							keyboardInputMsg.awtKeyModifiers
						),
						false
					);
				PortfolioServer.this.endOfUpdateFrame();
			}
		}
		
		public void remoteServerCallBack_shapeInputDeviceStateChanged(ShapeInputDeviceState sids) {
			synchronized(PortfolioServer.this.stateManager) {
				if(!PortfolioServer.this.clientIdToShapeCursor.containsKey(sids.clientId)) {				
					Cursor c = PortfolioServer.this.stateManager.opCreateCursorAutogenerateId();
					c.opSetCursorOptions(new Color(128,0,128,200), Cursor.DISPLAYTYPE_SHAPE);
					PortfolioServer.this.clientIdToShapeCursor.put(sids.clientId, c);
				}
				clientIdToShapeCursor.get(sids.clientId).opCursorShape(sids.polygonsForShapeCursor);
                PortfolioServer.this.endOfUpdateFrame();
			}
		}
		
		public void remoteServerCallBack_pointInputDeviceStateChanged(PointInputDeviceState pointInputMsg) {
			synchronized(PortfolioServer.this.stateManager) {
				PointInputDevice pid = new PointInputDevice(new Person(pointInputMsg.clientId, pointInputMsg.personId), pointInputMsg.pointInputDeviceType);
				
                
				// get Pen from our records				
				if(!PortfolioServer.this.pidToPidDataReadOnly.containsKey(pid)) {				
					PointInputDeviceData newPidData = new PointInputDeviceData(
						PortfolioServer.this.stateManager.opCreateCursorAutogenerateId()
					);
					newPidData.positionAndButtonsKnownWhenLastEventCreated = false;
					newPidData.cursor.opSetCursorOptions(Color.RED, Cursor.DISPLAYTYPE_TRAIL);
					PortfolioServer.this.pidToPidDataReadOnly.put(pid, newPidData);
				}
				PointInputDeviceData pidData = pidToPidDataReadOnly.get(pid);
				if(pidData.fdopModePortfolio!=null) {
										
					/* portfolio is in fdop mode */
					if(
						pointInputMsg.positionAndButtonsKnown
						&& (pidData.fdopModeButton==-1 || isButtonDown(pointInputMsg.buttons,pidData.fdopModeButton) )	
					) {
						if(
							pointInputMsg.DESKx != pidData.DESKxWhenLastFDOPeventCreated 
							|| pointInputMsg.DESKy != pidData.DESKyWhenLastFDOPeventCreated
							|| pointInputMsg.buttons != pidData.buttonsWhenLastFDOPeventCreated
                            || pointInputMsg.extra != pidData.extraWhenLastFDOPeventCreated
						) {
							PortfolioEvent e = new PortfolioEvent(
								pidData.fdopModeButton,
								pointInputMsg.buttons,
								pid.person,
								pid,
								pointInputMsg.DESKx,
								pointInputMsg.DESKy,
								0.0,
								0.0,
                                pointInputMsg.extra,
								PortfolioEvent.EVENT_PID_FDOPMODE,
								pointInputMsg.positionAndButtonsKnown,
								false,
								0,'\000',0
							);
							if(
								!pidData.fdopModePortfolio.commonBehaviour.customProcessFDOPevent(
									pidData.fdopModePortfolio,
									e,
									pidData.PORTxWhenFDOPmodeEntered, 
									pidData.PORTyWhenFDOPmodeEntered
								)
							) {
								pidData.fdopModePortfolio.customProcessFDOPevent(
									e,
									pidData.PORTxWhenFDOPmodeEntered, 
									pidData.PORTyWhenFDOPmodeEntered
								);
							} else {
								// processed by common behaviour so no need to send to portfolio
							}
							pidData.DESKxWhenLastFDOPeventCreated = pointInputMsg.DESKx;
							pidData.DESKyWhenLastFDOPeventCreated = pointInputMsg.DESKy;
							pidData.buttonsWhenLastFDOPeventCreated = pointInputMsg.buttons;
                            pidData.extraWhenLastFDOPeventCreated = pointInputMsg.extra;
							pidData.cursor.opCursorPos(pointInputMsg.DESKx, pointInputMsg.DESKy, true);
							
							
						} else {
							// no change!
						}
					} else {
						// either posn not known or buttons not known or button is not down
						// exit fdop mode
						
						// but first set the data values that need to be used by leaveFDOPmode
						// to the most recent known value - because it's almost like an event 
						// took place and if we don't then we run into problems. 
						pidData.DESKxWhenLastFDOPeventCreated = pointInputMsg.DESKx;
						pidData.DESKyWhenLastFDOPeventCreated = pointInputMsg.DESKy;
						pidData.buttonsWhenLastFDOPeventCreated = pointInputMsg.buttons;
						
						PortfolioServer.this.leaveFDOPmodeForPointInputDevice(pid);
					}
				} else {
					
					
					if(pointInputMsg.positionAndButtonsKnown) {
						if(
							!pidData.positionAndButtonsKnownWhenLastEventCreated
							|| pointInputMsg.DESKx != pidData.DESKxWhenLastEventCreated
							|| pointInputMsg.DESKy != pidData.DESKyWhenLastEventCreated
							|| pointInputMsg.buttons != pidData.buttonsWhenLastEventCreated
                            || pointInputMsg.extra != pidData.extraWhenLastEventCreated
						) {
							Portfolio P = 
								PortfolioServer.this.rootPortfolio.getPortfolioAtCoordinates(pointInputMsg.DESKx, pointInputMsg.DESKy);
							
							Set<Portfolio> thisEventIsInsidePortfolios = 
								P==null ? new HashSet<Portfolio>() : P.getThisAndAnscestorsSet();
														
							// generate enter and exit events!
							// toopt
							for(Portfolio pp: pidData.portfoliosInsideWhenLastEventCreated) {
								if(!thisEventIsInsidePortfolios.contains(pp)) {
									// we dont bubble exit messages
									PortfolioEvent e = new PortfolioEvent(
											0,
											pointInputMsg.buttons,
											pid.person,
											pid,
											pointInputMsg.DESKx,
											pointInputMsg.DESKy,
											pidData.DESKxWhenLastEventCreated,
											pidData.DESKyWhenLastEventCreated,
                                            pointInputMsg.extra,
											PortfolioEvent.EVENT_PID_EXIT,
											pointInputMsg.positionAndButtonsKnown,
											pidData.positionAndButtonsKnownWhenLastEventCreated,
											0,'\000',0
										);
									dispatchEventToPortfolioWithoutBubble(pp,e,false);
								}
							}
							for(Portfolio pp: thisEventIsInsidePortfolios) {
								if(!pidData.portfoliosInsideWhenLastEventCreated.contains(pp)) {
									// we don't bubble enter messages
									PortfolioEvent e = new PortfolioEvent(
											0,
											pointInputMsg.buttons,
											pid.person,
											pid,
											pointInputMsg.DESKx,
											pointInputMsg.DESKy,
											pidData.DESKxWhenLastEventCreated,
											pidData.DESKyWhenLastEventCreated,
                                            pointInputMsg.extra,
											PortfolioEvent.EVENT_PID_ENTER,
											pointInputMsg.positionAndButtonsKnown,
											pidData.positionAndButtonsKnownWhenLastEventCreated,
											0,'\000',0
										);
									dispatchEventToPortfolioWithoutBubble(pp,e,false);
								}
							}
							
							if(
								!pidData.positionAndButtonsKnownWhenLastEventCreated
								|| pointInputMsg.DESKx != pidData.DESKxWhenLastEventCreated
								|| pointInputMsg.DESKy != pidData.DESKyWhenLastEventCreated	
                                || pointInputMsg.extra != pidData.extraWhenLastEventCreated
							) {
								// generate move event
								if(P!=null) {
									// we don't bubble move messages
									PortfolioEvent e = new PortfolioEvent(
												0,
												pointInputMsg.buttons,
												pid.person,
												pid,
												pointInputMsg.DESKx,
												pointInputMsg.DESKy,
												pidData.DESKxWhenLastEventCreated,
												pidData.DESKyWhenLastEventCreated,
                                                pointInputMsg.extra,
												PortfolioEvent.EVENT_PID_MOVE,
												pointInputMsg.positionAndButtonsKnown,
												pidData.positionAndButtonsKnownWhenLastEventCreated,
												0,'\000',0
										);
									dispatchEventToPortfolioWithoutBubble(P,e,false);
								} else {
									// no need to send a move message since we're not over a tile.
								}
								
								// if we move with a button down and we're not in fdop mode (ie not fast movements) then
								// surpress cursor.
								if(pointInputMsg.buttons==0) {
									pidData.cursor.opCursorPos(pointInputMsg.DESKx, pointInputMsg.DESKy, true);
								} else {
                                    if(pidData.cursor.getMostRecentPosition()!=null && pidData.cursor.getMostRecentPosition().visible) {
                                        pidData.cursor.opCursorPos(pointInputMsg.DESKx, pointInputMsg.DESKy, false);
                                    } else {
                                        // already invisible
                                    }
								}
							} else {
								// no need to generate move event
							}
							
							if(P!=null) {
								for(int buttonNumber=0; buttonNumber<32; buttonNumber++) {
									int buttonStateChanged = hasButtonStateChanged(
											pidData.positionAndButtonsKnownWhenLastEventCreated,
											pidData.buttonsWhenLastEventCreated, 
											pointInputMsg.buttons, 
											buttonNumber);
									if(buttonStateChanged==1) {
										// button released
										
										dispatchEventToPortfolioAndBubble(
											P,
											new PortfolioEvent(
												buttonNumber,
												pointInputMsg.buttons,
												pid.person,
												pid,
												pointInputMsg.DESKx,
												pointInputMsg.DESKy,
												0,
												0,
                                                pointInputMsg.extra,
												PortfolioEvent.EVENT_PID_RELEASE,
												pointInputMsg.positionAndButtonsKnown,
												false,
												0,'\000',0
											),
											false
										);
										
										// todo we shouldn't generate click if we've moved whlie button down
										dispatchEventToPortfolioAndBubble(
											P,
											new PortfolioEvent(
												buttonNumber,
												pointInputMsg.buttons,
												pid.person,
												pid,
												pointInputMsg.DESKx,
												pointInputMsg.DESKy,
												0,
												0,
                                                pointInputMsg.extra,
												PortfolioEvent.EVENT_PID_CLICK,
												pointInputMsg.positionAndButtonsKnown,
												false,
												0,'\000',0
											),
											false
										);
										
									} else if(buttonStateChanged==2) {
                                        
                                        // notify press outside portfolios
                                        Iterator<Portfolio> portsToNotify = pidData.portfoliosToNotifyWhenPressedOutside.iterator();
                                        while(portsToNotify.hasNext()) {
                                            Portfolio portToNotify = portsToNotify.next();
                                            if(P==portToNotify) {
                                                // do nothing
                                            } else {
                                                portsToNotify.remove();
                                                dispatchEventToPortfolioAndBubble(
                                                        portToNotify,
                                                        new PortfolioEvent(
                                                            buttonNumber,
                                                            pointInputMsg.buttons,
                                                            pid.person,
                                                            pid,
                                                            pointInputMsg.DESKx,
                                                            pointInputMsg.DESKy,
                                                            0,
                                                            0,
                                                            pointInputMsg.extra,
                                                            PortfolioEvent.EVENT_PID_PRESSOUTSIDE,
                                                            pointInputMsg.positionAndButtonsKnown,
                                                            false,
                                                            0,'\000',0
                                                        ),
                                                        false
                                                    );
                                            }
                                        }
                                        
										dispatchEventToPortfolioAndBubble(
											P,
											new PortfolioEvent(
												buttonNumber,
												pointInputMsg.buttons,
												pid.person,
												pid,
												pointInputMsg.DESKx,
												pointInputMsg.DESKy,
												0,
												0,
                                                pointInputMsg.extra,
												PortfolioEvent.EVENT_PID_PRESS,
												pointInputMsg.positionAndButtonsKnown,
												false,
												0,'\000',0
											),
											false
										);
									} else {
										// do nothing, no change in button state for this button
									}
									
								} // end of for loop
							} else {
								// do nothing - no point looking for button changes cos we
								// have no portfolio to send messages to.
							}
							
							pidData.portfoliosInsideWhenLastEventCreated = thisEventIsInsidePortfolios;
						} else {
							// do nothing - no state change in or position, buttons
						}
					} else {
						// position and buttons unknown!
						if(pidData.positionAndButtonsKnownWhenLastEventCreated) {
							
							// we knew our location before but now we don't.							
							
							pidData.cursor.opCursorPos(pointInputMsg.DESKx, pointInputMsg.DESKy, false);			
							
							// 	generate exit events 
							for(Portfolio pp: pidData.portfoliosInsideWhenLastEventCreated) {
									// we don't bubble exit events
									PortfolioEvent e = new PortfolioEvent(
											0,
											pointInputMsg.buttons,
											pid.person,
											pid,
											pointInputMsg.DESKx,
											pointInputMsg.DESKy,
											pidData.DESKxWhenLastEventCreated,
											pidData.DESKyWhenLastEventCreated,
                                            pointInputMsg.extra,
											PortfolioEvent.EVENT_PID_EXIT,
											pointInputMsg.positionAndButtonsKnown,
											pidData.positionAndButtonsKnownWhenLastEventCreated,
											0,'\000',0
										);
									dispatchEventToPortfolioWithoutBubble(pp,e,false);
							}
							
							pidData.portfoliosInsideWhenLastEventCreated = new HashSet<Portfolio>();
							
						} else {
							// we didn't know our location before and we still don't
							// so don't do anything
						}
						
					}
		
					// update our records
					pidData.positionAndButtonsKnownWhenLastEventCreated = pointInputMsg.positionAndButtonsKnown;
					pidData.buttonsWhenLastEventCreated = pointInputMsg.buttons;
					pidData.DESKxWhenLastEventCreated = pointInputMsg.DESKx;
					pidData.DESKyWhenLastEventCreated = pointInputMsg.DESKy;
                    pidData.extraWhenLastEventCreated = pointInputMsg.extra;
		
				}
			
				
				PortfolioServer.this.endOfUpdateFrame();
			}
		}
		
	}
    

}
