/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.util;


import java.awt.DisplayMode;
import java.awt.Frame;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.util.LinkedList;

import javax.media.opengl.GL;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLCanvas;
import javax.media.opengl.GLEventListener;
import javax.media.opengl.glu.GLU;

import t3.hrd.renderer.Projector;
import t3.hrd.renderer.ProjectorConfig;
import t3.hrd.renderer.ProjectorTransforms;
import t3.hrd.renderer.Projector.ProjectorOpeningException;
import t3.hrd.state.JOGLHelper;

import com.sun.opengl.util.GLUT;


/**
 * 
 * Diagnostic utility to identify and test graphics cards.
 * 
 * @author pjt40
 *
 */
public class RunDisplayDiag {

	
	private static String display;
	private static GLU glu=new GLU();
	
	public static void main(String[] args) {

		try {	
			
			System.out.println("==================");
			System.out.println("Diagnostic utility");
			System.out.println("==================");
			
			GraphicsDevice[] ds = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices();
			int i = -1;
			
			LinkedList<Frame> frames = new LinkedList<Frame>();
			LinkedList<GLCanvas> canvases = new LinkedList<GLCanvas>();
			
			for(GraphicsDevice d: ds) {
				i++;

				display = "GraphicsDevice Index: "+i+"; and IDString: "+d.getIDstring();
				
				
				System.out.println("");
				System.out.println("");
				System.out.println("========================");
				System.out.println("");
				System.out.println("");
				System.out.println(display);
				System.out.println("Fullscreen support?: "+d.isFullScreenSupported());
				System.out.println("Default device?: "+(d==GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()));
				System.out.println("Display Mode Change support not in full screen?: "+d.isDisplayChangeSupported() );
				System.out.println("Accelerated Memory: "+d.getAvailableAcceleratedMemory()+" Bytes"  );
				GraphicsConfiguration gc = d.getDefaultConfiguration();
				System.out.println("Current graphics configuration: ("
						+gc.getBounds().x+" "
						+gc.getBounds().y+" "
						+gc.getBounds().width+" "
						+gc.getBounds().height+" "
						+") "
						+gc.getColorModel().toString()
					);
				DisplayMode dm = d.getDisplayMode();
				System.out.println("Current display mode: "+dm.getWidth()+"px * "+dm.getHeight()+"px * "+dm.getBitDepth()+"bits at "+dm.getRefreshRate()+"Hz" );
				
				ProjectorConfig w = new ProjectorConfig();
				w.graphicsDeviceIndex = i;
				w.window_fullScreenExclusive = false;
				
				System.out.println("Attempting to open OpenGL window");
				try {
					Frame f = Projector.createOGLWindow(w, new MyGLRenderer(), null);
					GLCanvas canvas = (GLCanvas) f.getComponent(0);
					canvas.display();
                    canvas.swapBuffers();
					Thread.currentThread().sleep(1000);
					canvas.display();
                    canvas.swapBuffers();
					Thread.currentThread().sleep(1000);
					
					frames.add(f);
					canvases.add(canvas);
					
				} catch(ProjectorOpeningException e) {
					System.out.println("Error opening window. Error follows:");
					e.printStackTrace(System.out);
					continue;
				}
					
			}

			for(GLCanvas c: canvases) {
				c.display();
			}
			Thread.currentThread().sleep(1000);
			for(GLCanvas c: canvases) {
				c.display();
			}
			Thread.currentThread().sleep(1000);
			for(Frame f: frames) {
				f.dispose();
			}
			
			System.out.println("");
			System.out.println("");
			System.out.println("Done. Exiting...");
			
		} catch(Throwable e) {
			//includes exceptions and assertionerrors
			e.printStackTrace();
			System.exit(1);
		}
	}	
	
	
	private static class MyGLRenderer implements GLEventListener {
		
		
		public void init(GLAutoDrawable drawable) {
			// called by our canvas
			GL gl = drawable.getGL();
			gl.setSwapInterval(1);
			gl.glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
			gl.glShadeModel(gl.GL_FLAT);
			JOGLHelper.oglThrowExceptionIfError(gl, glu);
		}
			
		public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height) {
			// called by our canvas
			GL gl = drawable.getGL();
			System.out.println("OpenGL vendor: "+gl.glGetString(GL.GL_VENDOR));
			System.out.println("OpenGL version: "+gl.glGetString(GL.GL_VERSION));
			System.out.println("OpenGL renderer: "+gl.glGetString(GL.GL_RENDERER));
			System.out.println("OpenGL extensions: "+gl.glGetString(GL.GL_EXTENSIONS));
			// no need to call glViewport - it's done automatically			
			gl.glMatrixMode( GL.GL_PROJECTION ); 
			gl.glLoadIdentity();	
			gl.glFrustum(
					ProjectorTransforms.openGlView_left, 
					ProjectorTransforms.openGlView_left+ProjectorTransforms.openGlView_width, 
					ProjectorTransforms.openGlView_bottom, 
					ProjectorTransforms.openGlView_bottom+ProjectorTransforms.openGlView_height, 
					ProjectorTransforms.openGlView_near, 
					ProjectorTransforms.openGlView_far
					);
			gl.glMatrixMode(gl.GL_MODELVIEW);
			gl.glLoadIdentity(); 
			JOGLHelper.oglThrowExceptionIfError(gl, glu);
		}
			
			
		public void display(GLAutoDrawable drawable) {
			// called by our canvas
			// we must now redraw because if we don't then
			// the display will not necessarily be stable because of back buffering
		
			GL gl = drawable.getGL();				

			gl.glClear( GL.GL_COLOR_BUFFER_BIT );
			gl.glColor3f(1.0f,1.0f,1.0f);
			gl.glRasterPos3d(-0.1,0.0,ProjectorTransforms.openGlView_z); // set position
			GLUT g = new GLUT();
			g.glutBitmapString(GLUT.BITMAP_HELVETICA_18, display);
			gl.glRasterPos3d(-0.1,-0.1,ProjectorTransforms.openGlView_z); // set position
			g.glutBitmapString(GLUT.BITMAP_HELVETICA_12, "OpenGL vendor: "+gl.glGetString(GL.GL_VENDOR));
			gl.glRasterPos3d(-0.1,-0.2,ProjectorTransforms.openGlView_z); // set position
			g.glutBitmapString(GLUT.BITMAP_HELVETICA_12, "OpenGL renderer: "+gl.glGetString(GL.GL_RENDERER));
			
			JOGLHelper.oglThrowExceptionIfError(gl, glu);
			
		}
		
		public void displayChanged(GLAutoDrawable drawable,
				boolean modeChanged,
				boolean deviceChanged)
		{
		}
		
	}	
	
}
