/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;


import java.awt.Color;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;



/**
 * 
 * A StateManager allows you to create, destroy and reorder Links, Tiles and Cursors.
 * <p>
 * Each Link, Tile and Cursor has an id. You can create them with specific ids, or allow
 * the system to auto-generate ids for you. This class has methods to get a Link, Tile
 * or Cursor object given its id, if it exists. You can reorder Links and Tiles by
 * specifying the new ordering either by id or by the actual Link and Tile objects themselves.
 * <p> 
 * Threading notes: this class is not thread-safe. You must use some kind of locking scheme
 * if you use it in a multithreaded environment.
 *   
 * @author pjt40
 *
 */
public class StateManager {

	
	private static final Logger logger = Logger.getLogger("t3.hrd.state");
	
	public static final int MAX_ORDEREDELEMENT_ID = 5000;
	public static final int MAX_CURSOR_ID = 500;
	
	private java.util.List<OrderedElement> tilesAndLinksInOrder;
	public java.util.List<OrderedElement> tilesAndLinksInOrderReadOnly;
	// tilesInOrder: head of the list is the back-most tile.
	
	private ArrayList<OrderedElement> tilesAndLinksById;
	public final boolean usePowersOfTwoForTileImages;
	public final boolean backTiles;
	public final int maxDimensionForTileImages;
	public final int splitDimensionForTileImages;
	protected final StateListener stateListener;
	private int nextOrderedElementId;
	
	private ArrayList<Cursor> cursorsById;
	private Set<Cursor> cursors;
	public final Set<Cursor> cursorsReadOnly;
	private int nextCursorId;
	
	
	
	/**
	 * Creates a StateManager.
	 * @param stateListener		Callbacks that allow customisation/
	 * @param usePowersOfTwo	Iff this is true then tile image dimensions will be changed to be powers of two.
	 * @param backTiles			Iff this is true then tiles will be backed.
	 * @param maxDimensionForTileImages		Maximum width or height of a tile before it will be split, or 0 for no splitting.
	 * @param splitDimensionForTileImages	Width or height of component tiles into which tiles are split, or 0 for no splitting.
	 */
	public StateManager(StateListener stateListener, boolean usePowersOfTwo, boolean backTiles, int maxDimensionForTileImages, int splitDimensionForTileImages) {
		
		this.stateListener = stateListener;
		
		if(!backTiles && (usePowersOfTwo || maxDimensionForTileImages!=0)) {
			throw new IllegalArgumentException("Cannot have non-backed tiles with usePowersOfTwo or non-zero maxDimensionForTileImages");
		}
		if(maxDimensionForTileImages!=0 && maxDimensionForTileImages<splitDimensionForTileImages) {
			throw new IllegalArgumentException("If maxDimensionForTileImages is non-zero then it must be greater equal to splitDimensionForTileImages");
		}
		if(maxDimensionForTileImages!=0 && JOGLHelper.getNearestPowerOfTwoGE(splitDimensionForTileImages)!=splitDimensionForTileImages) {
			throw new IllegalArgumentException("If maxDimensionForTileImages is non-zero then splitDimensionForTileImages must be a power of two");
		}
		
		this.usePowersOfTwoForTileImages = usePowersOfTwo;
		this.backTiles = backTiles;
		this.maxDimensionForTileImages = maxDimensionForTileImages;
		this.splitDimensionForTileImages = splitDimensionForTileImages;
				
		tilesAndLinksInOrder = new LinkedList<OrderedElement>();
		this.tilesAndLinksInOrderReadOnly = Collections.unmodifiableList(tilesAndLinksInOrder);
		tilesAndLinksById = new ArrayList<OrderedElement>(MAX_ORDEREDELEMENT_ID);
		for(int i=0; i<MAX_ORDEREDELEMENT_ID; i++) { tilesAndLinksById.add(null); }
		this.nextOrderedElementId =100;
		
		this.cursorsById = new ArrayList<Cursor>(MAX_CURSOR_ID);
		for(int i=0; i<MAX_CURSOR_ID; i++) { this.cursorsById.add(null); }
		this.cursors = new HashSet<Cursor>();
		this.cursorsReadOnly = Collections.unmodifiableSet(this.cursors);
		this.nextCursorId =100;
	}
	
	public OrderedElement getTileOrLinkByIdForOp(int id) {
		if(id<0 || id>MAX_ORDEREDELEMENT_ID) {
			throw new IllegalArgumentException("Id too big or too small: "+id);
		}
		OrderedElement t = tilesAndLinksById.get(id);
		if(t==null) { 
			throw new IllegalArgumentException("No such tile or link with id "+id);
		}
		return t;
	}
	
	public Tile getTileByIdForOp(int id) {		
		OrderedElement t = getTileOrLinkByIdForOp(id);
		if(!(t instanceof Tile)) { 
			throw new IllegalArgumentException("No such tile with id "+id);
		}
		return (Tile)t;
	}
	
	public Link getLinkByIdForOp(int id) {
		OrderedElement t = getTileOrLinkByIdForOp(id);
		if(!(t instanceof Link)) { 
			throw new IllegalArgumentException("No such linkId with id "+id);
		}
		return (Link)t;
	}
    
    public void opLogMessage(String message) {
        logger.info("OpLogMessage: "+message);
        stateListener.callback_opLogMessage(message);
    }
		
	/**
	 * Creates a new Tile. New Tiles are not visible.
	 * @param elId
	 * @param w
	 * @param h
	 * @param flags
	 * @return
	 */
	public Tile opCreateTile(int elId, int w, int h, int flags) {
		if(elId<0 || elId>MAX_ORDEREDELEMENT_ID) {
			throw new IllegalArgumentException("Tileid too big or too small: "+elId);
		}
		if(tilesAndLinksById.get(elId) != null) {
			throw new IllegalArgumentException("Tile or link already exists with id "+elId);
		}
		Tile t = this.factoryReturnNewTileObject(this.stateListener, elId, w,h, flags, usePowersOfTwoForTileImages);
		tilesAndLinksById.set(elId, t);
		tilesAndLinksInOrder.add(t);
		assert !t.isVisible();
		stateListener.callback_createdTile(t, w, h, flags);
		return t;
	}
	
	public Tile opCreateTileAutogenerateId(int w, int h, int flags) {
		// need to increment nextTileId before we call opCreateTile
		// because it might call this function itself.
		int curTileId = nextOrderedElementId++;
		return this.opCreateTile(curTileId, w, h, flags);
	}
	
	/**
	 * Creates and returns a link with the specified parameters. 
	 * @param elId
	 * @param displayType
	 * @param c
	 * @param tStandardRectToDESKrectA
	 * @param tStandardRectToDESKrectB
	 * @return
	 */
	public Link opCreateLink(int elId, int displayType, Color c, ScaRotTraTransformImmutable tStandardRectToDESKrectA, ScaRotTraTransformImmutable tStandardRectToDESKrectB) {
		if(elId<0 || elId>MAX_ORDEREDELEMENT_ID) {
			throw new IllegalArgumentException("Linkid too big or too small: "+elId);
		}
		if(tilesAndLinksById.get(elId) != null) {
			throw new IllegalArgumentException("Tile or link already exists with id "+elId);
		}
		Link t = this.factoryReturnNewLinkObject(this.stateListener, elId, displayType, c, tStandardRectToDESKrectA, tStandardRectToDESKrectB);
		tilesAndLinksById.set(elId, t);
		tilesAndLinksInOrder.add(t);
		stateListener.callback_createdLink(t, displayType, c, tStandardRectToDESKrectA, tStandardRectToDESKrectB);
		return t;
	}
	
	public Link opCreateLinkAutogenerateId(int displayType, Color c, ScaRotTraTransformImmutable tStandardRectToDESKrectA, ScaRotTraTransformImmutable tStandardRectToDESKrectB) {
		// need to increment nextTileId before we call opCreateTile
		// because it might call this function itself.
		int curId = nextOrderedElementId++;
		return this.opCreateLink(curId, displayType, c, tStandardRectToDESKrectA, tStandardRectToDESKrectB);
	}
	
	public void opDestroyTile(Tile t) {
		if(t.isVisible()) {
			throw new IllegalArgumentException("Cannot destroy a visible tile");
		}
		t.destroyed();
		tilesAndLinksById.set(t.elementId,null);
		tilesAndLinksInOrder.remove(t);
		stateListener.callback_destroyedTile(t);
	}
	
	public void opDestroyLink(Link l) {
		l.destroyed();
		tilesAndLinksById.set(l.elementId,null);
		tilesAndLinksInOrder.remove(l);
		stateListener.callback_destroyedLink(l);
	}
	
	/**
	 * Reorder the tiles and links by specifying an array of elementIds. The first element
	 * of the array is the backmost element. You must not specify the elementIds of any tiles
	 * that were created as a result of the splitting process. Any that you do not specify will be
	 * added at the front.
	 * @param orderById
	 */
	public void opReorderTilesAndLinksById(int[] orderById)  {
		// first el of the array is the back-most tile.
		// you must not specify any divtiles here - only tiles you actually created yourself.
		List<OrderedElement> order = new LinkedList<OrderedElement>();
		for(int id: orderById) {
			order.add(this.getTileOrLinkByIdForOp(id));
		}
		opReorderTilesAndLinks(order);
	}
	
	/**
	 * Reorder the tiles and links by specifying an array of OrderedElements. The first element
	 * of the array is the backmost element. You must not specify the elementIds of any tiles
	 * that were created as a result of the splitting process. Any that you do not specify will be
	 * added at the front.
	 * @param order
	 */
	public void opReorderTilesAndLinks(List<OrderedElement> order)  {
		// head of the list is the back-most tile.
		// you must not specify any divtiles here - only tiles you actually created yourself.
		// and you don't have to specify all of them
		ListIterator<OrderedElement> li = order.listIterator();		
		while(li.hasNext()) {
			OrderedElement t = li.next();
			if(t instanceof Tile && ((Tile)t).tileType==Tile.TILETYPE_SPLIT) {
				for(Tile[] tts: ((Tile)t).getSplitComponentsPackedByColForReadOnly()) {
					for(Tile tt: tts) {
						li.add(tt);
					}
				}
			}
		}		
		int nTilesAndLinksTotal = this.tilesAndLinksInOrder.size();
		this.tilesAndLinksInOrder.clear();
		Set<OrderedElement> tilesAndLinksRemainingToAdd = new HashSet<OrderedElement>( tilesAndLinksById );
		tilesAndLinksRemainingToAdd.remove(null); // need to remove the null elements
		this.tilesAndLinksInOrder.addAll(order);
		tilesAndLinksRemainingToAdd.removeAll(tilesAndLinksInOrder);
		this.tilesAndLinksInOrder.addAll(tilesAndLinksRemainingToAdd);
		if(this.tilesAndLinksInOrder.size()>nTilesAndLinksTotal) {
			throw new IllegalArgumentException("Some tiles or links appeared twice in tile order, or some div tiles appeared.");
		}
		assert this.tilesAndLinksInOrder.size() == nTilesAndLinksTotal;
		stateListener.callback_changedTilesAndLinksOrder(tilesAndLinksInOrder);
	}	
	
	/**
	 * Prints a stack trace and calls System.exit(1);
	 * @param e
	 */
	public void fatalError(Throwable e) {
		if(logger!=null) {
			logger.log(Level.SEVERE,"Caught exception and about to exit",e);
		}
		e.printStackTrace();
		System.exit(1);
	}
	
	protected Tile factoryReturnNewTileObject(StateListener tilesListener, int tileId, int w, int h, int flags, boolean usePowersOfTwo) {
		return new Tile(this, tileId, w,h, flags);
	}
	
	protected Link factoryReturnNewLinkObject(StateListener tilesListener, int linkId, int displayType, Color c, ScaRotTraTransformImmutable tStandardRectToDESKrectA, ScaRotTraTransformImmutable tStandardRectToDESKrectB) {
		return new Link(this, linkId, displayType, c, tStandardRectToDESKrectA, tStandardRectToDESKrectB);
	}
	

	
	
	/********* CURSORS **************/
	

	public Cursor opCreateCursor(int id)  {
		if(id<0 || id>MAX_CURSOR_ID) {
			throw new IllegalArgumentException("Cursor id too big or too small: "+id);
		}
		if(cursorsById.get(id) != null) {
			throw new IllegalArgumentException("Cursor already exists with id "+id);
		}
		
		Cursor c =  factoryreturnNewCursorObject(id);
		cursorsById.set(id, c);
		cursors.add(c);
		stateListener.callback_createdCursor(c);
		return c;
	}
	
	public Cursor opCreateCursorAutogenerateId()  {
		return this.opCreateCursor(this.nextCursorId++);
	}
	
	public void opDestroyCursor(Cursor c)  {
		cursorsById.set(c.getCursorId(),null);
		cursors.remove(c);		
		stateListener.callback_destroyedCursor(c);
	}
	
	
	public Cursor getCursorByIdForOp(int id)  {
		if(id<0 || id>MAX_CURSOR_ID) {
			throw new IllegalArgumentException("Cursor id too big or too small: "+id);
		}
		Cursor t = cursorsById.get(id);
		if(t==null) { 
			throw new IllegalArgumentException("No such cursor with id "+id);
		}
		return t;
	}
	
	protected Cursor factoryreturnNewCursorObject(int id)  {
		return new Cursor(stateListener, id, Color.WHITE,0);
	}
}
