/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.renderer;


import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import javax.media.opengl.GL;

import t3.hrd.state.Cursor;
import t3.hrd.state.Link;
import t3.hrd.state.OrderedElement;
import t3.hrd.state.ScaRotTraTransformImmutable;
import t3.hrd.state.StateListener;
import t3.hrd.state.Tile;
import t3.hrd.state.UnwarpedRect;
import t3.hrd.state.Cursor.CursorShapePolygons;



/* 
 * Threading notes:
 * This class is thread-safe provided you enforce your own multiple reader single writer
 * routines on its operations.
 */

class StateListenerForHRDRenderer implements StateListener {

	private static final Logger logger = Logger.getLogger("t3.hrd.renderer");
	
	
	Set<Link> linksWhoseAffineTransformsOrVisibilityHaveChanged;
	Set<Tile> backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged;
	Set<Tile> backedTilesWhoseVisibilityHasChanged;
	Set<TileContentUpdate> tileContentUpdatesForBackedTiles;
	Map<Projector, Long> cursorLastMovedOnProjector;
	boolean elementOrderHasChanged;
	
	
	private boolean needToRefreshAllBecauseOfCursors;
	private HRDRenderer hrd;
	
		
	StateListenerForHRDRenderer(HRDRenderer hrd) {
		linksWhoseAffineTransformsOrVisibilityHaveChanged = new HashSet<Link>();
		backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged = new HashSet<Tile>();
		backedTilesWhoseVisibilityHasChanged = new HashSet<Tile>();
		tileContentUpdatesForBackedTiles = new HashSet<TileContentUpdate>();
		elementOrderHasChanged = false;
		this.hrd = hrd;
		this.cursorLastMovedOnProjector = new HashMap<Projector, Long>();
		this.needToRefreshAllBecauseOfCursors = false;
	}
	

	public void callback_repaintUnbackedTile(Tile tile, Rectangle r, BufferedImage update, Graphics2D g) {
		assert false;
	}

	public void callback_opLogMessage(String m) {
        this.needToRefreshAllBecauseOfCursors = true;
        this.hrd.infoString = m;
    }
    
	public void callback_createdTile(Tile tile, int w, int h, int flags) {}
	public void callback_destroyedTile(Tile tile) {}
	
	public void callback_updatedTileContents(Tile tile, int x, int y, BufferedImage update, int compressionHint) {
		if(tile.tileType!=Tile.TILETYPE_SPLIT) {
			logger.fine("Number of entries in tileContentUpdates = "+tileContentUpdatesForBackedTiles.size()+" while for updating for tile "+tile);
			tileContentUpdatesForBackedTiles.add( new TileContentUpdate(tile,x,y,update));
		}
	}	
	
	public void callback_updatedTileContentsByCopyingFromOtherTile(Tile tile, int sx, int sy, int dx, int dy, int w, int h, Tile source) {
		assert tile.tileType==Tile.TILETYPE_BACKED;
		assert source.tileType == Tile.TILETYPE_BACKED;
		logger.fine("Number of entries in tileContentUpdates = "+tileContentUpdatesForBackedTiles.size()+" while for updating for tile "+tile);
		// update shares same data buffer as source's tile image
		BufferedImage update = source.getImageForReadingOnlyNoPainting().getSubimage(sx,sy,w,h);
		tileContentUpdatesForBackedTiles.add( new TileContentUpdate(tile,sx,sy,update));
	}
	
	public void callback_updatedTileAff(Tile t, double DESKcentreX, double DESKcentreY, double DESKwidth, double DESKheight, double thetaClockwise) {
		if(t.tileType!=Tile.TILETYPE_SPLIT) {
			this.backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged.add(t);
		}
	}
	
	public void callback_updatedTileVisibility(Tile t, boolean v) {
		if(t.tileType!=Tile.TILETYPE_SPLIT) {
			backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged.add(t);
			backedTilesWhoseVisibilityHasChanged.add(t);
		}
	}
	
	public void callback_createdUnwarpedRect(Tile tile,UnwarpedRect r) {
		if(tile.tileType!=Tile.TILETYPE_SPLIT) {
			backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged.add(tile);
		}
	}
	/*
	public void callback_destroyedUnwarpedRect(Tile tile,Rectangle r) {
		if(tile.tileType!=Tile.TILETYPE_SPLIT) {
			backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged.add(tile);
		}
	}*/
	

	public void callback_destroyedUnwarpedRectsIntersecting(Tile tile,Rectangle r) {
		if(tile.tileType!=Tile.TILETYPE_SPLIT) {
			backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged.add(tile);			
		}
	}
	
	public void callback_changedTilesAndLinksOrder(List<OrderedElement> elsInOrder) {
		this.elementOrderHasChanged = true;
	}
	
	void clear() {
		linksWhoseAffineTransformsOrVisibilityHaveChanged.clear();
		backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged.clear();
		backedTilesWhoseVisibilityHasChanged.clear();
		tileContentUpdatesForBackedTiles.clear();
		elementOrderHasChanged = false;
		this.needToRefreshAllBecauseOfCursors = false;
	}
		

	public static class TileContentUpdate {
		final public BufferedImage image;
		final Tile tile;
		final int x, y;
		public TileContentUpdate(Tile tile, int x, int y, BufferedImage image) {
			this.tile=tile; this.x=x; this.y=y; this.image=image;
		}
	}
	
	public void callback_cursorPos(Cursor c, double DESKx, double DESKy, boolean visible) {
		for(Projector p: hrd.projectors) {
			if(p.transforms.rDESKdeskSpaceAlignedNonBlackedRect.contains(DESKx,DESKy)) {
				// the new point is on projector p
				// trail period +200 because we want the trail to disappear
                // (this means we assume a minimum of 5fps)
				cursorLastMovedOnProjector.put(p, System.currentTimeMillis()+Cursor.TRAIL_PERIOD+200);
			}				
		}	
	}
	
	public void callback_cursorShape(Cursor c, CursorShapePolygons polygonsForShapeCursor) {
		for(Projector p: hrd.projectors) {
			for(Rectangle2D rDESKboundingBox:  polygonsForShapeCursor.rDESKboundingBoxes) {
				if(p.transforms.rDESKdeskSpaceAlignedNonBlackedRect.intersects(rDESKboundingBox)) {
					cursorLastMovedOnProjector.put(p, System.currentTimeMillis()+200);
				}
			}
		}
	}
	
	public void callback_createdCursor(Cursor c) {}
	
	public void callback_updatedCursorOptions(Cursor c) {
		this.needToRefreshAllBecauseOfCursors = true;
	}
	
	public void callback_destroyedCursor(Cursor c) {
		this.needToRefreshAllBecauseOfCursors = true;
	}	
	
	
	
	public void oglDrawCursorsOnProjector(GL gl, Projector p) {
		// toopt don't need to draw all of them, only the ones that are on the projector...
		for(Cursor c: hrd.stateManager.cursorsReadOnly) {
			c.oglDrawCursor(gl, p);
		}
	}
	
	
	public boolean projectorNeedsRefreshingBecauseOfCursors(Projector p) {
		if(this.needToRefreshAllBecauseOfCursors) {
			return true;
		} else {
			Long t = cursorLastMovedOnProjector.get(p);
			return t!=null && System.currentTimeMillis()<=t.longValue();
		}
	}


	public void callback_createdLink(Link l, int displayType, Color c, ScaRotTraTransformImmutable tUniformRectToDESKrectA, ScaRotTraTransformImmutable tUniformRectToDESKrectB) {
		this.linksWhoseAffineTransformsOrVisibilityHaveChanged.add(l);
	}


	public void callback_destroyedLink(Link l) {
		this.linksWhoseAffineTransformsOrVisibilityHaveChanged.add(l);
	}


	public void callback_updatedLinkAff(Link l,ScaRotTraTransformImmutable tUniformRectToDESKrectA, ScaRotTraTransformImmutable tUniformRectToDESKrectB) {
		this.linksWhoseAffineTransformsOrVisibilityHaveChanged.add(l);
	}	
	
	

	
	
}
	
