/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.renderer;


import java.awt.Color;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.imageio.ImageIO;
import javax.media.opengl.GL;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLEventListener;
import javax.media.opengl.glu.GLU;

import com.sun.opengl.util.GLUT;

import t3.hrd.state.JOGLHelper;
import t3.hrd.state.Link;
import t3.hrd.state.OrderedElement;
import t3.hrd.state.Tile;
import t3.hrd.state.UnwarpedRect;
import Jama.Matrix;

class ProjectorOpenGLCanvasRepainter implements GLEventListener {

	
	final Projector projector;

	
	/**
	 * By present, we mean that the texture was created in this projector's opengl
	 * context. A texture created in another projector's context and shared through
	 * to this projector's context will not appear in this map.
	 */
	private Map<Tile,TextureOnProjector> tilesToTexturesPresentOnThisProjector;
	
	private TextureOnProjector blendTexture;
	private int frameCountMod;
	
	/*private static final double maxAngleForUnwarpedRects = (0.57*(Math.PI*2.0/360.0));
	private static final double minSfForUnwarpedRects = 0.90;
	private static final double maxSfForUnwarpedRects = 1.30;*/
	
    private static final double maxAngleForUnwarpedRects = (0.75*(Math.PI*2.0/360.0));
    private static final double minSfForUnwarpedRects = 0.90;
    private static final double maxSfForUnwarpedRects = 1.05;
    
    
	
	private static final Logger logger = Logger.getLogger("t3.hrd.renderer");
	private static final GLU glu = new GLU();
    private static final GLUT glut = new GLUT();
	
	ProjectorOpenGLCanvasRepainter(Projector p) {
		this.projector = p;
		this.tilesToTexturesPresentOnThisProjector = new HashMap<Tile,TextureOnProjector>();

		this.blendTexture = null;
		this.frameCountMod=0;
	}
	
	/**
	 * Returns null if tile t's texture not present on this projector
	 * @param t
	 * @return
	 */
	TextureOnProjector getTexturePresentOnThisProjectorForTile(Tile t) {
		return tilesToTexturesPresentOnThisProjector.get(t);
	}
	
	public void init(GLAutoDrawable drawable) {
		// called by our canvas
		GL gl = drawable.getGL();
		logger.info("OpenGL vendor: "+gl.glGetString(GL.GL_VENDOR));
		logger.info("OpenGL version: "+gl.glGetString(GL.GL_VERSION));
		logger.info("OpenGL renderer: "+gl.glGetString(GL.GL_RENDERER));
		logger.fine("OpenGL extensions: "+gl.glGetString(GL.GL_EXTENSIONS));		
		
		gl.setSwapInterval(0);
		gl.glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
		gl.glShadeModel(gl.GL_FLAT);
		gl.glPixelStorei(gl.GL_UNPACK_ALIGNMENT,1);
		gl.glEnable(gl.GL_BLEND);
		gl.glBlendFunc(gl.GL_SRC_ALPHA,gl.GL_ONE_MINUS_SRC_ALPHA);
		gl.glEnable(gl.GL_TEXTURE_2D);
		gl.glTexEnvf(gl.GL_TEXTURE_ENV,gl.GL_TEXTURE_ENV_MODE,gl.GL_REPLACE);
		gl.glDisable(gl.GL_TEXTURE_2D);
		JOGLHelper.oglThrowExceptionIfError(gl, glu);
	}
	
	
		
	public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height) {
		// called by our canvas
		GL gl = drawable.getGL();
		// no need to call glViewport - it's done automatically			
		gl.glMatrixMode( GL.GL_PROJECTION ); 
		gl.glLoadIdentity();	
		gl.glFrustum(
				ProjectorTransforms.openGlView_left, 
				ProjectorTransforms.openGlView_left+ProjectorTransforms.openGlView_width, 
				ProjectorTransforms.openGlView_bottom, 
				ProjectorTransforms.openGlView_bottom+ProjectorTransforms.openGlView_height, 
				ProjectorTransforms.openGlView_near, 
				ProjectorTransforms.openGlView_far
				);
		gl.glMatrixMode(gl.GL_MODELVIEW);
		gl.glLoadIdentity(); 
		JOGLHelper.oglThrowExceptionIfError(gl, glu);
	}
	
			
		
	public void display(GLAutoDrawable drawable) {
		// called by our canvas
		// we must now redraw because if we don't then
		// the display will not necessarily be stable because of back buffering
	
		this.frameCountMod++;
		if(this.frameCountMod>=10) { this.frameCountMod=0; }
		GL gl = drawable.getGL();
		
		if(this.projector.hrdRenderer.isLastOpenGLRedrawBeforeClosing()) {			
			this.oglLastRedrawBeforeClosingSoDestroyAllTextures(gl);		
		} else {
		
			long t2 = System.currentTimeMillis();
			
			JOGLHelper.oglThrowExceptionIfError(gl, glu);
			this.oglCreateAndDestroyTexturesForTiles(gl);
			
			long t3 = System.currentTimeMillis();
			
			JOGLHelper.oglThrowExceptionIfError(gl, glu);
			this.oglUpdateContentsOfTexturesForTiles(gl);
			JOGLHelper.oglThrowExceptionIfError(gl, glu);
			
			long t5 = System.currentTimeMillis();
			
			gl.glClear( GL.GL_COLOR_BUFFER_BIT );
			this.oglDrawOrderedElements(gl);
			
			long t6 = System.currentTimeMillis();
			
			projector.hrdRenderer.stateListener.oglDrawCursorsOnProjector(gl, projector);
	
			long t7 = System.currentTimeMillis();			
			
			projector.hrdRenderer.callBacks.callback_oglPaintOverlay(gl, projector);
	
			long t8 = System.currentTimeMillis();	
			
			this.oglDrawBlackOrBlend(gl);
            

            this.oglDrawInfoString(gl);

			long t9 = System.currentTimeMillis();
			//if(t9-t1>70) {
				//logger.info("Long refresh: "+this+": "+(t2-t1)+" "+(t3-t2)+" "+(t4-t3)+" "+(t5-t4)+" "+(t6-t5)+" "+(t7-t6)+" "+(t8-t7)+" "+(t9-t8)+" ");
			//}
		}
			
		JOGLHelper.oglThrowExceptionIfError(gl, glu);
			
	}
	
	
		
	/** This method handles things if display depth changes */
	public void displayChanged(GLAutoDrawable drawable,
			boolean modeChanged,
			boolean deviceChanged){
	}
	
	
	private void oglLastRedrawBeforeClosingSoDestroyAllTextures(GL gl) {
		for(TextureOnProjector texture: this.tilesToTexturesPresentOnThisProjector.values()) {
			logger.info("Destroyed tile texture on projector "+this.projector);
			JOGLHelper.oglDestroyTexture(gl,texture.textureName);
		}
		if(this.blendTexture!=null) {
			logger.info("Destroyed blend texture on projector "+this.projector);
			JOGLHelper.oglDestroyTexture(gl,this.blendTexture.textureName);
		}
	}
	
	
	private void oglCreateAndDestroyTexturesForTiles(GL gl) {
		for(Tile tile: projector.hrdRenderer.stateListener.backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged) {
			assert tile.tileType==Tile.TILETYPE_BACKED;
			TextureOnProjector texture = this.getTexturePresentOnThisProjectorForTile(tile);
			boolean tileTextureShouldBeOnThisProjector = tileTextureShouldBePresentOnThisProjector(tile);
			if(texture==null && tileTextureShouldBeOnThisProjector) {
				// create the texture and fill it
				logger.fine("Created texture on projector "+this.projector+" for tile "+tile);
				int textureName = JOGLHelper.oglCreateTexture(
						gl,
						tile.imageW,
						tile.imageH,
						tile.getImageForReadingOnlyNoPainting()
					);
				TextureOnProjector newTexture = new TextureOnProjector(textureName, tile.imageW, tile.imageH, tile.tileWidth, tile.tileHeight);
				this.tilesToTexturesPresentOnThisProjector.put(tile, newTexture);
			} else if(texture!=null && !tileTextureShouldBeOnThisProjector) {
				// destroy the texture				
				JOGLHelper.oglDestroyTexture(gl,texture.textureName);
				logger.fine("Destroyed texture on projector "+this.projector+" for tile "+tile);
				this.tilesToTexturesPresentOnThisProjector.remove(tile);				
			} else {
				// no need to do anything
			}
		}
	}	
	
	
	
	private boolean oglUpdateContentsOfTexturesForTiles(GL gl) {
		/* toopt
		 * When a tile is created, (as opposed to just being moved onto a different
		 * projector) we actually end up sending the bitmap data down to the head twice:
		 * once in reponse to the tile creation and once in response to its first update.
		 * We could actually amalgamate the two by saying that if a tile has just been created on this projector then 
		 * there's no need to carry out its updates on this projector. But this shouldn't
		 * actually happen that often.
		 */		
		boolean changed = false;
		for(StateListenerForHRDRenderer.TileContentUpdate tileContentUpdate:projector.hrdRenderer.stateListener.tileContentUpdatesForBackedTiles) {
			assert tileContentUpdate.tile.tileType==Tile.TILETYPE_BACKED;
			TextureOnProjector texture = this.tilesToTexturesPresentOnThisProjector.get(tileContentUpdate.tile);
			if(texture!=null) {
				changed = true;
				//logger.info("Updating texture on projector "+this+" for tile "+tileContentUpdate.tile);
				gl.glBindTexture(gl.GL_TEXTURE_2D, texture.textureName);
				JOGLHelper.oglUpdateTextureContents(
						gl,
						tileContentUpdate.x,
						tileContentUpdate.y,
						tileContentUpdate.image.getWidth(),
						tileContentUpdate.image.getHeight(),
						tileContentUpdate.image);
			} else {
				// no need to do anything. Tile's texture is not present on this projector.
			}
		}
		return changed;
	}
	
	private void oglDrawOrderedElements(GL gl) {
		// toopt don't go through this list every time. have our own list of tilesinorder
		// for just the tiles on our display
		for(OrderedElement el:projector.hrdRenderer.stateManager.tilesAndLinksInOrderReadOnly) {
			if(
				el instanceof Tile 
				&& ((Tile)el).tileType==Tile.TILETYPE_BACKED 
				&& this.projector.tilesVisibleInThisProjectorsDeskSpaceImmutable.contains((Tile)el)
			) {
				TextureOnProjector texture = this.getTextureOnProjectorForTileUsingSharingIfConfigured((Tile)el);
				assert texture != null;
				oglDrawTextureForTile(gl, (Tile)el, texture);
			} else if(el instanceof Link){
				oglDrawLink(gl,(Link)el);
			}
		}
	}
	
	private void oglDrawTextureForTile(GL gl, Tile tile, TextureOnProjector texture) {

		assert tile.tileType==Tile.TILETYPE_BACKED;
		
		if(tile.isFlagSet(Tile.FLAG_NO_TRANSFORM)) {
			oglDrawTextureForUnwarpedRect(gl,tile,texture, new java.awt.geom.Rectangle2D.Double(0,0,tile.tileWidth,tile.tileHeight));			
		} else {
			
			//long t1= System.currentTimeMillis();
			
			// toopt use opengl matrix stack to do these in hardware
			Matrix mTILEtoTOGL = projector.transforms.mDESKtoTOGL.times(tile.getmTILEtoDESK());
			Matrix[] mTOGLcornersClockwise = JOGLHelper.applyMatrixToSetOfMatrices(mTILEtoTOGL, tile.getmTILEcornersClockwise());

			//long t2= System.currentTimeMillis();

			gl.glEnable(gl.GL_TEXTURE_2D);
            //gl.glHint(gl.GL_PERSPECTIVE_CORRECTION_HINT, gl.GL_FASTEST);
			gl.glBindTexture(gl.GL_TEXTURE_2D, texture.textureName);
			
			// from time to time, check it's actually resident!
			if(this.frameCountMod==0) {
				if(!JOGLHelper.oglCurrentBoundTextureIsResident(gl)) {
					logger.warning("Tile texture is not resident on projector! Tile="+tile+" Projector="+this.projector);
				}
			}

			//long t3= System.currentTimeMillis();
			gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_LINEAR);
			gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_LINEAR);
			gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_WRAP_S, gl.GL_CLAMP_TO_EDGE );
			gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_WRAP_T, gl.GL_CLAMP_TO_EDGE );

			//long t4= System.currentTimeMillis();
			gl.glBegin(gl.GL_QUADS);
			gl.glTexCoord2d(texture.fracLeftX,texture.fracBottomY); 	JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[0]);
			gl.glTexCoord2d(texture.fracLeftX,texture.fracTopY); 		JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[1]);
			gl.glTexCoord2d(texture.fracRightX,texture.fracTopY); 		JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[2]);
			gl.glTexCoord2d(texture.fracRightX,texture.fracBottomY); 	JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[3]);
			
			//long t5= System.currentTimeMillis();
			gl.glEnd();
			gl.glDisable(gl.GL_TEXTURE_2D);
			

			//long t6= System.currentTimeMillis();
			
			if(tile.unwarpedRectsWordsReadOnly.size()>0 || tile.unwarpedRectsLinesReadOnly.size()>0) {
				final int shouldDrawUnwarped = this.shouldDrawUnwarpedRectsForTile(tile);
				if(shouldDrawUnwarped==0) {
					// do nothing. angle is too great
				} else if(shouldDrawUnwarped==1) {
					for(UnwarpedRect r: tile.unwarpedRectsWordsReadOnly) {
						oglDrawBlankPolyForUnwarpedRect(gl,tile,r.r, r.backgroundARGB);	
					}
					for(UnwarpedRect r: tile.unwarpedRectsWordsReadOnly) {
						oglDrawTextureForUnwarpedRect(gl,tile,texture,r.r);	
					}
				} else if(shouldDrawUnwarped==2) { 
					for(UnwarpedRect r: tile.unwarpedRectsLinesReadOnly) {
						oglDrawBlankPolyForUnwarpedRect(gl,tile,r.r, r.backgroundARGB);	
					}
					for(UnwarpedRect r: tile.unwarpedRectsLinesReadOnly) {
						oglDrawTextureForUnwarpedRect(gl,tile,texture,r.r);	
					}
				} else {
					assert false;
				}
			} else {
				// don't draw as no unwarped rects
			}

			//long t7= System.currentTimeMillis();
			//System.out.println("Drew tile "+(t2-t1)+" "+(t3-t2)+" "+(t4-t3)+" "+(t5-t4)+" "+(t6-t5)+" "+(t7-t6));
		
		}
		
	}
	
	
	private int shouldDrawUnwarpedRectsForTile(Tile tile) {
		
		/*
		 *  returns:
		 *  	0: no
		 *  	1: yes, draw words but not lines
		 *  	2: yes, draw lines but not words
		 */
		
		final boolean FORCE_RETURN = false;
		final int FORCE_RETURN_INT = 2;
		
		if(FORCE_RETURN) {
			return FORCE_RETURN_INT;
		}
		
		// get diagonal points
		
		Matrix mTILEtoFB = this.projector.transforms.mDESKtoFBT.times(tile.getmTILEtoDESK());
		final Matrix mFBtopLeftOfTile = mTILEtoFB.times(tile.getmTILEcornersClockwise()[0]);
		final Matrix mFBbtmRightOfTile = mTILEtoFB.times(tile.getmTILEcornersClockwise()[2]);
		final Matrix mFBtopRightOfTile = mTILEtoFB.times(tile.getmTILEcornersClockwise()[3]);
		final Matrix mFBbtmLeftOfTile = mTILEtoFB.times(tile.getmTILEcornersClockwise()[1]);

		// check rotation
		/*
		 * OLD WAY - average two diagonals.
		 *
		final double fbDiagAngle1 = 
			JOGLHelper.getPolarAngleOfLineBetween2hmPoints(
				mFBtopLeftOfTile,
				mFBbtmRightOfTile
			);
		final double fbDiagAngle2 = 
			JOGLHelper.getPolarAngleOfLineBetween2hmPoints(
				mFBbtmLeftOfTile,
				mFBtopRightOfTile
			);		
		final double tileDiagAngle1 = Math.atan2(-tile.tileHeight, tile.tileWidth);
		final double tileDiagAngle2 = -tileDiagAngle1;
		final double tileToFbRotn1 = fbDiagAngle1-tileDiagAngle1;
		final double tileToFbRotn2 = fbDiagAngle2-tileDiagAngle2;
		final double tileToFbRotnAvg = 0.5*(tileToFbRotn1+tileToFbRotn2);
		double tileToFbRotnModPiByTwo = tileToFbRotnAvg % (Math.PI/2.0);
		if(tileToFbRotnModPiByTwo<0) {
			tileToFbRotnModPiByTwo+=Math.PI/2.0;
		}
		*/
		
		// NEW WAY - care much more about horiz angle.
		final double fbAngle1 = 
			JOGLHelper.getPolarAngleOfLineBetween2hmPoints(
				mFBbtmLeftOfTile,
				mFBbtmRightOfTile
			);
		final double fbAngle2 = 
			JOGLHelper.getPolarAngleOfLineBetween2hmPoints(
				mFBtopLeftOfTile,
				mFBtopRightOfTile
			);
		double tileToFbRotnModPiByTwo = (0.5*(fbAngle1+fbAngle2)) % (Math.PI/2.0);
		if(tileToFbRotnModPiByTwo<0) {
			tileToFbRotnModPiByTwo+=Math.PI/2.0;
		}		
		if(tileToFbRotnModPiByTwo>maxAngleForUnwarpedRects && tileToFbRotnModPiByTwo<Math.PI/2.0-maxAngleForUnwarpedRects) {
			return 0;
		}
				
		// check scale
		final double fbWarpedDiagonal = JOGLHelper.getDistanceBetween2hmPoints(
				 mFBtopLeftOfTile,
				 mFBbtmRightOfTile
			);
		final double fbUnwarpedDiagonal = Math.sqrt(tile.tileWidth*tile.tileWidth+tile.tileHeight*tile.tileHeight);
		final double scaleFactorWarpedToUnwarped = fbUnwarpedDiagonal/fbWarpedDiagonal;
		//System.out.println(scaleFactorWarpedToUnwarped);
		if( scaleFactorWarpedToUnwarped<minSfForUnwarpedRects || scaleFactorWarpedToUnwarped>maxSfForUnwarpedRects) {
			return 0;
		}
		
		//	if( this.projector.transforms.aDESKnonBlackedArea.contains(tile.getrDESKboundingBox())) {
		if( this.projector.transforms.rDESKdeskSpaceAlignedNonBlackedRect.contains(tile.getrDESKboundingBox())) {
			return 2;
		} else {
			return 1;
		}	
	}
	
	
	private void oglDrawBlankPolyForUnwarpedRect(GL gl, Tile tile, Rectangle2D unwarpedRect, int bgARGB) {
		
		// need to be careful here!
		// integer tile coords correspond to the top left hand ocrner of the corresponding pixel
		// so to make sure we get teh whole rectangle we need to add 1 to the maxx and maxy coordinates
		// as below.
		// seems to work a little better if you add an extra 1 on top of this to the maxy coord 
		
		final Matrix[] mTILEcornersClockwise = new Matrix[] {
				JOGLHelper.getMFromD(unwarpedRect.getX()-1, unwarpedRect.getY(), 1),
				JOGLHelper.getMFromD(unwarpedRect.getX()-1, unwarpedRect.getY()+unwarpedRect.getHeight()+2, 1),
				JOGLHelper.getMFromD(unwarpedRect.getX()+unwarpedRect.getWidth()+2, unwarpedRect.getY()+unwarpedRect.getHeight()+2, 1),
				JOGLHelper.getMFromD(unwarpedRect.getX()+unwarpedRect.getWidth()+2, unwarpedRect.getY(), 1)
			};
		final Matrix mTILEtoTOGL = projector.transforms.mDESKtoTOGL.times(tile.getmTILEtoDESK());
		final Matrix[] mTOGLcornersClockwise = JOGLHelper.applyMatrixToSetOfMatrices(mTILEtoTOGL, mTILEcornersClockwise);

		gl.glDisable(gl.GL_TEXTURE_2D);
		JOGLHelper.oglColorFromARGBint(gl, bgARGB);
		gl.glBegin(gl.GL_QUADS);
		JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[0]);
		JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[1]);
		JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[2]);
		JOGLHelper.oglVertex4dFromM(gl,mTOGLcornersClockwise[3]);
		gl.glEnd();
	}
	
	private void oglDrawTextureForUnwarpedRect(GL gl, Tile tile, TextureOnProjector texture, Rectangle2D unwarpedRect) {
				
		Matrix mTILEunwarpedRectCentre = JOGLHelper.getMFromD(
				new double[] { 
						unwarpedRect.getX()+unwarpedRect.getWidth() / 2.0, 
						unwarpedRect.getY()+unwarpedRect.getHeight()/ 2.0,
						1.0
				}
			);	
		// compute centerInOGLSpace
		Matrix mFBunwarpedRectCentre = 
			projector.transforms.mDESKtoFB.times(
					tile.getmTILEtoDESK().times(mTILEunwarpedRectCentre)
				);
		
		// unhomogenise
		mFBunwarpedRectCentre.set(0,0,mFBunwarpedRectCentre.get(0,0)/mFBunwarpedRectCentre.get(3,0));
		mFBunwarpedRectCentre.set(1,0,mFBunwarpedRectCentre.get(1,0)/mFBunwarpedRectCentre.get(3,0));
		mFBunwarpedRectCentre.set(2,0,mFBunwarpedRectCentre.get(2,0)/mFBunwarpedRectCentre.get(3,0));
		mFBunwarpedRectCentre.set(3,0,1.0);
				
		final int nTILEtoFBrotateClockwise = (int)Math.round((tile.getTILEtoDESKrotationClockwise()+projector.transforms.UOGLtoWOGLrotation)/(Math.PI/2.0)); 
		
		// compute width and height in framebuffer space, rememebering to swap if we rotate
		// these should be whole numbers
		final double FBwidth = ( nTILEtoFBrotateClockwise%2==0) ? unwarpedRect.getWidth() : unwarpedRect.getHeight();
		final double FBheight= ( nTILEtoFBrotateClockwise%2==0) ? unwarpedRect.getHeight(): unwarpedRect.getWidth();

		// compute bottom left and top right in pixels
		double FBleft = Math.round(mFBunwarpedRectCentre.get(0,0)-FBwidth/2.0),
			FBbottom = Math.round(mFBunwarpedRectCentre.get(1,0)-FBheight/2.0),
			FBright = FBleft+FBwidth,
			FBtop = FBbottom+FBheight,
			FBz = mFBunwarpedRectCentre.get(2,0);
				
		if(Tile.TILEtoDESKflipsHoriz ^ projector.transforms.UOGLtoWOGLflipsHoriz) {
			double t = FBleft;
			FBleft = FBright;
			FBright = t;
		}
		
		if(Tile.TILEtoDESKflipsVert ^ projector.transforms.UOGLtoWOGLflipsVert) {
			double t = FBbottom;
			FBbottom = FBtop;
			FBtop = t;
		}		
				
		Matrix[] mFBunwarpedRectCornersClockwise = new Matrix[] {
			JOGLHelper.getMFromD( new double[] {FBleft, FBbottom, FBz, 1.0}),
			JOGLHelper.getMFromD( new double[] {FBleft, FBtop, FBz, 1.0  }),
			JOGLHelper.getMFromD( new double[] {FBright, FBtop, FBz, 1.0  }),
			JOGLHelper.getMFromD( new double[] {FBright, FBbottom, FBz, 1.0  })
		};
		
		if(nTILEtoFBrotateClockwise==0) {
			// no need to do anything
		} else {
			mFBunwarpedRectCornersClockwise = JOGLHelper.rotateArray(mFBunwarpedRectCornersClockwise,nTILEtoFBrotateClockwise);
		}
		
		Matrix[] mTOGLunwarpedRectCornersClockwise = JOGLHelper.applyMatrixToSetOfMatrices( projector.transforms.mFBtoTOGL, mFBunwarpedRectCornersClockwise);
			
		gl.glEnable(gl.GL_TEXTURE_2D);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_NEAREST);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_NEAREST);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_WRAP_S, gl.GL_CLAMP);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_WRAP_T, gl.GL_CLAMP);
		gl.glBindTexture(gl.GL_TEXTURE_2D, texture.textureName);
		gl.glBegin(gl.GL_QUADS);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord((int)unwarpedRect.getX())/(double)texture.w,texture.getTextureYCoordFromTileYCoord((int)unwarpedRect.getY())/(double)texture.h); 						
		JOGLHelper.oglVertex4dFromM(gl,mTOGLunwarpedRectCornersClockwise[0]);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord((int)unwarpedRect.getX())/(double)texture.w,texture.getTextureYCoordFromTileYCoord((int)unwarpedRect.getY()+(int)unwarpedRect.getHeight())/(double)texture.h); 			
		JOGLHelper.oglVertex4dFromM(gl,mTOGLunwarpedRectCornersClockwise[1]);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord((int)unwarpedRect.getX()+(int)unwarpedRect.getWidth())/(double)texture.w,texture.getTextureYCoordFromTileYCoord((int)unwarpedRect.getY()+(int)unwarpedRect.getHeight())/(double)texture.h); 
		JOGLHelper.oglVertex4dFromM(gl,mTOGLunwarpedRectCornersClockwise[2]);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord((int)unwarpedRect.getX()+(int)unwarpedRect.getWidth())/(double)texture.w,texture.getTextureYCoordFromTileYCoord((int)unwarpedRect.getY())/(double)texture.h); 			
		JOGLHelper.oglVertex4dFromM(gl,mTOGLunwarpedRectCornersClockwise[3]);
		gl.glEnd();	
		gl.glDisable(gl.GL_TEXTURE_2D);
	}
	
	
	private void oglDrawLink(GL gl, Link link) {
		JOGLHelper.oglColor(gl, link.color);
		for(List<Matrix> l:link.getmDESKpolyPoints()) {
			
			gl.glBegin(gl.GL_POLYGON);
			for(Matrix m: l) {
				JOGLHelper.oglVertex4dFromM(gl,this.projector.transforms.mDESKtoTOGL.times(m));
			}
			gl.glEnd();
		}
	}
	
	
	
	private void oglDrawBlackOrBlend(GL gl) {
		if(this.blendTexture==null) {
			this.oglCreateBlendTexture(gl);
		} else {
			// no need to create blend texture as it already exists
		}
		this.oglDrawBlendTexture(gl);
	}
	
	
	private void oglDrawBlendTexture(GL gl) {
		assert this.blendTexture !=null;
		TextureOnProjector texture = this.blendTexture;		
		gl.glEnable(gl.GL_TEXTURE_2D);
		gl.glBindTexture(gl.GL_TEXTURE_2D, this.blendTexture.textureName);		
		//	from time to time, check it's actually resident!
		if(this.frameCountMod==0) {
			if(!JOGLHelper.oglCurrentBoundTextureIsResident(gl)) {
				logger.warning("Blend texture is not resident on projector! Projector="+this.projector);
			}
		}		
		gl.glBegin(gl.GL_QUADS);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord(0)/(double)texture.w,texture.getTextureYCoordFromTileYCoord(0)/(double)texture.h); 						
		JOGLHelper.oglVertex4dFromM(gl,projector.transforms.mTOGLcornersOfProjectorLimitClockwise[0]);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord(0)/(double)texture.w,texture.getTextureYCoordFromTileYCoord(texture.extH)/(double)texture.h); 			
		JOGLHelper.oglVertex4dFromM(gl,projector.transforms.mTOGLcornersOfProjectorLimitClockwise[1]);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord(texture.extW)/(double)texture.w,texture.getTextureYCoordFromTileYCoord(texture.extH)/(double)texture.h); 
		JOGLHelper.oglVertex4dFromM(gl,projector.transforms.mTOGLcornersOfProjectorLimitClockwise[2]);
		gl.glTexCoord2d(texture.getTextureXCoordFromTileXCoord(texture.extW)/(double)texture.w,texture.getTextureYCoordFromTileYCoord(0)/(double)texture.h); 			
		JOGLHelper.oglVertex4dFromM(gl,projector.transforms.mTOGLcornersOfProjectorLimitClockwise[3]);
		gl.glEnd();			
        gl.glDisable(gl.GL_TEXTURE_2D);
	}
	

	private void oglCreateBlendTexture(GL gl) {
		
		assert this.blendTexture == null; 
		int displayW = projector.transforms.fbWidth;
		int displayH = projector.transforms.fbHeight;
		int textureW = JOGLHelper.getNearestPowerOfTwoGE(displayW);
		int textureH = JOGLHelper.getNearestPowerOfTwoGE(displayH);
		BufferedImage b = JOGLHelper.createNewBufferedImageForTexture(textureW,textureH, false);
		this.projector.transforms.paintBlendImage(
            this.projector.hrdRenderer.blendOptions,
			b.createGraphics(),
            this.projector.hrdRenderer.projectors,
            false
		);
        if(ProjectorTransforms.DEBUG_OUTPUT_BLEND_IMAGES) {
    		try {
    			ImageIO.write(b,"png",new File("blend-"+this.hashCode()+"-.png"));
    		} catch(Exception e) {
    			throw new RuntimeException(e);
    		}
        }
		int textureName = JOGLHelper.oglCreateTexture(gl,textureW,textureH,b);
		this.blendTexture = new TextureOnProjector(textureName, textureW, textureH, displayW, displayH);
		
	}
    
    private void oglDrawInfoString(GL gl) {
        String s = this.projector.hrdRenderer.infoString;
        if(s!=null && !s.equals("")) {
            JOGLHelper.oglColor(gl, Color.WHITE); 
            gl.glRasterPos3d(-0.1,0.0,ProjectorTransforms.openGlView_z); // set position
            glut.glutBitmapString(GLUT.BITMAP_HELVETICA_18, s);
            
        } else {
            // don't bother
        }
    }

	

	private TextureOnProjector getTextureOnProjectorForTileUsingSharingIfConfigured(Tile t) {
		if(this.projector.hrdRenderer.shareTexturesBetweenContextsPrimaryProjector==null) {
			return this.getTexturePresentOnThisProjectorForTile(t);
		} else {
			return this.projector.hrdRenderer.shareTexturesBetweenContextsPrimaryProjector.openGLcanvasRepainter.getTexturePresentOnThisProjectorForTile(t);
		}
	}
	
	
	/**
	 * By present we mean that the texture should be created in this projector's
	 * opengl context. If textures are shared between contexts then this will only
	 * return true for the primary projector.
	 * @param t
	 * @return
	 */
	private boolean tileTextureShouldBePresentOnThisProjector(Tile t) {
		assert t.tileType==Tile.TILETYPE_BACKED;
		return 
			(
				this.projector.hrdRenderer.shareTexturesBetweenContextsPrimaryProjector==this.projector
				&& (t.isVisible() || t.isFlagSet(Tile.FLAG_TEXTUREALWAYSRESIDENTEVENWHENTILEINVISIBLE) )
			) || (
				this.projector.hrdRenderer.shareTexturesBetweenContextsPrimaryProjector==null	
				&& (
					t.isFlagSet(Tile.FLAG_TEXTUREALWAYSRESIDENTEVENWHENTILEINVISIBLE)
					|| (t.isFlagSet(Tile.FLAG_TEXTUREALWAYSRESIDENTWHENTILEVISIBLE) && t.isVisible())
					|| this.projector.tilesVisibleInThisProjectorsDeskSpaceImmutable.contains(t)
				)
			);
	}
	
	
	
	
}
