/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PATCHTREELEAF_H
#define PATCHTREELEAF_H

#include "patchtree.h"
#include "types.h"

namespace snurbs {

class KnotInterval;

/**
 * There is a PatchTreeLeaf at the bottom of every PatchTree branch created by
 * BinaryPatchTree and QuadPatchTree. The form of the leaf depends on whether
 * the mesh is primal (created by a PrimalMeshBuilder), or dual (created by a
 * DualMeshBuilder).
 * @see PrimalPatchTreeLeaf
 * @see DualPatchTreeLeaf
 *
 * @ingroup Mesh
 */
class PatchTreeLeaf : public PatchTree
{
    friend class PrimalMeshBuilder;
    friend class DualMeshBuilder;

public:
    /** PatchTreeLeaf constructor. */
    PatchTreeLeaf(void);

    /**
     * @name Getter functions
     * @{
     */
    KnotInterval *getKnotInterval(Orientation orientation) const;

    PatchTree *&getChild(Direction dir);

    PatchTree *&getParent(void);
    /** @} */

    bool isLeaf(void) const;

    bool isRoot(void) const;

    bool splits(Orientation orientation) const;

    void setParent(PatchTree *newParent);

    Direction dirFromParent(void) const;

    Direction childDir(const PatchTree *child) const;

    /**
     * @return @c true if there are no Vertices which return @c false to
     * Vertex::isReal(). (Imaginary Vertex objects are used to mark
     * PatchTreeLeaf objects which exist on boundaries purely to provide
     * knot interval information)
     */
    virtual bool isReal(void) const = 0;

    /**
     * @name Subdivision functions
     * @{
     */
    /**
     * Makes requests to insert knots into #horizontal and #vertical
     * KnotInterval objects, so that regular regions tend to uniformity
     */
    virtual void makeInsertionRequests(void) = 0;

    /**
     * The refine stage is carried out in two parts, of which this is the
     * first. It is responsible for splitting PatchTree nodes.
     * @return The new @c PatchTree* that should replace @c this
     */
    virtual PatchTree *refineTree(void) = 0;

    /**
     * The refine stage is carried out in two parts, of which this is the
     * second. It is responsible for making sure that all vertices are
     * allocated, and is allowed to assume that all neighbouring PatchTree
     * objects have also been split.
     * @param degree The degree to subdivide at (see Mesh::degree)
     */
    virtual void refineVertices(unsigned char degree) = 0;

    /**
     * Make refine stage adjustments to the Vertex::Replacement objects
     * referenced from this leaf, given that there is an extraordinary
     * PatchNode of valency @c valency in Direction @c extNode.
     */
    virtual void extraordinaryRefine(unsigned char degree,
                                     Direction extNode,
                                     unsigned char valency) = 0;

    /**
     * Recursively carry out a smoothing stage of a subdivision step.
     * @param degree The degree to subdivide at (see Mesh::degree)
     * @param stage The subdivision stage: ranges from 1 to @f$\lfloor@f$
     * @c degree / 2 @f$\rfloor@f$
     */
    virtual void smooth(unsigned char degree, unsigned char stage) = 0;

    /**
     * Make smoothing stage adjustments to the Vertex::Replacement objects
     * referenced from this leaf, given that there is an extraordinary
     * PatchNode of valency @c valency in Direction @c extNode.
     */
    virtual VertexPrecision extraordinarySmooth(unsigned char degree,
                                                unsigned char stage,
                                                Direction extNode,
                                                unsigned char valency) = 0;
    /** @} */

protected:
    /** The PatchTree object that references this PatchTreeLeaf as a child. */
    PatchTree    *parent;

    /**
     * The KnotInterval object specifying the knot spacing across top and
     * bottom (North and South) edges.
     */
    KnotInterval *horizontal;

    /**
     * The KnotInterval object specifying the knot spacing across left and
     * right (West and East) edges.
     */
    KnotInterval *vertical;
};

inline KnotInterval *PatchTreeLeaf::getKnotInterval(Orientation orientation)
    const
{
    switch (orientation)
    {
        case HORIZONTAL:
            return horizontal;
        case VERTICAL:
            return vertical;
    }
    return NULL;
}

inline PatchTree *&PatchTreeLeaf::getParent(void)
{
    return parent;
}

inline bool PatchTreeLeaf::isLeaf(void) const
{
    return true;
}

inline bool PatchTreeLeaf::isRoot(void) const
{
    return false;
}

inline bool PatchTreeLeaf::splits(Orientation /* orientation */) const
{
    return false;
}

inline void PatchTreeLeaf::setParent(PatchTree *newParent)
{
    parent = newParent;
}

}

#endif
