/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef GLWIDGET_H
#define GLWIDGET_H

#include <QGLWidget>
#include <snurbs.h>

class GLWidget : public QGLWidget
{
    Q_OBJECT

public:
    GLWidget(QWidget *parent = 0);
    ~GLWidget(void);

    enum VectorType
    {
        EPS, PDF, PGF
    };

    enum MeshType
    {
        PLY, UVF
    };

    void   getModelViewMatrix   (float *mvMatrix) const;
    void   setModelViewMatrix   (float *mvMatrix);
    GLuint getSelectedFace      (void) const;
    GLuint getSelectedEdge      (void) const;
    void   selectKnotFromDetails(uint face, uint edge);

public slots:
    void loadFile           (const  QString &fileName);
    void saveFile           (const  QString &fileName, MeshType   type);
    void saveVectorImage    (const  QString &fileName, VectorType type);
    void changeSubdivLevel  (int    newLevel);
    void changeDegree       (int    newDegree);
    void changeKnotSpacing  (double newInterval);
    void changeStripeDensity(int    newDensity, bool update = true);
    void changeIsoparamSubd (int    newSubdivs, bool update = true);
    void setWireframe       (void);
    void setFlat            (void);
    void setSmooth          (void);
    void setIsophotes       (void);
    void setIsoparams       (void);
    void drawControlMesh    (bool   enabled);

signals:
    void knotSpacingChanged (double newInterval);
    void subdivLevelChanged (int    newLevel);
    void numVertsChanged    (int    numVerts);
    void numFacesChanged    (int    numFaces);
    void numKnotIntsChanged (int    numKnots);

protected:
    void initializeGL       (void);
    void paintGL            (void);
    void resizeGL           (int width, int height);

    void mousePressEvent    (QMouseEvent *event);
    void mouseMoveEvent     (QMouseEvent *event);
    void wheelEvent         (QWheelEvent *event);

private:
    void drawSelControl     (bool selected);
    void sphereProject      (float x, float y, float (&v)[3]);
    void setScale           (double newScale);
    void selectVertex       (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
    void selectFace         (void);
    void vertexCallback     (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
    void wireframeVertex    (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
    void wireframeFace      (void);
    void texCoordCallback   (GLfloat s, GLfloat t);
    void makeControlObject  (void);
    void makeSubdivObject   (void);
    void deleteSubdivObject (void);

    enum ShadeModel
    {
        WIREFRAME, FLAT, SMOOTH, ISOPHOTES, ISOPARAMS
    };

    snurbs::Mesh          *lowDegreeMesh;
    snurbs::Mesh          *controlMesh;
    GLuint                 controlObject;
    snurbs::Mesh          *subdivMesh;
    GLuint                 subdivObject;
    GLuint                 isoparamTexture;
    bool                   texFirstFace;
    GLfloat                firstS;
    GLfloat                firstT;
    float                  max[3];
    float                  min[3];
    float                  median[3];
    float                  boundingCube;
    int                    nVerts;
    GLuint                 selectedFace;
    GLuint                 selectedEdge;

    double                 selectCloseLoop[4];
    bool                   selectFirstVert;
    snurbs::KnotInterval  *selectedKnot;
    snurbs::KnotInterval  *lowDegreeKnot;

    int           subdivLevel;
    int           subdivDegree;
    ShadeModel    shadeModel;
    bool          drawControl;
    float         scale;
    QPoint        lastPos;

    float         lightAmbient[4];
    float         lightDiffuse[4];
    float         lightSpecular[4];
    float         lightPosition[4];

    std::vector<GLdouble> wireframeVerts;

    static const double minScale;
    static const double scaleDragSpeed;
    static const double scaleWheelSpeed;
    static const double rotateSpeed;
    static const double backClipPlane;
    static const double stepBackAmount;
};

inline GLuint GLWidget::getSelectedFace(void) const
{
    return selectedFace;
}

inline GLuint GLWidget::getSelectedEdge(void) const
{
    return selectedEdge;
}

#endif
