/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cassert>
#include <cmath>
#include <QtGui>
#include <QtOpenGL>
#include "gl2ps/gl2ps.h"
#include "glwidget.h"
#include "knotinterval.h"

const double GLWidget::minScale        = 0.05;
const double GLWidget::scaleDragSpeed  = 1 / 200.0;
const double GLWidget::scaleWheelSpeed = 1 / 800.0;
const double GLWidget::rotateSpeed     = 90.0;
const double GLWidget::backClipPlane   = 15.0;
const double GLWidget::stepBackAmount  = 10.0;

using namespace snurbs;
using namespace std::tr1::placeholders;

GLWidget::GLWidget(QWidget *parent) : QGLWidget(parent)
{
    controlObject = 0;
    subdivObject  = 0;
    scale         = 1;
    subdivDegree  = 3;
    shadeModel    = FLAT;
    controlMesh   = new Mesh(subdivDegree);
    subdivMesh    = NULL;
    lowDegreeMesh = NULL;
    selectedKnot  = NULL;
    lowDegreeKnot = NULL;
    drawControl   = false;
    selectedFace  = 0xffffffff;
    selectedEdge  = 0xffffffff;

    median[0]     = 0;
    median[1]     = 0;
    median[2]     = 0;

    lightAmbient[0]  = 0.6;
    lightAmbient[1]  = 0.6;
    lightAmbient[2]  = 0.6;
    lightAmbient[3]  = 1;

    lightDiffuse[0]  = 0.8;
    lightDiffuse[1]  = 0.8;
    lightDiffuse[2]  = 0.8;
    lightDiffuse[3]  = 1;

    lightPosition[0] = -1;
    lightPosition[1] = 1;
    lightPosition[2] = 2;
    lightPosition[3] = 0;

    lightSpecular[0] = 1;
    lightSpecular[1] = 1;
    lightSpecular[2] = 1;
    lightSpecular[3] = 1;
}

GLWidget::~GLWidget(void)
{
    deleteSubdivObject();
    if (glIsList(controlObject))
    {
        glDeleteLists(controlObject, 1);
    }
    if (lowDegreeMesh != NULL)
    {
        delete lowDegreeMesh;
        lowDegreeMesh = NULL;
    }
}

void GLWidget::getModelViewMatrix(float *mvMatrix) const
{
    glGetFloatv(GL_MODELVIEW_MATRIX, mvMatrix);
}

void GLWidget::setModelViewMatrix(float *mvMatrix)
{
    glLoadIdentity();
    glMultMatrixf(mvMatrix);
}

void GLWidget::selectKnotFromDetails(uint face, uint edge)
{
    MeshFlattener *meshFlattener = MeshFlattener::create(*controlMesh);
    selectedKnot = meshFlattener->getKnotIntervalForEdge(face, edge);
    emit knotSpacingChanged(selectedKnot->getInterval());

    if (lowDegreeMesh != NULL)
    {
        MeshFlattener::dispose(meshFlattener);
        meshFlattener = MeshFlattener::create(*lowDegreeMesh);
        lowDegreeKnot = meshFlattener->getFlattenedKnotNumber(
                                    controlMesh->getKnotIndex(selectedKnot));
    }

    makeControlObject();
    updateGL();
    MeshFlattener::dispose(meshFlattener);
}

void GLWidget::loadFile(const QString &fileName)
{
    deleteSubdivObject();
    if (lowDegreeMesh != NULL)
    {
        delete lowDegreeMesh;
        lowDegreeMesh = NULL;
    }
    delete controlMesh;
    controlMesh = new Mesh(subdivDegree);
    MeshBuilder *meshBuilder = MeshBuilder::create(*controlMesh);
    PlyReader *plyReader = new PlyReader(*meshBuilder);

    try
    {
        plyReader->read(fileName.toStdString());
    }
    catch (std::exception &e)
    {
        MeshBuilder::dispose(meshBuilder);
        delete plyReader;
        delete subdivMesh;
        controlMesh->clear();
        subdivMesh = new Mesh(*controlMesh);
        makeControlObject();
        makeSubdivObject();
        emit subdivLevelChanged(0);
        emit numVertsChanged(0);
        emit numFacesChanged(0);
        emit numKnotIntsChanged(0);
        emit knotSpacingChanged(-1);
        updateGL();
        QMessageBox::critical(this, "Error loading file", e.what());
        return;
    }

    MeshBuilder::dispose(meshBuilder);
    makeControlObject();

    if (controlMesh->hasBoundary())
    {
        if (lowDegreeMesh == NULL)
        {
            lowDegreeMesh = new Mesh(3);
        }
        else
        {
            lowDegreeMesh->clear();
        }

        meshBuilder = MeshBuilder::create(*lowDegreeMesh);
        delete plyReader;
        plyReader = new PlyReader(*meshBuilder);
        plyReader->read(fileName.toStdString());
        MeshBuilder::dispose(meshBuilder);

        // This extra copy gets the order of the KnotInterval objects correct
        // in the Mesh::knots vectors, so that mousePressEvent can work
        controlMesh->clear();
        meshBuilder = MeshBuilder::create(*controlMesh);
        MeshFlattener *flattener = MeshFlattener::create(*lowDegreeMesh);
        flattener->flatten(*meshBuilder);
        MeshFlattener::dispose(flattener);
        MeshBuilder::dispose(meshBuilder);
    }

    delete plyReader;
    delete subdivMesh;
    subdivMesh  = new Mesh(*controlMesh);
    subdivLevel = 0;
    selectedKnot = NULL;
    lowDegreeKnot = NULL;
    emit subdivLevelChanged(subdivLevel);
    emit numVertsChanged(subdivMesh->getNumVerts());
    emit numFacesChanged(subdivMesh->getNumFaces());
    emit numKnotIntsChanged(subdivMesh->getNumKnots());
    emit knotSpacingChanged(-1);
    glLoadIdentity();
    makeSubdivObject();
    if (boundingCube <= 0)
    {
        boundingCube = 1;
        median[0] = 0;
        median[1] = 0;
        median[2] = 0;
    }
    scale = 0.75 / boundingCube;
    glScalef(scale, scale, scale);
    glTranslatef(-median[0], -median[1], -median[2]);
    updateGL();
}

void GLWidget::saveFile(const QString &fileName, MeshType type)
{
    if (subdivMesh != NULL)
    {
        std::ofstream file(fileName.toAscii().constData());
        MeshFlattener *meshFlattener = MeshFlattener::create(*subdivMesh);
        meshFlattener->setCompatible(true);

        FlatMeshHandler *handler = NULL;
        switch (type)
        {
            case PLY:
                handler = new PlyWriter(&file);
                break;
            case UVF:
                handler = new UvfWriter(&file);
        }
        meshFlattener->flatten(*handler);
        delete handler;

        MeshFlattener::dispose(meshFlattener);
        file.close();
    }
}

void GLWidget::saveVectorImage(const QString &fileName, VectorType type)
{
    FILE *fp;
    int state = GL2PS_OVERFLOW, buffsize = 2 * 1024 * 1024;
    int format = 0;

    switch (type)
    {
        case PDF:
            format = GL2PS_PDF;
            break;
        case EPS:
            format = GL2PS_EPS;
            break;
        case PGF:
            format = GL2PS_PGF;
            break;
    }

    glEnable(GL_CULL_FACE);
    if (shadeModel == WIREFRAME)
    {
        glDisable(GL_STENCIL_TEST);
    }

    fp = fopen(fileName.toAscii().constData(), "wb");
    while(state == GL2PS_OVERFLOW)
    {
        gl2psBeginPage("SubdNURBS Surface", "SubdNURBS", NULL, format,
                       GL2PS_SIMPLE_SORT,
                       GL2PS_TIGHT_BOUNDING_BOX,
                       GL_RGBA, 0, NULL, 0, 0, 0,
                       buffsize, fp, fileName.toAscii().constData());
        paintGL();
        state = gl2psEndPage();
        buffsize *= 2;
    }
    fclose(fp);

    glDisable(GL_CULL_FACE);
    if (shadeModel == WIREFRAME)
    {
        glEnable(GL_STENCIL_TEST);
    }
}

void GLWidget::changeSubdivLevel(int newLevel)
{
    if (newLevel != subdivLevel && subdivMesh != NULL)
    {
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        if (newLevel < subdivLevel)
        {
            delete subdivMesh;
            subdivMesh = new Mesh(*controlMesh);
            subdivLevel = 0;
        }

        while (subdivLevel < newLevel)
        {
            subdivMesh->subdivide();
            ++subdivLevel;
        }

        float oldBC = boundingCube;
        float oldMedian[3];
        oldMedian[0] = median[0];
        oldMedian[1] = median[1];
        oldMedian[2] = median[2];
        deleteSubdivObject();
        makeSubdivObject();
        if (boundingCube <= 0)
        {
            boundingCube = oldBC;
            median[0] = oldMedian[0];
            median[1] = oldMedian[1];
            median[2] = oldMedian[2];
        }
        makeCurrent();
        glTranslatef(oldMedian[0] - median[0],
                     oldMedian[1] - median[1],
                     oldMedian[2] - median[2]);
        setScale(scale * oldBC / boundingCube);
        updateGL();
        emit subdivLevelChanged(subdivLevel);
        emit numVertsChanged(subdivMesh->getNumVerts());
        emit numFacesChanged(subdivMesh->getNumFaces());
        emit numKnotIntsChanged(subdivMesh->getNumKnots());
        QApplication::restoreOverrideCursor();
    }
}

void GLWidget::changeDegree(int newDegree)
{
    if (newDegree != subdivDegree && subdivMesh != NULL)
    {
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        Mesh *copyFromMesh;
        if (lowDegreeMesh != NULL)
        {
            copyFromMesh = lowDegreeMesh;
        }
        else
        {
            copyFromMesh = controlMesh;
        }
        Mesh *newControlMesh = new Mesh(newDegree);

        MeshBuilder *meshBuilder = MeshBuilder::create(*newControlMesh);
        MeshFlattener *flattener = MeshFlattener::create(*copyFromMesh);
        flattener->flatten(*meshBuilder);
        MeshFlattener::dispose(flattener);
        MeshBuilder::dispose(meshBuilder);

        delete controlMesh;
        controlMesh = newControlMesh;
        delete subdivMesh;
        subdivMesh = new Mesh(*controlMesh);

        int level = subdivLevel;

        while (level)
        {
            subdivMesh->subdivide();
            --level;
        }

        float oldBC = boundingCube;
        float oldMedian[3];
        oldMedian[0] = median[0];
        oldMedian[1] = median[1];
        oldMedian[2] = median[2];
        deleteSubdivObject();
        makeSubdivObject();
        selectedKnot = NULL;
        lowDegreeKnot = NULL;
        emit knotSpacingChanged(-1);
        makeControlObject();
        if (lowDegreeMesh != NULL)
        {
            emit numVertsChanged(subdivMesh->getNumVerts());
            emit numFacesChanged(subdivMesh->getNumFaces());
        }
        if (boundingCube <= 0)
        {
            boundingCube = oldBC;
            median[0] = oldMedian[0];
            median[1] = oldMedian[1];
            median[2] = oldMedian[2];
        }
        makeCurrent();
        glTranslatef(oldMedian[0] - median[0],
                     oldMedian[1] - median[1],
                     oldMedian[2] - median[2]);
        setScale(scale * oldBC / boundingCube);
        updateGL();
        QApplication::restoreOverrideCursor();
    }

    subdivDegree = newDegree;
}

void GLWidget::changeKnotSpacing(double newInterval)
{
    if (selectedKnot != NULL && selectedKnot->getInterval() != newInterval)
    {
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        selectedKnot->setInterval(newInterval);
        if (lowDegreeKnot != NULL)
        {
            lowDegreeKnot->setInterval(newInterval);
        }
        delete subdivMesh;
        subdivMesh = new Mesh(*controlMesh);
        int level = subdivLevel;

        while (level)
        {
            subdivMesh->subdivide();
            --level;
        }
        deleteSubdivObject();
        makeSubdivObject();
        emit numVertsChanged(subdivMesh->getNumVerts());
        emit numFacesChanged(subdivMesh->getNumFaces());
        emit numKnotIntsChanged(subdivMesh->getNumKnots());
        //emit knotSpacingChanged(newInterval);
        updateGL();
        QApplication::restoreOverrideCursor();
    }
}

void GLWidget::changeStripeDensity(int newDensity, bool update)
{
    static const int stripeRes = 1024;
    static const double pi = 3.14159265358979323846;
    GLfloat stripes[stripeRes];
    for (int i = 0; i < stripeRes; ++i)
    {
        stripes[i] = 0.5 + sin(2 * pi * i * newDensity / stripeRes);
        stripes[i] = (stripes[i] < 0 ? 0 : stripes[i]);
        stripes[i] = (stripes[i] > 1 ? 1 : stripes[i]);
    }
    glTexImage1D(GL_TEXTURE_1D, 0, GL_LUMINANCE, stripeRes,
                 0, GL_LUMINANCE, GL_FLOAT, stripes);

    if (update)
    {
        updateGL();
    }
}

void GLWidget::changeIsoparamSubd(int newSubdivs, bool update)
{
    static const int texRes = 256;
    GLfloat tex[texRes][texRes];

    glBindTexture(GL_TEXTURE_2D, isoparamTexture);

    for (int i = 0; i < texRes; ++i)
    {
        for (int j = 0; j < texRes; ++j)
        {
            tex[i][j] = 1;
        }
    }

    for (int i = 0; i < 3; ++i)
    {
        for (int j = 0; j < texRes; ++j)
        {
            tex[i][j] = 0;
            tex[j][i] = 0;
        }
    }

    for (int i = 252; i < texRes; ++i)
    {
        for (int j = 0; j < texRes; ++j)
        {
            tex[i][j] = 0;
            tex[j][i] = 0;
        }
    }

    if (newSubdivs > 0)
    {
        for (int i = 127; i < 129; ++i)
        {
            for (int j = 0; j < texRes; ++j)
            {
                tex[i][j] = 0;
                tex[j][i] = 0;
            }
        }
    }

    if (newSubdivs > 1)
    {
        for (int i = 64; i < 66; ++i)
        {
            for (int j = 0; j < texRes; ++j)
            {
                tex[i][j] = 0;
                tex[j][i] = 0;
            }
        }

        for (int i = 190; i < 192; ++i)
        {
            for (int j = 0; j < texRes; ++j)
            {
                tex[i][j] = 0;
                tex[j][i] = 0;
            }
        }
    }

    glTexImage2D(GL_TEXTURE_2D, 0, GL_LUMINANCE, texRes, texRes,
                 0, GL_LUMINANCE, GL_FLOAT, tex);

    if (update)
    {
        updateGL();
    }
}

void GLWidget::setWireframe(void)
{
    if (shadeModel != WIREFRAME)
    {
        shadeModel = WIREFRAME;
        glShadeModel(GL_SMOOTH);
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        glDisable(GL_LIGHTING);
        glDisable(GL_TEXTURE_1D);
        glDisable(GL_TEXTURE_2D);
        glEnable(GL_STENCIL_TEST);

        deleteSubdivObject();
        makeSubdivObject();
        updateGL();
    }
}

void GLWidget::setFlat(void)
{
    if (shadeModel != FLAT)
    {
        shadeModel = FLAT;
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glEnable(GL_LIGHTING);
        glDisable(GL_TEXTURE_1D);
        glDisable(GL_TEXTURE_2D);
        glDisable(GL_STENCIL_TEST);
        glShadeModel(GL_FLAT);

        deleteSubdivObject();
        makeSubdivObject();
        updateGL();
    }
}

void GLWidget::setSmooth(void)
{
    if (shadeModel != SMOOTH)
    {
        shadeModel = SMOOTH;
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glEnable(GL_LIGHTING);
        glDisable(GL_TEXTURE_1D);
        glDisable(GL_TEXTURE_2D);
        glDisable(GL_STENCIL_TEST);
        glShadeModel(GL_SMOOTH);

        deleteSubdivObject();
        makeSubdivObject();
        updateGL();
    }
}

void GLWidget::setIsophotes(void)
{
    if (shadeModel != ISOPHOTES)
    {
        shadeModel = ISOPHOTES;
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glEnable(GL_LIGHTING);
        glDisable(GL_STENCIL_TEST);
        glShadeModel(GL_SMOOTH);
        glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
        glDisable(GL_TEXTURE_2D);

        glEnable(GL_TEXTURE_1D);
        glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

        deleteSubdivObject();
        makeSubdivObject();
        updateGL();
    }
}

void GLWidget::setIsoparams(void)
{
    if (shadeModel != ISOPARAMS)
    {
        shadeModel = ISOPARAMS;
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glEnable(GL_LIGHTING);
        glDisable(GL_STENCIL_TEST);
        glShadeModel(GL_SMOOTH);
        glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
        glDisable(GL_TEXTURE_1D);

        glEnable(GL_TEXTURE_2D);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

        deleteSubdivObject();
        makeSubdivObject();
        updateGL();
    }
}

void GLWidget::drawControlMesh(bool enabled)
{
    if (drawControl != enabled)
    {
        drawControl = enabled;
        if (selectedKnot)
        {
            selectedKnot = NULL;
            lowDegreeKnot = NULL;
            emit knotSpacingChanged(-1);
            makeControlObject();
        }
        updateGL();
    }
}

void GLWidget::initializeGL(void)
{
    glClearColor(1, 1, 1, 0);
    glShadeModel(GL_FLAT);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glEnable(GL_NORMALIZE);
    glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
    glLightfv(GL_LIGHT0, GL_AMBIENT,  lightAmbient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE,  lightDiffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, lightSpecular);
    glEnable (GL_LIGHT0);
    glLoadIdentity();
    changeStripeDensity(8, false);
    glGenTextures(1, &isoparamTexture);
    changeIsoparamSubd(2, false);
    swapBuffers();
}

void GLWidget::paintGL(void)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glPushMatrix();
    // Pre-multiply by translation
    GLdouble currentMatrix[16] = {0};
    glGetDoublev(GL_MODELVIEW_MATRIX, currentMatrix);
    glLoadIdentity();
    glTranslated(0.0, 0.0, -stepBackAmount);
    glMultMatrixd(currentMatrix);
    if (!drawControl || subdivLevel > 0)
    {
        if (shadeModel == ISOPARAMS)
        {
            glTexCoord2f(firstS, firstT);
            glBindTexture(GL_TEXTURE_2D, isoparamTexture);
        }
        glCallList(subdivObject);
    }
    if (drawControl)
    {
        glPushAttrib(GL_ALL_ATTRIB_BITS);

        glDisable(GL_TEXTURE_1D);
        glDisable(GL_TEXTURE_2D);
        glEnable(GL_BLEND);
        gl2psEnable(GL2PS_BLEND);
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glEnable(GL_LIGHTING);
        glDisable(GL_STENCIL_TEST);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        gl2psBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        glDepthMask(GL_FALSE);

        GLfloat params[4];
        params[0] = 0.3; params[1] = 0.3; params[2] = 0.4; params[3] = 1;
        glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, params);
        params[0] = 0; params[1] = 0; params[2] = 0; params[3] = 0.2;
        glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, params);
        glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, params);

        glCallList(controlObject);

        glDisable(GL_BLEND);
        gl2psDisable(GL2PS_BLEND);
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        glDisable(GL_LIGHTING);
        glShadeModel(GL_SMOOTH);
        qglColor(QColor("#666"));

        glCallList(controlObject);

        glPopAttrib();
    }
    glPopMatrix();
}

void GLWidget::resizeGL(int width, int height)
{
    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    double dbMin = 2 * qMin(width, height);
    glOrtho(-width  / dbMin, width  / dbMin,
            -height / dbMin, height / dbMin, 0, backClipPlane);
    glMatrixMode(GL_MODELVIEW);
}

void GLWidget::mousePressEvent(QMouseEvent *event)
{
    if (drawControl && (event->buttons() & Qt::RightButton))
    {
        static const unsigned int BUFSIZE = 64;
        GLuint selectBuf[BUFSIZE];
        glSelectBuffer(BUFSIZE, selectBuf);
        glRenderMode(GL_SELECT);
        glInitNames();
        glPushName(0);

        GLint viewport[4];
        glGetIntegerv(GL_VIEWPORT, viewport);
        glPushMatrix();
        // Pre-multiply by translation
        GLdouble currentMatrix[16] = {0};
        glGetDoublev(GL_MODELVIEW_MATRIX, currentMatrix);
        glLoadIdentity();
        glTranslated(0.0, 0.0, -stepBackAmount);
        glMultMatrixd(currentMatrix);
        glMatrixMode(GL_PROJECTION);
        glPushMatrix();
        glLoadIdentity();
        gluPickMatrix(event->x(), height() - event->y(), 10.0, 10.0, viewport);
        double dbMin = 2 * qMin(width(), height());
        glOrtho(-width()  / dbMin, width()  / dbMin,
                -height() / dbMin, height() / dbMin, 0, backClipPlane);

        MeshFlattener *meshFlattener = MeshFlattener::create(*controlMesh);
        meshFlattener->setCompatible(true);
        selectFirstVert = true;
        meshFlattener->receiveVertices(
            std::tr1::bind(&GLWidget::selectVertex, this, _1, _2, _3, _4));
        meshFlattener->receiveFlatNormals(
            std::tr1::bind(&GLWidget::selectFace,   this));
        meshFlattener->flattenFaces();
        selectFace();
        glPopName();
        glPopName();

        glPopMatrix();
        GLint hits = glRenderMode(GL_RENDER);
        glMatrixMode(GL_MODELVIEW);
        glPopMatrix();

        GLuint minZ         = 0xffffffff;
        selectedFace        = 0xffffffff;
        selectedEdge        = 0xffffffff;

        for (GLint i = 0; i < hits; ++i)
        {
            assert(selectBuf[i*5] == 2);
            if (selectBuf[i*5 + 1] < minZ)
            {
                minZ = selectBuf[i*5 + 1];
                selectedFace = selectBuf[i*5 + 3];
                selectedEdge = selectBuf[i*5 + 4];
            }
        }

        if (selectedFace != 0xffffffff)
        {
            selectedKnot = meshFlattener->getKnotIntervalForEdge(selectedFace,
                                                                 selectedEdge);
            emit knotSpacingChanged(selectedKnot->getInterval());

            if (lowDegreeMesh != NULL)
            {
                MeshFlattener::dispose(meshFlattener);
                meshFlattener = MeshFlattener::create(*lowDegreeMesh);
                lowDegreeKnot = meshFlattener->getFlattenedKnotNumber(
                                    controlMesh->getKnotIndex(selectedKnot));
            }
        }
        else
        {
            selectedKnot  = NULL;
            lowDegreeKnot = NULL;
            emit knotSpacingChanged(-1);
        }
        makeControlObject();
        updateGL();
        MeshFlattener::dispose(meshFlattener);
    }

    lastPos = event->pos();
}

void GLWidget::drawSelControl(bool selected)
{
    glEnd();

    GLfloat params[4] = {0.3, 0.3, 0.3, 1};

    if (selected)
    {
        params[0] = 1;
        params[1] = 0;
        params[2] = 0;
    }
    else
    {
        params[2] = 0.4;
    }
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, params);
    glBegin(GL_QUADS);
}

void GLWidget::sphereProject(float x, float y, float (&v)[3])
{
    float r = qMin(width(), height());
    v[0] = (2 * x  - width()) / r;
    v[1] = (height() - 2 * y) / r;
    float d = v[0] * v[0] + v[1] * v[1];
    if (d > 0.5)
    {
        v[2] = 0.5 / sqrt(d);
    }
    else
    {
        v[2] = sqrt(1 - d);
    }
}

void GLWidget::mouseMoveEvent(QMouseEvent *event)
{
    if (event->buttons() & Qt::LeftButton)
    {
        float oldPos[3], newPos[3], axis[3], diff[3];

        sphereProject(lastPos.x(), lastPos.y(), oldPos);
        sphereProject( event->x(),  event->y(), newPos);

        axis[0] = oldPos[1] * newPos[2] - oldPos[2] * newPos[1];
        axis[1] = oldPos[2] * newPos[0] - oldPos[0] * newPos[2];
        axis[2] = oldPos[0] * newPos[1] - oldPos[1] * newPos[0];

        diff[0] = oldPos[0] - newPos[0];
        diff[1] = oldPos[1] - newPos[1];
        diff[2] = oldPos[2] - newPos[2];

        // Pre-multiply by rotation matrix
        GLdouble currentMatrix[16] = {0};
        makeCurrent();
        glGetDoublev(GL_MODELVIEW_MATRIX, currentMatrix);
        glLoadIdentity();
        glRotatef(rotateSpeed *
                  sqrt(diff[0]*diff[0] + diff[1]*diff[1] + diff[2]*diff[2]),
                  axis[0], axis[1], axis[2]);
        glMultMatrixd(currentMatrix);
        updateGL();
    }
    else if (event->buttons() & Qt::RightButton)
    {
        if (!drawControl)
        {
            int dy = event->y() - lastPos.y();
            setScale(scale - dy * scaleDragSpeed / boundingCube);
        }
    }
    lastPos = event->pos();
}

void GLWidget::wheelEvent(QWheelEvent *event)
{
    setScale(scale + event->delta() * scaleWheelSpeed / boundingCube);
}

void GLWidget::setScale(double newScale)
{
    if (newScale < minScale * (1 / boundingCube))
    {
        newScale = minScale * (1 / boundingCube);
    }

    if (newScale != scale)
    {
        double ratio = newScale / scale;

        // Pre-multiply by scaling matrix
        GLdouble currentMatrix[16] = {0};
        makeCurrent();
        glGetDoublev(GL_MODELVIEW_MATRIX, currentMatrix);
        glLoadIdentity();
        glScaled(ratio, ratio, ratio);
        glMultMatrixd(currentMatrix);

        updateGL();
        scale = newScale;
    }
}

void GLWidget::selectVertex(GLdouble x, GLdouble y, GLdouble z, GLdouble w)
{
    static GLuint edgeName = 0;

    if (selectFirstVert)
    {
        selectCloseLoop[0] = x;
        selectCloseLoop[1] = y;
        selectCloseLoop[2] = z;
        selectCloseLoop[3] = w;
        selectFirstVert = false;
        edgeName = 0;
    }
    else
    {
        glVertex4d(x, y, z, w);
        glEnd();
    }
    glLoadName(edgeName);
    ++edgeName;
    glBegin(GL_LINES);
    glVertex4d(x, y, z, w);
}

void GLWidget::selectFace(void)
{
    static GLuint faceName = 0;

    if (selectFirstVert == false)
    {
        glVertex4dv(selectCloseLoop);
        glEnd();
        glPopName();
        selectFirstVert = true;
        ++faceName;
    }
    else
    {
        faceName = 0;
    }

    glLoadName(faceName);
    glPushName(0);
}

void GLWidget::vertexCallback(GLdouble x, GLdouble y, GLdouble z, GLdouble w)
{
    glVertex4d(x, y, z, w);

    if (x > max[0]) max[0] = x;
    if (y > max[1]) max[1] = y;
    if (z > max[2]) max[2] = z;

    if (x < min[0]) min[0] = x;
    if (y < min[1]) min[1] = y;
    if (z < min[2]) min[2] = z;

    median[0] += x;
    median[1] += y;
    median[2] += z;

    ++nVerts;
}

void GLWidget::wireframeVertex(GLdouble x, GLdouble y, GLdouble z, GLdouble w)
{
    vertexCallback(x, y, z, w);

    wireframeVerts.push_back(x);
    wireframeVerts.push_back(y);
    wireframeVerts.push_back(z);
    wireframeVerts.push_back(w);
}

void GLWidget::wireframeFace(void)
{
    if (!wireframeVerts.empty())
    {
        std::vector<GLdouble>::iterator iter = wireframeVerts.begin();

        glEnd();

        glColor3f(1, 1, 1);
        glStencilFunc(GL_EQUAL, 0, 1);
        glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

        glBegin(GL_QUADS);

        while (iter != wireframeVerts.end())
        {
            glVertex4d(*iter, *(iter + 1), *(iter + 2), *(iter + 3));
            iter += 4;
        }

        glEnd();

        glColor3f(0, 0, 0);
        glStencilFunc(GL_ALWAYS, 0, 1);
        glStencilOp(GL_INVERT, GL_INVERT, GL_INVERT);
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

        glBegin(GL_QUADS);

        iter = wireframeVerts.begin();
        while (iter != wireframeVerts.end())
        {
            glVertex4d(*iter, *(iter + 1), *(iter + 2), *(iter + 3));
            iter += 4;
        }
        wireframeVerts.clear();
    }
}

void GLWidget::texCoordCallback(GLfloat s, GLfloat t)
{
    // This works around an annoying bug in one GL implementation (on aneto).
    // It seems that if the first call to glTexCoord is before the first call
    // to glVertex within a display list, then all texture coordinates get set
    // to (0, 0). So we just store the first texture coordinates, and apply
    // them in paintGL() instead (i.e. outside of the display list).
    if (!texFirstFace)
    {
        glTexCoord2f(s, t);
    }
    else
    {
        firstS = s;
        firstT = t;
        texFirstFace = false;
    }
}

void GLWidget::makeSubdivObject(void)
{
    makeCurrent();
    subdivObject = glGenLists(1);
    glNewList(subdivObject, GL_COMPILE);

    if (subdivMesh != NULL)
    {
        if (shadeModel == WIREFRAME)
        {
            glClear(GL_STENCIL_BUFFER_BIT);
            glStencilFunc(GL_ALWAYS, 0, 1);
            glStencilOp(GL_INVERT, GL_INVERT, GL_INVERT);
            glColor3f(0, 0, 0);
        }
        else
        {
            GLfloat specComp[4] = {0.25, 0.25, 0.25, 1};
            glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specComp);
            glMateriali(GL_FRONT_AND_BACK, GL_SHININESS, 30);

            GLfloat ambientParams[4] = {0.2, 0.2, 0.2, 1};
            if (shadeModel == ISOPHOTES || shadeModel == ISOPARAMS)
            {
                ambientParams[0] = 0.6;
                ambientParams[1] = 0.6;
                ambientParams[2] = 0.6;
            }
            glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, ambientParams);
            GLfloat diffuseParams[4] = {0.8, 0.8, 0.8, 1};
            glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diffuseParams);
        }

        nVerts    = 0;
        max[0]    = -std::numeric_limits<float>::infinity();
        max[1]    = -std::numeric_limits<float>::infinity();
        max[2]    = -std::numeric_limits<float>::infinity();
        min[0]    =  std::numeric_limits<float>::infinity();
        min[1]    =  std::numeric_limits<float>::infinity();
        min[2]    =  std::numeric_limits<float>::infinity();
        median[0] = 0; median[1] = 0; median[2] = 0;

        MeshFlattener *meshFlattener = MeshFlattener::create(*subdivMesh);
        meshFlattener->setCompatible(true);

        if (shadeModel == ISOPHOTES)
        {
            glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
            glEnable(GL_TEXTURE_GEN_S);
        }
        else
        {
            glDisable(GL_TEXTURE_GEN_S);
        }

        glBegin(GL_QUADS);

        switch (shadeModel)
        {
            case WIREFRAME:
                wireframeVerts.reserve(16);
                meshFlattener->receiveVertices(
                    std::tr1::bind(&GLWidget::wireframeVertex,
                                   this, _1, _2, _3, _4));
                meshFlattener->receiveFlatNormals(
                    std::tr1::bind(&GLWidget::wireframeFace, this));
                break;

            case FLAT:
                meshFlattener->receiveVertices(
                    std::tr1::bind(&GLWidget::vertexCallback, this,
                                                              _1, _2, _3, _4));
                meshFlattener->receiveFlatNormals(&glNormal3d);
                break;

            case ISOPARAMS:
                texFirstFace = true;
                meshFlattener->receivePatchCoords(
                    std::tr1::bind(&GLWidget::texCoordCallback, this, _1, _2));
            case ISOPHOTES:
            case SMOOTH:
                meshFlattener->receiveVertices(
                    std::tr1::bind(&GLWidget::vertexCallback, this,
                                                              _1, _2, _3, _4));
                meshFlattener->receiveSmoothNormals(&glNormal3d);
                break;
        }

        meshFlattener->flattenFaces();

        if (shadeModel == WIREFRAME)
        {
            wireframeFace();
        }

        glEnd();

        MeshFlattener::dispose(meshFlattener);

        // Actually calculates a mean of face medians, rather than a mean of
        // vertices. But that's good enough for our purposes.
        median[0] /= nVerts;
        median[1] /= nVerts;
        median[2] /= nVerts;

        boundingCube = max[0] - min[0];
        if (max[1] - min[1] > boundingCube) boundingCube = max[1] - min[1];
        if (max[2] - min[2] > boundingCube) boundingCube = max[2] - min[2];
    }

    glEndList();
}

void GLWidget::makeControlObject(void)
{
    makeCurrent();
    if (glIsList(controlObject))
    {
        glDeleteLists(controlObject, 1);
    }
    controlObject = glGenLists(1);
    glNewList(controlObject, GL_COMPILE);
    MeshFlattener *meshFlattener = MeshFlattener::create(*controlMesh);
    meshFlattener->setCompatible(true);
    if (selectedKnot != NULL)
    {
        meshFlattener->notifyOnKnot(selectedKnot,
                                    std::tr1::bind(&GLWidget::drawSelControl,
                                                   this, true),
                                    std::tr1::bind(&GLWidget::drawSelControl,
                                                   this, false));
    }
    glBegin(GL_QUADS);
    meshFlattener->receiveVertices   (&glVertex4d);
    meshFlattener->receiveFlatNormals(&glNormal3d);
    meshFlattener->flattenFaces();
    glEnd();
    MeshFlattener::dispose(meshFlattener);
    glEndList();
}

void GLWidget::deleteSubdivObject(void)
{
    makeCurrent();
    if (glIsList(subdivObject))
    {
        glDeleteLists(subdivObject, 1);
    }
}
