/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef VERTEX_H
#define VERTEX_H

#include <ostream>
#include "types.h"

/** @file */
// This file appears in the documentation because of the global function
// std::ostream &operator<<(std::ostream &os, const Vertex &vertex)
// defined here

namespace snurbs {

/**
 * A 4D Vertex with x, y and z co-ordinates as well as a rational weight.
 *
 * @ingroup Mesh
 */
class Vertex
{
    friend std::ostream &operator<<(std::ostream &os, const Vertex &vertex);

public:
    /**
     * Vertex constructor.
     * @param x x co-ordinate
     * @param y y co-ordinate
     * @param z z co-ordinate
     * @param w w co-ordinate (i.e. rational weight).
     */
    Vertex(VertexPrecision x,
           VertexPrecision y,
           VertexPrecision z,
           VertexPrecision w);

    /**
     * Vertex constructor with no arguments: initialises vertex to (0, 0, 0, 1)
     * and also assigns a #replacement
     */
    Vertex(void);

    /** Vertex destructor: deletes #replacement if it exists. */
    ~Vertex(void);

    /**
     * @name Getter functions
     * @{
     */
    /** Returns the X co-ordinate for this vertex. */
    VertexPrecision getX  (void) const;
    /** Returns the Y co-ordinate for this vertex. */
    VertexPrecision getY  (void) const;
    /** Returns the Z co-ordinate for this vertex. */
    VertexPrecision getZ  (void) const;
    /** Returns the W co-ordinate (i.e.\ rational weight) for this vertex. */
    VertexPrecision getW  (void) const;
    /** Returns @c true if this vertex carries a geometric position. */
    bool            isReal(void) const;
    /** @} */

    /**
     * Called if this vertex has fallen `off the edge' of a boundary.
     */
    void setNotReal(void);

    /**
     * @name Operators
     * @{
     */
    /** Vector subtraction. */
    const Vertex operator-(const Vertex &v) const;
    /** Vector addition. */
    const Vertex operator+(const Vertex &v) const;
    /**
     * Vector cross-product in 3D (requires both operands to have w=0).
     * @param v The right operand
     */
    const Vertex operator*(const Vertex &v) const;
    /** @} */

    /**
     * @return @c true if #replacement is not equal to NULL, @c false otherwise
     */
    bool hasReplacement(void) const;

    /**
     * Throws away the replacement position held in #replacement.
     */
    void discardReplacement(void);

    /**
     * @return the denominator from #replacement, if it exists.
     */
    VertexPrecision getDenominator(void) const;

    /**
     * Make a contribution to the #replacement Vertex with a specified weight.
     * #replacement is allocated if it is not already available. This function
     * handles the multiplication by @c weight , so the vertex should be passed
     * unmultiplied.
     * @param v The Vertex making the contribution
     * @param weight The weight associated with this contribution
     */
    void receiveContribution(const Vertex &v, VertexPrecision weight);

    /**
     * Overwrite #coords with the co-ordinates of the #replacement vertex.
     * @param freeReplacement If @c true, also frees memory associated with
     * #replacement. @c replace(true) should be called for all vertices at the
     * very end of a subdivision step.
     */
    void replace(bool freeReplacement = false);

    /** Project to w=1. */
    const Vertex project(void) const;

private:
    /**
     * Holds the information associated with each vertex during subdivision
     * to construct a replacement.
     *
     * @ingroup Support
     */
    struct Replacement
    {
        /**
         * The replacement vertex co-ordinates.
         * Don't use a Vertex object, as we don't need a Vertex::replacement
         * pointer here.
         */
        VertexPrecision coords[4];
        /** The sum of weights by which replacement must be normalised. */
        VertexPrecision denominator;
    };

    /**
     * A pointer to a Vertex::Replacement struct that is normally NULL, but is
     * used during subdivision to store a replacement vertex.
     */
    Replacement *replacement;

    /** Vertex co-ordinates. */
    VertexPrecision coords[4];

    /** Initialize the #replacement pointer. */
    void initReplacement(void);
};

inline VertexPrecision Vertex::getX  (void) const { return coords[0]; }
inline VertexPrecision Vertex::getY  (void) const { return coords[1]; }
inline VertexPrecision Vertex::getZ  (void) const { return coords[2]; }
inline VertexPrecision Vertex::getW  (void) const { return coords[3]; }

inline bool Vertex::hasReplacement(void) const
{
    return replacement != NULL;
}

inline VertexPrecision Vertex::getDenominator(void) const
{
    if (hasReplacement())
    {
        return replacement->denominator;
    }
    else
    {
        return -1;
    }
}

/** Stream << operator for Vertex objects. */
std::ostream &operator<<(std::ostream &os, const Vertex &vertex);

}

#endif
