/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef UVFWRITER_H
#define UVFWRITER_H

#include <fstream>
#include <sstream>
#include "flatmeshhandler.h"

namespace snurbs {

/**
 * A FlatMeshHandler that writes out a very simple file format, used to gain
 * access to the (u, v) parameter space values associated with each vertex of a
 * face. This format was invented to export models created by libsnurbs to
 * Blender: there is an import script for Blender that can import models in
 * this format including the (u, v) coordinates.
 *
 * The file format has a header: a single line consisting of
 *   @verbatim #numverts #numfaces @endverbatim
 * followed by \#numverts lines, each of which is in the format
 *   @verbatim x y z @endverbatim
 * followed by \#numfaces lines, each of which is in the format
 *   @verbatim vertexindex vertexu vertexv @endverbatim
 * (repeated for each vertex in the face).
 *
 * @ingroup Support
 */
class UvfWriter : public FlatMeshHandler
{
public:
    /** UvfWriter constructor. */
    UvfWriter(std::ostream *fileStream);
    void addVertex(VertexPrecision x,
                   VertexPrecision y,
                   VertexPrecision z,
                   VertexPrecision w);
    void addToFace(unsigned int vertNum);
    void addTexCoords(KnotPrecision u, KnotPrecision v);
    void closeFace(void);
    void finishKnotIntervals(void);

private:
    /**
     * The @c std::ostream used to represent the target file.
     */
    std::ostream *file;

    /** Holds the number of calls to writeVertex. */
    unsigned int numVertices;
    /** Holds the number of calls to closeFace. */
    unsigned int numFaces;

    /**
     * Holds the contents of the file until the header is written in
     * finishKnotIntervals(). This buffer is then copied into the target file.
     */
    std::stringstream buffer;
};

}

#endif
