/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <boost/checked_delete.hpp>
#include <cassert>
#include <tr1/functional>
#include "quadpatchtree.h"

namespace snurbs {

using namespace boost;
using namespace std;
using namespace tr1::placeholders;

QuadPatchTree::QuadPatchTree(PatchTree *child1,
                             PatchTree *child2,
                             PatchTree *child3,
                             PatchTree *child4) : PatchTree()
{
    children[0] = child1;
    children[1] = child2;
    children[2] = child3;
    children[3] = child4;

    children[0]->setParent(this);
    children[1]->setParent(this);
    children[2]->setParent(this);
    children[3]->setParent(this);
}

QuadPatchTree::~QuadPatchTree(void)
{
    for_each(children.begin(), children.end(), checked_delete<PatchTree>);
}

PatchTree *&QuadPatchTree::getChild(Direction dir)
{
    if (dir == ANY)
    {
        return children[0];
    }
    else
    {
        return children[toIndex(dir)];
    }
}

PatchTree::Direction QuadPatchTree::dirFromParent(void) const
{
    return parent->childDir(this);
}

PatchTree::Direction QuadPatchTree::childDir(const PatchTree *child) const
{
    tr1::array<PatchTree *, 4>::const_iterator iter =
        find(children.begin(), children.end(), child);

    if (iter == children.end())
    {
        throw child_not_found();
    }
    else
    {
        return toDir(distance(children.begin(), iter));
    }
}

void QuadPatchTree::stream(ostream &os, unsigned char level) const
{
    streamDepthMarkers(os, level);
    os << " " << this << endl;

    for_each(children.begin(), children.end(),
             tr1::bind(&PatchTree::stream, _1, tr1::ref(os), level + 1));
}

}
