/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PRIMALPATCHTREELEAF_H
#define PRIMALPATCHTREELEAF_H

#include <cassert>
#include <tr1/array>
#include "patchtreeleaf.h"
#include "types.h"
#include "vertex.h"

namespace snurbs {

/**
 * Contains pointers to the four vertices which lie on the four corners of
 * this PatchTreeLeaf.
 *
 * @todo Some of the methods in PrimalPatchTreeLeaf will be identical for
 * DualPatchTreeLeaf (e.g.\ getRefineComb?), and should therefore be in
 * PatchTreeLeaf instead. Something to sort out before looking at
 * DualPatchTreeLeaf.
 *
 * @ingroup Mesh
 */
class PrimalPatchTreeLeaf : public PatchTreeLeaf
{
    friend class PrimalMeshBuilder;

public:
    /** PrimalPatchTreeLeaf constructor. */
    PrimalPatchTreeLeaf(void);

    /** Getter for #vertices. */
    Vertex* const &getVertex(unsigned char index) const;

    bool isReal(void) const;

    /**
     * @return @c true if this PrimalPatchTreeLeaf is entirely imaginary (i.e.
     * Vertex::isReal() returns @c false for each Vertex in #vertices). Note
     * that isImag() and isReal() can both return @c false
     */
    bool isImag(void) const;

    /**
     * Calculate an unnormalized face normal for the quadrilateral face
     * specified by the four #vertices. If the face is non-planar, returns a
     * best-fit plane (see Graphics Gems III, 5.5).
     * @param[out] normal The array in which to receive the calculated normal
     */
    void getNormal(std::tr1::array<VertexPrecision, 3> &normal) const;

    /**
     * @name Subdivision functions
     * @{
     */
    void            makeInsertionRequests();
    PatchTree      *refineTree           ();
    void            refineVertices       (unsigned char degree);
    void            extraordinaryRefine  (unsigned char degree,
                                          Direction extNode,
                                          unsigned char valency);
    void            smooth               (unsigned char degree,
                                          unsigned char stage);
    VertexPrecision extraordinarySmooth  (unsigned char degree,
                                          unsigned char stage,
                                          Direction extNode,
                                          unsigned char valency);
    /**
     * Make final stage adjustments to the Vertex::Replacement objects
     * referenced from this leaf, given that there is an extraordinary
     * PatchNode of valency 3 in Direction @c extNode.
     * @param degree      The subdivision degree
     * @param extNode     The PatchTree::Direction of the valency 3 vertex.
     * @param weightsProd The product of weights from the EV to its successor
     *                    points throughout the refine and smooth process.
     */
    void       extraVal3          (unsigned char degree,
                                   Direction extNode,
                                   double weightsProd);
    /** @} */

    /**
     * Used to implement
     * operator<<(std::ostream &os, const PatchTreeRoot &patch);
     */
    void stream(std::ostream &os, unsigned char level) const;

private:
    /**
     * Pointers to the four vertices which lie on the corners of this
     * PatchTreeLeaf.
     *  - @c vertices[0] is the North-East vertex
     *  - @c vertices[1] is the South-East vertex
     *  - @c vertices[2] is the South-West vertex
     *  - @c vertices[3] is the North-West vertex
     */
    std::tr1::array<Vertex *, 4> vertices;

    /**
     * Returns the weight of a univariate affine combination used in the refine
     * step, where the direction of the calculated weight depends:
     *  - on whether @c orientation is HORIZONTAL or VERTICAL
     *  - on which side of the #parent this leaf lies.
     * @param orientation The orientation in which to calculate this weight
     * @param degree The subdivision degree. Affects the number of steps taken
     * by PatchTreeWalker::knotSum
     */
    KnotPrecision getRefineComb(Orientation orientation,
                                unsigned char degree) const;

    /**
     * Uses a PatchTreeWalker to calculate all the necessary properties of the
     * knot configuration in a given PatchTree::Direction. This function uses
     * just one PatchTreeWalker and makes use as efficient as possible. Values
     * are returned in the final seven reference parameters.
     * @param direction The PatchTree::Direction in which to calculate sums of
     * knot intervals
     * @param degree The subdivision degree
     * @param stage The subdivision smoothing stage number
     * @param[out] nearSum The interval in the knot vector from the
     * @c direction edge of this PrimalPatchTreeLeaf, to the first knot we may
     * need to insert in this step
     * @param[out] farSum The interval in the knot vector from the
     * @c direction edge of this PrimalPatchTreeLeaf, to the second knot we may
     * need to insert in this step
     * @param[out] nearEndSum The interval in the knot vector from the
     * @c direction edge of this PrimalPatchTreeLeaf, to the most distant knot
     * in the blossom of points on the edge opposite to @c direction
     * @param[out] farEndSum The interval in the knot vector from the
     * @c direction edge of this PrimalPatchTreeLeaf, to the most distant knot
     * in the blossom of points on the @c direction edge
     * @param[out] furthestEndSum The interval in the knot vector from the
     * @c direction edge of this PrimalPatchTreeLeaf, to the most distant knot
     * in the blossom of points on the edge lying just past the @c direction
     * edge
     * @param[out] near Whether the knot calculated in @c nearSum was first
     * inserted in the current subdivision step
     * @param[out] far Whether the knot calculated in @c farSum was first
     * inserted in the current subdivision step
     */
    void calcKnotSums(Direction direction,
                      unsigned char degree,
                      unsigned char stage,
                      KnotPrecision &nearSum,
                      KnotPrecision &farSum,
                      KnotPrecision &nearEndSum,
                      KnotPrecision &farEndSum,
                      KnotPrecision &furthestEndSum,
                      bool &near,
                      bool &far) const;

    /**
     * Uses calcKnotSums() to calculate, for a given @c orientation, the
     * univariate affine combinations for each of the two edges of the
     * PrimalPatchTreeLeaf perpendicular to @c orientation to themselves, and
     * to the opposite edge. Values are returned in the final four reference
     * parameters. We @em could break this function into one which calculates
     * the affine combinations for just @em one edge, but we would then lose
     * the efficiency in calcKnotSums().
     * @param orientation The PatchTree::Orientation to work in
     * @param degree The subdivision degree
     * @param stage The subdivision smoothing stage number
     * @param[out] LL The affine combination from the Left (North or West)
     * edge to the Left (North or West) edge
     * @param[out] RL The affine combination from the Right (South or East)
     * edge to the Left (North or West) edge
     * @param[out] RR The affine combination from the Right (South or East)
     * edge to the Right (South or East) edge
     * @param[out] LR The affine combination from the Left (North or West)
     * edge to the Rught (South or East) edge
     */
    void getCombsForOrientation(Orientation orientation,
                                unsigned char degree,
                                unsigned char stage,
                                VertexPrecision &LL,
                                VertexPrecision &RL,
                                VertexPrecision &RR,
                                VertexPrecision &LR) const;

    /**
     * Uses getCombsForOrientation() for both PatchTree::HORIZONTAL and
     * PatchTree::VERTICAL to calculate the bivariate affine combinations
     * from each vertex in the PrimalPatchTreeLeaf to every other vertex.
     * Values are returned in the final four parameters, each of which should
     * be of type VertexPrecision[4], or @c NULL to specify that a certain set
     * of affine combinations are not required.
     * @param degree The subdivision degree
     * @param stage The subdivision smoothing stage number
     * @param[out] fromNE The affine combinations from the North-East vertex
     * to all other vertices
     * @param[out] fromSE The affine combinations from the South-East vertex
     * to all other vertices
     * @param[out] fromSW The affine combinations from the South-West vertex
     * to all other vertices
     * @param[out] fromNW The affine combinations from the North-West vertex
     * to all other vertices
     */
    void getSmoothCombs(unsigned char degree,
                        unsigned char stage,
                        VertexPrecision *fromNE,
                        VertexPrecision *fromSE,
                        VertexPrecision *fromSW,
                        VertexPrecision *fromNW) const;
};

inline Vertex* const &PrimalPatchTreeLeaf::getVertex(unsigned char index) const
{
    assert(index < 4);
    return vertices[index];
}

inline bool PrimalPatchTreeLeaf::isReal(void) const
{
    return (vertices[0]->isReal() &&
            vertices[1]->isReal() &&
            vertices[2]->isReal() &&
            vertices[3]->isReal());
}

inline bool PrimalPatchTreeLeaf::isImag(void) const
{
    return !(vertices[0]->isReal() ||
             vertices[1]->isReal() ||
             vertices[2]->isReal() ||
             vertices[3]->isReal());
}

}

#endif
