/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PRIMALMESHBUILDER_H
#define PRIMALMESHBUILDER_H

#include <map>
#include <vector>
#include "meshbuilder.h"

namespace snurbs {

class HalfEdge;
class KnotInterval;
class PatchNode;
class PatchTreeRoot;
class PatchTreeWalker;
class PrimalPatchTreeLeaf;
class Vertex;

/**
 * Used to build a primal Mesh (i.e.\ one with odd @link Mesh::degree degree
 * @endlink)
 *
 * @todo Some of the methods in PrimalMeshBuilder will be identical for
 * DualMeshBuilder (e.g. linkKnotStrip), and should therefore be in MeshBuilder
 * instead. Something to sort out before looking at DualMeshBuilder.
 *
 * @ingroup Support
 */
class PrimalMeshBuilder : public MeshBuilder
{
public:
    /**
     * PrimalMeshBuilder constructor.
     * @param mesh The mesh to receive objects.
     */
    PrimalMeshBuilder(Mesh &mesh) : MeshBuilder(mesh) {};

    void addVertex(VertexPrecision x,
                   VertexPrecision y,
                   VertexPrecision z,
                   VertexPrecision w);
    void startFace(unsigned int numEdges);
    void addToFace(unsigned int vertNum);
    void closeFace(void);
    /**
     * In this function, we make sure that every HalfEdge has a pair (which
     * requires constructing the boundary), and assign a HalfEdge pointer to
     * every PatchNode.
     */
    void finishFaces(void);
    /**
     * Specify the knot interval along the strip of quadrilateral faces which
     * runs perpendicular to the edge specified by vertex1-vertex2.
     */
    void addKnotInterval(unsigned int vertex1,
                         unsigned int vertex2,
                         KnotPrecision value);
    void finishKnotIntervals(void);

private:
    /**
     * The #pairMap which locates HalfEdge::pair pointers uses a key which is
     * this ordered pair of PatchNode pointers.
     */
    typedef std::pair<PatchNode *, PatchNode *>                PairMapKey;
    /**
     * The type of #pairMap, used for finding HalfEdge::pair pointers.
     */
    typedef std::map <PairMapKey, HalfEdge *>                  PairMap;
    /**
     * The type of #knotFlipMap, used for storing the orientation of
     * knot intervals.
     */
    typedef std::map <PatchTreeRoot *, std::pair<bool, bool> > KnotFlipMap;

    /**
     * Temporary list of vertices which we need to index for addtoFace() and
     * addKnotInterval().
     */
    std::vector <Vertex *> vertices;
    /**
     * Keeps track of the current patch tree leaf being processed.
     */
    PrimalPatchTreeLeaf *  currentFace;
    /**
     * Keeps track of the current patch tree root being processed.
     */
    PatchTreeRoot *        currentPatch;
    /**
     * Used to index into PrimalPatchTreeLeaf::vertices in addToFace().
     */
    unsigned char          currentVertIndex;
    /**
     * Used to match up HalfEdge objects with HalfEdge::pair pointers.
     */
    PairMap                pairMap;
    /**
     * Used to store the orientations of knot intervals with respect to
     * PatchTreeRoot objects, before normalising the orientations in
     * finishKnotIntervals().
     */
    KnotFlipMap            knotFlipMap;

    /**
     * Uses #pairMap to match up a HalfEdge with its HalfEdge::pair.
     */
    void findPair(HalfEdge *edge);
    /**
     * Uses any edges left in #pairMap to construct HalfEdge objects for any
     * boundaries.
     */
    void addBoundary(const PairMap::value_type &unmatched);
    /**
     * Assigns HalfEdge::next pointers to all boundary HalfEdge objects.
     */
    static void linkBoundary(const PairMap::value_type &unmatched);
    /**
     * Helper function used by cullBoundary()
     */
    void cullVertex(const PatchTreeWalker &walker);
    /**
     * Deletes and removes pointers to vertices too close to the boundary.
     */
    void cullBoundary(const PairMap::value_type &unmatched);
    /**
     * Assigns KnotInterval pointers to strips of PatchTreeRoot objects.
     */
    void linkKnotStrip(HalfEdge*     const edge,
                       KnotInterval* const knotInterval,
                       bool                defaultOrientation);
    /**
     * Checks that the leaf associated with a PatchTreeRoot has both
     * horizontal and vertical knot intervals. This will usually only fail to
     * be the case when no knot spacings were provided at all. Any missing knot
     * intervals are given the value 1.
     * @param patch The PatchTreeRoot to check.
     * @see finishKnotIntervals().
     */
    void addMissingIntervals(const PatchTreeRoot* const patch);

    /**
     * Overloaded version of addKnotInterval that is provided the HalfEdge
     * corresponding to a KnotInterval, so there is no need to search for it.
     */
    void addKnotInterval(HalfEdge* startEdge, KnotPrecision value);
};

}

#endif
