/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PLYREADER_H
#define PLYREADER_H

#include <string>
/** @file
 * ply/byte_order.hpp needs @<algorithm@> but doesn't include it. We get away
 * with it here because we've already included @<string@>.
 */
#include <ply.hpp>

namespace snurbs {

class MeshBuilder;

/**
 * Reads a mesh in from a Stanford PLY file. Note: for a non-uniform surface,
 * the PLY file must also have knot spacing information.
 *
 * Sample header:
 * @verbatim
ply
format ascii 1.0
comment Sample PLY header for libsnurbs
element vertex 8
property float32 x
property float32 y
property float32 z
comment The w coordinate can be omitted
property float32 w
comment Faces can reference undefined vertices,
comment purely to define boundary knot intervals
element face 6
property list uint8 int32 vertex_index
comment Any missing knot_intervals will be
comment assumed to have value 1
element knot_interval 3
property int32 vertex1
property int32 vertex2
property float32 value
end_header
@endverbatim
 *
 * @ingroup Support
 */
class PlyReader
{
public:
    /**
     * PlyReader constructor.
     * @param meshBuilder The MeshBuilder object to make calls on.
     */
    PlyReader(MeshBuilder &meshBuilder);
    /**
     * Sets up libply with the correct callbacks and calls
     * ply::ply_parser::parse to read in a PLY file.
     * @param file A string with a valid filename for a PLY file to read.
     */
    void read(const std::string& file);

private:
    /** The MeshBuilder object to make calls on (and thereby build a mesh). */
    MeshBuilder &meshBuilder;
    /**
     * Stores the co-ordinates of a vertex as they arrive. We initialize
     * currentVertex with a w co-ordinate of `1', so PLY files with only x, y
     * and z co-ordinates are set up correctly.
     */
    ply::float32 currentVertex[4];
    /**
     * Stores the vertex indices for a knot_interval element as they arrive.
     */
    ply::int32   currentKnotEdge[2];
    /**
     * Stores the value property of the knot_interval element as it arrives.
     */
    ply::float32 currentKnotInterval;

    /**
     * The number of vertices specified by the PLY file. Used to call
     * MeshBuilder::finishVertices().
     */
    std::size_t numVertices;
    /**
     * The number of faces specified by the PLY file. Used to call
     * MeshBuilder::finishFaces().
     */
    std::size_t numFaces;

    /**
     * @name Element callbacks
     * @{
     */
    void startVertex(void);
    void endVertex(void);
    void startKnotInterval(void);
    void endKnotInterval(void);

    ply::ply_parser::element_callbacks_type
        elementDefinition(const std::string& name, std::size_t number);
    /** @} */

    /**
     * @name Scalar property callbacks
     * @{
     */
    void vertexCoord(unsigned int dimension, ply::float32 value);
    void knotIntervalVertex(unsigned int number, ply::int32 vertex);
    void knotIntervalValue(ply::float32 interval);
    /** @} */

    // Doxygen can't handle these specialised template members in a
    // non-template class. So we have to exclude them from the documentation.
    /** @cond */
    /*
     * This template definition is for the function which assigns function
     * pointers to specific pairs of element and property names. Also see
     * ListPropertyDefinition.
     */
    template <typename ScalarType>
        std::tr1::function <void (ScalarType)>
            scalarPropertyDefinition(const std::string& elementName,
                                     const std::string& propertyName);
    /** @endcond */

    /**
     * @name List property callbacks
     * @{
     */

    /**
     * For processing a list of vertex indices, libply uses a tuple of three
     * function pointers, for the begin, element, and end callbacks. This
     * typedef simplifies the construction of this specific tuple of
     * function pointers.
     */
    typedef std::tr1::tuple < std::tr1::function<void (ply::uint8)>,
                              std::tr1::function<void (ply::int32)>,
                              std::tr1::function<void ()> >
                                faceVertexIndexCallback;

    void faceVertexIndexBegin(ply::uint8 length);
    void faceVertexIndexElement(ply::int32 index);
    void faceVertexIndexEnd(void);
    /** @} */

    // Doxygen can't handle these specialised template members in a
    // non-template class. So we have to exclude them from the documentation.
    /** @cond */
    /*
     * This template definition is for the function which assigns function
     * pointers to specific pairs of element and property names. It differs
     * from scalarPropertyDefinition because of the return type.
     */
    template <typename SizeType, typename ScalarType>
        std::tr1::tuple < std::tr1::function<void (SizeType)>,
                          std::tr1::function<void (ScalarType)>,
                          std::tr1::function<void ()> >
            listPropertyDefinition(const std::string& elementName,
                                   const std::string& propertyName);
    /** @endcond */
};

}

#endif
