/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "plyreader.h"
#include "meshbuilder.h"

namespace snurbs {

using namespace ply;
using namespace std;
using namespace tr1::placeholders;

PlyReader::PlyReader(MeshBuilder &meshBuilder) :
        meshBuilder(meshBuilder),
        numVertices(0),
        numFaces(0)
{
}

// Doxygen can't handle these specialised template members in a non-template
// class. So we have to exclude them from the documentation.
/** @cond */

/* We specialise the template for `float32' (float) properties. */
template <>
    tr1::function <void (float32)>
        PlyReader::scalarPropertyDefinition(const string &elementName,
                                            const string &propertyName)
{
    if (elementName == "vertex")
    {
        if (propertyName == "x")
        {
            return tr1::bind(&PlyReader::vertexCoord, this, 0, _1);
        }
        else if (propertyName == "y")
        {
            return tr1::bind(&PlyReader::vertexCoord, this, 1, _1);
        }
        else if (propertyName == "z")
        {
            return tr1::bind(&PlyReader::vertexCoord, this, 2, _1);
        }
        else if (propertyName == "w")
        {
            return tr1::bind(&PlyReader::vertexCoord, this, 3, _1);
        }
    }
    else if (elementName == "knot_interval")
    {
        if (propertyName == "value")
        {
            return tr1::bind(&PlyReader::knotIntervalValue, this, _1);
        }
    }
    return NULL;
}

/* And then for `int32' (integer) properties. */
template <>
    tr1::function <void (int32)>
        PlyReader::scalarPropertyDefinition(const string &elementName,
                                            const string &propertyName)
{
    if (elementName == "knot_interval")
    {
        if (propertyName == "vertex1")
        {
            return tr1::bind(&PlyReader::knotIntervalVertex, this, 0, _1);
        }
        else if (propertyName == "vertex2")
        {
            return tr1::bind(&PlyReader::knotIntervalVertex, this, 1, _1);
        }
    }
    return NULL;
}

/* listPropertyDefinition only needs to be specialised for one specific
 * instance: returning an instance of faceVertexIndexCallback.
 */
template <>
    PlyReader::faceVertexIndexCallback
        PlyReader::listPropertyDefinition(const string &elementName,
                                          const string &propertyName)
{
    if (elementName == "face")
    {
        if (propertyName == "vertex_index" ||
            propertyName == "vertex_indices")
        {
            return faceVertexIndexCallback(
                tr1::bind(&PlyReader::faceVertexIndexBegin, this, _1),
                tr1::bind(&PlyReader::faceVertexIndexElement, this, _1),
                tr1::bind(&PlyReader::faceVertexIndexEnd, this) );
        }
    }
    return faceVertexIndexCallback(NULL, NULL, NULL);
}

/** @endcond */

void PlyReader::read(const string& file)
{
    ply_parser plyParser;

    // Element definitions callback
    plyParser.element_definition_callback(
        tr1::bind(&PlyReader::elementDefinition, this, _1, _2));


    // Set up the callbacks for scalar properties.
    ply_parser::scalar_property_definition_callbacks_type
        scalarPropertyDefinitionCallbacks;

    // Account for both the float32 and int32 specialisations.
    at<float32>(scalarPropertyDefinitionCallbacks) =
        tr1::bind(&PlyReader::scalarPropertyDefinition<float32>, this, _1, _2);
    at<int32>(scalarPropertyDefinitionCallbacks) =
        tr1::bind(&PlyReader::scalarPropertyDefinition<int32>, this, _1, _2);

    // Assign the callbacks object to plyParser.
    plyParser.scalar_property_definition_callbacks(
        scalarPropertyDefinitionCallbacks);


    // Set up the callbacks for list properties.
    ply_parser::list_property_definition_callbacks_type
        listPropertyDefinitionCallbacks;

    // Only one type of list, so only one specialisation here.
    at<uint8, int32>(listPropertyDefinitionCallbacks) =
        tr1::bind(&PlyReader::listPropertyDefinition<uint8, int32>,
                  this, _1, _2);

    // Assign the callbacks object to plyParser.
    plyParser.list_property_definition_callbacks(
        listPropertyDefinitionCallbacks);

    plyParser.parse(file);

    // Make sure we always call meshBuilder.finishKnotIntervals(), even if
    // there are no knot intervals defined in the file.
    meshBuilder.finishKnotIntervals();
}

void PlyReader::startVertex(void)
{
    currentVertex[0] = 0;
    currentVertex[1] = 0;
    currentVertex[2] = 0;
    currentVertex[3] = 1;
}

void PlyReader::endVertex(void)
{
    meshBuilder.addVertex(currentVertex[0],
                          currentVertex[1],
                          currentVertex[2],
                          currentVertex[3]);
    --numVertices;

    if (numVertices == 0)
    {
        meshBuilder.finishVertices();
    }
}

void PlyReader::startKnotInterval(void)
{
    currentKnotInterval = defaultKnotSpacing;
    currentKnotEdge[0]  = 0;
    currentKnotEdge[1]  = 0;
}

void PlyReader::endKnotInterval(void)
{
    meshBuilder.addKnotInterval(currentKnotEdge[0],
                                currentKnotEdge[1],
                                currentKnotInterval);
}

ply::ply_parser::element_callbacks_type
    PlyReader::elementDefinition(const string& name, std::size_t number)
{
    if (name == "vertex")
    {
        numVertices = number;
        return ply_parser::element_callbacks_type(
                    tr1::bind(&PlyReader::startVertex, this),
                    tr1::bind(&PlyReader::endVertex, this));
    }
    else if (name == "face")
    {
        numFaces = number;
    }
    else if (name == "knot_interval")
    {
        return ply_parser::element_callbacks_type(
                    tr1::bind(&PlyReader::startKnotInterval, this),
                    tr1::bind(&PlyReader::endKnotInterval, this));
    }
    return ply_parser::element_callbacks_type(NULL, NULL);
}

void PlyReader::vertexCoord(unsigned int dimension, ply::float32 value)
{
    currentVertex[dimension] = value;
}

void PlyReader::knotIntervalVertex(unsigned int number, ply::int32 vertex)
{
    currentKnotEdge[number] = vertex;
}

void PlyReader::knotIntervalValue(ply::float32 interval)
{
    currentKnotInterval = interval;
}

/**
 * Called at the start of processing every face.
 */
void PlyReader::faceVertexIndexBegin(ply::uint8 length)
{
    meshBuilder.startFace(length);
}

/**
 * Called for each vertex in the list that is given for each face.
 */
void PlyReader::faceVertexIndexElement(ply::int32 index)
{
    meshBuilder.addToFace(index);
}

/**
 * Called at the end of processing every face.
 */
void PlyReader::faceVertexIndexEnd(void)
{
    meshBuilder.closeFace();
    --numFaces;
    if (numFaces == 0)
    {
        meshBuilder.finishFaces();
    }
}

}
