/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef HALFEDGE_H
#define HALFEDGE_H

#include <ostream>
#include <stdexcept>

/** @file */

namespace snurbs {

class PatchNode;
class PatchTreeRoot;

/**
 * Each HalfEdge is part of a linked list that surrounds each PatchTreeRoot.
 * The HalfEdge objects and their pointers provide access to the arbitrary
 * topology of the base mesh.
 *
 * @ingroup Mesh
 */
class HalfEdge
{
    friend class PrimalMeshBuilder;

public:
    /**
     * HalfEdge constructor.
     * @param node  The PatchNode at the end of this HalfEdge.
     * @param patch The PatchTreeRoot adjacent to this HalfEdge.
     */
    HalfEdge(PatchNode *node, PatchTreeRoot *patch);

    /**
     * @name Getter functions
     * @{
     */
    HalfEdge      *getPair (void) const;
    HalfEdge      *getNext (void) const;
    PatchNode     *getNode (void) const;
    PatchTreeRoot *getPatch(void) const;
    /** @} */

    /**
     * Finds the previous edge in the loop. So for a HalfEdge object @c he ,
     * <tt>he.prev()->next == he</tt>
     */
    HalfEdge *prev(void);

    /**
     * @param edge The target edge.
     * @return The number of steps from this edge to the target edge,
     * following #next pointers.
     * @throws std::runtime_error If the target edge is not found.
     */
    unsigned char stepsTo(const HalfEdge* const edge);

    /**
     * This is an abstract base class used to traverse a mesh using HalfEdge
     * pointers. Derived classes provide specific traversal types.
     *
     * @ingroup Iterators
     */
    class Iterator : public std::iterator<std::input_iterator_tag,
                                          HalfEdge* const>
    {
    public:
        /**
         * Iterator constructor.
         * @param init The starting edge.
         */
        Iterator(HalfEdge *init);

        /**
         * Dereference an iterator object to access the HalfEdge pointer that
         * the iterator currently points to.
         */
        HalfEdge* const &operator* (void) const;
        /**
         * Tests two Iterator objects for equality. The default implementation
         * is to equate Iterator objects only if they point to the same
         * HalfEdge, but that can be overridden (e.g. see PatchStripIterator).
         * @param iter The Iterator to compare.
         */
        virtual    bool  operator==(const Iterator &iter);
        /** Inequality operator for Iterator objects. */
                   bool  operator!=(const Iterator &iter);
        /** Preincrement operator for Iterator objects. */
        virtual    void  operator++(void) = 0;

    protected:
        /** The current position of the iterator. */
        HalfEdge *edge;
    };

    /**
     * Iterator that traverses the edges surrounding a PatchNode.
     *
     * @ingroup Iterators
     */
    class NodeIterator : public Iterator
    {
    public:
        /**
         * NodeIterator constructor.
         * @param init The HalfEdge that the iterator should be initialised
         * pointing to.
         */
        NodeIterator(HalfEdge *init);
        void operator++(void);
    };

    /**
     * Iterator that traverses the edges surrounding a PatchTreeRoot.
     *
     * @ingroup Iterators
     */
    class PatchIterator : public Iterator
    {
    public:
        /**
         * PatchIterator constructor.
         * @param init The HalfEdge that the iterator should be initialised
         * pointing to.
         */
        PatchIterator(HalfEdge *init);
        void operator++(void);
    };

    /**
     * Iterator that traverses the edges along a strip of PatchTreeRoot
     * objects.
     *
     * @ingroup Iterators
     */
    class PatchStripIterator : public Iterator
    {
    public:
        /**
         * PatchStripIterator constructor.
         * @param init The HalfEdge that the iterator should be initialised
         * pointing to.
         */
        PatchStripIterator(HalfEdge *init);
        void operator++(void);
        /**
         * Replaces the default Iterator::operator==() implementation (that
         * just checks if the iterator has looped back to the beginning) with
         * one that also checks if the iterator has hit a boundary.
         */
        bool operator==(const Iterator &iter);
    };

private:
    /**
     * Points to the next HalfEdge in the linked list surrounding the
     * PatchTreeRoot patch.
     */
    HalfEdge      *next;
    /**
     * Points to the HalfEdge pointing in the opposite direction along the same
     * edge.
     */
    HalfEdge      *pair;
    /**
     * Points to the PatchNode at the 'end' of this HalfEdge.
     */
    PatchNode     *node;
    /**
     * Points to the PatchTreeRoot surrounded by the linked list of HalfEdges.
     */
    PatchTreeRoot *patch;
};

/**
 * Version of std::find() that preincrements the iterator before dereferencing
 * it for the first time. This makes it suitable for use on iterators that
 * loop back to the start. We also need just one iterator to specify the whole
 * range, instead of the STL version which takes two iterators.
 * @param iter The iterator used as the beginning and end of the range to
 * iterate over.
 * @param fn The function which specifies the iterator position to search for
 */
template <typename _Iterator, typename _Function>
inline _Iterator find_circular(_Iterator iter, _Function fn)
{
    _Iterator end(iter);
    do
    {
        if (fn(*iter))
        {
            return iter;
        }
        ++iter;
    } while (iter != end);

    throw std::runtime_error("Not found");
}

/**
 * Version of std::for_each() that preincrements the iterator before
 * dereferencing it for the first time. This makes it suitable for use on
 * iterators that loop back to the start. We also need just one iterator to
 * specify the whole range, instead of the STL version which takes two
 * iterators.
 * @param iter The iterator used as the beginning and end of the range to
 * iterate over.
 * @param fn The function applied to each position of the iterator
 */
template <typename _Iterator, typename _Function>
inline _Iterator for_each_circular(_Iterator iter, _Function fn)
{
    _Iterator end(iter);
    do
    {
        fn(*iter);
        ++iter;
    } while (iter != end);

    return iter;
}

/**
 * Version of std::count_if() that preincrements the iterator before
 * dereferencing it for the first time. This makes it suitable for use on
 * iterators that loop back to the start. We also need just one iterator to
 * specify the whole range, instead of the STL version which takes two
 * iterators.
 * @param iter The iterator used as the beginning and end of the range to
 * iterate over.
 * @param fn The function used to test dereferenced iterators before counting
 */
template <typename _Iterator, typename _Function>
inline int count_if_circular(_Iterator iter, _Function fn)
{
    _Iterator end(iter);
    int result = 0;
    do
    {
        if (fn(*iter))
        {
            ++result;
        }
        ++iter;
    } while (iter != end);

    return result;
}

/**
 * Version of std::accumulate() that preincrements the iterator before
 * dereferencing it for the first time. This makes it suitable for use on
 * iterators that loop back to the start. We also need just one iterator to
 * specify the whole range, instead of the STL version which takes two
 * iterators.
 * @param iter The iterator used as the beginning and end of the range to
 * iterate over.
 * @param init The initial value to accumulate from.
 * @param fn The binary function used to accumulate whilst iterating.
 */
template <typename _Iterator, typename _Type, typename _Function>
inline _Type accumulate_circular(_Iterator iter, _Type init, _Function fn)
{
    _Iterator end(iter);
    do
    {
        init = fn(init, *iter);
        ++iter;
    } while (iter != end);

    return init;
}

inline HalfEdge      *HalfEdge::getPair (void) const { return pair;  };
inline HalfEdge      *HalfEdge::getNext (void) const { return next;  };
inline PatchNode     *HalfEdge::getNode (void) const { return node;  };
inline PatchTreeRoot *HalfEdge::getPatch(void) const { return patch; };

inline HalfEdge* const &HalfEdge::Iterator::operator*(void) const
{
    return edge;
}

inline HalfEdge::Iterator::Iterator(HalfEdge *init) : edge(init)
{
}

inline bool HalfEdge::Iterator::operator==(const Iterator &iter)
{
    return (edge == iter.edge);
}

inline bool HalfEdge::Iterator::operator!=(const Iterator &iter)
{
    return !(*this == iter);
}

inline HalfEdge::NodeIterator::NodeIterator(HalfEdge *init) : Iterator(init)
{
}

inline void HalfEdge::NodeIterator::operator++(void)
{
    edge = edge->pair->next;
}

inline HalfEdge::PatchIterator::PatchIterator(HalfEdge *init) : Iterator(init)
{
}

inline void HalfEdge::PatchIterator::operator++(void)
{
    edge = edge->next;
}

inline HalfEdge::PatchStripIterator::PatchStripIterator(HalfEdge *init) :
                                                         Iterator(init)
{
}

inline void HalfEdge::PatchStripIterator::operator++(void)
{
    edge = edge->pair->next->next;
}

inline bool HalfEdge::PatchStripIterator::operator==(const Iterator &iter)
{
    const PatchStripIterator *psIter =
            static_cast<const PatchStripIterator *>(&iter);
    return (edge == psIter->edge) ||
           (edge->patch == NULL) ||
           (psIter->edge->patch == NULL);
}

/**
 * Stream << operator for HalfEdge objects.
 */
std::ostream &operator<<(std::ostream &os, const HalfEdge &edge);

}

#endif
