/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef FLATMESHHANDLER_H
#define FLATMESHHANDLER_H

#include "types.h"

namespace snurbs {

/**
 * An abstract class used to implement classes that accept a mesh in flattened
 * form (e.g.\ read from a file, or output from a MeshFlattener). Default
 * empty implementations are provided for all methods, so derived classes only
 * need to override the methods they require.
 *
 * @ingroup Support
 */
class FlatMeshHandler
{
public:
    /**
     * FlatMeshHandler destructor.
     * This is made pure virtual so that FlatMeshHandler cannot be
     * instantiated.
     */
    virtual ~FlatMeshHandler(void) = 0;

    /**
     * Called before any other functions in FlatMeshHandler
     */
    virtual void startMesh(void);

    /**
     * Called to add a control point (Vertex) to the mesh.
     */
    virtual void addVertex(VertexPrecision x,
                           VertexPrecision y,
                           VertexPrecision z,
                           VertexPrecision w);

    /**
     * finishVertices is called when all vertices have been added.
     */
    virtual void finishVertices(void);


    /**
     * A face is added to the mesh by calling startFace(), using addtoFace() to
     * add zero-indexed vertices, and finally calling closeFace().
     */
    virtual void startFace(unsigned int numEdges);

    /**
     * @param vertNum The index of the vertex to add to the current face. The
     * first vertex added has index 0. If @f$\ge@f$ 0 and less than the
     * number of calls to addVertex(), this references a vertex added using
     * addVertex(). Otherwise, @c vertNum references an imaginary vertex which
     * can be used to add additional knot intervals to boundary edges.
     */
    virtual void addToFace(unsigned int vertNum);

    /**
     * Adds texture coordinates to the vertex which was just added with
     * addToFace().
     * @param u The u-coordinate of the vertex <em>in the current face</em>.
     * @param v The v-coordinate of the vertex <em>in the current face</em>.
     */
    virtual void addTexCoords(KnotPrecision u,
                              KnotPrecision v);

    /**
     * Finishes constructing the current face.
     */
    virtual void closeFace(void);

    /**
     * finishFaces is called when all faces have been added.
     */
    virtual void finishFaces(void);


    /**
     * Called to add a knot interval. The interval is specified by vertex1 and
     * vertex2, but the interpretation depends on whether the mesh is primal or
     * dual.
     * @see PrimalMeshBuilder::addKnotInterval
     * @see DualMeshBuilder::addKnotInterval
     * @param vertex1 Specify an edge with two vertex indices.
     * @param vertex2 Specify an edge with two vertex indices.
     * @param interval The knot spacing on this interval.
     */
    virtual void addKnotInterval(unsigned int vertex1,
                                 unsigned int vertex2,
                                 KnotPrecision interval);

    /**
     * finishKnotIntervals is called when all knot intervals have been added.
     */
    virtual void finishKnotIntervals(void);
};

// Default no-op implementations (at least some of which will be overridden in
// derived classes)
inline      FlatMeshHandler::~FlatMeshHandler   (void)                       {}
inline void FlatMeshHandler::startMesh          (void)                       {}
inline void FlatMeshHandler::addVertex          (VertexPrecision /*x*/,
                                                 VertexPrecision /*y*/,
                                                 VertexPrecision /*z*/,
                                                 VertexPrecision /*w*/)      {}
inline void FlatMeshHandler::finishVertices     (void)                       {}
inline void FlatMeshHandler::startFace          (unsigned int /*numEdges*/)  {}
inline void FlatMeshHandler::addToFace          (unsigned int /*vertNum*/)   {}
inline void FlatMeshHandler::addTexCoords       (KnotPrecision /*u*/,
                                                 KnotPrecision /*v*/)        {}
inline void FlatMeshHandler::closeFace          (void)                       {}
inline void FlatMeshHandler::finishFaces        (void)                       {}
inline void FlatMeshHandler::addKnotInterval    (unsigned int /*vertex1*/,
                                                 unsigned int /*vertex2*/,
                                                 KnotPrecision /*interval*/) {}
inline void FlatMeshHandler::finishKnotIntervals(void)                       {}

}

#endif
