/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <boost/checked_delete.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/casts.hpp>
#include <boost/lambda/lambda.hpp>
#include <limits>
#include <numeric>
#include <tr1/functional>
#include "halfedge.h"
#include "knotinterval.h"
#include "mesh.h"
#include "meshflattener.h"
#include "meshbuilder.h"
#include "patchnode.h"
#include "patchtreeleaf.h"
#include "patchtreeroot.h"
#include "vertex.h"

namespace snurbs {

using namespace boost;
using namespace std;
using namespace tr1::placeholders;

Mesh::Mesh(unsigned char degree) : degree(degree)
{
}

Mesh::Mesh(const Mesh &mesh) : degree(mesh.degree)
{
    MeshFlattener *flattener = MeshFlattener::create(mesh);
    MeshBuilder   *builder   = MeshBuilder::create(*this);
    flattener->flatten(*builder);
    MeshFlattener::dispose(flattener);
    MeshBuilder::dispose(builder);
}

Mesh::~Mesh(void)
{
    clear();
}

bool Mesh::hasBoundary(void) const
{
    return find_if(halfEdges.begin(), halfEdges.end(),
                   !lambda::bind(&HalfEdge::getPatch, lambda::_1))
           != halfEdges.end();
}

void Mesh::clear(void)
{
    /* Primal meshes have shared Vertex pointers, so we need to make sure we're
     * operating on a list of unique Vertex pointers.
     */
    forEachVertexUnique(checked_delete<Vertex>, this);

    /* Deleting the rest of the mesh objects is easier, as the quadtree
     * structure always makes sure that we delete each object just once.
     */
    for_each(patchNodes.begin(), patchNodes.end(), checked_delete<PatchNode>);
    for_each( halfEdges.begin(),  halfEdges.end(), checked_delete<HalfEdge>);
    for_each(   patches.begin(),    patches.end(),
                                               checked_delete<PatchTreeRoot>);
    for_each(     knots.begin(),      knots.end(),
                                               checked_delete<KnotInterval>);

    // Clear all lists
    patchNodes.clear();
    halfEdges.clear();
    patches.clear();
    knots.clear();
}

void Mesh::subdivide(void)
{
    bool needAnotherStep;

    bool largeIntervalsFirst = true;
    bool boundedCurvature    = true;

    do
    {
        KnotPrecision minInterval = numeric_limits<KnotPrecision>::infinity();
        KnotPrecision maxInterval = 0;
        KnotIterator  end(knots.end());

        // Find minimum non-zero and maximum knot intervals
        // At the same time, we put in an insertion request at the midpoint
        // of every interval
        for (KnotIterator iter(knots.begin()); iter != end; ++iter)
        {
            if ((*iter)->getInterval() > maxInterval)
            {
                maxInterval = (*iter)->getInterval();
            }
            if ((*iter)->getInterval() < minInterval &&
                (*iter)->getInterval() > 0)
            {
                minInterval = (*iter)->getInterval();
            }

            (*iter)->insertionRequest((*iter)->getInterval() / 2);
        }

        // Remove any knot insertion requests where the interval is less than
        // twice the minimum non-zero interval
        if (largeIntervalsFirst && maxInterval >= 2 * minInterval)
        {
            needAnotherStep = true;
            for (KnotIterator iter(knots.begin()); iter != end; ++iter)
            {
                if ((*iter)->getInterval() < 2 * minInterval)
                {
                    (*iter)->clearRequests();
                }
            }
        }
        else
        {
            needAnotherStep = false;

            // Remove knot insertion requests for intervals adjacent to
            // extraordinary points, and make insertion requests to create
            // locally uniform regions
            if (boundedCurvature)
            {
                for_each(patchNodes.begin(), patchNodes.end(),
                         mem_fun(&PatchNode::findSpokeSetMin));

                for_each(patchNodes.begin(), patchNodes.end(),
                         mem_fun(&PatchNode::removeRequests));

                for_each(patchNodes.begin(), patchNodes.end(),
                         mem_fun(&PatchNode::requestNewKnots));
            }
        }

        // Subdivide all KnotInterval objects (so that leftChild and rightChild
        // are allocated) wherever toBeSubdivided() returns true.
        for_each(KnotIterator(knots.begin()), KnotIterator(knots.end()),
                 mem_fun(&KnotInterval::subdivide));

        // Then subdivide all leaf objects using the newly-created KnotInterval
        // objects. Pointers to existing vertices are preserved, but for primal
        // meshes, we can't calculate any new vertices yet:
        // PatchTreeWalker::knotSum needs all child and parent pointers to be
        // defined and consistent, which they won't be until this call
        // finishes. So we leave pointers to all new vertices as NULL.
        transform(RefineTreeIterator(patches.begin()),
                  RefineTreeIterator(patches.end()),
                  RefineTreeIterator(patches.begin()),
                  lambda::bind(&PatchTreeLeaf::refineTree,
                      lambda::ll_static_cast<PatchTreeLeaf *>(lambda::_1)));

        // If primal, make a second pass over the mesh. Where vertex pointers
        // are NULL, we allocate a new vertex and update the three other
        // pointers which share the vertex. We sort out the positioning for the
        // new vertices at the same time.
        if (isPrimal())
        {
            for_each(PatchIterator(patches.begin()),
                     PatchIterator(patches.end()),
                     lambda::bind(&PrimalPatchTreeLeaf::refineVertices,
                     lambda::ll_static_cast<PrimalPatchTreeLeaf *>(lambda::_1),
                                  degree));
        }

        // Positions aren't quite correct around extraordinary vertices... this
        // call corrects them.
        if (boundedCurvature)
        {
            for_each(patchNodes.begin(), patchNodes.end(),
                     tr1::bind(&PatchNode::extraordinaryRefine, _1, degree));
        }

        // We can now update vertices with normalised versions of their
        // Replacement structs, and prepare for the smoothing stages.
        if (isPrimal())
        {
            for_each(PrimalVertexIterator(patches.begin()),
                     PrimalVertexIterator(patches.end()),
                     tr1::bind(&Vertex::replace, _1, false));
        }
        else
        {
            for_each(DualVertexIterator(patches.begin()),
                     DualVertexIterator(patches.end()),
                     tr1::bind(&Vertex::replace, _1, false));
        }

        // Collect the output of PatchIterator(patches.begin()) into a vector
        // so that we can use OpenMP below
        vector<PatchTreeLeaf *> leaves;
        for(PatchIterator iter(patches.begin());
            iter != PatchIterator(patches.end()); ++iter)
        {
            leaves.push_back(static_cast<PatchTreeLeaf *>(*iter));
        }
        int numLeaves = leaves.size();

        // For each smoothing stage
        for (unsigned char stage = 1; stage <= degree / 2; ++stage)
        {
            // Update replacement vertices in parallel
            #pragma omp parallel for schedule(dynamic)
            for (int i = 0; i < numLeaves; ++i)
            {
                leaves[i]->smooth(degree, stage);
            }

            // Make adjustments around extraordinary vertices
            if (boundedCurvature)
            {
                for_each(patchNodes.begin(), patchNodes.end(),
                         tr1::bind(&PatchNode::extraordinarySmooth,
                                    _1, degree, stage));
            }

            // Update vertices with replacements
            if (isPrimal())
            {
                for_each(PrimalVertexIterator(patches.begin()),
                         PrimalVertexIterator(patches.end()),
                         tr1::bind(&Vertex::replace, _1, false));
            }
            else
            {
                for_each(DualVertexIterator(patches.begin()),
                         DualVertexIterator(patches.end()),
                         tr1::bind(&Vertex::replace, _1, stage == degree / 2));
            }
        }

        // If primal, valency 3 vertices need an extra smoothing stage
        if (isPrimal())
        {
            if (boundedCurvature)
            {
                for_each(patchNodes.begin(), patchNodes.end(),
                         tr1::bind(&PatchNode::valency3Adjust, _1, degree));
            }

            for_each(PrimalVertexIterator(patches.begin()),
                     PrimalVertexIterator(patches.end()),
                     tr1::bind(&Vertex::replace, _1, true));
        }
    } while (needAnotherStep);
}

int Mesh::getNumVerts(void) const
{
    UniquePrimalVertexLister list(patches.begin(), patches.end());
    return count_if(list.begin(), list.end(), tr1::bind(&Vertex::isReal, _1));
}

int Mesh::getNumFaces(void) const
{
    if (isPrimal())
    {
        return count_if(PatchIterator(patches.begin()),
                        PatchIterator(patches.end()),
                        lambda::bind(&PatchTreeLeaf::isReal,
                        lambda::ll_static_cast<PatchTreeLeaf *>(lambda::_1)));
    }
    else
    {
        return 0;
    }
}

int Mesh::getNumKnots(void) const
{
    return count_if(KnotIterator(knots.begin()), KnotIterator(knots.end()),
                    lambda::constant(true));
}

Mesh::KnotIterator::KnotIterator(vector<KnotInterval *>::const_iterator begin)
    : vectorIter(begin),
      current   (NULL)
{
}

void Mesh::KnotIterator::operator++(void)
{
    while (!childPath.empty() && childPath.top() == RIGHT)
    {
        childPath.pop();
        current = current->getParent();
    }

    if (childPath.empty())
    {
        vectorIter++;
        current = NULL;
    }
    else
    {
        current = current->getParent();
        childPath.pop();
        current = current->getRightChild();
        childPath.push(RIGHT);
    }
}

KnotInterval*& Mesh::KnotIterator::operator*(void)
{
    if (current == NULL)
    {
        current = *vectorIter;
    }
    while (current->getLeftChild() != NULL)
    {
        current = current->getLeftChild();
        childPath.push(LEFT);
    }
    return current;
}

Mesh::PatchIterator::PatchIterator(vector<PatchTreeRoot *>::const_iterator
                                   begin)
    : patchIter(begin),
      current  (NULL)
{
}

void Mesh::PatchIterator::operator++(void)
{
    if (!current->isRoot())
    {
        PatchTree::Direction d = current->dirFromParent();

        while (d == PatchTree::NORTH_WEST || d == PatchTree::SOUTH ||
               d == PatchTree::WEST || d == PatchTree::CHILD)
        {
            current = current->getParent();
            if (current->isRoot())
            {
                break;
            }
            else
            {
                d = current->dirFromParent();
            }
        }
    }

    if (current->isRoot())
    {
        patchIter++;
        current = NULL;
    }
    else
    {
        PatchTree::Direction d = current->dirFromParent();
        current = current->getParent();
        current = current->getChild(++d);
    }
}

PatchTree*& Mesh::PatchIterator::operator*(void)
{
    if (current == NULL)
    {
        current = *patchIter;
    }
    while (!current->isLeaf())
    {
        if (!(current->splits(PatchTree::HORIZONTAL) ||
              current->splits(PatchTree::VERTICAL)))
        {
            current = current->getChild(PatchTree::CHILD);
        }
        else
        {
            current = current->getChild(PatchTree::NORTH_EAST);
        }
    }

    return current;
}

Mesh::RefineTreeIterator::RefineTreeIterator
    (vector<PatchTreeRoot *>::iterator begin) : PatchIterator(begin)
{
}

Mesh::VertexIterator::VertexIterator(vector<PatchTreeRoot *>::const_iterator
                                     begin) : patchIter(PatchIterator(begin))
{
}

Mesh::PrimalVertexIterator::PrimalVertexIterator(
    vector<PatchTreeRoot *>::const_iterator begin) : VertexIterator(begin),
                                                     vertexNum(0)
{
}

Mesh::UniquePrimalVertexLister::UniquePrimalVertexLister(
    vector<PatchTreeRoot *>::const_iterator begin,
    vector<PatchTreeRoot *>::const_iterator end)
{
    vertices.insert(PrimalVertexIterator(begin), PrimalVertexIterator(end));
}

Mesh::DualVertexIterator::DualVertexIterator(
    vector<PatchTreeRoot *>::const_iterator begin) : VertexIterator(begin)
{
}

unsigned int Mesh::getKnotIndex(KnotInterval *knot) const
{
    vector<KnotInterval *>::const_iterator it = find(knots.begin(),
                                                     knots.end(), knot);
    return distance(knots.begin(), it);
}

KnotInterval *Mesh::getKnotNumber(unsigned int index) const
{
    return knots[index];
}

ostream &operator<<(ostream &os, const Mesh &mesh)
{
    os << mesh.patchNodes.size() << " PatchNode objects" << endl;
    os << mesh.halfEdges.size() << " HalfEdge objects" << endl;
    os << mesh.patches.size() << " PatchTreeRoot objects" << endl;
    os << mesh.knots.size() << " KnotInterval objects" << endl;
    os << endl;

    os << "PatchNodes" << endl;
    os << "----------" << endl;
    for_each(mesh.patchNodes.begin(), mesh.patchNodes.end(),
             os << *lambda::_1);
    os << endl;

    os << "HalfEdges" << endl;
    os << "---------" << endl;
    for_each(mesh.halfEdges.begin(),  mesh.halfEdges.end(), os << *lambda::_1);
    os << endl;

    os << "PatchTreeRoots" << endl;
    os << "--------------" << endl;
    for_each(mesh.patches.begin(),    mesh.patches.end(),   os << *lambda::_1);
    os << endl;

    os << "KnotIntervals" << endl;
    os << "-------------" << endl;
    for_each(Mesh::KnotIterator(mesh.knots.begin()),
             Mesh::KnotIterator(mesh.knots.end()),   os << *lambda::_1);
    os << endl;

    os << "Vertices" << endl;
    os << "--------" << endl;
    forEachVertexUnique((os << *lambda::_1), &mesh);
    os << endl;

    return os;
}

}
