/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <boost/checked_delete.hpp>
#include <cassert>
#include "binarypatchtree.h"

namespace snurbs {

using namespace boost;
using namespace std;

BinaryPatchTree::BinaryPatchTree(Orientation  splitType,
                                 PatchTree   *child1,
                                 PatchTree   *child2) : PatchTree(),
                                                        splitType(splitType)
{
    children[0] = child1;
    children[1] = child2;

    children[0]->setParent(this);
    children[1]->setParent(this);
}

BinaryPatchTree::~BinaryPatchTree(void)
{
    for_each(children.begin(), children.end(), checked_delete<PatchTree>);
}

PatchTree *&BinaryPatchTree::getChild(Direction dir)
{
    switch(splitType)
    {
        case VERTICAL:
            if (dir & NORTH)
            {
                return children[0];
            }
            else if (dir & SOUTH)
            {
                return children[1];
            }
            break;

        case HORIZONTAL:
            if (dir & WEST)
            {
                return children[0];
            }
            else if (dir & EAST)
            {
                return children[1];
            }
            break;
    }

    throw invalid_child_direction();
}

PatchTree::Direction BinaryPatchTree::dirFromParent(void) const
{
    return parent->childDir(this);
}

PatchTree::Direction BinaryPatchTree::childDir(const PatchTree *child) const
{
    if (child == children[0])
    {
        switch(splitType)
        {
            case VERTICAL:
                return NORTH;
            case HORIZONTAL:
                return WEST;
        }
    }
    else if (child == children[1])
    {
        switch(splitType)
        {
            case VERTICAL:
                return SOUTH;
            case HORIZONTAL:
                return EAST;
        }
    }
    throw child_not_found();
}

void BinaryPatchTree::stream(ostream &os, unsigned char level) const
{
    streamDepthMarkers(os, level);
    os << " " << this << " : ";
    if (splitType == VERTICAL)
    {
        os << "NORTH-SOUTH";
    }
    else
    {
        assert(splitType == HORIZONTAL);
        os << "WEST-EAST";
    }
    os << endl;

    children[0]->stream(os, level + 1);
    children[1]->stream(os, level + 1);
}

}
