/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PATCHTREEROOT_H
#define PATCHTREEROOT_H

#include "abstractunarypatchtree.h"

/** @file */
// This file appears in the documentation because of the global function
// std::ostream &operator<<(std::ostream &os, const PatchTreeRoot &patch)
// defined here

namespace snurbs {

class HalfEdge;
class PatchTreeLeaf;

/**
 * Forms the base mesh, connected together by @link PatchNode PatchNodes
 * @endlink and @link HalfEdge HalfEdges @endlink.
 *
 * @ingroup Mesh
 */
class PatchTreeRoot : public AbstractUnaryPatchTree
{
    friend class PrimalMeshBuilder;

public:
    /**
     * PatchTreeRoot constructor.
     * @param patchTreeLeaf A PatchTreeRoot is always initialized with a
     * PatchTreeLeaf as the #child PatchTree. Subsequent subdivision replaces
     * this child with a BinaryPatchTree or a QuadPatchTree.
     * @param halfEdge One of the HalfEdge objects that surround this
     * PatchTreeRoot. This defines which HalfEdge is `north' for this
     * PatchTree.
     */
    PatchTreeRoot(PatchTreeLeaf *patchTreeLeaf, HalfEdge *halfEdge);

    bool isLeaf(void) const;

    bool isRoot(void) const;

    Direction dirFromParent(void) const;

    /**
     * @name Getter functions
     * @{
     */
    HalfEdge *getEdge(void) const;

    PatchTree *&getParent(void);

    void setParent(PatchTree *newParent);
    /** @} */

    void stream(std::ostream &os, unsigned char level) const;

private:
    /** The patch is oriented such that edge is the northern HalfEdge. */
    HalfEdge  *edge;
};

inline bool PatchTreeRoot::isLeaf(void) const
{
    return false;
}

inline bool PatchTreeRoot::isRoot(void) const
{
    return true;
}

inline HalfEdge *PatchTreeRoot::getEdge(void) const
{
    return edge;
}

/**
 * Stream << operator for PatchTreeRoot objects.
 */
std::ostream &operator<<(std::ostream &os, const PatchTreeRoot &patch);

}

#endif
