/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PATCHTREE_H
#define PATCHTREE_H

#include <ostream>
#include <stdexcept>

namespace snurbs {

/**
 * PatchTree is an abstract base class for the mixed @link QuadPatchTree quad
 * @endlink, @link BinaryPatchTree binary @endlink and @link UnaryPatchTree
 * unary @endlink trees that form each surface patch.
 *
 * @ingroup Mesh
 */
class PatchTree
{
public:
    /**
     * PatchTree destructor must be virtual so that derived classes can
     * polymorphically delete child pointers
     */
    virtual ~PatchTree(void) {};

    /**
     * Used to specify one of the two KnotInterval objects associated with
     * every PatchTreeLeaf, or to specify the type of binary split in a
     * BinaryPatchTree.
     */
    enum Orientation {
        HORIZONTAL, /**< Horizontal */
        VERTICAL    /**< Vertical   */
    };

    /**
     * The directions used to navigate a PatchTree.
     */
    enum Direction
    {
        NORTH = 1, /**< The North or top    edge/child */
        EAST  = 2, /**< The East  or right  edge/child */
        SOUTH = 4, /**< The South or bottom edge/child */
        WEST  = 8, /**< The West  or left   edge/child */

        NORTH_EAST = NORTH | EAST, /**< The North-East or    top-right child */
        SOUTH_EAST = SOUTH | EAST, /**< The South-East or bottom-right child */
        SOUTH_WEST = SOUTH | WEST, /**< The South-West or bottom-left  child */
        NORTH_WEST = NORTH | WEST, /**< The North-West or    top-left  child */

        /**
         * Used to specify the sole child pointer of a PatchTreeRoot or
         * UnaryPatchTree.
         */
        CHILD = 0,

        /**
         * Used in PatchTreeWalker::getInterval() to ask for an unspecified
         * child.
         */
        ANY = 15
    };

    /**
     * Convert a Direction to an index into QuadPatchTree::children or
     * PrimalPatchTreeLeaf::vertices
     */
    static unsigned char toIndex(Direction dir);

    /**
     * Convert an index into QuadPatchTree::children or
     * PrimalPatchTreeLeaf::vertices into a Direction
     */
    static Direction toDir(unsigned char index);

    /**
     * @return @c true if an object of type PatchTreeLeaf (or derived),
     * @c false otherwise.
     */
    virtual bool isLeaf(void) const = 0;

    /**
     * @return @c true if an object of type PatchTreeRoot, @c false otherwise.
     */
    virtual bool isRoot(void) const = 0;

    /**
     * @return @c true if an object that represents a split in the given
     * @c orientation, @c false otherwise.
     */
    virtual bool splits(Orientation orientation) const = 0;

    /**
     * Returns a child in the direction specified by @c dir, or throws an
     * invalid_argument exception if there is no valid child to return
     * (e.g.\ if a BinaryPatchTree with BinaryPatchTree::splitType VERTICAL
     * has getChild called with PatchTree::Direction EAST).
     */
    PatchTree* const &getChild(Direction dir) const;

    /**
     * Returns a child in the direction specified by @c dir, or throws an
     * invalid_argument exception if there is no valid child to return
     * (e.g.\ if a BinaryPatchTree with BinaryPatchTree::splitType VERTICAL
     * has getChild called with PatchTree::Direction EAST).
     */
    virtual PatchTree*& getChild(Direction dir) = 0;

    /**
     * Returns the PatchTree parent, unless called on a PatchTreeRoot object.
     * In which case, throws a logic_error exception.
     */
    PatchTree* const &getParent(void) const;

    /**
     * Returns the PatchTree parent, unless called on a PatchTreeRoot object.
     * In which case, throws a logic_error exception.
     */
    virtual PatchTree*& getParent(void) = 0;

    /**
     * Sets the PatchTree parent unless called on a PatchTreeRoot object, in
     * which case throws a logic_error exception.
     */
    virtual void setParent(PatchTree *newParent) = 0;

    /**
     * For any PatchTree with a parent, returns the PatchTree::Direction @c d
     * such that <tt>this == parent->getChild(d)</tt>
     */
    virtual Direction dirFromParent(void) const = 0;
    /**
     * For any PatchTree with children, returns the PatchTree::Direction @c d
     * such that <tt>child == getChild(d)</tt>
     */
    virtual Direction childDir(const PatchTree *child) const = 0;

    /**
     * Used to implement
     * operator<<(std::ostream &os, const PatchTreeRoot &patch)
     */
    virtual void stream(std::ostream &os, unsigned char level) const = 0;

protected:
    /**
     * Used to implement stream(std::ostream &os, unsigned char level) const
     */
    inline void streamDepthMarkers(std::ostream &os, unsigned char level) const
        { for (unsigned char l = 0; l < level; ++l) { os << " |"; } }
};

/**
 * Exception type thrown by PatchTree::getChild()
 *
 * @ingroup Exception
 */
class invalid_child_direction : public std::invalid_argument
{
public:
    /** invalid_child_direction constructor. */
    invalid_child_direction() :
        invalid_argument("No child in that direction") {};
};

/**
 * Exception type thrown by PatchTree::childDir()
 *
 * @ingroup Exception
 */
class child_not_found : public std::invalid_argument
{
public:
    /** child_not_found constructor. */
    child_not_found() : invalid_argument("No child matching that pointer") {};
};

/**
 * Thrown if any function finds a boundary and is thus unable to continue
 *
 * @ingroup Exception
 */
class hit_boundary : public std::runtime_error
{
public:
    /** hit_boundary constructor. */
    hit_boundary() : runtime_error("Hit the boundary") {};
};

inline unsigned char PatchTree::toIndex(Direction dir)
{
    switch (dir)
    {
        case NORTH_EAST:
            return 0;
        case SOUTH_EAST:
            return 1;
        case SOUTH_WEST:
            return 2;
        case NORTH_WEST:
            return 3;
        default:
            throw std::runtime_error("No corresponding index");
    }
}

inline PatchTree::Direction PatchTree::toDir(unsigned char index)
{
    switch (index)
    {
        case 0:
            return NORTH_EAST;
        case 1:
            return SOUTH_EAST;
        case 2:
            return SOUTH_WEST;
        case 3:
            return NORTH_WEST;
    }

    throw std::runtime_error("Invalid index");
};

inline PatchTree* const &PatchTree::getChild(Direction dir) const
{
    return const_cast<PatchTree *>(this)->getChild(dir);
}

inline PatchTree* const &PatchTree::getParent(void) const
{
    return const_cast<PatchTree *>(this)->getParent();
}

}

#endif
