/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PATCHNODE_H
#define PATCHNODE_H

#include <ostream>
#include "halfedge.h"
#include "patchtreewalker.h"
#include "types.h"

namespace snurbs {

class KnotInterval;

/**
 * PatchNode objects link up the PatchTreeRoot patches to form a mesh structure
 * with adjacency information provided by HalfEdge objects. PatchNode is not a
 * carrier of geometric information: for dual meshes, there is a PatchNode for
 * each @b face in the base mesh, whereas for primal meshes, each PatchNode
 * corresponds to a <b>control point</b>. The geometric positions of control
 * points are stored in Vertex objects, not in @link PatchNode PatchNodes
 * @endlink.
 *
 * @ingroup Mesh
 */
class PatchNode
{
    friend class PrimalMeshBuilder;

public:
    /** PatchNode constructor. */
    PatchNode(void);

    /** Getter for #edge */
    HalfEdge     *getEdge        (void) const;

    /** Getter for #spokeSetHead */
    PatchNode    *getSpokeSetHead(void) const;

    /** Getter for #minInterval */
    KnotPrecision getMinInterval (void) const;

    /**
     * Return the edge pointing to a PatchNode.
     * @param node A pointer to the target PatchNode.
     */
    HalfEdge *getEdgeTo(const PatchNode* const node) const;

    /**
     * The valency of this PatchNode.
     * @return The number of HalfEdge objects @c e with <tt>e.node == this</tt>
     */
    unsigned char valency(void) const;

    /**
     * Whether this PatchNode lies on a boundary.
     */
    bool onBoundary(void) const;

    /**
     * Whether this PatchNode is extraordinary (takes boundary into account)
     * @return @c true if this is a non-boundary node with irregular valency,
     * or a boundary node with valency greater than three. @c false otherwise.
     */
    bool extraordinary(void) const;

    /**
     * If this PatchNode is an extraordinary node (valency not equal to 4),
     * this method makes a request on every adjacent KnotInterval to insert
     * a new knot which is a uniform distance away from the node.
     */
    void requestNewKnots(void);

    /**
     * Updates the #minInterval held in the #spokeSetHead, if the interval
     * currently held there is greater than a KnotInterval surrounding this
     * PatchNode.
     */
    void findSpokeSetMin(void);

    /**
     * Calls KnotInterval::clearRequests() on every interval adjacent to this
     * PatchNode.
     */
    void removeRequests(void);

    /**
     * Make extraordinary adjustments to adjacent PatchTreeLeaf objects for
     * the refine stage.
     */
    void extraordinaryRefine(unsigned char degree);

    /**
     * Make extraordinary adjustments to adjacent PatchTreeLeaf objects for
     * a smoothing stage.
     */
    void extraordinarySmooth(unsigned char degree, unsigned char stage);

    /**
     * Makes the necessary adjustments to valency 3 nodes. Does nothing if
     * valency() is anything other than 3, or if onBoundary() returns @c true
     * @param degree The subdivision degree
     */
    void valency3Adjust(unsigned char degree);

    /**
     * Sets up the equivalence class of extraordinary points with spokes in
     * common.
     */
    void setUpSpokeSets(void);

private:
    /**
     * Points to one of the @link HalfEdge HalfEdges @endlink emanating from
     * this PatchNode.
     */
    HalfEdge *edge;

    /**
     * The representative PatchNode in the equivalence class of extraordinary
     * points with common spokes.
     */
    PatchNode *spokeSetHead;

    /**
     * The next PatchNode in the equivalence class of extraordinary points
     * with common spokes.
     */
    PatchNode *spokeSetNext;

    /**
     * The minimum non-zero knot interval surrounding this PatchNode.
     */
    KnotPrecision minInterval;

    /**
     * The cumulative product of weights from an extraordinary point to its
     * sucessor during a subdivision step. Used to determine how much a point
     * should move in the additional smoothing step for valency 3.
     * @see valency3Adjust()
     */
    double cumProdWeights;

    /**
     * Used to calculate #cumProdWeights
     */
    double evStageWeight;

    /**
     * Iterates over the PatchTreeLeaf objects adjacent to a PatchNode, by
     * means of a PatchTreeWalker. The PatchTreeWalker methods can then be used
     * to access KnotInterval or PatchTreeLeaf objects with the correct
     * implicit rotation applied.
     *
     * @ingroup Iterators
     */
    class SurroundingLeafIterator :
        public std::iterator<std::input_iterator_tag, const PatchTreeWalker>
    {
    public:
        /** SurroundingLeafIterator constructor */
        SurroundingLeafIterator(HalfEdge *edge);

        /**
         * @name Operators
         * @{
         */
        void operator++(void);
        const PatchTreeWalker &operator*(void);
        bool operator==(const SurroundingLeafIterator &iter);
        bool operator!=(const SurroundingLeafIterator &iter);
        /** @} */

    private:
        /**
         * The HalfEdge::NodeIterator used to traverse the HalfEdge objects
         * surrounding a PatchNode.
         */
        HalfEdge::NodeIterator nodeIter;
        /**
         * The PatchTreeWalker used to find the PatchTreeLeaf adjacent to a
         * PatchNode. Initialised using #nodeIter.
         */
        PatchTreeWalker walker;
    };
};

inline HalfEdge *PatchNode::getEdge(void) const
{
    return edge;
}


inline PatchNode *PatchNode::getSpokeSetHead(void) const
{
    return spokeSetHead;
}

inline KnotPrecision PatchNode::getMinInterval(void) const
{
    return minInterval;
}

/** Stream << operator for PatchNode objects. */
std::ostream &operator<<(std::ostream &os, const PatchNode &node);

}

#endif
