﻿// Uncomment this for the multi-run, timed configuration that should not be used with the CUDA profiler
//#define NO_PROFILER

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using VideoLib.Parameters;
using VideoLib.Stereo.GpGpu;
using System.Threading;
using Utils;

namespace StereoLibTester
{
    /// <summary>
    /// Static methods for a simple testing program, testing the functionality of VideoLib.Stereo.GpGpu
    /// </summary>
    /// <remarks>
    /// <para/>The stereo pipeline setup is defined in <see cref="ConfigureDisparityEstimator"/>.
    /// <para/>Try running with (command line arguments):
    /// <para>- tsukuba/left.png tsukuba/right.png 16 tsukuba/_out.png tsukuba/true.png 16 tsukuba/mask-all.png</para>
	/// <para>- venus/left.png venus/right.png 20 venus/_out.png venus/true.png 8 venus/mask-all.png</para>
	/// <para>- teddy/left.png teddy/right.png 64 teddy/_out.png teddy/true.png 4 teddy/mask-all.png</para>
	/// <para>- cones/left.png cones/right.png 64 cones/_out.png cones/true.png 4 cones/mask-all.png</para>
    /// <para>- hydra2/left.png hydra2/right.png 128 hydra2/_out.png</para>
    /// <para>- hydra3/left.png hydra3/right.png 100 hydra3/_out.png</para>
    /// <para>- hydra4/left.png hydra4/right.png 80 hydra4/_out.png</para>
    /// </remarks>
    public class StereoLibTesterProgram
    {
        #region Stereo Configuration

        private const int NframeIterations = 10;
        private const int ImageShift = 0; //100
        private static Func<DisparityEstimatorBase> EstimatorCreator = new Func<DisparityEstimatorBase>(DisparityEstimatorBase.CreateStandard);

        private static void ConfigureDisparityEstimator(DisparityEstimatorBase estimator, int nDisparities, string techniqueName)
        {
            // Set which techniques are to be used - max. of 1 per TechniqueType
            //SetTechnique(estimator, TechniqueType.Downsampler, "ImageDownsamplerFactory");
            //SetTechnique(estimator, TechniqueType.Upsampler, "JointBilateralUpsamplerFactory");
            //SetParameter(estimator, "ImageDownsampler", "DownsamplingFactor", 2);

            //SetTechnique(estimator, TechniqueType.PreProcessor, "ResidualPreProcessorFactory");
            //SetParameter(estimator, "ResidualPreProcessor", "Sigma", 4.0);
            //SetParameter(estimator, "ReMixingFactor", "Sigma", 0.15);

            SetTechnique(estimator, TechniqueType.CostComputer, "SadCostComputerFactory");
            SetParameter(estimator, "SadCostComputer", "NDisparityValues", nDisparities);
            SetParameter(estimator, "SadCostComputer", "CostGradient", 5.0f);
            SetParameter(estimator, "SadCostComputer", "CostLimit", 5.0f);

			//// HBP
			//SetTechnique(estimator, TechniqueType.Optimizer, "HbpOptimizerFactory");
			////SetParameter(estimator, "HbpOptimizer", "SmoothingGradient", 0.8);
			////SetParameter(estimator, "HbpOptimizer", "SmoothingLimit", 1.4f);
			//SetParameter(estimator, "HbpOptimizer", "SmoothingGradient", 0.4f);
			//SetParameter(estimator, "HbpOptimizer", "SmoothingLimit", 6.5f);
			//SetParameter(estimator, "HbpOptimizer", "NIterationsPerLevel", 7);

			// select cost aggregation technique
			if(techniqueName.ToLower().StartsWith("y"))
				SetTechnique(estimator, TechniqueType.Aggregator, "YoonKweonAggregatorFactory");
			else if (techniqueName.ToLower() == "dcb" || techniqueName.ToLower() == "naivedcb")
				SetTechnique(estimator, TechniqueType.Aggregator, "NaiveDcbAggregatorFactory");
			else if (techniqueName.ToLower() == "dcbgrid")
				SetTechnique(estimator, TechniqueType.Aggregator, "DcbGridAggregatorFactory");
			else if (techniqueName.ToLower() == "dcbgrid2")
				SetTechnique(estimator, TechniqueType.Aggregator, "DcbGrid2AggregatorFactory");
			else if (techniqueName.ToLower() == "tdcbgrid")
			{
				SetTechnique(estimator, TechniqueType.Aggregator, "TdcbGridAggregatorFactory");
				//SetParameter(estimator, "TdcbGridAggregator", "Weighting", 4);
			}

			//SetTechnique(estimator, TechniqueType.Maximizer, "SimpleMaximizerFactory");
			SetTechnique(estimator, TechniqueType.Maximizer, "SubPixelMaximizerFactory");

            //SetTechnique(estimator, TechniqueType.Aggregator, "DcbGridAggregatorFactory");
            //SetTechnique(estimator, TechniqueType.Aggregator, "ShiftAggregatorFactory");
            //SetTechnique(estimator, TechniqueType.GuidedGlobalOptimize, "GuidedHbpOptimizerFactory");
            //SetParameter(estimator, "GuidedHbpOptimizer", "OcclusionTolerance", 2.0f);
            //SetParameter(estimator, "ShiftAggregator", "BoxFilterHsize", 16);
            //SetParameter(estimator, "ShiftAggregator", "MinFilterHsize", 16);

            /*SetTechnique(estimator, TechniqueType.CostComputer, "SadByteCostComputerFactory");
            SetTechnique(estimator, TechniqueType.Optimizer, "ByteHbpOptimizerFactory");
            SetTechnique(estimator, TechniqueType.Maximizer, "SimpleBytesMaximizerFactory");
            SetParameter(estimator, "SadByteCostComputer", "NDisparityValues", nDisparities);
            SetParameter(estimator, "ByteHbpOptimizer", "NIterationsPerLevel", 6);*/
        }

        #endregion

        // Layout & path settings
        private const int ParamCharacterPadding = 35;
        private const int RunSeparatorWidth = 50;
        private const string DataDirectoryPrefix = ""; // "../../../data/";

		private static bool flip = false;

        /// <summary>
        /// Application entry point
        /// </summary>
        /// <param name="args">Command line arguments</param>
        public static void Main(string[] args)
        {
			//if(args.Length < 4)
			//{
			//    //Console.WriteLine("Usage: StereoLibTester.exe left_im_path right_im_path n_disparities " +
			//    //    "[out_im_path] [true_im_path] [true_im_scale] [mask_im_path] [out_error_im_path]");
			//    Console.WriteLine("Usage: StereoLibTester.exe left_im_path right_im_path n_frames n_disparities " +
			//        "[out_im_path] [technique_name] [flip]");
			//    return;
			//}

			////
			//// Read in the command line arguments

			//string lImagePath = args[0];
			//string rImagePath = args[1];

			//int nFrames = Int32.Parse(args[2]);
			//int nDisparities = Int32.Parse(args[3]);

			//string outImagePath = (args.Length > 4 ? args[4] : "disparities.png");
			////string trueImagePath = (args.Length > 4 ? args[4] : null);
			////int trueScaleFactor = (args.Length > 5 ? Int32.Parse(args[5]) : 0);
			////string maskImagePath = (args.Length > 6 ? args[6] : null);
			////string errorImagePath = (args.Length > 7 ? args[7] : null);
			//string techniqueName = (args.Length > 5 ? args[5] : "dcbgrid");
			//int noiseLevel = (args.Length > 6 ? Int32.Parse(args[6]) : 10);
			//flip = (args.Length > 7 ? true: false);

//---- 2010-03-06 for stereo still images (e.g. LeftOnlyStereo) -----------------------------------

			//if (args.Length < 4)
			//{
			//    Console.WriteLine("Usage: StereoLibTester.exe left_im_path right_im_path n_disparities " +
			//        "[out_im_path] [technique_name] [flip]");
			//    return;
			//}

			//// Read in the command line arguments
			//string lImagePath = args[0];
			//string rImagePath = args[1];
			//int nDisparities = Int32.Parse(args[2]);
			//string outImagePath = (args.Length > 3 ? args[3] : "disparities.png");
			//string techniqueName = (args.Length > 4 ? args[4] : "dcbgrid");
			//flip = (args.Length > 5 ? true : false);

			//int nFrames = 1;
			//int noiseLevel = 0;

//---- 2010-03-12 for stereo videos (e.g. RunStereoVideo) -----------------------------------------

			if (args.Length < 4)
			{
				Console.WriteLine("Usage: StereoLibTester.exe left_im_path right_im_path n_frames n_disparities " +
					"[out_im_path] [technique_name] [flip]");
				return;
			}

			string lImagePath = args[0];
			string rImagePath = args[1];

			int nFrames = Int32.Parse(args[2]);
			int nDisparities = Int32.Parse(args[3]);

			string outImagePath = (args.Length > 4 ? args[4] : "disparities.png");
			string techniqueName = (args.Length > 5 ? args[5] : "dcbgrid");
			int noiseLevel = (args.Length > 6 ? Int32.Parse(args[6]) : 0);
			flip = (args.Length > 7 ? true : false);

//---- finished reading in command line arguments -------------------------------------------------
			GenerateDepthMaps(lImagePath, rImagePath, outImagePath, nFrames, nDisparities, techniqueName, noiseLevel);

			////
			//// Accuracy checking

			//if (trueImagePath != null)
			//{
			//    Bitmap tImage = new Bitmap(DataDirectoryPrefix + trueImagePath);

			//    // Checking can only be performed with an image of the same size
			//    if (tImage.Width != lImage.Width || tImage.Height != lImage.Height)
			//    {
			//        Console.WriteLine("Error: The true disparity image must be the same size (currently true:({0},{1}), stereo:({2},{3}))",
			//            tImage.Width, tImage.Height, lImage.Width, lImage.Height);
			//        return;
			//    }

			//    // Masking is optional
			//    Bitmap maskImage = null;
			//    if (maskImagePath != null)
			//    {
			//        maskImage = new Bitmap(DataDirectoryPrefix + maskImagePath);

			//        // Masking can only be performed with an image of the same size
			//        if (maskImage.Width != lImage.Width || maskImage.Height != lImage.Height)
			//        {
			//            Console.WriteLine("Error: The true disparity image must be the same size (currently mask:({0},{1}), stereo:({2},{3}))",
			//                maskImage.Width, maskImage.Height, lImage.Width, lImage.Height);
			//            return;
			//        }
			//    }

			//    string maskImageName = (maskImagePath == null ? "unmasked" : Path.GetFileNameWithoutExtension(maskImagePath));

			//    // Evaluate the accuracy of the depth map, and write out the results
			//    EvaluateDepthMap(rawResult, lImage.Width, lImage.Height, tImage, maskImage, maskImageName, errorImagePath, scale, trueScaleFactor, 1.0f);
			//}

			//Console.WriteLine();
			//FinishIfDebug();
        }

        private static void CopyInInputs(Bitmap lImage, Bitmap rImage, byte[] rawImageL, byte[] rawImageR)
        {
            // Copy out raw bitmap data
            BitmapData imageLdata = lImage.LockBits(new Rectangle(Point.Empty, lImage.Size), ImageLockMode.ReadOnly, lImage.PixelFormat);
            BitmapData imageRdata = rImage.LockBits(new Rectangle(Point.Empty, lImage.Size), ImageLockMode.ReadOnly, rImage.PixelFormat);

            int bppL = (lImage.PixelFormat == PixelFormat.Format32bppArgb ? 4 : 3);
            int bppR = (rImage.PixelFormat == PixelFormat.Format32bppArgb ? 4 : 3);

            unsafe
            {
                byte* pLdata = (byte*)imageLdata.Scan0.ToPointer();
                byte* pRdata = (byte*)imageRdata.Scan0.ToPointer();

                fixed (byte* pOutL = &rawImageL[0], pOutR = &rawImageR[0])
                {
                    for (int y = 0; y < imageLdata.Height; ++y)
                        for (int x = 0; x < imageLdata.Width; ++x)
                        {
                            *((uint*)pOutL + lImage.Width * y + x) = MakeUintFromBytes(*(pLdata + imageLdata.Stride * y + bppL * x + 2),
                                *(pLdata + imageLdata.Stride * y + bppL * x + 1),
                                *(pLdata + imageLdata.Stride * y + bppL * x));

                            int sx = x - ImageShift;
                            *((uint*)pOutR + lImage.Width * y + x) = (sx <= 0 ? 0 : MakeUintFromBytes(*(pRdata + imageRdata.Stride * y + bppR * sx + 2),
                                *(pRdata + imageRdata.Stride * y + bppR * sx + 1),
                                *(pRdata + imageRdata.Stride * y + bppR * sx)));
                        }
                }
            }
            lImage.UnlockBits(imageLdata);
            rImage.UnlockBits(imageRdata);
        }

        private static void SaveResult(byte[] rawResult, int width, int height, string outImagePath)
        {
            Bitmap result = new Bitmap(width, height, PixelFormat.Format32bppRgb);

            // Copy in resultant raw bitmap data
            BitmapData resultData = result.LockBits(new Rectangle(0, 0, width, height), ImageLockMode.WriteOnly, PixelFormat.Format32bppRgb);
            unsafe
            {
                uint* pOutData = (uint*)resultData.Scan0.ToPointer();

                fixed (byte* pResult = &rawResult[0])
                {
                    float* rdata = (float*)pResult;

					if (flip)
					{
						for (int y = 0; y < resultData.Height; ++y)
							for (int x = 0; x < resultData.Width; ++x)
								pOutData[(resultData.Stride / 4) * y + x] = MakeUintFromFloat(rdata[width * y + (resultData.Width - 1 - x)]);
					}
					else
					{
						for (int y = 0; y < resultData.Height; ++y)
							for (int x = 0; x < resultData.Width; ++x)
								pOutData[(resultData.Stride / 4) * y + x] = MakeUintFromFloat(rdata[width * y + x]);
					}
                }
            }
            result.UnlockBits(resultData);

            // Save the resultant bitmap
            result.Save(outImagePath);
        }

		private static void GenerateDepthMaps(string lImagePath, string rImagePath, string outImagePath, int frames, int nDisparities, string techniqueName, int noiseLevel)
		{
			using (DisparityEstimatorBase dEstimator = EstimatorCreator())
			{
				// Set up the disparity estimator
				ConfigureDisparityEstimator(dEstimator, nDisparities, techniqueName);

				for (int frame = 1; frame < frames + 1; frame++)
				{
					// Load the images
					Bitmap lImage = new Bitmap(DataDirectoryPrefix + String.Format(lImagePath, frame));
					Bitmap rImage = new Bitmap(DataDirectoryPrefix + String.Format(rImagePath, frame));

					if (flip)
					{
						lImage.RotateFlip(RotateFlipType.RotateNoneFlipX);
						rImage.RotateFlip(RotateFlipType.RotateNoneFlipX);
					}

					// Depth-map generation can only be performed on two equal-size images
					if (lImage.Width != rImage.Width || lImage.Height != rImage.Height)
					{
						Console.WriteLine("Error: Input images must be the same size (currently ({0},{1}) and ({2},{3}))",
							lImage.Width, lImage.Height, rImage.Width, rImage.Height);
						return;
					}

					// Only one format is currently understoond
					if ((lImage.PixelFormat != PixelFormat.Format24bppRgb && lImage.PixelFormat != PixelFormat.Format32bppArgb) ||
						(rImage.PixelFormat != PixelFormat.Format24bppRgb && rImage.PixelFormat != PixelFormat.Format32bppArgb))
					{
						Console.WriteLine("Error: Input images must be either 24 or 32 bits-per-pixel RGB");
						return;
					}

					// add noise
					if (noiseLevel > 0)
					{
						//ApplyAdditiveUniformNoise(lImage, noiseLevel, 2 * frame + (flip ? 0 : 1));
						//ApplyAdditiveUniformNoise(rImage, noiseLevel, 2 * frame + (flip ? 1 : 0));
						ApplyAdditiveGaussianNoise(lImage, noiseLevel, 2 * frame + (flip ? 0 : 1));
						ApplyAdditiveGaussianNoise(rImage, noiseLevel, 2 * frame + (flip ? 1 : 0));
					}

					// for saving noisy input images (for supplementary videos)
					// NB. can comment out rest of this function
					if (flip)
					{
						lImage.RotateFlip(RotateFlipType.RotateNoneFlipX);
						lImage.Save(DataDirectoryPrefix + String.Format(outImagePath, frame));
						lImage.RotateFlip(RotateFlipType.RotateNoneFlipX);
					}
					else
					{
						lImage.Save(DataDirectoryPrefix + String.Format(outImagePath, frame));
					}

					//byte[] rawImageL = new byte[sizeof(int) * lImage.Width * lImage.Height];
					//byte[] rawImageR = new byte[sizeof(int) * lImage.Width * lImage.Height];
					//byte[] rawResult = new byte[sizeof(float) * lImage.Width * lImage.Height];
					//int width = lImage.Width;
					//int height = lImage.Height;
					//float scale = 1.0f / ((float)nDisparities);

					//// Do the actual copying and processing
					//CopyInInputs(lImage, rImage, rawImageL, rawImageR);
					//Console.WriteLine("Processing frame {0} of {1} ({2}x{3}) with {4} disparity values ...", frame, frames, width, height, nDisparities);

					//// Do a seperate first run (this includes initialization)
					//DateTime start = DateTime.Now;
					//dEstimator.ProcessFrame(rawImageL, rawImageR, width, height, rawResult);
					//Console.WriteLine("Processed frame 0 in {0}", DateTime.Now - start);
					////start = DateTime.Now;
					////dEstimator.ProcessFrame(rawImageL, rawImageR, width, height, rawResult);
					////Console.WriteLine("Processed frame 1 in {0}\r\n", DateTime.Now - start);
					//PrintSeparator();

					//// If not profiling, do a number of runs for average performance
					//TimeDepthMaps(dEstimator, rawImageL, rawImageR, width, height, nDisparities, scale, rawResult);
					//SaveResult(rawResult, width, height, DataDirectoryPrefix + String.Format(outImagePath, frame));
				}

                // Write out the settings used
                Console.WriteLine("\r\nSettings:");
                PrintConfiguration(dEstimator);

                PrintSeparator();
            }
		}

		private static void ApplyAdditiveUniformNoise(Bitmap image, int noiseLevel, int seed)
		{
			Random random = new Random(seed);
			BitmapData data = new BitmapData();
			image.LockBits(new Rectangle(Point.Empty, image.Size), ImageLockMode.ReadWrite, image.PixelFormat, data);
			unsafe
			{
				byte* pImage = (byte*)data.Scan0.ToPointer();
				for (int i = 0; i < data.Stride * data.Height; i++)
				{
					double newVal = (int)(*pImage) + noiseLevel * (random.NextDouble() - 0.5) + 0.5;
					if (newVal > 255) *pImage = 255;
					else if (newVal < 0) *pImage = 0;
					else *pImage = (byte)newVal;
					pImage++;
				}
			}
			image.UnlockBits(data);
		}

		private static void ApplyAdditiveGaussianNoise(Bitmap image, int sigma, int seed)
		{
			GaussianRandom random = new GaussianRandom(seed);
			BitmapData data = new BitmapData();
			image.LockBits(new Rectangle(Point.Empty, image.Size), ImageLockMode.ReadWrite, image.PixelFormat, data);
			unsafe
			{
				byte* pImage = (byte*)data.Scan0.ToPointer();
				for (int i = 0; i < data.Stride * data.Height; i++)
				{
					double newVal = (int)(*pImage) + random.NextGaussian(0.0, sigma) + 0.5;
					if (newVal > 255) *pImage = 255;
					else if (newVal < 0) *pImage = 0;
					else *pImage = (byte)newVal;
					pImage++;
				}
			}
			image.UnlockBits(data);
		}

        [Conditional("NO_PROFILER")]
        private static void TimeDepthMaps(DisparityEstimatorBase dEstimator, byte[] imageL, byte[] imageR, 
            int width, int height, int nDisparities, float scale, byte[] result)
        {
            // Do a number of runs to obtain an averaged performance figure
			Stopwatch watch = new Stopwatch();
			TimeSpan total = TimeSpan.Zero;
            for (int i = 0; i < NframeIterations; ++i)
            {
				// start stopwatch
				watch.Reset();
				watch.Start();
				DateTime start = DateTime.Now;

				// run computation
                dEstimator.ProcessFrame(imageL, imageR, width, height, result);

				// stop stopwatch
				watch.Stop();
				TimeSpan taken = watch.Elapsed;
				total += taken;

				// print status
                Console.WriteLine("Processed frame {0} in {1}", i + 2, taken);
                PrintSeparator();
            }

            // Write out the performance results
            Console.WriteLine("Performance ({0}x{1}x{2}):", width, height, nDisparities);
            Console.WriteLine("\r\n  {0} ms\r\n  {1} fps", (total.TotalMilliseconds / NframeIterations).ToString("#0.000"), 
                (NframeIterations / total.TotalSeconds).ToString("#0.000"));
        }

        private static unsafe void EvaluateDepthMap(byte[] rawResult, int width, int height, Bitmap tImage, Bitmap maskImage, string maskName, string errorImagePath, 
            float dispScale, float trueScale, float threshold)
        {
            uint numBadPixels = 0;
            uint numPixels = 0;

            if(maskImage == null)
            {
                // Iterate over every pixel, counting error pixels
                fixed (byte* pResult = &rawResult[0])
                {
                    float* result = (float*)pResult;

                    for (int y = 0; y < height; ++y)
                        for (int x = 0; x < width; ++x)
                        {
                            if (Math.Abs(result[width * y + x] / dispScale - (float)tImage.GetPixel(x, y).B / trueScale) > threshold + 1e-4f)
                                ++numBadPixels;
                            ++numPixels;
                        }
                }
            }
            else
            {
                // Iterate over all masked pixels, counting error pixels
                fixed (byte* pResult = &rawResult[0])
                {
                    float* result = (float*)pResult;

                    for (int y = 0; y < height; ++y)
                        for (int x = 0; x < width; ++x)
                            if (maskImage.GetPixel(x, y).B == 255) // Only consider white pixels (on a black-and-white mask)
                            {
                                if (Math.Abs(result[width * y + x] / dispScale - (float)tImage.GetPixel(x, y).B / trueScale) > threshold + 1e-4f)
                                    ++numBadPixels;
                                ++numPixels;
                            }
                }
            }

            Console.WriteLine("\r\nResults:");
            Console.WriteLine("\r\n  Percentage bad pixels ({0}): {1}%", maskName, (100.0f * (float)numBadPixels / (float)numPixels).ToString("#0.#"));
        }

        #region Helper functions

        [Conditional("NO_PROFILER")]
        private static void FinishIfDebug()
        {
			//Console.Write("Press any key to exit...");
			//Console.ReadKey(true);
        }

        private static void SetTechnique(DisparityEstimatorBase estimator, TechniqueType type, string name)
        {
            TechniqueDescription desc = estimator.GetTechniques(type).Find(x => (x.TypeName == name));
            if (desc.Name == null)
            {
                Console.WriteLine("Configuration Error: Factory \"{0}\" does not exist - ignoring this technique setting.", name);
                return;
            }
            estimator.SetTechnique(desc);
        }

        private static void SetParameter(DisparityEstimatorBase estimator, string nodeName, string paramName, object value)
        {
            // Find the node
            List<IStereoNode> nodes = estimator.ActiveParamObjects.FindAll(x => (x.GetType().Name == nodeName));
            if (nodes == null || nodes.Count == 0)
            {
                Console.WriteLine("Configuration Error: Node \"{0}\" does not exist - ignoring this parameter setting.", nodeName);
                return;
            }

            // Find the parameter
            ParameterInfo info;
            if (!ParameterReflector.TryFindParameter(nodes[0].GetType(), paramName, ref info))
            {
                Console.WriteLine("Configuration Error: Parameter \"{0}\" for node \"{1}\" does not exist - ignoring this parameter setting.", paramName, nodeName);
                return;
            }

            foreach (IStereoNode node in nodes)
            {
                // Set the parameter value
                info.SetValue(node, value);

                if (node.FollowerNodes != null)
                    foreach (IStereoNode follower in node.FollowerNodes)
                        info.SetValue(follower, value);
            }
        }

        private static uint MakeUintFromFloat(float val)
		{
			//uint uVal = (uint)(val * 255.0f) & 0xFF; // no clipping? no rounding?
			uint uVal = (uint)Math.Max(0, Math.Min((int)(255.0f * val + 0.5f), 255));
            return uVal * (1 + (1 << 8) + (1 << 16));
        }

        private static uint MakeUintFromBytes(byte r, byte g, byte b)
        {
            return (((uint)r & 0xFF) << 16) + (((uint)g & 0xFF) << 8) + ((uint)b & 0xFF);
        }

        private static void PrintSeparator()
        {
			Console.WriteLine(Environment.NewLine + "{0}" + Environment.NewLine, new String('-', RunSeparatorWidth));
        }

        private static void PrintConfiguration(DisparityEstimatorBase dEstimator)
        {
            PrintConfiguration(dEstimator, TechniqueType.PreProcessor);
            PrintConfiguration(dEstimator, TechniqueType.CostComputer);
            PrintConfiguration(dEstimator, TechniqueType.Aggregator);
            PrintConfiguration(dEstimator, TechniqueType.Maximizer);
        }

        private static void PrintConfiguration(DisparityEstimatorBase dEstimator, TechniqueType type)
        {
            IStereoNode node = dEstimator.ActiveParamObjects.Find(x => x.Type == type);

            if (node != null)
            {
                List<ParameterInfo> paramsInfo = ParameterReflector.EnumerateParameters(node.GetType());

                // Print out a header, then a list of all the parameter values
                Console.WriteLine("\r\n  {0}- {1}", type, node.GetType().Name);
                foreach (ParameterInfo info in paramsInfo)
                    Console.WriteLine("    {0}{1}= {2}", 
                        info.Name, new string(' ', Math.Max(0, ParamCharacterPadding - info.Name.Length)), 
                        info.GetValue(node));
            }
        }

        #endregion

        #region Currently unused

        private static void PrintTechniqueList(DisparityEstimatorBase estimator, TechniqueType type, char id)
        {
            List<TechniqueDescription> descs = estimator.GetTechniques(type);

            Console.WriteLine("{0}: {1} techniques:", id, Enum.GetName(typeof(TechniqueType), type));
            for(int i = 0; i < descs.Count; ++i)
                Console.WriteLine("\t{0}: {1}", i, descs[i].Name);

            Console.WriteLine();
        }

        #endregion
    }
}
