#include "RPii.h"

/*Prototypes*/
//Process Functions
int Process_Mesg_ACC(struct Log_Control *Control, struct cwiid_acc_mesg acc_mesg);
int Process_State_ACC(struct Log_Control *Control);
int Process_Mesg_BTN(struct Log_Control *Control, uint16_t BTNPressed);
int Process_EndOfLog( struct Log_Control *Control, char* filename);
float Process_Time_diff(struct timeval t1, struct timeval t2);
char* Process_CreateTimestamp(void);
int Process_OpenFilename(struct files *files, char *filename);
float Process_Calibrate_Value(struct Log_Control *Control, int CalAxis);
//Control Functions
int Toggle_ACC_Mesg(struct RPii *RPii);
int Toggle_ACC_prnts(struct PRNTACC *tempprnt);
int Control_Change_Settings(struct RPii *RPii);
int RPii_Set_Ctrl_Defaults(struct RPii *RPii);

/*MACROS*/
// Used to toggle the output to the wii. e.g for turning LEDs on.
#define toggle_bit(bf,b)	\
	(bf) = ((bf) & b)		\
	       ? ((bf) & ~(b))	\
	       : ((bf) | (b))
	       
//Calibration Defaults which are initialised
#define defaultmx 	-0.3773
#define defaultcx 	47.54
#define defaultmy	-0.3702
#define defaultcy	45.34
#define defaultmz	-0.3924
#define defaultcz	48.66

/*****Process Functions********/
//
//
//A function to process the ACC messages, storing the callback mesg in Control.Acc_Data and passing Control to Print_ACC
//
int Process_Mesg_ACC(struct Log_Control *Control, struct cwiid_acc_mesg acc_mesg){
		
	//Assign acc_mesg to Acc_Data in Control
	// mesgStream_on is false, then the wiimote will not be sending any ACC_mesgs, so this function will not be called.
	Control->Acc_Data.X = (uint8_t)acc_mesg.acc[CWIID_X];
	Control->Acc_Data.Y = (uint8_t)acc_mesg.acc[CWIID_Y];
	Control->Acc_Data.Z = (uint8_t)acc_mesg.acc[CWIID_Z];
	return 1;
}
//
//A function to process the ACC messages using get state
//
int Process_State_ACC(struct Log_Control *Control){

	//Assign the data collected in wiimote.state to Acc_Data in Control
	Control->Acc_Data.X = (uint8_t)Control->state.acc[CWIID_X];
	Control->Acc_Data.Y = (uint8_t)Control->state.acc[CWIID_Y];
	Control->Acc_Data.Z = (uint8_t)Control->state.acc[CWIID_Z];
	return 1;
}
//
//A function to process btn calls from the Wiimote.
//
int Process_Mesg_BTN(struct Log_Control *Control, uint16_t BTNPressed){
	
	//Change Flags appropriately
	switch ( BTNPressed ) {
	
	case 0:
		//printf("All buttons are released\n");
		break;
	
	case CWIID_BTN_A:
		printf("Button A has been pressed\n");
		if (!Control->log_flags.Enable_Log && !Control->log_flags.calibrating){
			//only enable logging if Chnge_Set is not Set			
			if(!Control->log_flags.Chnge_Set){	
				printf("LOGGING STARTS IN:\n");
				Print_Countdown(500000, 5);
				//enable logging
				Control->log_flags.Enable_Log= true;
				printf("LOGGING STARTED\n");
			}
			else {
				printf("Exit Settings before starting Logging\n");
			}
		}
		Control->btn_flags.A_flag = true;
		break;
	
	case CWIID_BTN_B:
		printf("Button B has been pressed\n");
		
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.B_flag = true;
		}
	
		//set Exit_BTN for 'exit_type_satisfied'
		if (Control->log_flags.Enable_Log) {
			Control->log_flags.Exit_BTN = true;
		}
		
		break;
	
	case CWIID_BTN_HOME:
		//If we are in calibrating mode, it must be finnished before exit
		if(Control->log_flags.calibrating){
			printf("Please finnish the Calibrating routine before exiting");
			break;
		}
		//If we are in Chnge_Settings mode, exit
		if (Control->log_flags.Chnge_Set){
			Control->log_flags.Chnge_Set = false;
		}
		//If home is pressed mid-logging, end the log
		if(Control->log_flags.Enable_Log && Control->log_flags.dataFile_Open){
			Control->log_flags.Enable_Log = false;
			Process_EndOfLog(Control, "TERMINATED LOG");
		}
		//Then exit the Log_Loop function
		Control->log_flags.End_Loop = true;
		printf("Home has been pressed, terminating program\n");
		break;
		
	case CWIID_BTN_UP:
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.UP_flag = true;
		}
		break;
		
	case CWIID_BTN_RIGHT:
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.RIGHT_flag = true;
		}
		break;
		
	case CWIID_BTN_LEFT:
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.LEFT_flag = true;
		}
		break;
		
	case CWIID_BTN_DOWN:
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.DOWN_flag = true;
		}
		break;
		
	case CWIID_BTN_1:
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.ONE_flag = true;
		}
		break;
		
	case CWIID_BTN_2:
		if( Control->log_flags.Chnge_Set){
			Control->btn_flags.TWO_flag = true;
		}
		break;
		
	case 0x1010: //The input for the plus and minus button pressed simultaneously
		
		if (!Control->log_flags.Enable_Log && !Control->log_flags.calibrating){
		
			if(!Control->log_flags.Chnge_Set){
				//Enable settings to be changed
				Control->log_flags.Chnge_Set = true;
			}
			else {
				Control->log_flags.Chnge_Set = false;
			}
		}
		else {
			printf("Settings cannot be changed until other events are closed\n");
		}
	
		break;
		
	case 0x0003: // The input for 1 and 2 pressed together
	
		if (!Control->log_flags.Enable_Log && !Control->log_flags.Chnge_Set){
		
			Control->log_flags.calibrating = true;
			
		}
		else {
			printf("Cannot calibrate until other events are closed\n");
		}
	
		break;

	
	default:
		//printf("Another Button has been pressed\n");
		break;
	}
	
	return 0;
}
//
//A function which closes the dataFile in Log_Control, and saves the Sample.count,
//and time of sampling etc in a new line of infoFile. 
//
int Process_EndOfLog(struct Log_Control *Control, char* filename){
	//TODO: complete this function
	
	//Check that structures have been declared
	if(Control == NULL){
		fprintf(stderr, "Error, Pointer to Control struct is NULL\n");
		exit(-1);
	}
	//Check that structures have been declared
	if(filename == NULL){
		fprintf(stderr, "Error, Pointer to filenameis NULL, cannot update infoFile\n");
		exit(-1);
	}
	
	//Write to infoFile...
	//Open infoFile
	Control->files.infoFile = fopen("infoFile", "a");
	if (Control->files.infoFile == NULL) {
		fprintf(stderr, "Could not open info file for writing.\n");
		exit(-1);
	}
	// Check if infoFile is empty, if it is, write a header line.
	fseek(Control->files.infoFile, 0 , SEEK_END);
	if (ftell(Control->files.infoFile)==0){
		printf("No infoFile found, creating a new infoFile\n");
		fprintf(Control->files.infoFile, "FileName, SampleCount, TimeElapsed\n");
	}
	
	//Find recording length in seconds and print
	float TimeElapsed = Process_Time_diff( Control->startTime, Control->sampleTime)/1000000;
	printf ("%d samples taken in %f seconds\n\n",Control->samples.count, TimeElapsed );
	
	fprintf(Control->files.infoFile, " %s, %d, %f\n", filename, Control->samples.count, TimeElapsed);
	
	//Close dataFile and change dataFile_Open
	fclose(Control->files.dataFile);
	fclose(Control->files.infoFile);
	Control->log_flags.dataFile_Open = false;
	return 0;
}
	

//
// A function  to return the time between two time variables
//
float Process_Time_diff(struct timeval t1, struct timeval t2){
	/* Due to the overflow of the tv_usec variable, care must be taken when subtracting*/
	
	float secdiff = t2.tv_sec - t1.tv_sec;
	float usecdiff = t2.tv_usec - t1.tv_usec;
	
	if (usecdiff < 0){
		usecdiff += 1000000;
		secdiff -= 1;
	}
	
	float uTdiff = 1000000*secdiff + usecdiff;
	return uTdiff;
}
//
//A function to create a timestamp; returns the TimeStamp
//
char* Process_CreateTimestamp(void){
	//create a variable to store the time and filename in
	char* filename;
	time_t currentTime;
	struct tm *tm;
	//get the time and structure
	currentTime = time(NULL);
	tm = localtime(&currentTime);
	//Allocate memory to this filename so that it is not overwritten. MAKE SURE THIS IS DESTROYED!
	filename = (char*)malloc(30);
	//write string as WiiData_Day.Month.Year_Hour:Min:Sec
	sprintf(filename, "WiiData_%d-%d-%d_%d-%d-%d.csv", tm->tm_mday, tm->tm_mon+1, tm->tm_year+1900, tm->tm_hour, tm->tm_min, tm->tm_sec);
	return filename;
}
//
//A function to open a file given a filename
//
int Process_OpenFilename(struct files *files, char *filename){
	//Open/create a file of name filename
	files->dataFile = fopen(filename, "w");
	if (files->dataFile == NULL) {
		fprintf(stderr, "Could not open data file for writing.\n");
		exit(-1);
	}
	return 0;
}
//
//A function to change a value between to 0 and 255, into ms^-2
//
float Process_Calibrate_Value(struct Log_Control *Control, int CalAxis){
	float m = 0;
	float c = -1;
	uint8_t value;
	float Acc_ms;
	
	//Depending on CalAxis, assign different values to the variables
	//For X axis, the first bit is set, for Y axis, the second bit and for the Z axis, the 3rd
	// For NOCAL values, bit 5 is also set, this causes 'value' to be returned without calibrating
	if ( CalAxis & CALX){
		value = Control->Acc_Data.X;
		if( CalAxis & 0x10){
			return value;
		}
		m = Control->Cal_Consts.mx;
		c = Control->Cal_Consts.cx;
	}
	
	if ( CalAxis & CALY ) {
		value = Control->Acc_Data.Y;
		if( CalAxis & 0x10){
			return value;
		}
		m = Control->Cal_Consts.my;
		c = Control->Cal_Consts.cy;
	}
	
	if ( CalAxis & CALZ ) {
		value = Control->Acc_Data.Z;
		if( CalAxis & 0x10){
			return value;
		}
		m = Control->Cal_Consts.mz;
		c = Control->Cal_Consts.cz;
	}

	//Using y = mx + c, where m and c have been saved previously, compute y from x.
	Acc_ms = m*value + c;
	
	return Acc_ms;
}
//
//
/*********Control Functions*********/
//
//A function to set control defaults
//
int RPii_Set_Ctrl_Defaults(struct RPii *RPii){
	
	//Initialise samples variables
	RPii->Control.samples.count = 0;
	RPii->Control.samples.max = 100;
	RPii->Control.samples.wait = 100000; //default of 10 samples per second
	RPii->Control.samples.LogDuration = 15000000;
	RPii->Control.samples.wait_offset = 0;
	RPii->Control.samples.wait_adjusted = 0;
	
	//Set Acc_data to -1 to show if no data has been collected
	RPii->Control.Acc_Data.X = -1;
	RPii->Control.Acc_Data.Y = -1;
	RPii->Control.Acc_Data.Z = -1;
	
	//Initialise btn_flags
	RPii->Control.btn_flags.A_flag = false;
	RPii->Control.btn_flags.B_flag = false;
	RPii->Control.btn_flags.HOME_flag = false;
	RPii->Control.btn_flags.PLUS_flag = false;
	RPii->Control.btn_flags.MINUS_flag = false;
	RPii->Control.btn_flags.RIGHT_flag = false;
	RPii->Control.btn_flags.LEFT_flag = false;
	RPii->Control.btn_flags.UP_flag = false;
	RPii->Control.btn_flags.DOWN_flag = false;
	RPii->Control.btn_flags.ONE_flag = false;
	RPii->Control.btn_flags.TWO_flag = false;

	
	//Initialise logging flags
	RPii->Control.log_flags.dataFile_Open= false;
	RPii->Control.log_flags.Get_Sample = false;
	RPii->Control.log_flags.Enable_Log = false;
	RPii->Control.log_flags.End_Loop = false;
	RPii->Control.log_flags.Chnge_Set = false;
	RPii->Control.log_flags.calibrating = false;
	RPii->Control.log_flags.mesgStream_on = true;
	RPii->Control.log_flags.Exit_BTN = false;	
	RPii->Control.log_flags.PRNTACC.X= 1;
	RPii->Control.log_flags.PRNTACC.Y= 1;
	RPii->Control.log_flags.PRNTACC.Z= 1;
	RPii->Control.log_flags.Exit_type = EXIT_BUTTONB;
	
	//Initialise Calibration values
	RPii->Control.Cal_Consts.mx = defaultmx;
	RPii->Control.Cal_Consts.cx = defaultcx;
	RPii->Control.Cal_Consts.my = defaultmy;
	RPii->Control.Cal_Consts.cy = defaultcy;
	RPii->Control.Cal_Consts.mz = defaultmz;
	RPii->Control.Cal_Consts.cz = defaultcz;
	
	return 0;
}
//
//A function to toggle ACC messaging, and the mesgSteam on or off
//
int Toggle_ACC_Mesg(struct RPii *RPii){
	
	//initialise rpt_mode and update the latest flags from wiimote
	unsigned char rpt_mode = 0;
	RPii->Control = *(Pointto_Control_Data(RPii->wiimote));
	
	if( RPii->Control.log_flags.mesgStream_on ){
		//Toggle ACC reporting off as it has not been set.
		toggle_bit(rpt_mode, CWIID_RPT_BTN);		
		cwiid_set_rpt_mode(RPii->wiimote, rpt_mode);
		RPii->Control.log_flags.mesgStream_on = false;
		//Update wiimote
		RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		return 0;
	}
	else if (! RPii->Control.log_flags.mesgStream_on ){
		toggle_bit(rpt_mode, CWIID_RPT_ACC);
		toggle_bit(rpt_mode, CWIID_RPT_BTN);		
		cwiid_set_rpt_mode(RPii->wiimote, rpt_mode);
		RPii->Control.log_flags.mesgStream_on = true;
		//Update wiimote
		RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		return 1;
	}
	else{
		fprintf(stderr, "Error setting ACC Messaging");
		exit(-1);
	}
	return - 1;
}
	
//
//A function to toggle the axes in an PRNTACC struct, given an input between 1 and 6
//
int Toggle_ACC_prnts(struct PRNTACC *tempprnt){
	//Check that axes_comb is valid
	if (tempprnt->axes_comb <1 || tempprnt->axes_comb > 7){
		printf("Error setting Print Axes\n");
		tempprnt->axes_comb = 1;
		return 0;
	}
	printf("New Print Axes:");
	//Bitcheck the value
	int eval;
	eval = tempprnt->axes_comb & 0x01;
		if(eval){
			tempprnt->X = 1;
			printf(" X");
		}
		else{
			tempprnt->X = 0;
		}
	eval = tempprnt->axes_comb & 0x02;
		if(eval){
			tempprnt->Y = 1;
			printf(" Y");
		}
		else{
			tempprnt->Y = 0;
		}
	eval = tempprnt->axes_comb & 0x04;
		if(eval){
			tempprnt->Z = 1;
			printf(" Z");
		}
		else{
			tempprnt->Z = 0;
		}
		printf("\n");
	return -1;
}
// A function to toggle settings
//
int Control_Change_Settings(struct RPii *RPii){
	
	struct PRNTACC tempprnt = RPii->Control.log_flags.PRNTACC;
	int temp_Exit_type = RPii->Control.log_flags.Exit_type;
	int temp_max = RPii->Control.samples.max;
	int temp_wait = RPii->Control.samples.wait;
	float temp_LogDuration = (float)RPii->Control.samples.LogDuration;
	
	Print_EditSet_Ctrls();
	
	while (RPii->Control.log_flags.Chnge_Set){
		usleep(5000);
		
		//Get Latest wiimote data
		RPii->Control = *(struct Log_Control*)cwiid_get_data(RPii->wiimote);
		
		//If the UP_Flag is set, then toggle through the Exit_types	
		if (RPii->Control.btn_flags.UP_flag){
			if (temp_Exit_type == EXIT_BUTTONB){
				temp_Exit_type = EXIT_TIME;
				printf("New Exit type: TIME\n");
			}
			else if(temp_Exit_type == EXIT_TIME){
				temp_Exit_type = EXIT_SAMPLES;
				printf("New Exit type: SAMPLES\n");
			}
			else if(temp_Exit_type == EXIT_SAMPLES){
				temp_Exit_type = EXIT_BUTTONB;
				printf("New Exit type: BUTTONB\n");
			}
			else{
				printf("Error, Exit type reset to BUTTONB\n");
				temp_Exit_type = EXIT_BUTTONB;
			}
			RPii->Control.btn_flags.UP_flag = false;
		}
		
		//Toggle through Axes Selections(do somthing clever with binary later)
		// 1 = X, 2 = Y, 4 = Z, 3 = X, Y, 5 = X, Z, 6 = Y, Z, 7 = X,Y,Z
		if(RPii->Control.btn_flags.ONE_flag){
			tempprnt.axes_comb += 1;
			if (tempprnt.axes_comb >7){
				tempprnt.axes_comb = 1;
			}
			Toggle_ACC_prnts(&tempprnt);
			RPii->Control.btn_flags.ONE_flag = false;
		}
		if(RPii->Control.btn_flags.TWO_flag){
			tempprnt.axes_comb -= 1;
			if (tempprnt.axes_comb <1){
				tempprnt.axes_comb = 7;
			}
			Toggle_ACC_prnts(&tempprnt);
			RPii->Control.btn_flags.TWO_flag = false;
		}
		
		//Toggle through values for max samples
		if(RPii->Control.btn_flags.RIGHT_flag){
			temp_max = temp_max*10;
			if(temp_max > 10000){
				temp_max = 10;
			}
			printf("New Max Samples: %d\n", temp_max);
			RPii->Control.btn_flags.RIGHT_flag = false;
		}
			
		//Toggle through values for LogDuration
		if(RPii->Control.btn_flags.LEFT_flag){
			temp_LogDuration = temp_LogDuration+15000000;
			if(temp_LogDuration > 60000000){
				temp_LogDuration = 15000000;
			}
			printf("New Log Duration(secs): %f\n", temp_LogDuration/1000000);
			RPii->Control.btn_flags.LEFT_flag = false;
		}
		
		//Toggle through values for samples wait
		if(RPii->Control.btn_flags.DOWN_flag){
			temp_wait= temp_wait*10;
			if(temp_wait > 1000000){
				//Wiimote reports at 100HZ so there is no point in logging faster
				temp_wait = 10000;
			}
			printf("New Samples.wait(ms): %d\n", temp_wait/1000);
			RPii->Control.btn_flags.DOWN_flag = false;
		}
		//Update flag changes to wiimote.data
		cwiid_set_data(RPii->wiimote, (const void*)&(RPii->Control));
	}
	// Update All settings to RPii->Control
	RPii->Control.log_flags.PRNTACC = tempprnt;
	RPii->Control.log_flags.Exit_type = temp_Exit_type;
	RPii->Control.samples.max = temp_max;
	RPii->Control.samples.wait = temp_wait;
	RPii->Control.samples.LogDuration = temp_LogDuration;
	//Update RPii->Control to wiimote.data
	//cwiid_set_data(RPii->wiimote, (const void*)&(RPii->Control));
	printf("\nSETTINGS SAVED\n\n");
	printf("NEW ");
	Print_Ctrl_Settings(RPii->Control);
	return 0;
}
// 