#include "RPii.h"

/*Prototypes*/
//Top Level Main
int RPii_Initialise_Wiimote(struct RPii *RPii);
int RPii_LogAccApp(struct RPii *RPii);
int Log_Calibrate(struct RPii *RPii);
int Log_MotionLogger(struct RPii *RPii);
//Other Functions
bool exit_type_satisfied(struct Log_Control *Control);
	
// For Calibration protocol described in notebook, these arrays contain the Calibration Positions at which...
// We expect X to be zero for 0,2,4 & 5. (The array indices start at 0 so we must subtract 1 from each pos)
int xZeros [4] = { 0,2,4,5 };
// We expect Y to be zero for 0,1,2 & 3.
int yZeros [4] = { 0,1,2,3 };
// We expect Z to be zero for 1,3,4 & 5.
int zZeros [4] = { 1,3,4,5 };

//Calibration Defaults which are initialised
#define defaultmx 	-0.3773
#define defaultcx 	47.54
#define defaultmy	-0.3702
#define defaultcy	45.34
#define defaultmz	-0.3924
#define defaultcz	48.66

// We expect the wiimote axis to experience a positive or negative G
#define GPosX		1
#define GPosY		4
#define GPosZ	2
#define GNegX	3
#define GNegY	5
#define GNegZ	0

/*******Top Level Log Functions*************/
//
//
//Logs data using the default (initialized) settings.
//
int RPii_LogAccApp(struct RPii *RPii){
	
	//Check that wiimote has been initialised successfully
	if(RPii ==NULL || RPii->wiimote == NULL){
		fprintf(stderr, "Error, Pointer to wiimote struct is NULL, ensure that the wiimote has been intialised\n");
		exit(-1);
	}
	// Print out the wiimote Controls
	Print_App_Ctrls();
	
	//Get data from wiimote and deference
	RPii->Control = *(Pointto_Control_Data(RPii->wiimote));

	// Loop until program is ended by setting the End_Loop flag
	while ( !RPii->Control.log_flags.End_Loop) {
		//Sleep to reduce load on CPU
		usleep(100000);
		
		//Get Latest wiimote data
		RPii->Control = *(Pointto_Control_Data(RPii->wiimote));
		
		// If Chnge_Set has be set, then call a program to change settings appropriatly
		if ( RPii->Control.log_flags.Chnge_Set){
			//Run the user inteface which allows settings to be changed with wiimote buttons
			Control_Change_Settings(RPii);
			//Update the wiimote with the new controls
			RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		}
		
		//Check if Calibration needs to be called.
		if( RPii->Control.log_flags.calibrating){
			//run Log_Calibrate and printout if calibration fails
			if(!Log_Calibrate(RPii)){
				printf("Calibration unsucessful\n");
				Print_App_Ctrls();
			}
		}
		
		//Check if Recording has been initialised, run Log_Record if it has.
		if ( RPii->Control.log_flags.Enable_Log){
			//This will run so long as Enable_Log is true
			Log_MotionLogger(RPii);
		}
		
		//Update Control with the latest version of wiimote->data for the while loop to check
		RPii->Control = *(struct Log_Control*)cwiid_get_data(RPii->wiimote);
		}	
	
	Disconnect_wiimote(RPii->wiimote);
	printf("end of application\n");
	return 0;
}

//
//A function to initialise the Wii connection with the RPi
//
int RPii_Initialise_Wiimote(struct RPii *RPii){
	
	//Check that structures have been declared
	if(RPii == NULL){
		fprintf(stderr, "Error, Pointer to RPii struct is NULL(Log_Initialise)\n");
		exit(-1);
	}
	
	//Connect to the wiimote
	RPii_Connect_wiimote(&(RPii->wiimote));
	
	//Initialise samples variables
	RPii->Control.samples.count = 0;
	RPii->Control.samples.max = 100;
	RPii->Control.samples.wait = 100000; //default of 10 samples per second
	RPii->Control.samples.LogDuration = 15000000;
	
	//Set Acc_data to -1 to show if no data has been collected
	RPii->Control.Acc_Data.X = -1;
	RPii->Control.Acc_Data.Y = -1;
	RPii->Control.Acc_Data.Z = -1;
	
	//Initialise btn_flags
	RPii->Control.btn_flags.A_flag = false;
	RPii->Control.btn_flags.B_flag = false;
	RPii->Control.btn_flags.HOME_flag = false;
	RPii->Control.btn_flags.PLUS_flag = false;
	RPii->Control.btn_flags.MINUS_flag = false;
	RPii->Control.btn_flags.RIGHT_flag = false;
	RPii->Control.btn_flags.LEFT_flag = false;
	RPii->Control.btn_flags.UP_flag = false;
	RPii->Control.btn_flags.DOWN_flag = false;
	
	//Initialise logging flags
	RPii->Control.log_flags.Enable_Log = false;
	RPii->Control.log_flags.End_Loop = false;
	RPii->Control.log_flags.dataFile_Open= false;
	RPii->Control.log_flags.mesgStream_on = true;
	RPii->Control.log_flags.PRNTACC.X= 1;
	RPii->Control.log_flags.PRNTACC.Y= 1;
	RPii->Control.log_flags.PRNTACC.Z= 1;
	RPii->Control.log_flags.Exit_type = EXIT_BUTTONB;
	
	//Initialise Calibration values
	RPii->Control.Cal_Consts.mx = defaultmx;
	RPii->Control.Cal_Consts.cx = defaultcx;
	RPii->Control.Cal_Consts.my = defaultmy;
	RPii->Control.Cal_Consts.cy = defaultcy;
	RPii->Control.Cal_Consts.mz = defaultmz;
	RPii->Control.Cal_Consts.cz = defaultcz;
	
	//Assign Control pointer to wiimote variable 'data' which can be accessed in callback
	RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
	
	printf("Wiimote successfully connected, Initialisation complete\n\n");
	return 0;
}
//
// A function to Calibrate the wiimote
//
int Log_Calibrate(struct RPii *RPii){
	
	//Currently, calibration has only been designed for mesgStream_On = true;
	if (!RPii->Control.log_flags.mesgStream_on){
		printf("Cannot Calibrate unless mesgStream is On, using default calibration values instead");
		RPii->Control.log_flags.calibrating = false;
		return 0;
	}
	
	printf("Calibrating started\n");
	struct Acc_Data Pos[6];
	struct Acc_Data ZeroMean;
	struct Acc_Data GravPos;
	struct Acc_Data GravNeg;
	struct Acc_Data Data_Diff;
	struct Calibration_Consts Cal;
	struct Calibration_Consts Check;
	ZeroMean.X = 0;
	ZeroMean.Y = 0;
	ZeroMean.Z = 0;
	int j = 0;
	int i ;
	int k;
	
	//Loop through each position
	for ( i = 0; i < 6; i++){
		RPii->Control.btn_flags.A_flag = false;
		j = 0;
		RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		printf("Please put wiimote in Position %d, press A to take a sample\n", i);
		//Give 20secs for the wiimote to be put into position
		while ( RPii->Control.btn_flags.A_flag == false){
			usleep(100000);
			//Update RPii
			RPii->Control = *(Pointto_Control_Data(RPii->wiimote));
			if (j == 200){
				printf("Timeout has been reached, Exiting Calibration\n");
				RPii->Control.log_flags.calibrating = false;
				RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
				return 0;
			}
			j = j+1;
		}
		Print_Countdown(300000, 3);
		//printf("Taking sample for position %d\n", i);
		Pos[i] = RPii->Control.Acc_Data;
		printf("Values for position %d:	X:%d	Y:%d	Z:%d\n", i,Pos[i].X, Pos[i].Y, Pos[i].Z);
	}
	printf("All readings taken, computing calibration coefficients...\n");
	
	//Check that for all positions with '0' acceleration, that all readings were consistant
	//Take an average of zero readings:
	//xZeros[k] contains an integer from 0 to 5 which is a position at which the acc should be zero g.
	for (k = 0; k<4; k++){
		ZeroMean.X += Pos[xZeros[k]].X;
		ZeroMean.Y += Pos[yZeros[k]].Y;
		ZeroMean.Z += Pos[zZeros[k]].Z;
	}
	
	/*//Print out zero values if debugging
	for (k = 0; k<4; k++){
		printf("Zero.X %d: %d,	Zero.Y %d: %d, Zero.Z %d: %d\n", k, Pos[xZeros[k]].X, k, Pos[yZeros[k]].Y, k, Pos[zZeros[k]].Z);
	}*/
	
	ZeroMean.X = ZeroMean.X/4;
	ZeroMean.Y= ZeroMean.Y/4;
	ZeroMean.Z = ZeroMean.Z/4;
	/*printf("ZeroMean.X %d: %d,	ZeroMean.Y %d: %d, ZeroMean.Z %d: %d\n", k,ZeroMean.X, k, ZeroMean.Y, k, ZeroMean.Z);*/

	
	//If any of the 'Zero' values are more than 3 points different from the mean, assume an error in the calibration
	for (k = 0; k<4; k++){
		if ( abs( Pos[xZeros[k]].X - ZeroMean.X) >3 || abs(Pos[yZeros[k]].Y - ZeroMean.Y) >3 || abs( Pos[zZeros[k]].Z - ZeroMean.Z) > 3){
				printf("Error in setting calibration coefficients, readings for zero values were not consistant.\n");
				RPii->Control.log_flags.calibrating = false;
				RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
				return 0;
		}
	}
	
	//Assign relevant position values to GravPos and GravNeg
	GravPos.X = Pos[GPosX].X;
	GravPos.Y = Pos[GPosY].Y;
	GravPos.Z = Pos[GPosZ].Z;
	GravNeg.X = Pos[GNegX].X;
	GravNeg.Y = Pos[GNegY].Y;
	GravNeg.Z = Pos[GNegZ].Z;
	
	//Calculate difference between values and exit if zero
	Data_Diff.X = GravNeg.X-GravPos.X;
	Data_Diff.Y = GravNeg.Y-GravPos.Y;
	Data_Diff.Z = GravNeg.Z-GravPos.Z;
	
	if (Data_Diff.X ==0 || Data_Diff.Y ==0 ||Data_Diff.Z ==0){
		printf("Error in setting calibration coefficients, values of positive and negative 'g' were equal\n");
		RPii->Control.log_flags.calibrating = false;
		return 0;
	}
	
	//Calculate Coefficients
	//m = dp/dq where p is the change in ms^-s and q is the change in 'units'
	Cal.mz = -(2*GRAVITY)/(Data_Diff.Z);
	Cal.my = -(2*GRAVITY)/(Data_Diff.Y);
	Cal.mx = -(2*GRAVITY)/(Data_Diff.X);
	
	Cal.cz = GRAVITY-Cal.mz*GravPos.Z;
	Cal.cy = GRAVITY-Cal.my*GravPos.Y;
	Cal.cx = GRAVITY-Cal.mx*GravPos.X;
	
	//Now compare to the C which would have been calculated using the ZeroMean values ( c = -mx*ZeroMean, when acc(ms^-2) = 0)
	Check = Cal;
	Check.cz = -(ZeroMean.Z*Check.mz);
	Check.cy = -(ZeroMean.Y*Check.my);
	Check.cx = -(ZeroMean.X*Check.mx);
	
	//printf("Check.cx: %f,	Check.cz: %f,	Check.cz: %f,\n", Check.cx, Check.cy, Check.cz);
	//printf("Cal.cx: %f,	Cal.cz: %f,	Cal.cz: %f,\n", Cal.cx, Cal.cy, Cal.cz);

	
	if ( abs( Check.cx - Cal.cx) > 0.3 || abs( Check.cx - Cal.cx) > 0.3 ||abs( Check.cz - Cal.cz) > 0.3 ){
		printf("Warning, there may be a inconsistancy between the calibration gradient and the zero values\n");
		printf("For a more accurate calibration, calibrate again\n");
	}
	
	//Update Control
	printf("Calibration constants:\n mx:	%f\n cx:	%f\n my:	%f\n cy:	%f\n mz:	%f\n cz:	%f\n", Cal.mx, Cal.cx, Cal.my, Cal.cy, Cal.mz, Cal.cz);
	printf("Calibration Finnished\n\n");
	RPii->Control.Cal_Consts = Cal;
	RPii->Control.log_flags.calibrating = false;
	RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
	Print_App_Ctrls();
	
	return 1;
}

	

//
//A function to intialise and run the data logging
//
int Log_MotionLogger(struct RPii *RPii){
	
	//Get Data, and then reset Exit_BTN to ensure that its false before entering anymore loops, and re set.
	RPii->Control = *(Pointto_Control_Data(RPii->wiimote));
	RPii->Control.log_flags.Exit_BTN = false;
	RPii->Control.samples.wait_adjusted = RPii->Control.samples.wait;
	RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		
	//Enter this loop while logging is enabled
	while ( RPii->Control.log_flags.Enable_Log){
		
		//delay the loop by the sample.wait so that a sample frequency can be set
		usleep(RPii->Control.samples.wait_adjusted);
		
		//START: Find out how long the following commands take...
		gettimeofday( &RPii->Control.startLoop, NULL);
		
		//Get Latest data for dataFile_Open check and exit_type_satisfied
		RPii->Control = *(Pointto_Control_Data(RPii->wiimote));
		
		//Open file for writing to with timestamp if it hasnt already been opened.
		if ( !RPii->Control.log_flags.dataFile_Open){
			
			//reset startTime and Sample count, initialise wait_offset
			gettimeofday( &RPii->Control.startTime, NULL);
			RPii->Control.samples.count =0;				
			
			// Open File
			RPii->Control.log_flags.dataFile_Open  = true;
			printf("Opening New File for logging...\n");
			RPii->Control.files.filename = Process_CreateTimestamp();
			Process_OpenFilename(&RPii->Control.files, RPii->Control.files.filename);
			printf ("The Current data will be stored in file: %s\n\n", RPii->Control.files.filename);
			//Print out what is being logged to the terminal and produce header of dataFile
			Print_Ctrl_Settings(RPii->Control);
			Rec_dataFileHeader(&RPii->Control);//####change type of function so that input is not a pointer
			//Update the wiimote with the new controls
			RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		}
		
		
		// If any of the Exit_types are satisfied, disable logging and run through the exiting protocol
		if( exit_type_satisfied( &RPii->Control )){
			//End Logging by turning off Enable_Log
			RPii->Control.log_flags.Enable_Log=0;
			//Update infoFile
			Process_EndOfLog( &RPii->Control, RPii->Control.files.filename);
			//Flash and Reset LEDs to show that collection is finnished
			//LEDs_All_Flash(RPii);
			free(RPii->Control.files.filename);
			//Update the wiimote with the new controls
			RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		}
		else {
			
			//Choose functionality depending on mesgStream_on
			if( (RPii->Control.log_flags.mesgStream_on )){
				//Callback thread will have already updated Acc_Data, so all we need to do is Record
				Rec_Data(&RPii->Control);
			}
			else if( !RPii->Control.log_flags.mesgStream_on ){
				//Get newest state and save to Log_Control
				cwiid_get_state( RPii->wiimote, &(RPii->Control.state));
				Process_State_ACC(&(RPii->Control));
				Rec_Data(&(RPii->Control));
				RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
			}
		
			/*// Update LED's by passing in the sample.count which is an int.
			if( LEDs_Mod4( RPii, RPii->Control.samples.count) ){
				fprintf(stderr, "Report send error (led)\n");
			}*/
			
			//END: Find out how long the following commands take and subtract this from samples.wait
			gettimeofday( &RPii->Control.endLoop, NULL);
			RPii->Control.samples.wait_offset = (int)Process_Time_diff(RPii->Control.startLoop, RPii->Control.endLoop);
			RPii->Control.samples.wait_adjusted =  (RPii->Control.samples.wait) - RPii->Control.samples.wait_offset;
			if (RPii->Control.samples.wait_adjusted < 0){
				RPii->Control.samples.wait_adjusted= 0;
			}
			//Update the wiimote with the new controls
			RPii_Set_Ctrl_Data(RPii->wiimote, &(RPii->Control));
		}
	}
	return 1;
}
	
//
//
/********Other Functions********/
//
//A function to decide when to exit the main sampling loop
//
bool exit_type_satisfied(struct Log_Control *Control){
	int Exit_type = Control->log_flags.Exit_type;
	float TimeElapsed ;
	
	switch (Exit_type){
		
		case EXIT_TIME:
			
			TimeElapsed = Process_Time_diff( Control->startTime, Control->sampleTime);
			if(TimeElapsed > Control->samples.LogDuration){
					printf ("Max Time reached, logging finished\n");
				return true;
			}
			else {
				return false;
			}
			break;
		
		case EXIT_SAMPLES:
			
			if( Control->samples.count  >= Control->samples.max){
				printf ("Max Samples Count reached, logging finished\n");
				return true;
			}
			else {
				return false;
			}
			break;
			
		case EXIT_BUTTONB:
				
			if( Control->log_flags.Exit_BTN){
				printf ("Button B pressed, logging finished\n");
				return true;
			}
			else {
				return false;
			}
			break;
		
		default:
			fprintf(stderr, "invalid exit_type\n");
			exit(-1);

	};
}
//
