/*  -*- Mode: C;  -*-
 * File: list.h
 * Author: James Hall (jch1003@cl.cam.ac.uk)
 * Copyright (C) University of Cambridge Computer Laboratory, 1997
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ** PACKAGE: Any
 **
 ** FUNCTION: Generic list type.
 **
 ** HISTORY:
 ** Created: Wed Jul  2 12:29:04 1997 (jch1003)
 ** Last Edited: Fri May  1 14:08:28 1998 By James Hall
 **
    $Log: list.h,v $
    Revision 1.1  1998/10/20 22:30:40  iap10
    Initial revision

 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */
 
#ifndef _LIST_H_
#define _LIST_H_



/* A generic d.l. list type */

typedef struct list
{
  struct list *lnext;
  struct list *lprev;
} list_t;

typedef list_t listhdr_t;

#define L_INIT(lp) \
  MACRO_BEGIN      \
    (lp)->lnext = (lp)->lprev = (lp);\
  MACRO_END

#define L_EMPTY(lp) ((lp)->lnext == (lp))

#define L_REM(lp, itemp, field, type) \
  MACRO_BEGIN      \
    list_t *lprev = (itemp)->field.lprev; \
    list_t *lnext = (itemp)->field.lnext; \
                                           \
    list_t *prev_link = ((lprev) == (lp)) ? (lp) : &(((type *)(lprev))->field); \
    list_t *next_link = ((lnext) == (lp)) ? (lp) : &(((type *)(lnext))->field); \
    prev_link->lnext = lnext;\
    next_link->lprev = lprev;\
  MACRO_END

#define L_INS_HEAD(lp, itemp, field, type)\
  MACRO_BEGIN      \
    if (L_EMPTY((lp))) \
      { \
        (lp)->lnext = (lp)->lprev = (list_t *)(itemp); \
        (itemp)->field.lnext = (itemp)->field.lprev = (lp); \
      }\
    else \
      { \
        (itemp)->field.lnext = (lp)->lnext;\
        (itemp)->field.lprev = (lp);\
        ((type *)(lp)->lnext)->field.lprev = (list_t *)(itemp);\
        (lp)->lnext = (list_t *)(itemp); \
      }\
  MACRO_END

#define L_WALK(ptr, lp, field, type) \
  for ((ptr) = (type *)(lp)->lnext;  \
       (ptr) != (type *)(lp);       \
       (ptr) = (type *)ptr->field.lnext)

#define L_WALK_REV(ptr, lp, field, type) \
  for ((ptr) = (type *)(lp)->lprev;  \
       (ptr) != (type *)(lp);       \
       (ptr) = (type *)ptr->field.lprev)


#define L_INS_TAIL(lp, itemp, field, type) \
  MACRO_BEGIN                   \
    if (L_EMPTY((lp))) \
      {  \
        (lp)->lnext = (lp)->lprev = (list_t *)(itemp); \
        (itemp)->field.lnext = (itemp)->field.lprev = (lp); \
      }\
    else \
      { \
      (itemp)->field.lnext = (lp); \
      (itemp)->field.lprev = (lp)->lprev; \
      ((type *)(lp)->lprev)->field.lnext = (list_t *)(itemp); \
      (lp)->lprev = (list_t *)(itemp); \
      } \
  MACRO_END

#define L_MOVE_TO_TAIL(lp, itemp, field, type) \
  MACRO_BEGIN                   \
    L_REM((lp), (itemp), field, type); \
    L_INS_TAIL((lp), (itemp), field, type); \
  MACRO_END

#define L_MOVE_TO_HEAD(lp, itemp, field, type) \
  MACRO_BEGIN                   \
    L_REM((lp), (itemp), field, type); \
    L_INS_HEAD((lp), (itemp), field, type); \
  MACRO_END

#define L_INS_BEFORE(lp, befp, itemp, field, type) \
  MACRO_BEGIN                                      \
    list_t *back_link = (befp)->field.lprev;       \
    (itemp)->field.lnext = (list_t *)(befp);       \
    (itemp)->field.lprev = back_link;              \
    if (back_link == (lp))                           \
      back_link->lnext = (list_t *)(itemp);        \
    else                                           \
      ((type *)back_link)->field.lnext = (list_t *)(itemp);\
    (befp)->field.lprev =  (list_t *)(itemp);      \
  MACRO_END
  

#define L_INS_AFTER(lp, aftp, itemp, field, type)         \
  MACRO_BEGIN                                             \
    list_t *for_link = (aftp)->field.lnext;               \
    (itemp)->field.lprev = (list_t *)(aftp);              \
    (itemp)->field.lnext = for_link;                      \
    if (for_link == (lp))                                 \
      for_link->lprev == (list_t *)(itemp);               \
    else                                                  \
      ((type *)for_link)->field.lprev = (list_t *)(itemp);\
    (aftp)->field.lnext=  (list_t *)(itemp);              \
  MACRO_END 

#if 0 

/*
 *	List of abstract objects.  List is maintained
 *	within that object.
 *
 *	Supports fast removal from within the list.
 *
 *	How to declare a list of elements of type "foo_t":
 *		In the "*foo_t" type, you must have a field of
 *		type "list_chain_t" to hold together this list.
 *		There may be more than one chain through a
 *		"foo_t", for use by different lists.
 *
 *		Declare the list as a "listp_t" type.
 *
 *		Elements of the list (of type "foo_t", that is)
 *		are referred to by reference, and cast to type
 *		"list_entryp_t" within this module.
 */

/*
 *	A generic doubly-linked list (list).
 */

struct list_entry {
	struct list_entry	*next;		/* next element */
	struct list_entry	*prev;		/* previous element */
};

typedef struct list_entry	*listp_t;
typedef	struct list_entry	list_head_t;
typedef	struct list_entry	list_chain_t;
typedef	struct list_entry	*list_entryp_t;

/*
 *	enque puts "elt" on the "list" end.
 *	deque returns the first element in the "list".
 */

#define enque(list,elt)	enlist_tail(list, elt)
#define deque(list)		delist_head(list)

extern void		enlist_head();
extern void		enlist_tail();
extern list_entryp_t	delist_head();
extern list_entryp_t	delist_tail();
extern void		inslist();
extern void		remlist();
extern void		reminslist();

/*
 *	Macro:		list_init
 *	Function:
 *		Initialize the given list.
 *	Header:
 *		void list_init(l)
 *			listp_t		l;
 */
#define list_init(l)	((l)->next = (l)->prev = l)

/*
 *	Macro:		list_first
 *	Function:
 *		Returns the first entry in the list,
 *	Header:
 *		list_entryp_t list_first(l)
 *			listp_t	l;  
 */
#define list_first(l)	((l)->next)

/*
 *	Macro:		list_next
 *	Header:
 *		list_entryp_t list_next(lc)
 *			listp_t lc;
 */
#define list_next(lc)	((lc)->next)

/*
 *	Macro:		list_end
 *	Header:
 *		boolean_t list_end(l, le)
 *			listp_t l;
 *			list_entryp_t le;
 */
#define list_end(l, le)	((l) == (le))

#define list_empty(l)		list_end((l), list_first(l))

/*
 *	Macro:		list_enter
 *	Header:
 *		void list_enter(l, elt, type, field)
 *			listp_t l;
 *			<type> elt;
 *			<type> is what's in our list
 *			<field> is the chain field in (*<type>)
 */
#define list_enter(head, elt, type, field)			\
MACRO_BEGIN							\
	if (list_empty((head))) {				\
		(head)->next = (list_entryp_t) elt;		\
		(head)->prev = (list_entryp_t) elt;		\
		(elt)->field.next = head;			\
		(elt)->field.prev = head;			\
	}							\
	else {							\
		register list_entryp_t prev;			\
								\
		prev = (head)->prev;				\
		(elt)->field.prev = prev;			\
		(elt)->field.next = head;			\
		(head)->prev = (list_entryp_t)(elt);		\
		((type *)prev)->field.next = (list_entryp_t)(elt);\
	}							\
MACRO_END

/*
 *	Macro:		list_field [internal use only]
 *	Function:
 *		Find the list_chain_t (or listp_t) for the
 *		given element (thing) in the given list (head)
 */
#define list_field(head, thing, type, field)			\
		(((head) == (thing)) ? (head) : &((type *)(thing))->field)

/*
 *	Macro:		list_remove
 *	Header:
 *		void list_remove(l, le, type, field)
 *			arguments as in list_enter
 */
#define list_remove(head, elt, type, field)			\
MACRO_BEGIN							\
	register list_entryp_t	next, prev;			\
								\
	next = (elt)->field.next;				\
	prev = (elt)->field.prev;				\
								\
	list_field((head), next, type, field)->prev = prev;	\
	list_field((head), prev, type, field)->next = next;	\
MACRO_END

/*
 *	Macro:		list_assign
 */
#define list_assign(to, from, type, field)			\
MACRO_BEGIN							\
	((type)((from)->prev))->field.next = (to);		\
	((type)((from)->next))->field.prev = (to);		\
	*to = *from;						\
MACRO_END

#define list_remove_first(h, e, t, f)				\
MACRO_BEGIN							\
	e = (t *) list_first((h));				\
	list_remove((h), (e), t, f);				\
MACRO_END

#define list_remove_last(h, e, t, f)				\
MACRO_BEGIN							\
	e = (t) list_last((h));				\
	list_remove((h), (e), t, f);				\
MACRO_END

/*
 *	Macro:		list_enter_first
 *	Header:
 *		void list_enter_first(l, elt, type, field)
 *			listp_t l;
 *			<type> elt;
 *			<type> is what's in our list
 *			<field> is the chain field in (*<type>)
 */
#define list_enter_first(head, elt, type, field)		\
MACRO_BEGIN							\
	if (list_empty((head))) {				\
		(head)->next = (list_entryp_t) elt;		\
		(head)->prev = (list_entryp_t) elt;		\
		(elt)->field.next = head;			\
		(elt)->field.prev = head;			\
	}							\
	else {							\
		register list_entryp_t next;			\
								\
		next = (head)->next;				\
		(elt)->field.prev = head;			\
		(elt)->field.next = next;			\
		(head)->next = (list_entryp_t)(elt);		\
		((type)next)->field.prev = (list_entryp_t)(elt);\
	}							\
MACRO_END

/*
 *	Macro:		list_last
 *	Function:
 *		Returns the last entry in the list,
 *	Header:
 *		list_entryp_t list_last(l)
 *			listp_t	l;
 */
#define list_last(l)	((l)->prev)

/*
 *	Macro:		list_prev
 *	Header:
 *		list_entryp_t list_prev(lc)
 *			listp_t lc;
 */
#define list_prev(lc)	((lc)->prev)

/*
 *	Macro:		list_entryp_to_base
 *	Function:	calculates pointer to structure containing list_entry
 */
#define list_entryp_to_base(le, type, field) \
	((type *)((char *)(le) - (int)&((type *)0)->field))

#endif
#endif /* _LIST_H_ */

/*
 * end list.h
 */
