/* (C) Cambridge University Computer Laboratory, 2000
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

typedef unsigned int uint32_t;

#include "ecn.h"

#define ENABLE_ECN

void
die (const char *s)
{
    perror (s);
    exit (1);
}


uint32_t
get_ip (const char *s)
{
    struct hostent *hent;

    hent = gethostbyname (s);

    if (!hent)
    {
	printf ("%s: name lookup failed\n", s);
	exit (1);
    }

    return *(uint32_t*)hent->h_addr;
}


void
copydata (int in, int out)
{
    int nread;
    int towrite;
    int ret;
    char *p;
    char buf[1024];

    do {
	nread = read (in, buf, sizeof (buf));
	if (nread < 0)
	    die ("read");

	towrite = nread;
	p = buf;
	while (towrite)
	{
	    ret = write (out, p, towrite);
	    if (ret <= 0)
		die ("write");
	    p += ret;
	    towrite -= ret;
	}
    } while (nread != 0);
}


int
main (int argc, char **argv)
{
    int fd;
    int ret;
    uint32_t destIP;
    int port = htons (5000);
    struct sockaddr_in laddr;
    struct sockaddr_in raddr;
#ifdef ENABLE_ECN
    int tos;
#endif

    if (argc <= 1)
    {
	fprintf (stderr, "usage: %s <hostname> [ <port> ]\n"
		 "\tSend UDP traffic to <hostname> on <port> with the ECN\n"
		 "\taware TOS bit set\n", argv[0]);
	exit (1);
    }

    destIP = get_ip (argv[1]);

    if (argc > 2)
	port = htons (atoi (argv[2]));

    fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (fd < 0)
	die ("socket");


#ifdef ENABLE_ECN
    /* tell the world we're an ECN Capable Transport */
    tos = socket_gettos (fd);
    if (tos < 0)
	die ("gettos");
    socket_settos (fd, ((unsigned char)tos) | TOS_ECN_ECT);
#endif /* ENABLE_ECN */

    laddr.sin_family = AF_INET;
    laddr.sin_port = 0; /* any port */
    laddr.sin_addr.s_addr = 0;  /* all interfaces */
    ret = bind (fd, (void*)&laddr, sizeof (laddr));
    if (ret < 0)
	die ("bind");

    raddr.sin_family = AF_INET;
    raddr.sin_port = port;
    raddr.sin_addr.s_addr = destIP;
    ret = connect (fd, (void*)&raddr, sizeof (raddr));
    if (ret < 0)
	die ("connect");

    copydata (0, fd);

    return 0;
}
