/* (C) Cambridge University Computer Laboratory, 2002
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <string.h>
#include "stm0.h"
#include "stm_test.h"

/***********************************************************************/

/*
 * Algorithm configuration
 *
 * DO_STM_ABORT_SELF => cause current transaction to abort if it finds
 * it needs an owned location when committing.  Does not give any progress 
 * guarantee in the case of stuck transactions.
 *
 * DO_STM_DEFER_OTHER => cause current transaction to spin if it finds
 * another one in progress.  Again, no progress guarantee but if
 * commit steps are fast then it avoids aborting.
 *
 * DO_STM_ABORT_SELF => cause current transaction to cause others it
 * sees to abort.  Obstruction free non-blocking behaviour.
 */

#ifndef EXTERNAL_STM_CONFIG

/* #define DO_STM_ABORT_SELF   */
#define DO_STM_DEFER_OTHER  
/* #define DO_STM_ABORT_OTHER  */

#define STM_OWNERSHIP_RECORDS 65536

/***********************************************************************/

/*
 * Memory management configuration.  Un-comment one line to select the
 * scheme used to manage STM descriptors.
 *
 * Malloc / GC indicates how fresh storage space is obtained: malloc
 * fetches it in blocks of size MM_BLOCK_SIZE to per-thread local
 * allocators.  GC allocates descriptors individually under the 
 * assumption that the upstream allocator is optimised for
 * frequent small allocations.
 *
 * Simple => do not attempt to re-use storage space.
 *
 * RC => perform reference counting on descriptors.  Per-thread
 * free lists hold descriptors eligible for re-use.
 *
 * PD => 'private or discard' algorithm.  A thread continues to
 * re-use a descriptor until it has been seen by another thread,
 * at which point it is discarded and a fresh descriptor allocated.
 *
 * SMR => Michael's Safe Memory Re-use algorithm.  Implementation
 * following PODC 2002.
 *
 * PTB => Pass-the-buck ROP solution.  Implementation following 
 * DISC 2002.
 *
 * Note that MallocSimple and MallocPD allow neither de-allocation to
 * the OS nor guaranteed re-use.  They're here for comparison
 * rather than serious use.  MallocRC does not allow de-allocation
 * to the OS.
 */

#define DO_PRIMARY_MALLOC   
/* #define DO_PRIMARY_GC       */

/* #define DO_SECONDARY_SIMPLE */
#define DO_SECONDARY_RC     
/* #define DO_SECONDARY_PD     */
/* #define DO_SECONDARY_SMR    */
/* #define DO_SECONDARY_PTB    */

/***********************************************************************/

#endif /* EXTERNAL_STM_CONFIG */

/*
 * We still statically allocate and initialize guard structures, but 
 * scanning only involves 0..tctr so there's no run-time overhead in
 * that respect.
 */
#define MAX_THREADS 64

#define MAX_LOCATIONS 32

/*
 * Mapping from a heap pointer to an index in the array of ownership
 * records.
 */
#define OREC_HASH(ptr) ((((uintptr_t) (ptr)) >> 2) & (STM_OWNERSHIP_RECORDS - 1))

/*
 * Size of malloc() requests made by malloc-based allocators
 */

#define MALLOC_BLOCK_SIZE    16*1024

/* 
 * If a per-thread object cache reaches the high water mark then it places
 * items beyond the low water mark on a shared stealable queue.  Other
 * threads will steal the entire queue before doing a MALLOC.  NB: the
 * "LWM" amount of objects are saved by linear traversal of the object
 * cache lists.  It is intended as a simple safeguard against oscillating,
 * not as a decent sharing mechanism.
 */

#define OBJECT_CACHE_HWM         4*1024*1024
#define OBJECT_CACHE_LWM                2048

#if (OPT_CASPTRINT_AVAILABLE == 1)
#define DO_BALANCE_OBJECT_CACHES
#endif

/*
 * If DO_CACHE_OREC_STATUS is set then each thread keeps track of the
 * orecs involved in its current transaction.  This takes the form of
 * a table mapping from orec number to orec_cache_t.  This in turn
 * refers to the entry (if any) in the descriptor relating to that 
 * orec, or holds OREC_IS_ALIASED if there are many.
 */

#define DO_CACHE_OREC_STATUS

/*
 * If DO_SHARABLE_READS is set then each read then we special-case
 * reads held in the transaction descriptor so that they will not
 * generate writes to the shared locations or their ownership records.
 */

#define DO_SHARABLE_READS

/*
 * If DO_FAST_READONLY is set then track whether a transaction is
 * read-only and use a streamlined commit function
 */

#define DO_FAST_READONLY

/*
 * If DO_SPECIAL_CASE_1 is set then use special case code if we know
 * only one thread is running
 */

//#define DO_SPECIAL_CASE_1

/*
 * Number of objects that a thread de-allocates before performing
 * a scan when using the SMR or PTB algorithms.
 */

#define DEFERRED_FREE_BATCH_SIZE     128

#define GUARDS_PER_THREAD       (MAX_LOCATIONS + 2)
#define MAX_GUARDS (MAX_THREADS * GUARDS_PER_THREAD)
#define GUARD_PROTECTS(gp, op) ((gp) == (op))

#define GET_ST(_e) ((STMState_t *) GET_ENV_STATE(_e))
#define SET_ST(_e, _s) SET_ENV_STATE((_e), ((void *) (_s)))

/***********************************************************************/

/* Testing options.  None of these should be enabled for production use */

/*
 * OVERHEAD_ONLY causes casn to be a no-op, always returning false.
 * Used to test how long is spent getting to/from that function when
 * it is plumbed through to a Java native method.
 */

/* #define OVERHEAD_ONLY           */

/*
 * KILL_ON_DEALLOCATE causes de-allocated storage to be filled with
 * DEAD_VALUE at the point at which it could be returned to the OS.
 * Helps trap use-after-freeing errors.
 */

#ifdef DEBUG
#define KILL_ON_DEALLOCATE
#define do_debug 1
#else
#define do_debug 0
#endif

#define DO_TIMINGS 
#define DO_COUNTS

#define DO_ALLOCATION_COUNT

#define DO_LOCATION_COUNT

/* #define DO_DUMP_ORECS */

/***********************************************************************/

/* Convert 'public' options above into internal ones and test for sanity */

#ifdef DO_STM_ABORT_SELF
#define OPT_ABORT_SELF 1
#else
#define OPT_ABORT_SELF 0
#endif

#ifdef DO_STM_DEFER_OTHER 
#define OPT_DEFER_OTHER  1
#else
#define OPT_DEFER_OTHER  0
#endif

#ifdef DEBUG
#define OPT_DEBUG 1
#define OPT_DEBUG_MSG 0
#else
#define OPT_DEBUG 0
#define OPT_DEBUG_MSG 0
#endif

#ifdef DO_SHARABLE_READS
#define OPT_SHARABLE_READS 1
#else
#define OPT_SHARABLE_READS 0
#endif

/*
 * DO_PRIMARY_GC => may need to flush data structures on each collection
 */

#if (defined (DO_PRIMARY_GC))
#define OPT_NEED_POST_GC_HOOK 1
#define OPT_NEED_PRE_GC_HOOK 1
#else
#define OPT_NEED_POST_GC_HOOK 0
#define OPT_NEED_PRE_GC_HOOK 1
#endif

#ifdef DO_PRIMARY_MALLOC
#define MM_PRIMARY_SCHEME Malloc
#endif

#ifdef DO_PRIMARY_GC
#define MM_PRIMARY_SCHEME GC
#endif

#ifdef DO_SECONDARY_SIMPLE
#define MM_SECONDARY_SCHEME Simple
#endif

#ifdef DO_SECONDARY_RC
#define MM_SECONDARY_SCHEME RC
#endif

#ifdef DO_SECONDARY_PD
#define MM_SECONDARY_SCHEME PD
#endif

#ifdef DO_SECONDARY_SMR
#define MM_SECONDARY_SCHEME SMR
#endif

#ifdef DO_SECONDARY_PTB
#define MM_SECONDARY_SCHEME PTB
#endif

#define PASTE2(x,y) x ## y
#define PASTE(x,y) PASTE2(x,y)

#define PrimaryAllocateBytes   PASTE (PrimaryAllocateBytes, MM_PRIMARY_SCHEME)
#define PrimaryDeallocateObj   PASTE (PrimaryDeallocateObj, MM_PRIMARY_SCHEME)
#define PrimaryInitState       PASTE (PrimaryInitState, MM_PRIMARY_SCHEME)
#define PrimaryReclaimState    PASTE (PrimaryReclaimState, MM_PRIMARY_SCHEME)
#define PrimaryDoneGC          PASTE (PrimaryDoneGC, MM_PRIMARY_SCHEME)
#define PrimaryInitScheme      PASTE (PrimaryInitScheme, MM_PRIMARY_SCHEME)

#define MMAllocateBytes        PASTE (MMAllocateBytes, MM_SECONDARY_SCHEME)
#define MMDeallocateObj        PASTE (MMDeallocateObj, MM_SECONDARY_SCHEME)
#define MMGuardObj             PASTE (MMGuardObj, MM_SECONDARY_SCHEME)
#define MMReleaseObj           PASTE (MMReleaseObj, MM_SECONDARY_SCHEME)
#define MMInitState            PASTE (MMInitState, MM_SECONDARY_SCHEME)
#define MMReclaimState         PASTE (MMReclaimState, MM_SECONDARY_SCHEME)
#define MMDoneGC               PASTE (MMDoneGC, MM_SECONDARY_SCHEME)
#define MMInitScheme           PASTE (MMInitScheme, MM_SECONDARY_SCHEME)

/***********************************************************************/

#define stm_hard_assert(_x) do { if(!(_x)) stm_panic((#_x)); } while (0)

/***********************************************************************/

/*
 * Timing settings
 */

#define TIMING_ALLOCATION          0
#define TIMING_COMMIT              1
#define TIMING_ABORT               2
#define TIMING_WRITE               3
#define TIMING_READ                4
#define TIMING_READ_SLOW           5
#define TIMING_WAIT                6
#define TIMING_ASLEEP              7
#define TIMING_TEMP                8

#define NUM_TIMINGS 9

#define COUNT_COMMIT_SUCCEEDS      0
#define COUNT_COMMIT_FAILS         1
#define COUNT_ABORTS               2
#define COUNT_WAITS                3
#define COUNT_RC_UP                4
#define COUNT_RC_DOWN              5

#define COUNT_FAST_READ            6
#define COUNT_SEMIFAST_READ        7
#define COUNT_SLOW_READ            8

#define COUNT_FAST_WRITE           9
#define COUNT_SLOW_WRITE          10

#define COUNT_ACQUIRE_FAST_NB     11
#define COUNT_ACQUIRE_SLOW_NB     12

#define COUNT_RELEASE_FAST_NB     13
#define COUNT_RELEASE_SLOW_NB     14

#define COUNT_FAST_COMMIT         15

#define NUM_COUNTS 16

#ifdef DO_TIMINGS
#define START_TIMING(_i) { tick_t start ## _i = RDTICK(); 
#define END_TIMING(_st,_i) (_st) -> timings[_i] += RDTICK() - (start ## _i);  stm_assert (RDTICK() > (start ## _i)); }
#define TEMP_EXIT_TIMING(_i) { tick_t start_exit ## _i = RDTICK();
#define TEMP_ENTER_TIMING(_i) start ## _i += (RDTICK() - start_exit ## _i); }
#else
#define START_TIMING(_i) 
#define END_TIMING(_st,_i)
#define TEMP_EXIT_TIMING(_i)
#define TEMP_ENTER_TIMING(_i)
#endif

#ifdef DO_COUNTS
#define COUNT(_st,_i) (_st) -> counts[_i] ++;
#else
#define COUNT(_st,_i) /* nothing */
#endif

/* Used to assign unique idx values to STMState_t structs */
static int tctr = 0;

/***********************************************************************/
/*
 * Booleans
 */

#if (OPT_NEED_BOOL_T == 1)
typedef enum { FALSE = 0,
	       TRUE = 1 } bool_t;
#endif

/*
 * Dummy heap definitions when not actually available.  Lets us compile
 * as much of this file as possible to avoid ifdefs later and to aid testing.
 */

#if (OPT_GC_HEAP_AVAILABLE != 1)

static void *tentativelyAllocateBytes (void *e, int bytes)
{
  stm_panic (("GC heap used but not available"));
}

#define GC_WRITE_BARRIER(_a, _b) /* Nothing */

#endif /* OPT_GC_HEAP_AVAILABLE != 1 */

#if (OPT_MALLOC_HEAP_AVAILABLE != 1)
static void *dummy_malloc (int s) {
  stm_panic (("Malloc heap used but not available"));
}

static void dummy_free (void *v) {
  stm_panic (("Malloc heap used but not available"));
}

#define mallocBytes(_x) dummy_malloc (_x)
#define freeMallocObj(_x) dummy_free (_x)
#endif /* OPT_MALLOC_HEAP_AVAILABLE != 1 */

#if (OPT_CASPTRINT_AVAILABLE != 1)
static int CASPtrInt (void *a, void *op, void *np, int oi, int ni)
{
  stm_panic (("CASPtrInt used but not available"));
}
#endif /* OPT_CASPTRINT_AVAILABLE != 1 */

/***********************************************************************/

typedef struct STMDescriptor_t STMDescriptor_t;

static bool_t is_stm_descriptor (void *ptr);

/***********************************************************************/

/*
 * Secondary MM schemes that wish to use the common object-cache
 * functions must place a SecondaryStateOC_t at the start of their
 * own state structure and OCHeader_t at the end of their header 
 * structure.
 */

typedef struct OCHeader_t OCHeader_t;
typedef struct RCHeader_t RCHeader_t;
typedef struct PDHeader_t PDHeader_t;

struct OCHeader_t {
  OCHeader_t *next;
};

typedef struct {
  OCHeader_t *object_cache_1;
  OCHeader_t *object_cache_2;
} SecondaryStateOC_t;

typedef struct {
  char *block;
  int bytes_left;
} PrimaryStateMalloc_t;

struct RCHeader_t {
  volatile int  rc;
  OCHeader_t    och;
};

struct PDHeader_t {
  volatile uintptr_t isPrivate;
};

typedef struct {
  PDHeader_t *last_header;
} SecondaryStatePD_t;

typedef struct {
  SecondaryStateOC_t  oc;
  OCHeader_t         *dlist;
  int                 dcount;
} SecondaryStateSMR_t;

typedef struct {
  SecondaryStateOC_t  oc;
  OCHeader_t         *dlist;
  int                 dcount;
} SecondaryStatePTB_t;

typedef struct STMEntry_t STMEntry_t;

#ifdef DO_CACHE_OREC_STATUS
/* 
 * invalid_addr_entry must be an STMEntry_t whose address field is 
 * an address whose contents must never be updated through STM.  A pointer
 * to it is used to indicate an orec for which the descriptor holds
 * multiple entries.  
 *
 * Hence to if "addr" is the address we're seeking, 
 *
 *  st -> orec_cache[OREC_HASH(addr)] -> addr != addr
 *
 * can detect cache misses and aliasing in a single test.
 */
static STMEntry_t invalid_addr_entry;
#define OREC_IS_ALIASED ((STMEntry_t *) &invalid_addr_entry)
#define OREC_NOT_SEEN   ((STMEntry_t *) NULL)

#endif

typedef struct STMSleepers_t STMSleepers_t;

struct STMState_t {
  int                idx;
  STMDescriptor_t   *current_transaction;
  STMDescriptor_t   *last_transaction;
#ifdef DO_SPECIAL_CASE_1
  bool_t started_one_thread;
#endif
  union {
    SecondaryStateOC_t  oc;
    SecondaryStatePD_t  pd;
    SecondaryStateSMR_t smr;
    SecondaryStatePTB_t ptb;
  } secondary_state;
  bool_t               does_own[MAX_LOCATIONS];
  STMSleepers_t       *sleepers;
  STMMutex_t           mutex;
  STMCond_t            cond;
#ifdef DO_TIMINGS
  tick_t init_time;
  tick_t timings[NUM_TIMINGS];
#endif
#ifdef DO_COUNTS
  int counts[NUM_COUNTS];
#endif
#ifdef DO_ALLOCATION_COUNT
  int allocation_count;
#endif
#ifdef DO_LOCATION_COUNT
  int loc_count;
  int max_loc_count;
  int trans_count;
#endif
#ifdef DO_FAST_READONLY
  bool_t         current_is_read_only;
#endif
  int object_cache_1_size;
  int object_cache_2_size;
#ifdef DO_CACHE_OREC_STATUS
  STMEntry_t  *orec_cache[STM_OWNERSHIP_RECORDS];
#endif
  union {
    PrimaryStateMalloc_t malloc;
  } primary_state;
  STMState_t        *next_state;
  STMDescriptor_t   *waiting_with[MAX_LOCATIONS];
  STM_ENV     *env;
};

struct STMEntry_t {
  addr_t addr;
  word_t new_val;
  word_t new_version;
  word_t old_val;
  word_t old_version;
};

typedef struct OwnershipRecord_t OwnershipRecord_t;

struct STMSleepers_t {
  STMState_t *st; /* Must be a ptr to distinguish from descriptors */
  STMSleepers_t *next;
};

/* 
 * STM_ACTIVE => transaction has been started but not yet committed
 * or aborted.
 *
 * STM_COMMITTED => transaction has been selected as committed, updates
 * may still be in progress.
 *
 * STM_ABORTED => transaction has been selected as aborted, trasnactional
 * read/write may still occur until it notices.
 */

struct STMDescriptor_t {
  volatile STMResult_t status; 
  int                  length;
  STMState_t          *st;
  STMEntry_t           entries[MAX_LOCATIONS];
};

/*
 * Version information
 */

union STMVersionInfo_t {
  word_t            version; /* Must be odd */
  STMDescriptor_t * owner;   /* Aligned ptr */
};

#define VI_IS_VERSION(u) (((u).version) & 1)

/*
 * Ownership records
 */

struct OwnershipRecord_t {
  volatile union STMVersionInfo_t u;
  volatile int   updaters;
};

static OwnershipRecord_t *orecs;

#define OREC_OF(ptr) (&orecs[OREC_HASH(ptr)])

#define OREC_IS_OWNED(orec) (!(((orec) -> u.version) & 1))

#ifdef DO_SPECIAL_CASE_1
static bool_t one_thread = TRUE;
static STMDescriptor_t * volatile one_committing_stmd = NULL;
#endif

/***********************************************************************/

static int get_and_adjust_slow (volatile int * ptr, int delta);
static int get_and_adjust (volatile int * ptr, int delta);

static void *PrimaryAllocateBytes (STMState_t *st, int bytes);
static void  PrimaryDeallocateObj (STMState_t *st, void *ptr);
static void  PrimaryInitState (STMState_t *st);
static void  PrimaryReclaimState (STMState_t *st);
static void  PrimaryDoneGC (STMState_t *st);
static void  PrimaryInitScheme (void);

static void *MMAllocateBytes (STMState_t *st, int bytes);
static void  MMDeallocateObj (STMState_t *st, void *ptr);
static void *MMGuardObj (STMState_t *st, int g, void *ptr);
static void  MMReleaseObj (STMState_t *st, int g, void *ptr);
static void  MMInitState (STMState_t *st);
static void  MMReclaimState (STMState_t *st);
static void  MMDoneGC (STMState_t *st);
static void  MMInitScheme (void);

/***********************************************************************/

static int get_and_adjust_slow (volatile int * ptr, int delta)
{
  int old;
  int seen;

  NO_INLINE();
  old = *ptr;
  MB();
  while (TRUE) {
    seen = CASInt (ptr, old, old + delta);
    if (seen == old) break;
    old = seen;
  }
 
  return old;
}

static int get_and_adjust (volatile int * ptr, int delta)
{
  int old;
  
MB();
  old = *ptr;
  if (CASInt (ptr, old, old + delta) != old) {
    old = get_and_adjust_slow (ptr, delta);
  }
MB();

  return old;  
}

/***********************************************************************/

#define SEC_TO_PRI_ALIGN_BYTES(_b) (_b)
#define PRI_TO_SEC_ALIGN_PTR(_p) (_p)

static uintptr_t mm_header_size = 0;

/***********************************************************************/

/*
 * Primary: malloc
 */

static void *PrimaryAllocateBytesMalloc (STMState_t *st, 
					 int bytes) {
  void *result;
  PrimaryStateMalloc_t *psm;

  bytes = (bytes + REQ_NATIVE_ALIGNMENT - 1) & (~(REQ_NATIVE_ALIGNMENT - 1));
  stm_assert (bytes >= ob);
  psm = &(st -> primary_state.malloc);

  stm_assert (bytes <= MALLOC_BLOCK_SIZE);
  if (psm -> bytes_left < bytes) {
    psm -> block = malloc(MALLOC_BLOCK_SIZE);
    psm -> bytes_left = MALLOC_BLOCK_SIZE;
  }

  result = psm -> block;
  psm -> block += bytes;
  psm -> bytes_left -= bytes;

#ifdef DO_ALLOCATION_COUNT
  st -> allocation_count += bytes;
#endif

  stm_assert (result != NULL);

  return result;
}

static void PrimaryDeallocateObjMalloc (STMState_t *st, void *ptr) {
  stm_panic (("PrimaryDeallocateObjMalloc not implemented"));
}

static void PrimaryInitStateMalloc (STMState_t *st) {
  PrimaryStateMalloc_t *psm;

  psm = &(st -> primary_state.malloc);
  psm -> block = NULL;
  psm -> bytes_left = 0;
}

static void PrimaryReclaimStateMalloc (STMState_t *st) {
  /* Should avoid losing the remaining space in the current block  */
  stm_panic (("PrimaryReclaimStateMalloc not implemented"));
}

static void PrimaryDoneGCMalloc (STMState_t *st) {
  /* Nothing */
}

static void PrimaryInitSchemeMalloc (void) {
  /* Nothing */
}

/***********************************************************************/

/*
 * Primary: GC
 */

static void *PrimaryAllocateBytesGC (STMState_t *st, int bytes) {
  void       *result;

  result = tentativelyAllocateBytes (st -> env, bytes);

#ifdef DO_ALLOCATION_COUNT
  st -> allocation_count += bytes;
#endif

  return result;
}

static void PrimaryDeallocateObjGC (STMState_t *st, void *ptr) {
  /* Nothing */
}

static void PrimaryInitStateGC (STMState_t *st) {
  /* Nothing */
}

static void PrimaryReclaimStateGC (STMState_t *st) {
  /* Nothing */
}

static void PrimaryDoneGCGC (STMState_t *st) {
#ifdef DO_CACHE_OREC_STATUS
  memset (st -> orec_cache, 0, sizeof(st -> orec_cache));
#endif
}

static void PrimaryInitSchemeGC (void) {
  /* Nothing */
}

/***********************************************************************/

/*
 * Secondary: information supplied by the client
 */

static int clientObjToBytes (void *ptr);
static void clientKillObject (void *ptr);

/*
 * Secondary: common object cache functions
 */

typedef struct {
  OCHeader_t *shared_oc;
  int size;
} OCBalanceInfo_t;

static volatile OCBalanceInfo_t *shared_oc_1;
static volatile OCBalanceInfo_t *shared_oc_2;

static OCHeader_t *OCObjToHeader (void *ptr)
{
  OCHeader_t *h;
  h = ((OCHeader_t *) ptr) - 1;
  return h;
}

static void *OCHeaderToObj (OCHeader_t *och)
{
  void *result;
  result = (void *) (och + 1);
  return result;
}

static void add_to_object_cache (STMState_t *st, void *ptr)
{
  OCHeader_t *och;
  och = OCObjToHeader (ptr);
  stm_assert (och -> next == NULL);

#ifdef DO_BALANCE_OBJECT_CACHES
  if ((st -> object_cache_1_size > OBJECT_CACHE_HWM) &&
      (shared_oc_1 -> shared_oc == NULL))
    {
      OCHeader_t *toch;
      OCHeader_t **ptoch;
      int keep;
      toch = st -> secondary_state.oc.object_cache_1;
      ptoch = &(st -> secondary_state.oc.object_cache_1);
      keep = 0;
      while (keep < OBJECT_CACHE_LWM) {
	keep += clientObjToBytes(OCHeaderToObj(toch));
	ptoch = &(toch -> next);
	toch = toch -> next;
	stm_assert (toch != NULL);
      }
      
      if ((CASPtrInt ((long long *) shared_oc_1, 
		      NULL, 
		      (void *) toch,
		      0,
		      st -> object_cache_1_size - keep)))
	{
	  *ptoch = NULL;
	  st -> object_cache_1_size = keep;
	}
    }

  if ((st -> object_cache_2_size > OBJECT_CACHE_HWM) &&
      (shared_oc_2 -> shared_oc == NULL))
    {
      OCHeader_t *toch;
      OCHeader_t **ptoch;
      int keep;
      toch = st -> secondary_state.oc.object_cache_2;
      ptoch = &(st -> secondary_state.oc.object_cache_2);
      keep = 0;
      while (keep < OBJECT_CACHE_LWM) {
	keep += clientObjToBytes(OCHeaderToObj(toch));
	ptoch = &(toch -> next);
	toch = toch -> next;
	stm_assert (toch != NULL);
      }
      
      if ((CASPtrInt ((long long *) shared_oc_2, 
		      NULL, 
		      (void *) toch,
		      0,
		      st -> object_cache_2_size - keep)))
	{
	  *ptoch = NULL;
	  st -> object_cache_2_size = keep;
	}
    }
#endif

  if (!is_stm_descriptor (ptr)) {
    stm_assert (clientObjToBytes (ptr) == sizeof (STMSleepers_t));
    st -> object_cache_1_size += clientObjToBytes (ptr);
    och -> next = st -> secondary_state.oc.object_cache_1;
    st -> secondary_state.oc.object_cache_1 = och;
  } else {
    stm_assert (clientObjToBytes (ptr) == sizeof (STMDescriptor_t));
    st -> object_cache_2_size += clientObjToBytes (ptr);
    och -> next = st -> secondary_state.oc.object_cache_2;
    st -> secondary_state.oc.object_cache_2 = och;
  }
}

static void *find_from_object_cache (STMState_t *st, int bytes)
{
  OCHeader_t  *result;
  OCHeader_t **ptr;

  if (bytes == sizeof (STMSleepers_t)) 
    {
      ptr = &(st -> secondary_state.oc.object_cache_1);
      result = *ptr;
#ifdef DO_BALANCE_OBJECT_CACHES
      if (result == NULL)
	{
	  int result_size;
	  do {
	    MB();
	    result = shared_oc_1 -> shared_oc;
	    result_size = shared_oc_1 -> size;
	  } while (!CASPtrInt((long long *) shared_oc_1, 
			      result, NULL,
			      result_size, 0));
	  if (result != NULL) {
	    st -> secondary_state.oc.object_cache_1 = result;
	    st -> object_cache_1_size = result_size;
	    result = *ptr;
	  }
	}
#endif
    } else {
      ptr = &(st -> secondary_state.oc.object_cache_2);
      result = *ptr;
#ifdef DO_BALANCE_OBJECT_CACHES
      if (result == NULL)
	{
	  int result_size;
	  do {
	    MB();
	    result = shared_oc_2 -> shared_oc;
	    result_size = shared_oc_2 -> size;
	  } while (!CASPtrInt((long long *) shared_oc_2, 
			      result, NULL,
			      result_size, 0));
	  if (result != NULL) {
	    st -> secondary_state.oc.object_cache_2 = result;
	    st -> object_cache_2_size = result_size;
	    result = *ptr;
	  }
	}
#endif
    }

  if (result != NULL)
    {
      *ptr = result -> next;
#ifdef DEBUG
      result -> next = NULL;
#endif
      if (bytes == sizeof (STMSleepers_t)) 
	{
	  st -> object_cache_1_size -= clientObjToBytes (OCHeaderToObj(result));
	} 
      else
	{
	  st -> object_cache_2_size -= clientObjToBytes (OCHeaderToObj(result));
	}
      return OCHeaderToObj(result);
    }
  else
    {
      return NULL;
    }  
}

static void *allocate_via_object_cache (STMState_t *st, int bytes)
{
  OCHeader_t  *result;

  result = find_from_object_cache (st, bytes);
  if (result == NULL)
    {
      result = PrimaryAllocateBytes (st, bytes + sizeof (OCHeader_t));
      if (result != NULL)
	{
#ifdef DEBUG
	  result -> next = NULL;
#endif
	  result = OCHeaderToObj (result);
	}
    }

  return result;
}

/* Discard cached objects */
static void drop_object_cache (STMState_t *st) 
{
#if (MM_PRIMARY_SCHEME != GC)
  if ((st -> secondary_state.oc.object_cache_1 != NULL) ||
      (st -> secondary_state.oc.object_cache_2 != NULL))
    {
      stm_panic (("Irretrievably dropping object cache contents"));
    }
#endif
  st -> secondary_state.oc.object_cache_1 = NULL;
  st -> secondary_state.oc.object_cache_2 = NULL;
}

/* Start-of-day initialization */
static void init_object_cache (STMState_t *st) 
{
  st -> secondary_state.oc.object_cache_1 = NULL;
  st -> secondary_state.oc.object_cache_2 = NULL;
}

/***********************************************************************/

/*
 * Secondary: Simple
 */

static void *MMAllocateBytesSimple (STMState_t *st, int bytes) {
  void *result;
  result = PrimaryAllocateBytes (st, bytes);
  return result;
}

static void MMDeallocateObjSimple (STMState_t *st, void *ptr) {
  /* Nothing */
}

static void *MMGuardObjSimple (STMState_t *st, int g, void *ptr) {
  return NULL;
}

static void MMReleaseObjSimple (STMState_t *st, int g, void *ptr) {
  /* Nothing */
}

static void MMInitStateSimple (STMState_t *st) {
  PrimaryInitState(st);
}

static void MMReclaimStateSimple (STMState_t *st) {
  PrimaryReclaimState(st);
}

static void MMDoneGCSimple (STMState_t *st) {
  PrimaryDoneGC (st);
}

static void MMInitSchemeSimple (void) {
  mm_header_size = 0;
  PrimaryInitScheme ();
}

/***********************************************************************/

/*
 * Secondary: RC
 */

static void *MMAllocateBytesRC (STMState_t *st, int bytes);
static void  MMDeallocateObjRC (STMState_t *st, void *ptr);
static void *MMGuardObjRC (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjRC (STMState_t *st, int g, void *ptr);

static RCHeader_t *RCObjToHeader (void *ptr) 
{
  RCHeader_t *h;
  h = ((RCHeader_t *) ptr) - 1;
  return h;
}

static void *RCHeaderToObj (RCHeader_t *h)
{
  void *result;
  result = (void *) (h + 1);
  return result;
}

static void *MMAllocateBytesRC (STMState_t *st, int bytes) {
  void        *result;
  RCHeader_t  *rch;
  RCHeader_t **ptr;

  result = find_from_object_cache(st, bytes);
  MB();
  if (result == NULL)
    {
      rch = PrimaryAllocateBytes (st, bytes + sizeof (RCHeader_t));
      if (rch == NULL) {
	return NULL;
      }
#ifdef DEBUG
      memset (rch, 0, bytes + sizeof (RCHeader_t));
#endif

      rch -> rc = 2;
      result = RCHeaderToObj (rch);
      MB();
    }
  else 
    {
      int old_rc;

      rch = RCObjToHeader (result);
      old_rc = get_and_adjust ((int *) &(rch -> rc), 1); 
      stm_assert ((old_rc & 1) == 1);
    }

  MB();

  return result;
}

static void MMDeallocateObjRC (STMState_t *st, void *ptr)
{
  COUNT(st, COUNT_RC_UP);
  MMReleaseObjRC (st, 0, ptr);
}

static void *MMGuardObjRC (STMState_t *st, int g, void *ptr)
{
  RCHeader_t *h;
  h = RCObjToHeader (ptr);
  get_and_adjust ((int *) &(h -> rc), 2);
  MB();
  COUNT(st, COUNT_RC_UP);
  return NULL;
}

static void MMReleaseObjRC (STMState_t *st, int g, void *ptr)
{
  RCHeader_t *h;
  int old_rc;
  int new_rc;

  COUNT(st, COUNT_RC_DOWN);

  h = RCObjToHeader (ptr);
  do
    {
      MB();
      old_rc = h -> rc;
      new_rc = old_rc - 2;
      if (new_rc == 0) 
	{
	  new_rc = 1;
	}
    }
  while (CASInt (&(h -> rc), old_rc, new_rc) != old_rc);

  if (old_rc == 1) {
    stm_panic (("freeing twice"));
  }

  if (((old_rc - new_rc) & 1) == 1)
    {
      stm_assert (old_rc == 2);
      MB();
#ifdef KILL_ON_DEALLOCATE
      clientKillObject (ptr);
#endif
      add_to_object_cache (st, ptr);
    }
}

static void MMInitStateRC (STMState_t *st) {
  PrimaryInitState (st);
  init_object_cache (st);
}

static void MMReclaimStateRC (STMState_t *st) {
  drop_object_cache (st);
  PrimaryReclaimState (st);
}

static void MMDoneGCRC (STMState_t *st) {
  drop_object_cache (st);
  PrimaryDoneGC (st);
}

static void MMInitSchemeRC (void) {
  mm_header_size = sizeof (RCHeader_t);
  PrimaryInitScheme ();
}

/***********************************************************************/

/*
 * Secondary: PD
 */

static void *MMAllocateBytesPD (STMState_t *st, int bytes);
static void  MMDeallocateObjPD (STMState_t *st, void *ptr);
static void *MMGuardObjPD (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjPD (STMState_t *st, int g, void *ptr);

static PDHeader_t *PDObjToHeader (void *ptr) 
{
  PDHeader_t *h;
  h = ((PDHeader_t *) ptr) - 1;
  return h;
}

static void *PDHeaderToObj (PDHeader_t *h)
{
  void *result;
  result = (void *) (h + 1);
  return result;
}

static void *MMAllocateBytesPD (STMState_t *st, int bytes) { 
  PDHeader_t *result;

  result = st -> secondary_state.pd.last_header;
  MB();
  if (result != NULL && result -> isPrivate == FALSE) {
    result = st -> secondary_state.pd.last_header = NULL;
  }

  if ((result == NULL) || 
      (clientObjToBytes (PDHeaderToObj (result)) != bytes))
    {
      result = PrimaryAllocateBytes (st, bytes + sizeof (PDHeader_t));
      if (result == NULL) {
	return NULL;
      }

      result -> isPrivate = TRUE;
    }
  
  return PDHeaderToObj (result);
}

static void MMDeallocateObjPD (STMState_t *st, void *ptr) {
  PDHeader_t *h;
  h = PDObjToHeader (ptr);
  st -> secondary_state.pd.last_header = h;
}

static void *MMGuardObjPD (STMState_t *st, int g, void *ptr) {
  PDHeader_t *h;
  h = PDObjToHeader (ptr);
  h -> isPrivate = FALSE;
  MB();
  return NULL;
}

static void MMReleaseObjPD (STMState_t *st, int g, void *ptr) {
  /* Nothing */
}

static void MMInitStatePD (STMState_t *st) {
  PrimaryInitState (st);
  st -> secondary_state.pd.last_header = NULL;
}

static void MMReclaimStatePD (STMState_t *st) {
  PrimaryReclaimState (st);
}

static void MMDoneGCPD (STMState_t *st) {
  st -> secondary_state.pd.last_header = NULL;
  PrimaryDoneGC (st);
}

static void MMInitSchemePD (void) {
  mm_header_size = sizeof (PDHeader_t);
  PrimaryInitScheme ();
}

/***********************************************************************/

/*
 * Secondary: SMR
 */

static void *MMAllocateBytesSMR (STMState_t *st, int bytes);
static void  MMDeallocateObjSMR (STMState_t *st, void *ptr);
static void *MMGuardObjSMR (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjSMR (STMState_t *st, int g, void *ptr);

/*
 * Hazard pointers are to objs rather than to our headers: we assume that
 * guard / unguard will be common and so defer conversions to scan
 */
static void *hp[MAX_GUARDS];

static void SMRFreeObj (STMState_t *st, void *obj) 
{
#ifdef KILL_ON_DEALLOCATE
  clientKillObject (obj);
#endif
#ifdef DEBUG
  OCObjToHeader(obj) -> next = NULL;
#endif
  add_to_object_cache (st, obj);
}

static void SMRScan (STMState_t *st)
{
  OCHeader_t *plist[MAX_GUARDS];
  OCHeader_t *new_dlist;
  OCHeader_t *t;
  int new_dcount;
  int i, j;
  int p;
  int old_tctr = tctr;

  NO_INLINE ();
  /* Stage 1 */
  p = 0;
  for (i = 0; i < old_tctr * GUARDS_PER_THREAD; i ++)
    {
      if (hp[i] != NULL) 
	{
	  t = OCObjToHeader (hp[i]);
	  plist[p++] = t;
	}
    }

  /* Stage 2 */

  /* 
   * For now we just use a linear search since there are only a small
   * number of pointers.
   */

  /* Stage 3 */
  t = st -> secondary_state.smr.dlist;
  new_dcount = 0;
  new_dlist = NULL;
  while (t != NULL) {
    OCHeader_t *next_t;

    next_t = t -> next;

    for (i = 0; i < p; i ++) {
      if (GUARD_PROTECTS (plist[i], t)) {
	t -> next = new_dlist;
	new_dlist = t;
	new_dcount ++;
	break;
      }
    }

    if (i == p)
      {
	SMRFreeObj (st, OCHeaderToObj (t));
      } 

    t = next_t;
  }

  /* Stage 4 */
  st -> secondary_state.smr.dlist = new_dlist;
  st -> secondary_state.smr.dcount = new_dcount;
}

static void *MMAllocateBytesSMR (STMState_t *st, int bytes) { 
  return allocate_via_object_cache (st, bytes);
}

static void MMDeallocateObjSMR (STMState_t *st, void *ptr) {
  OCHeader_t *h;
  OCHeader_t *old_dlist;

  h = OCObjToHeader (ptr);
  stm_assert (h -> next == NULL);
  old_dlist = st -> secondary_state.smr.dlist;
  h -> next = old_dlist;
  st -> secondary_state.smr.dlist = h;
  st -> secondary_state.smr.dcount ++;

  if (st -> secondary_state.smr.dcount >= DEFERRED_FREE_BATCH_SIZE) {
    st -> last_transaction = NULL;
    SMRScan (st);
  }
}

static void *MMGuardObjSMR (STMState_t *st, int g, void *ptr) {
  void *result;
  int   idx;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;
  result = hp[idx];
  hp[idx] = ptr;
  MB ();

  return result;
}

static void MMReleaseObjSMR (STMState_t *st, int g, void *ptr) {
  int idx;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  MB();
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;
  hp[idx] = NULL;
}

static void MMInitStateSMR (STMState_t *st) {
  int idx;
  int i;
  PrimaryInitState (st);
  init_object_cache (st);
  st -> secondary_state.smr.dcount = 0;
  st -> secondary_state.smr.dlist = NULL;
  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    hp[idx + i] = NULL;
  }
}

static void MMReclaimStateSMR (STMState_t *st) {
  int idx;
  int i;

  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    hp[idx + i] = NULL;
  }

  /* Should not lose any remaining items in the dlist */
  stm_panic (("MMReclaimStateSMR not implemented"));

  PrimaryReclaimState (st);
}

static void MMDoneGCSMR (STMState_t *st) {
  int idx;
  int i;

  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    hp[idx + i] = NULL;
  }

  st -> secondary_state.smr.dlist = NULL;
  st -> secondary_state.smr.dcount = 0;

  drop_object_cache (st);
  PrimaryDoneGC (st);
}

static void MMInitSchemeSMR (void) {
  int i;
  mm_header_size = sizeof (OCHeader_t);
  PrimaryInitScheme ();
  for (i = 0; i < MAX_GUARDS; i ++) {
    hp[i] = NULL;
  }
}

/***********************************************************************/

/*
 * Secondary: PTB
 */

static void *MMAllocateBytesPTB (STMState_t *st, int bytes);
static void  MMDeallocateObjPTB (STMState_t *st, void *ptr);
static void *MMGuardObjPTB (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjPTB (STMState_t *st, int g, void *ptr);

#define PTB_PACK(_hh) ((((long long) ((_hh).h) ) << 32) + ((_hh).ver))

typedef struct PTBHandoff_t PTBHandoff_t;

struct PTBHandoff_t {
  OCHeader_t *h;
  int         ver;
};

static OCHeader_t   **post;
static PTBHandoff_t  *handoff;

static void PTBFreeObj (STMState_t *st, void *obj) {
#ifdef KILL_ON_DEALLOCATE
  clientKillObject (obj);
#endif
#ifdef DEBUG
  OCObjToHeader(obj) -> next = NULL;
#endif
  add_to_object_cache (st, obj);
}

static void *MMAllocateBytesPTB (STMState_t *st, int bytes) { 
  return allocate_via_object_cache (st, bytes);
}

static OCHeader_t *PTBSearchForEscapee (OCHeader_t **hp, OCHeader_t *guard) {
  OCHeader_t *result;
  OCHeader_t *escapee;
  
  result = NULL;
  escapee = *hp;
  while (escapee != NULL) {
    if (GUARD_PROTECTS (guard, escapee)) {
      stm_assert (result == NULL);
      result = escapee;
#ifndef DEBUG
      break;
#endif
    }
    escapee = escapee -> next;
  }

  return result;
}

static void PTBInsertPtr (OCHeader_t **hp, OCHeader_t *ptr) {
  stm_assert (ptr -> next == NULL);
  ptr -> next = *hp;
  *hp = ptr;
}

static void PTBDeletePtr (OCHeader_t **hp, OCHeader_t *ptr) {
  OCHeader_t *h;

  h = *hp;
  while (h) {
    if (ptr == h) {
      *hp = h -> next;
      h -> next = NULL;
      return;
    }
    hp = &(h -> next);
    h = *hp;
  }

  stm_panic (("%p not found in PTB\n", ptr));
}


static void PTBScan (STMState_t *st) {
  OCHeader_t *vs;
  int i;
  int old_tctr = tctr;

  /*
   * Determine set to liberate
   */

  vs = st -> secondary_state.ptb.dlist;
  for (i = 0; i < old_tctr * GUARDS_PER_THREAD; i ++) {
    int            attempts = 0;
    PTBHandoff_t   ho;
    OCHeader_t    *v;
    OCHeader_t    *escapee;

    ho = handoff[i];
    RMB(); /* Ensure ho was in handoff when we read post */
    v = post[i];
    if (v != NULL && ((escapee = PTBSearchForEscapee (&vs, v)) != NULL)) {
      while (TRUE)
	{
	  PTBDeletePtr (&vs, escapee);
	  if (CASPtrInt ((int64_t *) &handoff[i],
			 (void *) ho.h, (void *) escapee, 
			 ho.ver, ho.ver + 1))
	    {
	      if (ho.h != NULL) PTBInsertPtr (&vs, ho.h);
	      break;
	    }
	  else /* CASPtrInt failed */
	    {
	      PTBInsertPtr (&vs, escapee);
	    }
	  attempts ++;
	  if (attempts == 3) break;
	  ho = handoff [i];
	  RMB ();
	  if (attempts == 2 && ho.h != NULL) break;
	  if (v != post[i]) break;
	}
    } else { /* Guard was stood down or not of our concern */
      if (ho.h != NULL &&
	  !GUARD_PROTECTS(v, ho.h))
	{
	  if (CASPtrInt ((int64_t *) &handoff[i],
			 ho.h, NULL,
			 ho.ver, ho.ver + 1)) 
	    {
	      PTBInsertPtr (&vs, ho.h);
	    }
	}
    }
  }

  /*
   * De-allocate all in vs
   */

  while (vs != NULL) {
    OCHeader_t *next_h;
    next_h = vs -> next;
    PTBFreeObj (st, OCHeaderToObj (vs));
    vs = next_h;
  }

  st -> secondary_state.ptb.dcount = 0;
  st -> secondary_state.ptb.dlist = NULL;
}

static void MMDeallocateObjPTB (STMState_t *st, void *ptr) {
  OCHeader_t *h;
  OCHeader_t *old_dlist;

  h = OCObjToHeader (ptr);
  old_dlist = st -> secondary_state.ptb.dlist;
  h -> next = old_dlist;
  st -> secondary_state.ptb.dlist = h;
  st -> secondary_state.ptb.dcount ++;

  if (st -> secondary_state.ptb.dcount >= DEFERRED_FREE_BATCH_SIZE) {
    PTBScan (st);
  }
}

static void *MMGuardObjPTB (STMState_t *st, int g, void *ptr) {
  OCHeader_t  *h;
  OCHeader_t  *old;
  int          idx;
  void        *result;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;
  
  h = OCObjToHeader (ptr);
  old = post[idx];
  post[idx] = h;
  MB ();

  result = (old == NULL) ? old : OCHeaderToObj (old);

  return result;
}

static void MMReleaseObjPTB (STMState_t *st, int g, void *ptr) {
  int idx;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;

  MB();
  post[idx] = NULL;
}

static void MMInitStatePTB (STMState_t *st) {

  if (sizeof (PTBHandoff_t) > 8) {
    stm_panic (("PTBHandoff_t too large for atomic update"));
  }

  PrimaryInitState (st);
  init_object_cache (st);
  st -> secondary_state.ptb.dlist = NULL;
  st -> secondary_state.ptb.dcount = 0;
}

static void MMReclaimStatePTB (STMState_t *st) {
  int idx;
  int i;
  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    post[idx + i] = NULL;
  }
  drop_object_cache (st);
  PrimaryReclaimState (st);
}

static void MMDoneGCPTB (STMState_t *st) {
  int idx;
  int i;

  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    post[idx + i] = NULL;
    handoff[idx + i].h = NULL;
    handoff[idx + i].ver = 0;
  }

  st -> secondary_state.ptb.dlist = NULL;
  st -> secondary_state.ptb.dcount = 0;

  drop_object_cache (st);
  PrimaryDoneGC (st);
}

static void MMInitSchemePTB (void) {
  int i;
  post = mallocBytes (MAX_GUARDS * sizeof (OCHeader_t *));
  handoff = mallocBytes (MAX_GUARDS * sizeof (PTBHandoff_t));
  mm_header_size = sizeof (OCHeader_t);
  PrimaryInitScheme ();
  for (i = 0; i < MAX_GUARDS; i ++) {
    post[i] = NULL;
    handoff[i].h = NULL;
    handoff[i].ver = 0;
  }
}

/***********************************************************************/

#ifdef DO_CACHE_OREC_STATUS
static void discard_ownership_cache (STMState_t *st,
				     STMDescriptor_t *stmd)
{
  int i, n;

  stm_assert (stmd != NULL);
  stm_assert (st != NULL);

  n = stmd -> length;
  for (i = 0; i < stmd -> length; i ++)
    {
      int idx;
      idx = OREC_HASH(stmd -> entries[i].addr);
      st -> orec_cache[idx] = OREC_NOT_SEEN;
    }
}
#else
static void discard_ownership_cache (STMState_t *st,
				     STMDescriptor_t *stmd)
{
  /* Nothing */
}
#endif

/*
 * Management of descriptors
 */

static bool_t is_stm_descriptor (void *ptr)
{
  STMDescriptor_t *tmp;
  bool_t result;

  tmp = (STMDescriptor_t *) ptr;
  if (tmp -> status >= STM_ACTIVE && 
      tmp -> status < BEYOND_LAST_STATUS)
    {
      result = TRUE;
    }
  else
    {
      result = FALSE;
    }
  return result;

}

static int descriptor_bytes (void)
{
  int result;
  result = sizeof (STMDescriptor_t);
  return result;
}

static int clientObjToBytes (void *ptr)
{
  int result;
  result = is_stm_descriptor (ptr) ? sizeof (STMDescriptor_t) : sizeof (STMSleepers_t);
  return result;
}

#define DEAD_VALUE ((void *) 0xDEADCA55)

static void clientKillObject (void *ptr)
{
  if (is_stm_descriptor (ptr))
    {
      STMDescriptor_t *stmd;
      int i;
      
      stmd = (STMDescriptor_t *) ptr;
      for (i = 0; i < MAX_LOCATIONS; i ++) {
	stm_assert ((OREC_OF(stmd -> entries[i].addr) -> u.owner) != stmd);
	stmd -> entries[i].addr = DEAD_VALUE;
	stmd -> entries[i].new_val = (word_t) DEAD_VALUE;
	stmd -> entries[i].new_version = (int) DEAD_VALUE;
	stmd -> entries[i].old_val = (word_t) DEAD_VALUE;
	stmd -> entries[i].old_version = (int) DEAD_VALUE;
      }
    }
  else
    {
      STMSleepers_t *stms;
      stms = (STMSleepers_t *) ptr;
      stms -> st = (STMState_t *) DEAD_VALUE;
      stms -> next = (STMSleepers_t *) DEAD_VALUE;
    }
}

/***********************************************************************/

static STMResult_t stm_set_status (STMDescriptor_t *stmd, 
				   STMResult_t expected_status,
				   STMResult_t desired_status)
{
  STMResult_t seen;

  stm_assert ((expected_status != STM_ABORTED) &&
	      (expected_status != STM_COMMITTED));

  MB ();

  seen = CASInt ((int volatile *) &(stmd -> status), 
		 expected_status, 
		 desired_status);

  MB();

  stm_assert (stmd -> status != STM_ACTIVE);

  return seen;
}

/* If we fail to add a registered sleeper then we must record that a
 * GC must take place.  Otherwise a thread can spin re-attempted a
 * transaction and never sleeping (because it can't) and never GCing
 * (because StartTransaction * always proceeds with a cached
 * desciptor.
 */

static volatile bool_t need_gc = FALSE;

static bool_t add_to_registered_sleepers (STMState_t *st,
					  STMState_t *master)
{
  STMSleepers_t *stms; 
  bool_t result;

  stms = MMAllocateBytes (st, sizeof (STMSleepers_t));
  if (stms != NULL) 
    {
      stms -> st = st;
      stms -> next = master -> sleepers;
      master -> sleepers = stms;
      result = FALSE;
    } 
  else 
    {
      need_gc = TRUE;
      result = TRUE;
    }
  return result;
}

static void wake_registered_sleepers (STMState_t *st)
{
  STMSleepers_t *stms;
  STMSleepers_t *next_stms;
  int i = 0;

  stms = st -> sleepers;
  while (stms != NULL)
    {
      STMState_t *other;
      STMDescriptor_t *od;
      other = stms -> st;
      od = other -> current_transaction;
      i ++;
      stm_assert (i < MAX_THREADS);
      if (od != NULL)
	{
	  void *og;
	  og = MMGuardObj (st, 0, od);
	  stm_assert (og == NULL);
	  if (other -> current_transaction == od)
	    {
	      STMMutexLock (&other -> mutex);
	      STMCondSignal (&other -> cond);
	      stm_set_status (od, STM_WAITING_2, STM_ABORTED);
	      STMMutexUnlock (&other -> mutex);
	    }
	  MMReleaseObj (st, 0, od);
	}
      next_stms = stms -> next;
      MMDeallocateObj (st, stms);
      stms = next_stms;
    }
  st -> sleepers = NULL;
}

/* 
 * Wake stmd if it is in deep sleep.  It is responsible for waking
 * its registered waiters.
 */

static void wake_sleeper (STMState_t *st, STMDescriptor_t *stmd)
{
  int seen;

  STMMutexLock (&stmd -> st -> mutex);

  seen = stm_set_status (stmd,
			 STM_WAITING_2,
			 STM_ABORTED);

  if (seen == STM_WAITING_2) STMCondSignal (&stmd -> st -> cond);

  STMMutexUnlock (&stmd -> st -> mutex);
}

/*......................................................................*/

static void release_ownership (STMDescriptor_t *stmd,
			       bool_t stmd_to_wait)
{
  int i;
  int n;
  bool_t committed;

  n = stmd -> length;
  committed = ((stmd -> status) == STM_COMMITTED);

  for (i = 0; i < n ; i++)
    {
      STMEntry_t        *entry = &(stmd -> entries [i]);
      OwnershipRecord_t *orec = OREC_OF (entry -> addr);
      STMDescriptor_t   *new_owner;

#ifdef DO_SHARABLE_READS
      if ((stmd_to_wait == FALSE) && 
	  (stmd -> entries[i].new_version == stmd -> entries[i].old_version))
	{
	  break; // No more writes after this entry
	}
#endif
      
      if (committed)
	{
	  new_owner = (void *) (stmd -> entries[i].new_version);
	} 
      else 
	{
	  new_owner = (void *) (stmd -> entries[i].old_version);
	}

      CASPtr ((addr_t) &(orec -> u.owner), stmd, new_owner);
    }
}

static bool_t can_deallocate (STMDescriptor_t *stmd)
{
  bool_t result;

  int i, n;

  result = TRUE;
  n = stmd -> length;
  for (i = 0; i < n; i ++)
    {
      STMEntry_t *t = &(stmd -> entries[i]);
      if (OREC_OF(t -> addr) -> u.owner == stmd)
	{
	  result = FALSE;
	  break;
	}
    }

  if (result == TRUE) {
    int status = stmd -> status;
    if (status == STM_COMMITTED) {
      result = (CASInt ((int volatile *) &(stmd -> status), 
			status, 
			STM_DEALLOCATED_COMMITTED) == status);
    } else if (status == STM_ABORTED) {
      result = (CASInt ((int volatile *) &(stmd -> status), 
			status, 
			STM_DEALLOCATED_ABORTED) == status);
    } else {
      result = FALSE;
      stm_assert (status == STM_DEALLOCATED_ABORTED || 
		  status == STM_DEALLOCATED_COMMITTED);
    }
  }

  return result;
}

static void release_ownership_nb_slow (STMState_t *st, 
				       STMDescriptor_t *stmd,
				       int i)
{
  int n;
  bool_t succeeded;

  NO_INLINE ();
  COUNT (st, COUNT_RELEASE_SLOW_NB);

  stm_assert (stmd -> status == STM_COMMITTED || stmd -> status == STM_ABORTED);
  succeeded = ((stmd -> status) == STM_COMMITTED);
    {
      STMEntry_t        *my_entry = &(stmd -> entries [i]);
      OwnershipRecord_t *orec = OREC_OF (my_entry -> addr);

      if (st -> does_own [i])
	{
	  STMEntry_t *update_entry;
	  STMDescriptor_t *exp_owner;
	  STMDescriptor_t *new_owner;
	  int exp_count, new_count;
	  word_t update_val;
	  
	  exp_owner = NULL;
	  do
	    {
	      int status;
	      update_entry = NULL;

	    retry_release:
	      if (exp_owner != NULL && exp_owner != stmd) {
		MMReleaseObj (st, 0, exp_owner);
	      }

	      MB();
	      exp_owner = orec -> u.owner;
	      if (exp_owner == stmd) {
		update_entry = my_entry;
	      } else {
		int update_n;
		int update_i;
		void *og;

		stm_assert ((((int)exp_owner) & 2) == 0);
		og = MMGuardObj (st, 0, exp_owner);
		stm_assert (og == NULL);
		if (orec -> u.owner != exp_owner) {
		  goto retry_release;
		}

		update_n = exp_owner -> length;
		for (update_i = 0; update_i < update_n; update_i ++)
		  {
		    STMEntry_t *t;
		    t = &(exp_owner -> entries[update_i]);
		    MB();
		    if (t -> addr == my_entry -> addr) {
		      update_entry = t;
		      break;
		    }
		  }

		stm_set_status (exp_owner, STM_ACTIVE, STM_ABORTED);
	      }
	      
	      stm_assert (update_entry != NULL);
	      stm_assert (exp_owner -> status != STM_ACTIVE);

	      status = exp_owner -> status;
	      if (status == STM_COMMITTED ||
		  status == STM_DEALLOCATED_COMMITTED) {
		update_val = update_entry -> new_val;
	      } else {
		update_val = update_entry -> old_val;
	      }

	      MB();
	      *(update_entry -> addr) = (void *) update_val;
	      GC_WRITE_BARRIER (update_entry -> addr, update_val);
	      MB();

	      exp_count = orec -> updaters;
	      new_count = exp_count - 1;
	      if (new_count == 0)
		{
		  if (status == STM_COMMITTED || status == STM_DEALLOCATED_COMMITTED)
		    {
		      new_owner = (void *) (update_entry -> new_version);
		    }
		  else
		    {
		      new_owner = (void *) (update_entry -> old_version);
		    }
		}
	      else 
		{
		  new_owner = exp_owner;
		}
	    }
	  while (!CASPtrInt ((long long *) &(orec -> u.owner), 
			     exp_owner, new_owner,
			     exp_count, new_count));


#ifdef DEBUG
	  if ((((int)new_owner) & 1) == 0)
	    {
	      int ikk = 0;
	      for (ikk = 0; ikk < new_owner -> length; ikk++) {
		stm_assert (new_owner -> entries[ikk].addr != DEAD_VALUE);
	      }
	    }
#endif

	  if (exp_owner != stmd)
	    {
	      if ((new_count == 0) && 
		  (can_deallocate (exp_owner)))
		{
		  MMReleaseObj (st, 0, exp_owner);
		  MMDeallocateObj (st, exp_owner);
		}
	      else
		{
		  MMReleaseObj (st, 0, exp_owner);
		}
	    }
	}
    }
}

static void release_ownership_nb (STMState_t *st, STMDescriptor_t *stmd)
{
  int i;
  int n;
  bool_t succeeded;

  n = stmd -> length;
  succeeded = ((stmd -> status) == STM_COMMITTED);
  for (i = 0; i < n ; i++)
    {
      if (st -> does_own [i])
	{
	  STMEntry_t        *my_entry = &(stmd -> entries [i]);
	  OwnershipRecord_t *orec     = OREC_OF (my_entry -> addr);
	  
	  if (orec -> u.owner == stmd)
	    {
	      word_t val = ((succeeded) ?
			    my_entry -> new_val : 
			    my_entry -> old_val);
	      int version =  ((succeeded) ?
			    my_entry -> new_version : 
			    my_entry -> old_version);

	      MB();
	      *(my_entry -> addr) = (void *) val;
	      GC_WRITE_BARRIER (my_entry -> addr, val);
	      MB();
	      
	      if (!CASPtrInt ((long long *) &(orec -> u.owner),
			      stmd, (void *) (word_t) version,
			      1, 0))
		{
		  release_ownership_nb_slow (st, stmd, i);
		}
	      else
		{
		  COUNT (st, COUNT_RELEASE_FAST_NB);
		}
	    } else {
	      release_ownership_nb_slow (st, stmd, i);
	    }
	}
    }
}

/*......................................................................*/

static bool_t check_reads (STM_ENV *env,
			   STMState_t *st,
			   STMDescriptor_t *stmd)
{
  bool_t result;
  int i, n;

  result = TRUE;

  MB();
  n = stmd -> length;
  for (i = 0; i < n; i ++)
    {
      STMEntry_t *entry;

      entry = &(stmd -> entries[i]);
      if (entry -> old_version == entry -> new_version)
	{
	  OwnershipRecord_t *orec;
	  union STMVersionInfo_t vi;

	  orec = OREC_OF (entry -> addr);
retry_check_location:
	  vi = orec -> u;
	  if (vi.version != entry -> old_version)
	    {
	      if (VI_IS_VERSION (vi)) 
		{
		  result = FALSE;
		  break;
		}
	      else
		{
		  STMDescriptor_t *owner = vi.owner;
		  int owner_status;
		  bool_t committed;
		  int version;
		  MMGuardObj (st, 0, owner);
		  if (orec -> u.owner != owner) {
		    MMReleaseObj (st, 0, owner);
		    goto retry_check_location;
		  }
		  owner_status = owner -> status;
#ifdef DO_SHARABLE_READS
		  if (owner_status == STM_READ_PHASE)
		    {
#if defined(DO_STM_DEFER_OTHER) || defined (DO_STM_ABORT_SELF)
		      MMReleaseObj (st, 0, owner);
		      goto retry_check_location;
#else
		      stm_set_status (owner, STM_READ_PHASE, STM_ABORTED);
		      MMReleaseObj (st, 0, owner);
		      goto retry_check_location;
#endif
		    }
#endif
		  
		  version = -1;
		  committed = ((owner_status == STM_COMMITTED) ||
			       (owner_status == STM_DEALLOCATED_COMMITTED));
		  
		  for (i = 0; i < owner -> length; i ++)
		    {
		      if (OREC_OF (owner -> entries[i].addr) == orec)
			{
			  STMEntry_t *entry;
			  entry = &(owner -> entries[i]);
			  version = committed ? entry -> new_version : entry -> old_version;
			  break;
			}
		    }
		  MMReleaseObj (st, 0, owner);

		  stm_assert (version != -1);
		  if (version != entry -> old_version) 
		    {
		      result = FALSE;
		      break;
		    }
		}
	    }
	}
    }
  MB();

  return result;
}

/*......................................................................*/

static bool_t acquire_ownership (STM_ENV *env,
				 STMState_t *st,
				 STMDescriptor_t *stmd, 
				 bool_t stmd_to_wait)
{
  int i, n;
  bool_t result;

  result = TRUE;
  n = stmd -> length;

  for (i = 0; i < n ; i++)
    {
      OwnershipRecord_t *orec;
      STMEntry_t        *entry;
      STMDescriptor_t *exp_owner, *new_owner;
      union STMVersionInfo_t vi;

      entry = &(stmd -> entries[i]);
#ifdef DO_SHARABLE_READS
      if ((!stmd_to_wait) && (entry -> old_version == entry -> new_version))
	{
	  break; // Will be dealt with in check_reads 
	}
#endif

      orec = OREC_OF (entry -> addr);
      MB();

      do
	{
retry:
	  vi = orec -> u;

	  exp_owner = vi.owner;
	  new_owner = stmd;

	  if (VI_IS_VERSION (vi) && vi.version != entry -> old_version)
	    {
	      /* This is not the version we are looking for */
	      result = FALSE;
	      goto done_acquire;
	    }

	  if ((!VI_IS_VERSION(vi)) && exp_owner != stmd)
	    {
	      /* This orec is already owned */
	      STMResult_t owners_status;
	      void *og;

	      MB();
	      og = MMGuardObj (st, i + 2, exp_owner);
	      stm_assert (og == NULL);
	      if (orec -> u.owner != exp_owner) {
		MMReleaseObj (st, i+2, exp_owner);
		goto retry;
	      }

	      owners_status = exp_owner -> status;

	      /* 
	       * Try to wake it if it is in light sleep -- either 
	       * we need to own the location (so must get it to abort)
	       * or we want to become a registered sleeper with it
	       * (so we want it in STM_WAITING_2).
	       */

	      if (owners_status == STM_WAITING_1)
		{
		  stm_set_status (exp_owner, STM_WAITING_1, STM_ABORTED);
		  MB();
		  owners_status = exp_owner -> status;
		}
	      
	      /*
	       * If it is in a deep sleep then discover the current
	       * version of the location it's sleeping on.  Then
	       * either:
	       *
	       *  (a) we want to sleep waiting for the same value to
	       *      change: register ourselves with the current owner,
	       *
	       *  (b) we want to update from that value: wake the sleeper
	       *      and try to take ownership of the location,
	       *
	       *  (c) we want to sleep or update from some other value:
	       *      we abort and leave the sleeper.	      
	       */

	      if (owners_status == STM_WAITING_2)
		{
		  int current_version;
		  int j;

		  current_version = -1;
		  for (j = 0; j < exp_owner -> length; j ++) {
		    if (OREC_OF(exp_owner -> entries[j].addr) == orec) {
		      current_version = exp_owner -> entries[j].old_version;
		      break; /* From searching for current_version */
		    }
		  }
		  stm_assert (current_version != -1);

		  if (stmd_to_wait) 
		    {
		      if (entry -> old_version == current_version)
			{
			  /* (a) We join sleeper */
			  st -> waiting_with[i] = exp_owner;
			  break; /* Go to next entry */
			}
		      else
			{
			  /* (c) Validation failed */
			  result = FALSE;
			  MMReleaseObj (st, i + 2, exp_owner);
			  goto done_acquire;
			}
		    }
		  else
		    {
		      if (entry -> old_version == current_version)
			{
			  /* (b) Wake the sleeper */
			  wake_sleeper (st, exp_owner);
			  owners_status = exp_owner -> status;
			}
		      else
			{
			  /* (c) Validation failed */
			  result = FALSE;
			  MMReleaseObj (st, i + 2, exp_owner);
			  goto done_acquire;
			}
		    }
		}
	      
	      /*
	       * If the current owner has aborted then we can try 
	       * to release their ownership of these locations.
	       */

	      if (owners_status == STM_ABORTED)
		{
                  release_ownership (exp_owner, stmd_to_wait);
		  MMReleaseObj (st, i+2, exp_owner);
		  goto retry; /* Retry this entry */
		}

	      /*
	       * The current owner is active or committed.  Decide 
	       * how to proceed on the basis of our configuration.
	       *
	       *  OPT_ABORT_SELF => we abort and hope that the current
	       *       owner will be out of the way if we retry.
	       */

	      if (OPT_ABORT_SELF) {
		result = FALSE;
		MMReleaseObj (st, i + 2, exp_owner);
		goto done_acquire;
	      } 

	      /*
	       *  OPT_DEFER_OTHER => we spin trying to gain ownership
	       *       of this location and hope that the current 
	       *       owner will be done soon.
	       */

	      stm_assert (OPT_DEFER_OTHER);
	      MMReleaseObj (st, i+2, exp_owner);
	      goto retry;
	    }
	}
      while (CASPtr ((addr_t) &(orec -> u.owner), 
		     exp_owner, 
		     new_owner) != exp_owner);

      stm_assert (stmd_to_wait || 
		  (exp_owner == stmd) ||
		  ((((word_t) exp_owner) & 1) == 1));

      stm_assert (stmd_to_wait || 
		  (exp_owner == stmd) || 
		  ((word_t) exp_owner) == entry -> old_version);

    }

 done_acquire:
  return result;
}



static bool_t merge_into_descriptor (STMState_t *st,
				     STMDescriptor_t *stmd,
				     STMDescriptor_t *from,
				     OwnershipRecord_t *orec)
{
  bool_t succeeded;
  bool_t result;
  int i;
  int n;
  int to_i;
  int to_n;
  int proto_length;
  int from_status;
  bool_t stmd_to_update;

  /* 
   * Is "stmd" proposing to make any updates?  If so then entries we
   * acquire through merging must take a new version number in order to
   * ensure consistency with any existing updates that use the same
   * ownership record.
   */

  proto_length = stmd -> length;
  stmd_to_update = FALSE;
  for (i = 0; i < proto_length; i ++)
    {
      if ((OREC_OF(stmd -> entries[i].addr) == orec) &&
	  (stmd -> entries[i].new_version != stmd -> entries[i].old_version))
	{
	  stmd_to_update = TRUE;
	  break;
	}
    }


  result = TRUE;
  from_status = from -> status;
  succeeded = ((from_status == STM_COMMITTED) ||
	       (from_status == STM_DEALLOCATED_COMMITTED));

  stm_assert (succeeded || 
	      (from_status == STM_ABORTED) ||
	      (from_status == STM_DEALLOCATED_ABORTED));

  to_n = stmd -> length;
  n = from -> length;
  for (i = 0; i < n; i ++)
    {
      STMEntry_t *from_entry;
      STMEntry_t *to_entry;
      int exp_version;

      from_entry = &(from -> entries[i]);
      exp_version = ((succeeded) ? 
		     from_entry -> new_version : 
		     from_entry -> old_version);


      if (OREC_OF (from_entry -> addr) == orec)
	{
	  /* from_entry is owned by the orec we're interested in */
	  for (to_i = 0; to_i < to_n; to_i++)
	    {
	      to_entry = &(stmd -> entries[to_i]);

	      if (OREC_OF (to_entry -> addr) == orec)
		{
		  /* to_entry is owned by the same orec, check consistency */
		  if (to_entry -> old_version != exp_version)
		    {
		      result = FALSE;
		      goto done_merge_into_descriptor;
		    }
		  else
		    {
		      if (to_entry -> addr == from_entry -> addr)
			{
			  break; /* from searching for to_i */
			}
		    }
		}
	    }

	  /* If we did not find a match then we need a new entry */
	  if (to_i == to_n)
	    {
	      stm_hard_assert ((proto_length < MAX_LOCATIONS));
	      to_entry = &(stmd -> entries[proto_length]);
	      to_entry -> addr = from_entry -> addr;
	      if (succeeded)
		{
		  to_entry -> old_val = from_entry -> new_val;
		  to_entry -> old_version = from_entry -> new_version;
		}
	      else
		{
		  to_entry -> old_val = from_entry -> old_val;
		  to_entry -> old_version = from_entry -> old_version;
		}    

	      to_entry -> new_val = to_entry -> old_val;
	      if (stmd_to_update) {
		to_entry -> new_version = to_entry -> old_version + 2;
	      } else {
		to_entry -> new_version = to_entry -> old_version;
	      }

	      st -> does_own[proto_length] = FALSE;

	      proto_length ++;
	    }
	}
    }

done_merge_into_descriptor:
  MB();
  stmd -> length = proto_length;

  return result;
}

static bool_t acquire_ownership_nb_slow (STM_ENV *env,
					 STMState_t *st,
					 STMDescriptor_t *stmd, 
					 int i,
					 bool_t stmd_to_wait)
{
  bool_t result = TRUE;
  OwnershipRecord_t *orec;
  STMEntry_t        *entry;
  STMDescriptor_t *exp_owner, *new_owner;
  int exp_count;
  union STMVersionInfo_t vi;

  NO_INLINE();

  entry = &(stmd -> entries[i]);
  orec = OREC_OF (entry -> addr);
  
  do
    {
    retry:
      vi = orec -> u;
      
      exp_owner = vi.owner;
      exp_count = orec -> updaters;

      new_owner = stmd;
      
      if (VI_IS_VERSION (vi) && vi.version != entry -> old_version)
	{
	  /* This is not the version we are looking for */
	  result = FALSE;
	  goto done_acquire;
	}
      
      if ((!VI_IS_VERSION(vi)) && exp_owner != stmd)
	{
	  /* This orec is already owned */
	  STMResult_t owners_status;
	  void *og;
	  
	  og = MMGuardObj (st, i + 2, exp_owner);
	  stm_assert (og == NULL);
	  if (orec -> u.owner != exp_owner) {
	    MMReleaseObj (st, i+2, exp_owner);
	    goto retry;
	  }
	  
	  owners_status = exp_owner -> status;
	  
	  stm_set_status (exp_owner, STM_ACTIVE, STM_ABORTED);
	  
	  stm_assert (exp_owner -> status != STM_ACTIVE);
	  
	  /* 
	   * Try to wake it if it is in light sleep -- either 
	   * we need to own the location (so must get it to abort)
	   * or we want to become a registered sleeper with it
	   * (so we want it in STM_WAITING_2).
	   */
	  
	retry_waking_sleeper:
	  MB();
	  owners_status = exp_owner -> status;
	  if (owners_status == STM_WAITING_1)
	    {
	      stm_set_status (exp_owner, 
			      STM_WAITING_1, 
			      STM_ABORTED);
	      owners_status = exp_owner -> status;
	    }
	  
	  stm_assert (exp_owner -> status != STM_ACTIVE);
	  stm_assert (exp_owner -> status != STM_WAITING_1);
	  
	  /*
	   * If it is in a deep sleep then discover the current
	   * version of the location it's sleeping on.  Then
	   * either:
	   *
	   *  (a) we want to sleep waiting for the same value to
	   *      change: register ourselves with the current owner,
	   *
	   *  (b) we want to update from that value: wake the sleeper
	   *      and try to take ownership of the location,
	   *
	   *  (c) we want to sleep or update from some other value:
	   *      we abort and leave the sleeper.	      
	   */

	  if (owners_status == STM_WAITING_2)
	    {
	      int current_version;
	      int j;
	      
	      current_version = -1;
	      for (j = 0; j < exp_owner -> length; j ++) {
		if (OREC_OF(exp_owner -> entries[j].addr) == orec) {
		  current_version = exp_owner -> entries[j].old_version;
		  break; /* From searching for current_version */
		}
	      }
	      stm_assert (current_version != -1);
	      
	      if (stmd_to_wait) 
		{
		  if (entry -> old_version == current_version)
		    {
		      /* (a) We join sleeper */
		      st -> waiting_with[i] = exp_owner;
		      break; /* Go to next entry */
		    }
		  else
		    {
		      /* (c) Validation failed */
		      result = FALSE;
		      MMReleaseObj (st, i + 2, exp_owner);
		      goto done_acquire;
		    }
		}
	      else
		{
		  if (entry -> old_version == current_version)
		    {
		      /* (b) Wake the sleeper */
		      wake_sleeper (st, exp_owner);
		      owners_status = exp_owner -> status;
		    }
		  else
		    {
		      /* (c) Validation failed */
		      result = FALSE;
		      MMReleaseObj (st, i + 2, exp_owner);
		      goto done_acquire;
		    }
		}
	    }
	  
	  MB();
	  owners_status = exp_owner -> status;
	  
#ifdef DO_SHARABLE_READS
	  if (owners_status == STM_READ_PHASE) {
	    stm_set_status (exp_owner, STM_READ_PHASE, STM_ABORTED);
	    owners_status = exp_owner -> status;
	    MB();
	  }
#endif
	  
	  stm_assert (owners_status != STM_ACTIVE);
	  stm_assert (owners_status != STM_WAITING_1);
	  stm_assert (owners_status != STM_WAITING_2);
	  
	  stm_assert ((owners_status == STM_ABORTED) ||
		      (owners_status == STM_COMMITTED) ||
		      (owners_status == STM_DEALLOCATED_ABORTED) ||
		      (owners_status == STM_DEALLOCATED_COMMITTED));
	  
	  if (stmd_to_wait) 
	    {
	      /* 
	       * If we're going to block anyway then wait for the
	       * current owner to get out of the way.  
	       *
	       * XXX Not sure what the 'right' semantics here are
	       * but this approach avoids anyone sleeping with
	       * updates merged into their descriptor -- i.e. 
	       * STM_WAITING_ people are always read-only.
	       */
	      
	      while (orec -> u.owner == exp_owner) {
		MB();
	      }
	      
	      MMReleaseObj (st, i+2, exp_owner);
	      goto retry;
	    }
	  
	  /*
	   * Merge previous owner's view into ours.
	   */
	  MB();
	  if (!merge_into_descriptor (st, stmd, exp_owner, orec)) {
	    /* The one we're merging into will doom us to fail. */
	    result = FALSE;
	    MMReleaseObj (st, i + 2, exp_owner);
	    goto done_acquire;
	  }
	  
	  if (CASPtrInt ((long long *) &(orec -> u.owner), 
			 exp_owner, new_owner,
			 exp_count, exp_count + 1))
	    {
	      MB();
	      st -> does_own[i] = TRUE;
	      MMReleaseObj (st, i+2, exp_owner);
	      break; /* from the while loop */
	    }
	  else
	    {
	      result = FALSE;
	      MMReleaseObj (st, i + 2, exp_owner);
	      goto done_acquire;
	    }
	}
      else
	{
	  if (CASPtrInt ((long long *) &(orec -> u.owner), 
			 exp_owner, new_owner,
			 exp_count, exp_count + 1))
	    {
	      MB();
	      st -> does_own[i] = TRUE;
	      break; /* from the while loop */
	    }
	}
    }
  while (TRUE);

 done_acquire:
  return result;
}


static bool_t acquire_ownership_nb (STM_ENV *env,
				    STMState_t *st,
				    STMDescriptor_t *stmd, 
				    bool_t stmd_to_wait)
{
  int i, n;
  bool_t result;

  result = TRUE;
  n = stmd -> length;

  i = 0;
  while (i < n)
    {
      OwnershipRecord_t *orec;
      STMEntry_t        *entry;

      entry = &(stmd -> entries[i]);
      orec = OREC_OF (entry -> addr);

#ifdef DO_SHARABLE_READS
      if ((!stmd_to_wait) && (entry -> old_version == entry -> new_version)) {
	i++;
	continue; // Will be dealt with in check_reads 
      }
#endif

      if (CASPtrInt ((long long *) &(orec -> u.owner), 
		     (void *) entry -> old_version, stmd,
		     0, 1))
	{
	  MB();
	  st -> does_own[i] = TRUE;
	  COUNT (st, COUNT_ACQUIRE_FAST_NB);
	}
      else if (!acquire_ownership_nb_slow (env, st, stmd, i, stmd_to_wait))
	{
	  result = FALSE;
	  COUNT (st, COUNT_ACQUIRE_SLOW_NB);
	  goto done_acquire;
	}
      else 
	{
	  n = stmd -> length;
	  COUNT (st, COUNT_ACQUIRE_SLOW_NB);
	}
      i ++;
    }

 done_acquire:
  return result;
}

static void sort_descriptor_entirely (STMDescriptor_t *stmd)
{
  OwnershipRecord_t *orec_i;
  OwnershipRecord_t *orec_j;
  int i, j, n;

  n = stmd -> length;
  for (i = 0; i < n - 1; i ++)
    {
      orec_i = OREC_OF(stmd -> entries[i].addr);
      for (j = i + 1; j < n; j ++)
	{
	  orec_j = OREC_OF(stmd -> entries[j].addr);
	  if (orec_i < orec_j)
	    {
	      STMEntry_t temp = stmd -> entries[j];
	      stmd -> entries[j] = stmd -> entries[i];
	      stmd -> entries[i] = temp;
	      orec_i = orec_j;
	    }
	}
    }
}

static void sort_descriptor_order_writes (STMDescriptor_t *stmd)
{
  OwnershipRecord_t *orec_i;
  OwnershipRecord_t *orec_j;
  bool_t write_i;
  bool_t write_j;
  int i, j, n;

  n = stmd -> length;
  for (i = 0; i < n - 1; i ++)
    {
      orec_i = OREC_OF(stmd -> entries[i].addr);
      write_i = (stmd -> entries[i].old_version != stmd -> entries[i].new_version);
      for (j = i + 1; j < n; j ++)
	{
	  orec_j = OREC_OF(stmd -> entries[j].addr);
	  write_j = (stmd -> entries[j].old_version != stmd -> entries[j].new_version);
	  if ((write_j && (!write_i)) ||
	      (write_j && write_i && orec_j < orec_i))	      
	    {
	      STMEntry_t temp = stmd -> entries[j];
	      stmd -> entries[j] = stmd -> entries[i];
	      stmd -> entries[i] = temp;
	      orec_i = orec_j;
	      write_i = write_j;
	    }
	}

      if (!write_i) {
	break;
      }
    }
}

static void sort_descriptor_segregate_writes (STMDescriptor_t *stmd)
{
  OwnershipRecord_t *orec_i;
  OwnershipRecord_t *orec_j;
  int i, j, n;

  n = stmd -> length;
  i = 0;
  j = n - 1;
  while (i < j)
    {
      bool_t write_i;
      bool_t write_j;

      write_i = (stmd -> entries[i].old_version != 
		 stmd -> entries[i].new_version);
      write_j = (stmd -> entries[j].old_version != 
		 stmd -> entries[j].new_version);

      if (write_i) {
	i++;
      } else if (!write_j) {
	j--;
      } else {
	STMEntry_t temp = stmd -> entries[j];
	stmd -> entries[j] = stmd -> entries[i];
	stmd -> entries[i] = temp;
	i ++;
	j --;    
      }  
    }
}

static void check_consistency (STMDescriptor_t *stmd)
{
  int i,n;
  n = stmd -> length;
  for (i = 0; i< n-1; i ++)
    {
      if (OREC_OF(stmd->entries[i].addr) == OREC_OF(stmd -> entries[i+1].addr))
	{
	  stm_assert ((stmd -> entries[i].old_version) ==
		      (stmd -> entries[i+1].old_version));
	  stm_assert ((stmd -> entries[i].new_version) == 
		      (stmd -> entries[i+1].new_version));
	}
    }
}


static void check_no_writes (STMDescriptor_t *stmd)
{
  int i, n;
  
  n = stmd -> length;
  for (i = 0; i < n ; i++)
    {
      STMEntry_t *entry;
      entry = &(stmd -> entries[i]);
      if (entry -> new_val != entry -> old_val) 
	{
	  stm_panic (("STMWait has updates"));
	}
    }
}

/*......................................................................*/

/* 
 * A transaction calls stm_sleep_lightly when it has 
 * entered STM_WAITING_1.  This provides it with an opportunity
 * to spin brielfy before blocking in a 'heavyweight' sleep.
 * The thread will be woken by having its status updated to
 * STM_ABORTED.
 */

static void stm_sleep_lightly (STMState_t *st, STMDescriptor_t *stmd)
{
  /* Nothing */
}

/*
 * A transaction calls stm_sleep_heavily when it has entered
 * STM_WAITING_2.  The thread is to sleep by blocking on its condition
 * varaible and will be woken by that cv being notifid.
 *
 * Before it can sleep it must register itself with any transactions
 * that own locations that it is interested in.  Once woken, it must
 * wake any other transactions registered with it.
 *
 * This method is called (and returns) holding stmd -> mutex.  
 */

static void stm_sleep_heavily (STMState_t *st, STMDescriptor_t *stmd)
{
  bool_t woken_before_sleeping;
  int i;
  int n;

  woken_before_sleeping = FALSE;

  /* 
   * Acquire mutexes on all transactions to register with.  Ensure
   * that they are in STM_WAITING_2.
   */
  
  n = stmd -> length;
  for (i = 0; i < n; i ++)
    {
      STMDescriptor_t *other;
      other = (st -> waiting_with[i]);
      if (other != NULL)
	{
	  STMMutexLock (&(other -> st -> mutex));
	  if (other -> status != STM_WAITING_2)
	    {
	      woken_before_sleeping = TRUE;
	    }
	}
    }
			
  if (!woken_before_sleeping)
    {
      /*
       * We now hold all the locks: register with anyone owning
       * a location that we're interested in.
       */

      for (i = 0; i < n; i ++)
	{
	  STMDescriptor_t *other;
	  other = (st -> waiting_with[i]);
	  if (other != NULL) {
	    woken_before_sleeping |= add_to_registered_sleepers (st, other -> st);
	  }
	}
    }

  /*
   * Release all the locks except for our own.  Anyone trying to wake
   * us will now block waiting to acquire that mutex.
   */

  for (i = 0; i < n; i ++)
    {
      STMDescriptor_t *other;
      other = (st -> waiting_with[i]);
      if (other != NULL) {
	STMMutexUnlock (&other -> st -> mutex);
      }
    }

  if (woken_before_sleeping)
    {
      /*
       * One of the other transaction we were supposed to wait for
       * got woken up before we registered with it.  We abort
       * and retry.
       */
      
      stm_set_status (stmd, STM_WAITING_2, STM_ABORTED);
    }
  else
    {
      /*
       * Now we block
       */

      stm_assert (st -> current_transaction == stmd);
      STMCondWait (&(st -> cond), &(st -> mutex));
      stm_set_status (stmd, STM_WAITING_2, STM_ABORTED);
      
      wake_registered_sleepers (st);
    }

  /*  stm_assert ((stmd == NULL) || (stmd -> status == STM_ABORTED));*/
}

/*......................................................................*/

void STMVolatileWriteValue (STM_ENV *env, addr_t addr, word_t val)
{
  bool_t did_start;

  do
    {
      did_start = STMStartTransaction (env);
      if (!did_start) stm_panic (("Transaction didn't start\n"));
      STMWriteValue (env, addr, val);
    }
  while (!STMCommitTransaction (env));
}

/*......................................................................*/

word_t STMVolatileReadValue (STM_ENV *env, addr_t addr)
{
  word_t result;
  bool_t did_start;

  do
    {
      did_start = STMStartTransaction (env);
      if (!did_start) stm_panic (("Transaction didn't start\n"));
      result = STMReadValue (env, addr);
    }
  while (!STMCommitTransaction (env));

  return result;	 
}

/*......................................................................*/

int STMStartTransaction (STM_ENV *env) 
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  bool_t           result;

  st = GET_ST (env);
  stm_assert (st -> current_transaction == NULL ||
	      st -> current_transaction -> length == 0);

  START_TIMING (TIMING_ALLOCATION);

  if (OPT_GC_HEAP_AVAILABLE && need_gc) {
    stmd = NULL;
    need_gc = FALSE;
  } else {
    if (st -> current_transaction == NULL)
      {
	stmd = MMAllocateBytes (st, descriptor_bytes ());
      }
    else 
      {
	stmd = st -> current_transaction;
      }
  }
  END_TIMING (st, TIMING_ALLOCATION);

  if (stmd != NULL)
    {
      result = TRUE;
      stmd -> status = STM_ACTIVE;
      stmd -> length = 0;
      stmd -> st = st;
      st -> current_transaction = stmd;
#ifdef DO_FAST_READONLY
      st -> current_is_read_only = TRUE;
#endif
    }
  else
    {
      fprintf (stderr, "xX\n");
      result = FALSE;
    }

  return result;
}

/*......................................................................*/

void STMWaitB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  int              i;
  int              n;
  bool_t           waiting;

  MB();

  st = GET_ST (env);
  START_TIMING (TIMING_WAIT);
  stmd = st -> current_transaction;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      /* Nothing: should retry immediately */
    }
  else
    {
      if (stmd -> status == STM_ACTIVE)
	{
	  n = stmd -> length;
	  
	  /* Sort entries to avoid deadlock */
	  sort_descriptor_entirely (stmd);
	  
	  /* Check no updates */
	  check_no_writes (stmd);
	  
	  /* Acquire ownership */
	  if (acquire_ownership (env, st, stmd, TRUE)) {
	    stm_set_status (stmd, STM_ACTIVE, STM_WAITING_1);
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }
	  
	  /* Wait if we need to */
	  if (stmd -> status == STM_WAITING_1)
	    {
	      stm_sleep_lightly (st, stmd);
	      
	      if (stmd -> status == STM_WAITING_1)
		{
		  STMMutexLock (&(st -> mutex));
		  stm_set_status (stmd, STM_WAITING_1, STM_WAITING_2);
		  if (stmd -> status == STM_WAITING_2)
		    {
		      TEMP_EXIT_TIMING (TIMING_WAIT);
		      {
			START_TIMING (TIMING_ASLEEP);
			stm_sleep_heavily (st, stmd);
			END_TIMING (st, TIMING_ASLEEP);
		      }
		      TEMP_ENTER_TIMING (TIMING_WAIT);
		    }
		  STMMutexUnlock (&(st -> mutex));
		}
	    }
	  
	  /* Release ownership and continue */
	  for (i = 0; i < stmd -> length ; i ++) 
	    {
	      STMEntry_t *entry = &(stmd -> entries[i]);
	      STMDescriptor_t *waiting_with = st -> waiting_with[i];
	      if (waiting_with != NULL) {
		MMReleaseObj (st, i + 2, waiting_with);
		st -> waiting_with[i] = NULL;
	      }
	    }
	  release_ownership (stmd, TRUE);
	}
    }

  END_TIMING (st, TIMING_WAIT);

  if (st -> current_transaction != NULL)
    {
       MMDeallocateObj (st, stmd);
    }

  st -> current_transaction = NULL;
}


void STMWaitNB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  int              i;
  int              n;
  bool_t           waiting;

  MB();

  st = GET_ST (env);
  START_TIMING (TIMING_WAIT);
  stmd = st -> current_transaction;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      /* Nothing: should retry immediately */
    }
  else
    {
      if (stmd -> status == STM_ACTIVE)
	{
	  void *og;
	  og = MMGuardObj (st, 1, stmd);
	  stm_assert (og == NULL);
	  stm_assert (stmd -> length > 0);
	  n = stmd -> length;

	  for (i = 0; i < n; i ++) {
	    st -> does_own[i] = FALSE;
	  }
      
	  /* Sort entries to avoid deadlock */
 	  sort_descriptor_entirely (stmd);
	  
	  /* Check no updates */
	  check_no_writes (stmd);

	  /* Acquire ownership */
	  if (acquire_ownership_nb (env, st, stmd, TRUE)) {
	    stm_set_status (stmd, STM_ACTIVE, STM_WAITING_1);
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }
	  
	  check_no_writes (stmd);

	  /* Wait if we need to */
	  if (stmd -> status == STM_WAITING_1)
	    {
	      stm_sleep_lightly (st, stmd);
	      
	      if (stmd -> status == STM_WAITING_1)
		{
		  STMMutexLock (&(st -> mutex));
		  stm_set_status (stmd, STM_WAITING_1, STM_WAITING_2);
		  if (stmd -> status == STM_WAITING_2)
		    {
		      TEMP_EXIT_TIMING (TIMING_WAIT);
		      stm_sleep_heavily (st, stmd);
		      TEMP_ENTER_TIMING (TIMING_WAIT);
		    }
		  STMMutexUnlock (&(st -> mutex));
		}
	    }

	  stm_assert (stmd -> status == STM_ABORTED);

	  /* Release ownership and continue */
	  for (i = 0; i < stmd -> length ; i ++) 
	    {
	      STMEntry_t *entry = &(stmd -> entries[i]);
	      STMDescriptor_t *waiting_with = st -> waiting_with[i];
	      if (waiting_with != NULL) {
		MMReleaseObj (st, i + 2, waiting_with);
		st -> waiting_with[i] = NULL;
	      }
	    }

	  release_ownership_nb (st, stmd);

 stm_assert (stmd -> status == STM_ABORTED);
	  if (can_deallocate (stmd)) {
	    MMReleaseObj (st, 1, stmd);
	    stm_assert (stmd -> status == STM_DEALLOCATED_ABORTED);
	    MMDeallocateObj (st, stmd);
	  } else {
	    MMReleaseObj (st, 1, stmd);
	  }
	}
      else
	{
	  MMDeallocateObj (st, stmd);
	}
    }

  END_TIMING (st, TIMING_WAIT);

  st -> current_transaction = NULL;
}

/*......................................................................*/

int STMCommitTransactionB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  bool_t           succeeded;

  MB();

  st = GET_ST (env);
  stmd = st -> current_transaction;
  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      return FALSE;
    }
  else
    {
      if (OPT_DEBUG_MSG) fprintf (stderr, "%p trying to commit", stmd);
      
      /*
       * Check whether we have already aborted e.g. during a transactional
       * read or transactional write operaion.
       */
      
      succeeded = FALSE;
      START_TIMING (TIMING_COMMIT);
      if (stmd -> status == STM_ACTIVE)
	{
	  int i, n;
	  n = stmd -> length;
	  

#ifdef DO_STM_DEFER_OTHER
	  /* Sort entries all entries to avoid deadlock */
#ifdef DO_SHARABLE_READS
	  sort_descriptor_order_writes (stmd); MB();
#else
	  /* Sort entries with writes internally ordered but at the start */
	  sort_descriptor_entirely (stmd); MB();
#endif
#else
#ifdef DO_SHARABLE_READS
	  /* Sort entries with writes internally unordered but at the start */
	  sort_descriptor_segregate_writes (stmd); MB();
#endif
#endif
	  
	  /* take owmership of each of the locations involved */
	  if (acquire_ownership (env, st, stmd, FALSE)) {
	    stm_set_status (stmd, STM_ACTIVE,
			    OPT_SHARABLE_READS ? STM_READ_PHASE : STM_COMMITTED);
#ifdef DO_SHARABLE_READS
	    /* 
	     * If OPT_SHARABLE_READS then check that read-only locations 
	     * hold the values that we want.
	     */
	    
	    if (check_reads (env, st, stmd)) {
	      stm_set_status (stmd, STM_READ_PHASE, STM_COMMITTED);
	    } else {
	      stm_set_status (stmd, STM_READ_PHASE, STM_ABORTED);
	    }
#endif
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }

  
	  /* make updates if necessary */
	  succeeded = ((stmd -> status) == STM_COMMITTED);
	  if (succeeded) 
	    {
	      for (i = 0; i < n ; i++)
		{
		  STMEntry_t        *entry;
		  entry = &(stmd -> entries[i]);
		  stm_assert (OREC_OF (entry -> addr) -> u.owner == stmd);
		  *(entry -> addr) = (void *) (entry -> new_val);
		  GC_WRITE_BARRIER (entry -> addr, entry -> new_val);
		  stm_assert (OREC_OF (entry -> addr) -> u.owner == stmd);
		}
	    }
	  
	  /* Release ownership and continue */
	  release_ownership (stmd, FALSE);
	}
      
      st -> current_transaction = NULL;
      MMDeallocateObj (st, stmd);
      END_TIMING (st, TIMING_COMMIT);
    }

  return succeeded;
}

int STMCommitTransactionNB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  bool_t           succeeded;
  int status;
  void *og;

  MB();

  st = GET_ST (env);
  stmd = st -> current_transaction;
  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      return FALSE;
    }
  else
    {
      int i, n;

      /*
       * Check whether we have already aborted e.g. during a transactional
       * read or transactional write operaion.
       */

      og = MMGuardObj (st, 1, stmd);
      stm_assert (og == NULL);
      stm_assert (stmd -> length > 0);
      succeeded = FALSE;
      START_TIMING (TIMING_COMMIT);
      
      n = stmd -> length;
      for (i = 0; i < n; i ++) {
	st -> does_own[i] = FALSE;
      }
      
      if (stmd -> status == STM_ACTIVE)
	{
	  /* take owmership of each of the locations involved */
	  if (acquire_ownership_nb (env, st, stmd, FALSE)) {
	    stm_set_status (stmd, STM_ACTIVE, 
			    OPT_SHARABLE_READS ? STM_READ_PHASE : STM_COMMITTED);
#ifdef DO_SHARABLE_READS
	    /* 
	     * If OPT_SHARABLE_READS then check that read-only locations 
	     * hold the values that we want.
	     */
	    
	    if (stmd -> status == STM_READ_PHASE) {
	      if (check_reads (env, st, stmd)) {
		stm_set_status (stmd, STM_READ_PHASE, STM_COMMITTED);
	      } else {
		stm_set_status (stmd, STM_READ_PHASE, STM_ABORTED);
	      }
	    }
#endif
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }

	  if (stmd -> status == STM_COMMITTED)
	    {
	      if (do_debug) check_consistency (stmd);
	    }

	  /* Release ownership and continue */
	  release_ownership_nb (st, stmd);
	}
      
      st -> current_transaction = NULL;
      END_TIMING (st, TIMING_COMMIT);

      st-> last_transaction = stmd;

      status = stmd -> status;
      stm_assert (status == STM_COMMITTED || 
		  status == STM_ABORTED ||
		  status == STM_DEALLOCATED_COMMITTED ||
		  status == STM_DEALLOCATED_ABORTED);

      if (status == STM_COMMITTED || status == STM_DEALLOCATED_COMMITTED) {
	succeeded = TRUE;
      }

      if (can_deallocate (stmd)) {
	MMReleaseObj (st, 1, stmd);
	stm_assert (stmd -> status == STM_DEALLOCATED_COMMITTED ||
		    stmd -> status == STM_DEALLOCATED_ABORTED);
	MMDeallocateObj (st, stmd);
      } else {
	MMReleaseObj (st, 1, stmd);
      }
      
      return succeeded;
    }
}


int STMCommitTransaction (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;

  int result;
  int i;

  st = GET_ST (env);
  stmd = st -> current_transaction;
  st -> last_transaction = NULL;

  if (stmd == NULL) { return FALSE; }

#ifdef DO_LOCATION_COUNT
  st -> trans_count ++;
  if (stmd -> length > st -> max_loc_count) {
    st -> max_loc_count = stmd -> length;
  }
  st -> loc_count += stmd -> length;
#endif

#ifdef DO_SPECIAL_CASE_1
  if (st -> started_one_thread)
    {
      one_committing_stmd = stmd;
      MB();
      if (one_thread) {
	int n = stmd -> length;
	int i;
	START_TIMING (TIMING_COMMIT);
#ifdef DO_CACHE_OREC_STATUS
	for (i = 0; i < n; i ++) {
	  *(word_t *)(stmd -> entries[i].addr) = stmd -> entries[i].new_val;
	  st->orec_cache[OREC_HASH(stmd->entries[i].addr)] = OREC_NOT_SEEN;
	}
#endif
	one_committing_stmd = NULL;
	END_TIMING (st, TIMING_COMMIT);
	return TRUE;
      } else {
	one_committing_stmd = NULL;
	st -> started_one_thread = FALSE;
	st -> current_transaction = NULL;
	discard_ownership_cache (st, stmd);
	MMDeallocateObj (st, stmd);
	return FALSE;
      }
    }
#endif

#ifdef DO_FAST_READONLY
  if (st -> current_is_read_only) {
    COUNT (st, COUNT_FAST_COMMIT);
    START_TIMING (TIMING_COMMIT);
    result = check_reads (env, st, stmd);
    discard_ownership_cache (st, stmd);
    stmd -> length = 0;
    END_TIMING (st, TIMING_COMMIT);
    goto done_commit;
  }
#endif

#ifdef DO_STM_ABORT_OTHER
  discard_ownership_cache (st, stmd);
  result = STMCommitTransactionNB (env);
#else
  discard_ownership_cache (st, stmd);
  result = STMCommitTransactionB (env);
#endif

 done_commit:
  if (result) {
    COUNT(GET_ST(env), COUNT_COMMIT_SUCCEEDS); 
    if (stmd != st ->last_transaction && st -> last_transaction != NULL) {
      stm_panic (("last %p not %p\n", stmd, st->last_transaction));
    }
  } else {
    COUNT(GET_ST(env), COUNT_COMMIT_FAILS); 
  }

  return result;
}

void STMWait (STM_ENV *env)
{
  STMState_t *st;
  STMDescriptor_t *stmd;

  st = GET_ST (env);
  stmd = st -> current_transaction;

#ifdef DO_LOCATION_COUNT
  if (stmd != NULL) {
    st -> trans_count ++;
    if (stmd -> length > st -> max_loc_count) {
      st -> max_loc_count = stmd -> length;
    }
    st -> loc_count += stmd -> length;
  }
#endif

  if (stmd != NULL) { discard_ownership_cache (st, stmd); }

#ifdef DO_SPECIAL_CASE_1
  if (st -> started_one_thread) {
    stm_assert (one_committing_stmd == NULL); // Could only be us
    st -> started_one_thread = FALSE;
    one_thread = FALSE;
    STMAbortTransaction (env);
    return;
  }
#endif

#ifdef DO_STM_ABORT_OTHER
  STMWaitNB (env);
#else
  STMWaitB (env);
#endif
  COUNT(GET_ST(env), COUNT_WAITS);
}

void STMAbortTransaction (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;

  st = GET_ST (env);
  stmd = st -> current_transaction;
  COUNT(GET_ST(env), COUNT_ABORTS);

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd != NULL) 
    {

#ifdef DO_LOCATION_COUNT
      if (stmd != NULL) {
	st -> trans_count ++;
	if (stmd -> length > st -> max_loc_count) {
	  st -> max_loc_count = stmd -> length;
	}
	st -> loc_count += stmd -> length;
      }
#endif
      
      /*
       * To abort a transaction we just update its status field:
       * descriptors are only made public during a commit operation.
       */
      
      START_TIMING (TIMING_ABORT);
      discard_ownership_cache (st, stmd);
      stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
      stm_assert (stmd -> status == STM_ABORTED);
      st -> current_transaction = NULL;
      stm_assert (stmd -> status == STM_ABORTED);
      MMDeallocateObj (st, stmd);
      END_TIMING (st, TIMING_ABORT);
    }
}

/*......................................................................*/

static word_t STMReadValue1 (STMState_t *st, 
			     addr_t addr, 
			     STMEntry_t **where)
{
  union STMVersionInfo_t vi_before, vi_after;
  word_t           result;
  OwnershipRecord_t *orec;
  int              version;
  int              wanted;
  int              i;
  int              n;
  STMDescriptor_t *stmd;

  NO_INLINE();

  START_TIMING (TIMING_READ_SLOW);

  stmd = st -> current_transaction;
  wanted = -1;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      /* 
       * We don't want to polute the space of the return values, so
       * give a plausible looking result and expose the abort when
       * the transaction tries to commit.
       */

      result = * ((word_t *) addr);
    }
 else
   {
#ifdef DO_CACHE_OREC_STATUS
     STMEntry_t *cached = st -> orec_cache[OREC_HASH(addr)];
     
     //     stm_assert (cached != addr); /* Handled by caller */
     
     if (cached != OREC_NOT_SEEN)
       {
	 if (cached == OREC_IS_ALIASED)
	   {
	     /* May be in our descriptor */
	     int n = stmd -> length;
	     for (i = 0; i < n; i ++)
	       {
		 if (OREC_OF(stmd -> entries[i].addr) == OREC_OF(addr))
		   {
		     wanted = stmd -> entries[i].old_version;
		     if (stmd -> entries [i].addr == addr)
		       {
			 result = stmd -> entries[i].new_val;
			 if (where != NULL) *where = &(stmd->entries[i]);
			 goto done_read;
		       }
		   }
	       }
	   }
	 else 
	   {
	     wanted = cached -> old_version;
	   }
       }
#else
     /* May be in our descriptor */
     int n = stmd -> length;
     for (i = 0; i < n; i ++)
       {
	 if (OREC_OF(stmd -> entries [i].addr) == OREC_OF(addr))
	   {
	     wanted = stmd -> entries[i].old_version;
	     if (stmd -> entries[i].addr == addr)
	       {
		 result = stmd -> entries[i].new_val;
		 if (where != NULL) *where = &(stmd->entries[i]);
		 goto done_read;
	       }
	   }
       }
#endif     
     
     /* Not yet read it: fetch it from memory */
     orec = OREC_OF(addr);
     do
       {

       retry_read_location:
	 vi_before = orec -> u;
	 MB();
	 if (VI_IS_VERSION (vi_before)) {
	   /* Ordinary read of version vi_before */
	   result = (word_t) *addr;
	   version = vi_before.version;
	 } else {
	   /* Read from an owned location */
	   STMDescriptor_t *owner = vi_before.owner;
	   int owner_status;
	   void *og;
	   bool_t committed;
	   int ost;

	   og = MMGuardObj (st, 0, owner);
	   stm_assert (og == NULL);
	   if (orec -> u.owner != owner) {
	     MMReleaseObj (st, 0, owner);
	     goto retry_read_location;
	   }

	   owner_status = owner -> status;
#ifdef DO_SHARABLE_READS
	   if (owner_status == STM_READ_PHASE) 
	     {
#if defined(DO_STM_DEFER_OTHER) || defined (DO_STM_ABORT_SELF)
	       MMReleaseObj (st, 0, owner);
	       goto retry_read_location;
#else
	       stm_set_status (owner, STM_READ_PHASE, STM_ABORTED);
	       MMReleaseObj (st, 0, owner);
	       goto retry_read_location;
#endif
	     }
#endif

           result = (word_t) *addr; 
	   MB();

	   version = -1;
	   committed = ((owner_status == STM_COMMITTED) ||
			(owner_status == STM_DEALLOCATED_COMMITTED));

	   for (i = 0; i < owner -> length; i ++)
	     {
	       if (OREC_OF (owner -> entries[i].addr) == orec)
		 {
		   STMEntry_t *entry;
		   entry = &(owner -> entries[i]);

		   /* Found correct ownership record */
		   version = committed ? entry -> new_version : entry -> old_version;
		   if (owner -> entries[i].addr == addr)
		     {
		       /* Found exactly matching address */
		       result = committed ? entry -> new_val : entry -> old_val;
		       break;
		     }
		 }
	     }

	   stm_assert (version != -1);
	   stm_assert ((version & 1) == 1);

	   MMReleaseObj (st, 0, owner);
	 }
	 MB();
	 vi_after = orec -> u;
       }
     while (vi_before.version != vi_after.version);

    
     /* Record value and version in descriptor */
     i = (stmd -> length) ++;
     stm_hard_assert ((i < MAX_LOCATIONS));

     stm_assert ((version & 1) == 1);
     if (wanted != -1 && version != wanted) 
       {
	 /* 
	  * We are now doomed to fail: we would have two entries
	  * with conflicting old version numbers.  
	  */

	 stm_set_status (stmd, STM_ACTIVE, STM_DEALLOCATED_ABORTED);
	 stm_assert (stmd -> status == STM_DEALLOCATED_ABORTED);
#ifdef DO_CACHE_OREC_STATUS
	 discard_ownership_cache (st, stmd);
#endif	 
	 MMDeallocateObj (st, stmd);
	 st -> current_transaction = NULL;
       }
     else
       {
	 stmd -> entries[i].addr = addr;
	 stmd -> entries[i].old_val = result;
	 stmd -> entries[i].new_val = result;
	 stmd -> entries[i].old_version = version;
	 stmd -> entries[i].new_version = version;
	 if (where != NULL) *where = &(stmd -> entries[i]);

#ifdef DO_CACHE_OREC_STATUS
	 if (cached == NULL) {
	   /* Cache was not in use: fill it */
	   st -> orec_cache[OREC_HASH(addr)] = &(stmd -> entries[i]);
	 } else {
	   /* Cache was in use: mark it aliased */
	   st -> orec_cache[OREC_HASH(addr)] = OREC_IS_ALIASED;
	 }
#endif
       }
   }
     
 done_read:
  END_TIMING (st, TIMING_READ_SLOW);
  return result;
}

#ifdef DO_CACHE_OREC_STATUS
static word_t STMReadValue0 (STMState_t *st, 
			     addr_t addr, 
			     STMEntry_t **where,
			     STMEntry_t **ct)
{
  union STMVersionInfo_t vi_before, vi_after;
  word_t           result;
  OwnershipRecord_t *orec;
  int              wanted;
  int              i;
  int              n;
  STMDescriptor_t *stmd;
  STMEntry_t      *entry;

  NO_INLINE();
  START_TIMING (TIMING_READ);

  orec = OREC_OF(addr);
  stmd = st -> current_transaction;
  vi_before = orec -> u;

  if ((VI_IS_VERSION(vi_before)) &&
      (*ct == OREC_NOT_SEEN) &&
      (stmd != NULL))
    {
     /* Not yet read it and its not owned: fetch it from memory */

     RMB();
     result = (word_t) *addr;
     RMB();
     vi_after = orec -> u;

     if (vi_before.version == vi_after.version)
       {
	 i = (stmd -> length) ++;
	 stm_hard_assert ((i < MAX_LOCATIONS));

	 entry = &(stmd -> entries[i]);
	 if (where != NULL) *where = entry;
	
	 entry -> addr = addr;
	 entry -> old_val = result;
	 entry -> new_val = result;
	 entry -> old_version = vi_before.version;
	 entry -> new_version = vi_after.version;

	 //	 ct -> addr = addr;
	 *ct = entry;

	 if (where == NULL) COUNT (st, COUNT_SEMIFAST_READ);
	 goto done_read;
       }
    }

  if (where == NULL) COUNT (st, COUNT_SLOW_READ); 
  result = STMReadValue1 (st, addr, where);

 done_read:
  END_TIMING (st, TIMING_READ);
  return result;
}
#endif

/*......................................................................*/

#ifdef DO_CACHE_OREC_STATUS
word_t STMReadValue (STM_ENV *env, addr_t addr)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  STMEntry_t      *ct;
  word_t           result;
  int              hash;

  st = GET_ST (env);
  hash = OREC_HASH(addr);
  ct = st -> orec_cache[hash];
  if (ct != OREC_NOT_SEEN && ct->addr == addr)
    {
      COUNT (st, COUNT_FAST_READ);
      result = ct -> new_val;
    }
  else
    {
#ifdef DO_SPECIAL_CASE_1
      if (st -> started_one_thread && ct == OREC_NOT_SEEN) 
	result = (word_t) (*addr);
      else 
#endif
	result = STMReadValue0 (st, addr, NULL, &(st -> orec_cache[hash]));
    }

  return result;
}
#else
word_t STMReadValue (STM_ENV *env, addr_t addr)
{
  STMState_t *st;
  word_t      result;

  st = GET_ST (env);
  COUNT (st, COUNT_SLOW_READ);
  result = STMReadValue1 (st, addr, NULL);
  return result;
}
#endif


/*......................................................................*/

void STMWriteValue0 (STMState_t *st, addr_t addr, word_t val
#ifdef DO_CACHE_OREC_STATUS
, STMEntry_t **ct
#endif
)
{
  STMDescriptor_t *stmd;
  STMEntry_t *where;
  int old_version;
  int new_version;

  NO_INLINE();

  /* stmd == NULL  <=>  GC occurred since transaction started */

  stmd = st -> current_transaction;
  if (stmd != NULL) 
    {
      int i;

#ifdef DO_CACHE_OREC_STATUS
      STMReadValue0 (st, addr, &where, ct);
#else
      STMReadValue1 (st, addr, &where);
#endif
      
      START_TIMING (TIMING_WRITE);
      /* Update it in the descriptor */
      old_version = where -> old_version;
      new_version = old_version + 2;
      where -> new_val = val;
      where -> new_version = new_version;
      stm_assert (((where -> new_version) & 1) == 1);
      stm_assert (((where -> old_version) & 1) == 1);

      for (i = 0; i < stmd -> length; i++)
	{
	  if (OREC_OF(stmd -> entries[i].addr) == OREC_OF(addr))
	    {
	      stmd -> entries[i].new_version = new_version;
	    }
	}

      END_TIMING (st, TIMING_WRITE);
    }

  return;
}

#ifdef DO_CACHE_OREC_STATUS
void STMWriteValue (STM_ENV *env, addr_t addr, word_t val)
{
  STMState_t      *st;
  STMEntry_t      *ct;
  STMEntry_t      *where;

  st = GET_ST (env);
  ct = st -> orec_cache[OREC_HASH(addr)];
#ifdef DO_FAST_READONLY
  st -> current_is_read_only = FALSE;
#endif
  if (ct != OREC_NOT_SEEN && ct -> addr == addr)
    {
      COUNT (st, COUNT_FAST_WRITE);
      ct -> new_val = val;
      ct -> new_version = ct -> old_version + 2;
    }
  else
    {
      COUNT (st, COUNT_SLOW_WRITE);
#ifdef DO_SPECIAL_CASE_1
      if (st -> started_one_thread && ct == OREC_NOT_SEEN) {
	STMDescriptor_t *stmd = st -> current_transaction;
	if (stmd != NULL)
	  {
	    int i = (stmd -> length ++);
	    stm_hard_assert (i < MAX_LOCATIONS);
	    stmd -> entries[i].addr = addr;
	    stmd -> entries[i].new_val = val;
	    stmd -> entries[i].old_version = 1;
	    st -> orec_cache[OREC_HASH(addr)] = &(stmd -> entries[i]);
	  }
      } else {
#endif
	STMWriteValue0 (st, addr, val, &(st -> orec_cache[OREC_HASH(addr)]));
#ifdef DO_SPECIAL_CASE_1
      }
#endif
    }

  return;
}
#else
void STMWriteValue (STM_ENV *env, addr_t addr, word_t val)
{
  STMState_t      *st;
  STMEntry_t      *where;

  st = GET_ST (env);
  COUNT (st, COUNT_SLOW_WRITE);
  STMWriteValue0 (st, addr, val);
}
#endif

/***********************************************************************/

/* Simple spin-lock to protect tctr, free_state, first_state*/

static volatile int state_management_lock = 0;

/* Head of a list of free state structures (st -> env == NULL) */
static STMState_t *free_state = NULL;

/* Head of a list of active state structures (st -> env != NULL) */
static STMState_t *first_state = NULL;

static volatile int active_threads = 0;

static void lock_state_management (void) {
  while (CASInt (&state_management_lock, 0, 1) != 0) {
    /* Nothing */
  }
}

static void unlock_state_management (void) {
  stm_assert (state_management_lock == 1);
  state_management_lock = 0;
}

void STMAddThread (STM_ENV *env) {
  STMState_t *st;

  /* Obtain state structure */
  st = NULL;
  lock_state_management ();
  if (free_state != NULL) {
    /* Re-use free state */
    st = free_state;
    free_state = st -> next_state;

    st -> next_state = first_state;
    first_state = st;

  } else {
    /* Allocate new structure and idx value */
    st = mallocBytes (sizeof (STMState_t));
    stm_assert(st != NULL);
    st -> idx = tctr;
    tctr ++;

    /* State for sleeping */
    STMMutexInit (&(st -> mutex));
    STMCondInit (&(st -> cond));
    st -> sleepers = NULL;

    if (st -> idx >= MAX_THREADS) {
      stm_panic (("Cannot handle more than %d threads", MAX_THREADS));
    }

    /* Initialize structure */
    st -> next_state = first_state;
    first_state = st;

    /* Inform memory management */
    MMInitState (st);

#ifdef DO_SPECIAL_CASE_1
    st -> started_one_thread = FALSE;
    if (st -> idx == 0) {
      st -> started_one_thread = TRUE;
    } else if (st -> idx == 1) {
      STMDescriptor_t *other;
      one_thread = FALSE;
      while (one_committing_stmd != NULL) {
	MB();
      }
    }
#endif
  }

  active_threads ++;
  unlock_state_management ();

  st -> current_transaction = NULL;

  /* Assign structure to env */
  SET_ST(env, st);
  st -> env = env;
  stm_assert (GET_ST (env) == st);

#ifdef DO_TIMINGS
  {
    int i;
    for (i = 0; i < NUM_TIMINGS; i ++)
      {
	st -> timings[i] = 0;
      }
    st -> init_time = RDTICK ();
  }
#endif

#ifdef DO_COUNTS
  {
    int i;
    for (i = 0; i < NUM_COUNTS; i ++)
      {
	st -> counts[i] = 0;
      }
  }
#endif

#ifdef DO_LOCATION_COUNT
  st -> trans_count = 0;
  st -> loc_count = 0;
  st -> max_loc_count = 0;
#endif

#ifdef DO_ALLOCATION_COUNT
  st -> allocation_count = 0;
#endif

  st -> object_cache_1_size = 0;
  st -> object_cache_2_size = 0;

#ifdef DO_CACHE_OREC_STATUS
  {
    int i;
    for (i = 0; i < STM_OWNERSHIP_RECORDS; i ++) {
      st -> orec_cache[i] = OREC_NOT_SEEN;
    }
  }
#endif

  memset (st -> waiting_with, 0, sizeof(st -> waiting_with));
}

/*......................................................................*/

static void dump_status (STMState_t *st)
{

#ifdef DO_TIMINGS
  {
    int i;
    tick_t tot = RDTICK() - (st -> init_time);
    
    fprintf(stderr, "%p ", st);
    for (i = 0; i < NUM_TIMINGS; i ++)
      {
	fprintf (stderr, "%2.2f ", ((float) (st -> timings[i]) / tot) * 100);
      }
  }
#endif

#ifdef DO_COUNTS
  {
    int i;
    fprintf(stderr,"(");
    for (i = 0;i < NUM_COUNTS; i ++)
      {
	if (i != 0) fprintf (stderr, " ");
	fprintf (stderr, "%d", st -> counts[i]);
      }
    fprintf(stderr,") ");
  }
#endif

#ifdef DO_ALLOCATION_COUNT
  {
    float n = st -> allocation_count;
    float n2 = (st -> object_cache_1_size) + (st -> object_cache_2_size);
    char *u = "bytes";
    char *u2 = "bytes";
    if (n > 1024) {
      n /= 1024;
      u = "Kbytes";
    }
    if (n > 1024) {
      n /= 1024;
      u = "Mbytes";
    }
    if (n > 1024) {
      n /= 1024;
      u = "Gbytes";
    }
    if (n2 > 1024) {
      n2 /= 1024;
      u2 = "Kbytes";
    }
    if (n2 > 1024) {
      n2 /= 1024;
      u2 = "Mbytes";
    }
    if (n2 > 1024) {
      n2 /= 1024;
      u2 = "Gbytes";
    }

    fprintf (stderr, "(Alloc %2.2f %s,  Keep %2.2f %s)", n, u, n2, u2);
  }
#endif

#ifdef DO_LOCATION_COUNT
  {
    fprintf (stderr, " Max %d,  Mean %d",
	     st -> max_loc_count,
	     (st -> trans_count == 0) ? 0 : ((st -> loc_count) / (st -> trans_count)));
  }
#endif

#if (defined(DO_ALLOCATION_COUNT) || defined (DO_TIMINGS))
  fprintf (stderr, "\n");
#endif
}

void snapshot (STM_ENV *env)
{
  fprintf (stderr, "\n");
  dump_status (GET_ST(env));
}

void STMRemoveThread (STM_ENV *env) {
  STMState_t  *st;
  STMState_t **tmp;

  st = GET_ST(env);
  stm_assert (st != NULL);
  lock_state_management ();
  active_threads --;

  dump_status(st);

#ifdef DO_DUMP_ORECS
  {
    if (active_threads == 0)
      {
	int i;
	fprintf (stderr, "Versions: ");
	for (i = 0; i < STM_OWNERSHIP_RECORDS; i ++)
	  {
	    fprintf (stderr, "%d ", orecs[i].u.version);
	  }
	fprintf (stderr, "\n");
      }
  }
#endif

  /* Remove from list of active structures */
  tmp = &first_state;
  while (*tmp != st) {
    tmp = &((*tmp) -> next_state);
    stm_assert (tmp != NULL);
  }
  *tmp = st -> next_state;

  /* Add to list of free structures */
  st -> next_state = free_state;
  free_state = st;
  unlock_state_management();

  /* Remove ownership from env */
  SET_ST (env, NULL);
  st -> env = NULL;
}

/*......................................................................*/

/* This is called before the GC attempts to suspend mutators */
void STMPreGCHook (void)
{
  if (OPT_NEED_PRE_GC_HOOK) {
    STMState_t *st;
    
    st = first_state;
    while (st != NULL) {
      STMDescriptor_t *stmd;
      stmd = st -> current_transaction;

      if (stmd != NULL)
	{
	  st -> current_transaction = NULL;
	}

      st = st -> next_state;
    }
  }
}

void STMPostGCHook (void)
{
  if (OPT_NEED_POST_GC_HOOK) {
    STMState_t *st;

    need_gc = FALSE;

    /* 
     * Iterate over active states 
     */

    st = first_state;
    while (st != NULL) {
      MMDoneGC (st);
      st = st -> next_state;
    }

    /* 
     * Iterate over free states -- e.g. PTB may still have values
     * in a handoff entry.
     */

    st = free_state;
    while (st != NULL) {
      MMDoneGC (st);
      st = st -> next_state;
    }

    /* 
     * Discard cached allocations
     */
#ifdef DO_BALANCE_OBJECT_CACHES
    shared_oc_1 -> shared_oc = NULL;
    shared_oc_1 -> size = 0;
    shared_oc_2 -> shared_oc = NULL;
    shared_oc_2 -> size = 0;
#endif
  }
}

/*......................................................................*/

void STMInit (void) 
{
  int i;

  /* 
   * Run-time sanity checks about properties we assume about the
   * layout of data structures.
   */

  stm_assert (sizeof (void *) == sizeof (uintptr_t));

#ifdef DO_SECONDARY_PTB
  /*
   * 2. The PTB handoff array should be 8-byte aligned (assuming,
   *    in turn, that that's the width a CASPtrInt acts on and that
   *    the hardware requires the alignment).
   */
  stm_assert ((((uintptr_t) &(handoff[0])) & 7) == 0);
#endif

  shared_oc_1 = mallocBytes (sizeof (OCBalanceInfo_t));
  shared_oc_2 = mallocBytes (sizeof (OCBalanceInfo_t));
  shared_oc_1 -> shared_oc = NULL;
  shared_oc_2 -> shared_oc = NULL;
  shared_oc_1 -> size = 0;
  shared_oc_2 -> size = 0;

#ifdef DO_BALANCE_OBJECT_CACHES
  /* 
   * 3. The object cache info should be 8-byte aligned as above
   */
  stm_assert ((((uintptr_t) (shared_oc_1)) & 7) == 0);
  stm_assert ((((uintptr_t) (shared_oc_2)) & 7) == 0);
#endif

#ifdef DO_CACHE_OREC_STATUS
  invalid_addr_entry.addr = (addr_t) &(invalid_addr_entry.addr);
#endif

  /*
   * Initialization
   */

  MMInitScheme ();
  orecs = mallocBytes (STM_OWNERSHIP_RECORDS * sizeof (OwnershipRecord_t));
  stm_assert ((((int)orecs) & 7) == 0);
  for (i = 0; i < STM_OWNERSHIP_RECORDS; i ++)
    {
      orecs[i].u.version = 1;
#ifdef DO_STM_ABORT_OTHER
      orecs[i].updaters = 0;
#endif
    }
}

void STMDone (void)
{
  /* To do */
}

/*......................................................................*/

long long STMTestRDTick(void)
{
  return RDTICK();
}

int STMTestCASInt (int volatile *a, int o, int n)
{
  return CASInt (a, o, n);
}

