/*
 *	msnl_port.c
 *	------------
 *
 * $Id: msnl_port.c,v 1.3 1993/05/25 15:18:41 rjb17 Exp $
 *
 * Copyright (c) 1993 Cambridge University Computer Laboratory.
 *
 */

#include "../h/types.h"
#include "../h/socket.h"
#include "../h/errno.h"

#include "msnl.h"
#include "msnl_port.h"

/* This file deals with allocation of msnl saps which unfortunately
 * must be done in the kernel on unix.
 */

unsigned int msnl_host;

#define MSNL_MAXPORTS	(256)
#define PORT_TRIES	(8)

struct {
    unsigned int	mp_value;
    unsigned int	mp_refs;
} msnl_ports[MSNL_MAXPORTS];

static unsigned int portcounter;
static int msnlactports;		/* number of active ports */

/*
 * ------------------------------------------------------------------------
 */

#ifdef __STDC__

static void advancePortCounter	();
static int  port_find		(unsigned int port);
static int  slot_find		();
static int  port_alloc		(unsigned int port);

#endif

/*
 * ------------------------------------------------------------------------
 */

static void advancePortCounter()
{
    unsigned char *ptr = (unsigned char *)&portcounter;
    
    if (++(ptr[3]) || ++(ptr[2]) || ++(ptr[1]) || ++(ptr[0]))
	return;
    
    ptr[1] = 1;
}

static int port_find(port)
    unsigned int		port;
{
    int			i;
    
    for (i=0; i<msnlactports; i++)
	if (msnl_ports[i].mp_value == port)
	    return i;
    return -1;
}

/*
 * slot_find: find a free slot in the table. This may mean updating the
 *            maximum used mark.
 */

static int slot_find()
{
    int			i;
    
    for (i=0; i<msnlactports; i++)
	if (msnl_ports[i].mp_value == 0)
	    return i;
    
    if (msnlactports + 1 >= MSNL_MAXPORTS)
	return -1;
    
    return msnlactports++;
}


static int port_alloc(port)
    unsigned int		port;
{
    int			i;
    
    if (port_find(port) != -1)
	return EADDRINUSE;
    
    if ((i = slot_find()) == -1)
	return ENOBUFS;

    msnl_ports[i].mp_value = port;
    msnl_ports[i].mp_refs = 1;

    return 0;
}

int msnl_sap(addr)
    struct sockaddr		*addr;
{
    struct sockaddr_msnl	*msaddr = (struct sockaddr_msnl *)addr;
    int				i;
    int				error;
    
    if (!msnl_host)
	return EPFNOSUPPORT;

    if (msaddr->sa_family != AF_MSNL)
	return EAFNOSUPPORT;

    if (msaddr->msnl_adr.m_host && msaddr->msnl_adr.m_host != msnl_host)
	return EADDRNOTAVAIL;

    msaddr->msnl_adr.m_host = msnl_host;

    if (msaddr->msnl_adr.m_port)
	return port_alloc(msaddr->msnl_adr.m_port);
    
    for (i=0; i<PORT_TRIES; i++)
    {
	advancePortCounter();
	if (!(error = port_alloc(portcounter)))
	{
	    msaddr->msnl_adr.m_port = portcounter;
	    return 0;
	}
    }
    return error;
}

/*
 * msnl_sap_again: called on sonewcon ish to increment the usage count
 */

void msnl_sap_again(addr)
    struct sockaddr		*addr;
{
    struct sockaddr_msnl	*msaddr = (struct sockaddr_msnl *)addr;
    int				i;

    if ((msaddr->sa_family != AF_MSNL) || !(msaddr->msnl_adr.m_port))
	panic("msnl_sap_again");
    
    if ((i = port_find(msaddr->msnl_adr.m_port)) == -1)
	panic("msnl_sap_again2");
    
    msnl_ports[i].mp_refs++;
}

void msnl_sap_free(addr)
    struct sockaddr		*addr;
{
    struct sockaddr_msnl	*msaddr = (struct sockaddr_msnl *)addr;
    int				i;

    if ((msaddr->sa_family != AF_MSNL) || !(msaddr->msnl_adr.m_port))
	panic("msnl_sap_free");
    
    if ((i = port_find(msaddr->msnl_adr.m_port)) == -1)
	panic("msnl_sap_free2");
    
    if (--msnl_ports[i].mp_refs)
	return;
    
    /* Got here then count now zero */
    msnl_ports[i].mp_value = 0;
}

/*
 * initialisation
 */

void msnl_set_host(addr)
    struct sockaddr		*addr;
{
    struct sockaddr_msnl	*msaddr = (struct sockaddr_msnl *)addr;
    
    msnl_host = msaddr->msnl_adr.m_host;
}

void msnl_port_init()
{
    bzero(msnl_ports, sizeof(msnl_ports));
    msnl_host = 0;

    ((unsigned char *)&portcounter)[0] = 0x00;
    ((unsigned char *)&portcounter)[1] = 0x01;
    ((unsigned char *)&portcounter)[2] = 0x00;
    ((unsigned char *)&portcounter)[3] = 0x00;

    msnlactports = 0;
}

