/*
 *	msnl_mbuf.c
 *	-----------
 *
 * $Id: msnl_mbuf.c,v 1.1 1993/10/04 10:41:25 rjb17 Exp $
 *
 * Copyright (c) 1993 Richard Black. All Rights Reserved.
 *
 * Permission is granted to University of Cambridge Computer Laboratory
 * to use in ultrix kernels.
 */

#include "../h/param.h"
#include "../h/systm.h"			/* min */
#include "../h/mbuf.h"

/*
 * ------------------------------------------------------------------------
 */

/*
 * Unix is such a dreadful shambles. In theory and practice the Mbuf chain
 * we get here could be in any old shape. This includes arbitary numbers
 * of mbufs with arbitary alignment, arbitary about of data, and normal
 * or cluster/loaned.
 *
 * We must output a chain where every mbuf has a multiple of 48 bytes in it
 * and is word aligned. Of course we want to copy as little of the data
 * as possible especialy for aligned clusters (disk blocks)
 *
 * Table is 4x2x2. There are actually some more states but they are generally
 * obvious eg initial conditions and end of buffer.
 *
 * Current:		Full		Ok	Needs, A	Needs
 * Next		-------------------------------------------------------------
 * - Cluster A	| Opt Create,	| Opt Create,	| Copy max up	| Copy max up
 *		| copy excess	| copy excess	|		|
 *		-------------------------------------------------------------
 * - Mbuf A	| Opt Create,	| Opt create,	| Copy max up	| Copy max up
 *		| copy excess	| copy excess	|		|
 *		-------------------------------------------------------------
 * - Cluster	| Create, copy	| Create, copy	| Copy max up	| Copy max up
 *		| max up	| max up	|		|
 *		-------------------------------------------------------------
 * - Mbuf	| Create, copy	| Create, copy	| Copy max up	| Copy max up
 *		| max up	| max up	|		|
 *		-------------------------------------------------------------
 *
 * "A" means Aligned. One could argue with a few of the table values perhaps
 * but mostly the ones that are for the silly cases. The above table
 * has a reasonably sensible decomposition. "Full" is max cells in an
 * ordinary mbuf, or some cells in a cluster. "OK" is some cells in an
 * ordinary mbuf space for at least one cell left.
 *
 * Note the mbuf list is always kept in such a way that if an allocate
 * fails we can always blow and return.
 */

#define STATE_FULL	1
#define STATE_NEEDS	2

int sane_trace = 0;

struct mbuf *msnl_saneify_mbufs(struct mbuf *s, int *plen)
{
    struct mbuf		*p = 0, *all = s;
    struct mbuf		*ma;
    int			state = STATE_FULL;
    char		*ptr;		/* only valid state needs */
    int			left;		/* only valid state needs */
    int			len;
    int			total = 0;
    int			cells;

    if (sane_trace)
    {
	printf("msnl_saneify_mbufs1: ");
	for (ma = all; ma; ma= ma->m_next)
	{
	    printf(" 0x%08X/0x%08X/%4d ", ma, ma->m_off, ma->m_len);
	}
	printf("\n");
    }

    while (s)
    {
	if (state == STATE_NEEDS)
	{
	    len = min(left, s->m_len);
	    bcopy(mtod(s, char *), ptr, len);

	    /* update target */
	    p->m_len	+= len;
	    left	-= len;
	    ptr		+= len;

	    if ((!(left)) || (len && !(left % 48)))
		state = STATE_FULL;

	    /* update source */
	    s->m_len -= len;
	    if (!(s->m_len))
	    {
		ma = s->m_next;
		m_free(s);
		s = ma;
		p->m_next = s;
	    }
	    else
		s->m_off += len;
	}
	else
	{
	    /* FULL: P must be perfect */
	    if (p)
		total += p->m_len;
	    
	    if (s->m_off & 3)
	    {
	    newoneandcopy:
		if (!(ma = m_get(M_DONTWAIT, MT_DATA)))
		{
		    m_freem(all);
		    return 0;
		}
		ma->m_next = s;
		if (p)
		    p->m_next = ma;
		else
		    all = ma;
		p = ma;
		state = STATE_NEEDS;
		ptr = mtod(p, char *);
		left = (MLEN / 48) * 48;
		ma->m_len = 0;
	    }
	    else
	    {
		cells = s->m_len / 48;
		len = s->m_len % 48;
		
		if (cells)
		{
		    if (len)
		    {
			if (!(ma = m_get(M_DONTWAIT, MT_DATA)))
			{
			    m_freem(all);
			    return 0;
			}
			ma->m_next = s->m_next;
			s->m_next = ma;
			bcopy(mtod(s, char *) + cells * 48, mtod(ma, char *),
			      len);
			s->m_len -= len;
			ma->m_len = len;

			/* move on again */
			state = STATE_NEEDS;
			ptr = mtod(ma, char *) + ma->m_len;
			left = ((MLEN / 48) * 48) - ma->m_len;
			
			p = ma;
			total += s->m_len;
			s = ma->m_next;

			if (p!=ma) panic("msnl_saneify_mbufs\n");
		    }
		    else
		    {
			p = s;
			s = s->m_next;
		    }
		}
		else
		{
		    if (s->m_off != MMINOFF)
			goto newoneandcopy;
		    
		    state = STATE_NEEDS;
		    ptr = mtod(s, char *);
		    left = ((MLEN / 48) * 48) - len;
		    
		    p = s;
		    s = s->m_next;
		}
	    }
	}
    } /* while */

    if (!p)
	p = all;
    if (state == STATE_NEEDS)
	p->m_len += left % 48;

    total += p->m_len;
    
    *plen = total;

    if (sane_trace)
    {
	printf("msnl_saneify_mbufs2: ");
	for (ma = all; ma; ma= ma->m_next)
	{
	    printf(" 0x%08X/0x%08X/%4d ", ma, ma->m_off, ma->m_len);
	}
	printf("\n");
    }

    return all;
}

/*
 * ------------------------------------------------------------------------
 */
