/*
 *	msnl_data.c
 *	-----------
 *
 * $Id: msnl_data.c,v 1.13 1993/06/09 12:10:57 rjb17 Exp $
 *
 * Copyright (c) 1993 Cambridge University Computer Laboratory.
 *
 */

/*
 * This file deals with receiving and transmission of data.
 *
 * Generic Unix devices will call us on the protocol input routine (edl)
 * but our own MSNL only devices (such as yes) call up directly
 * which is more efficient
 */

#include "../h/param.h"
#include "../h/errno.h"
#include "../h/systm.h"			/* min */
#include "../h/mbuf.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../net/net/if.h"
#include "../net/net/netisr.h"
#include "../netinet/in.h"		/* Needed for if_ether.h */
#include "../netinet/if_ether.h"	/* Silly place for it */

#include "msnl_sar.h"
#include "msnl_assoc.h"

/*
 * move these elsewhere!
 */

typedef struct mbuf MBUF;
typedef MBUF *MBUFP;

/*
 * ----------------------------------------------------------------------
 */

/* This should be somewhere sensible! */

#ifdef __STDC__
static void print_ether(unsigned char *e);
#endif

static void print_ether(e)
    unsigned char	*e;
{
    printf("%02X:%02X:%02X:%02X:%02X:%02X",e[0],e[1],e[2],e[3],e[4],e[5]);
}

/*
 * -------------------------------------------------------------------------
 */

struct ifqueue		edl_queue, fdl_queue;

struct mbuf *edl_ifinput(m, ifp, inqp, eh)
    MBUFP			m;
    struct ifnet		*ifp;
    struct ifqueue		**inqp;		/* NOT WANTED */
    struct ether_header		*eh;
{
    MBUFP			m0;
    EDLQUEUE			*q;
    int				s = splimp();
    
    if ((m0 = m_get(M_DONTWAIT, MT_DATA)) == 0)
    {
	IF_DROP(&edl_queue);
	m_freem(m);
	splx(s);
	return 0;
    }
    
    q = mtod(m0, EDLQUEUE *);
    q->data = m;
    q->ifp = ifp;
    bcopy((caddr_t)(eh->ether_shost), (caddr_t)((q->src)), 6);
    
    if (IF_QFULL(&edl_queue))
    {
	IF_DROP(&edl_queue);
	m_free(m0);
	m_freem(m);
    }
    else
    {
	IF_ENQUEUE(&edl_queue, m0);
	schednetisr(NETISR_MSNL);
    }
    splx(s);
    return 0;
}

void fdl_ifinput(m, ifp, associd)
    MBUFP			m;
    struct ifnet		*ifp;
    int				associd;
{
    FDLQUEUE			*q;
    int				s = splimp();
    
    /* Remember the Q gets put before the lead data block */

    q = mtod(m, FDLQUEUE *);
    q--;
    
    q->ifp = ifp;
    q->associd = associd;
    
    if (IF_QFULL(&fdl_queue))
    {
	IF_DROP(&fdl_queue);
	m_freem(m);
    }
    else
    {
	IF_ENQUEUE(&fdl_queue, m);
	schednetisr(NETISR_MSNL);
    }
    splx(s);
    return;
}

/*
 * ----------------------------------------------------------------------
 */

unsigned int edl_bad_reached = 0;

void edl_rx()
{
    MBUFP			m, mq, m1;
    EDLQUEUE			*q;
    int				s;
    int				associd;
    BlockEDL			*block;
    ASSOC			*assoc;
    int				cells;
    EDLSAR			*psar;		/* Packet FAS */
    EDLSAR			sar;
    int				start;
    int				len;
    
    s = splimp();
    IF_DEQUEUE(&edl_queue, mq);
    splx(s);
    
    q = mtod(mq, EDLQUEUE *);
    m = q->data;


    /* At this point in the code it always used to call m_pullup to
     * ensure that the whole thing was readable. However this has
     * the side effect of ensuring that the start of the data is not
     * in a cluster mbuf. We have no objection to leaving it in a cluster
     * if it is already so we only call it if necessary for our purposes.
     * Incidentally this has the side effect of not running into the
     * bug in riscix uipc_mbuf.c.
     */
    if (m->m_len < sizeof(BlockEDL))
	if ((m = m_pullup(m, sizeof(BlockEDL))) == 0)
	{
	    printf("failed to pullup\n");
	    goto bad;
	}
    
    block = mtod(m, BlockEDL *);

    /* Maybe one day we will have a sensible vci on the ethernet. 
     * Maybe one day we will all be rich and famous.
     */

    associd = (block->vci[0] << 8) | block->vci[1];
    if (associd & 0x3f)
    {
	/* I have observed some broken code doing this and I
	 * intend to catch it.
	 */
	printf("edl_rx(msnl): protocol violation from ");
	print_ether(q->src);
	printf("\n");
    bad:
	edl_bad_reached++;
	m_freem(m);
	m_free(mq);
	return;
    }
    associd >>= 6;
    
    /* Now we see what state the association is in and if this packet makes
     * any sense to us. Null ifp means we dont care.
     */

    if (((assoc = assoc_check(associd)) == 0)
	|| (assoc->as_ifp && (assoc->as_ifp != q->ifp)))
    {
	assoc_got_bogus(associd, q->ifp);
	goto bad;
    }

    /* calculate the raw length of the packet */
    len = 0;
    for ( m1=m; m1; m1=m1->m_next )
	len += m1->m_len;
    
    cells = block->count;

    if (cells * 32 != len - sizeof(BlockEDL))
	goto bad;

    psar = &block->sar[0];
    sar = *psar;
    start = sar.bits.startbit;
    sar.bits.startbit = 0;
    
    if (assoc->as_m_start)
    {
	if (sar.all != assoc->as_rxsar.edlsar.all)
	{
	    /* Not happy. Dump stuff got so far */
	    m_freem(assoc->as_m_start);
	    assoc->as_m_start = 0;
	    if (!start)
	    {
		/* The new one is no good either */
		goto bad;
	    }
	}
	/* Drop through to normal checks */
    }
    else
    {
	if (!start)
	{
	    /* A continuation frame when expecting a start */
	    goto bad;
	}
    }
    
    if (!cells || (cells > MAXEDLCELLS))
	goto bad;
    
    while (--cells)
    {
	if (!--sar.bits.sequence)
	    goto bad;
	psar++;
	if (sar.all != psar->all)
	    goto bad;
    }
    
    /* Here then the control information is as expected. 
     * Update the assoc structure as appropriate.
     */

    m->m_len -= sizeof(BlockEDL);
    m->m_off += sizeof(BlockEDL);

    if (sar.bits.sequence != 1)
    {
	/* Note that what we want to store is what is expected for the
	 * first cell of next time. This is one less than the last one
	 * of this time which is what we have now
	 */
	assoc->as_rxsar.edlsar = sar;
	assoc->as_rxsar.edlsar.bits.sequence--;
    }
    else
	assoc->as_rxsar.edlsar.all = 0;
    
    if (start)
    {
	if (sar.bits.sequence == 1)
	{
	    /* The whole shooting match in one go */
	    /* No munging to do */
	}
	else
	{
	    assoc->as_m_start = m;
	middle:
	    for (m1 = m; m1->m_next; m1 = m1->m_next);
	    assoc->as_m_current = m1;
	    m_free(mq);
	    return;
	}
    }
    else
    {
	assoc->as_m_current->m_next = m;
	if (sar.bits.sequence == 1)
	{
	    /* This makes a complete block */
	    m = assoc->as_m_start;
	    assoc->as_m_start = 0;
	}
	else
	{
	    goto middle;
	}
    }
    
    /* Here then want to pass up "m" */
    if (!assoc->as_rx_proc)
    {
	printf("edl_rx(msnl): No rx_proc (serious)!\n");
	goto bad;
    }
    
    (assoc->as_rx_proc)(m, assoc, (caddr_t)q);	/* Only reads q not keep */
    m_free(mq);
}

void fdl_rx()
{
    MBUFP			m;
    FDLQUEUE			*q;
    int				s;
    int				associd;
    ASSOC			*assoc;
    
    s = splimp();
    IF_DEQUEUE(&fdl_queue, m);
    splx(s);
    
    q = mtod(m, FDLQUEUE *);
    q--;
    
    associd = q->associd;

    if (((assoc = assoc_check(associd)) == 0)
	|| (assoc->as_ifp && (assoc->as_ifp != q->ifp)))
    {
	assoc_got_bogus(associd, q->ifp);
    bad:
	m_freem(m);
	return ;
    }

    if (!assoc->as_rx_proc)
    {
	printf("fdl_rx(msnl): No rx_proc (serious)!\n");
	goto bad;
    }
    
    (assoc->as_rx_proc)(m, assoc, (caddr_t)q);	/* Only reads q not keep */
}


/*
 * ----------------------------------------------------------------------
 */

/* This function is running at software interrupt level splnet
 * rather than the hardware level. (splimp). There could be stuff
 * on either edl or fdl queues and it could be valid or bogus
 */

void msnl_netisr()
{
    while (edl_queue.ifq_head)
	edl_rx();
    while (fdl_queue.ifq_head)
	fdl_rx();
    /* there may be additional things needing to be processed here such
     * as interface failure messages etc on some management queue
     */
}

/*
 * ----------------------------------------------------------------------
 */

int edl_tx(assoc, m)
    ASSOC		*assoc;
    MBUFP		m;
{
    int			len;
    MBUFP		m0;
    int			extra;
    EDLSAR		sar;

    len=0;
    for(m0 = m; m0; m0 = m0->m_next)
	len += m0->m_len;
    
    if ((extra = EDLPADNEEDED(len)))
    {
	m0 = m_get(M_DONTWAIT, MT_DATA);
	if (!m0)
	{
	    m_freem(m);
	    return ENOBUFS;
	}
	m0->m_len = extra;
	m_cat(m,m0);
	len += extra;
    }
    
    if (EDLCELLS(len) > MAXEDLSEQ)
    {
	m_freem(m);
	return EMSGSIZE;
    }
    
    assoc->as_txsar.edlsar.bits.assemble++;
    sar.all = assoc->as_txsar.edlsar.all;
    sar.bits.sequence = EDLCELLS(len);
    sar.bits.startbit = 1;
    
    while (sar.bits.sequence)
    {
	int		thisone,i;
	MBUFP		send, header;
	BlockEDL	*block;
	EDLSAR		*psar;
	
	thisone = min(sar.bits.sequence, MAXEDLCELLS);
	
	if (thisone == sar.bits.sequence)
	{
	    send = m;
	    m = 0;
	}
	else
	{
	    send = m_copy(m, 0, thisone * BYTESPEREDLCELL);
	    m_adj(m, thisone * BYTESPEREDLCELL);
	}
	header = m_get(M_DONTWAIT, MT_DATA);
	if (!header || !send)
	{
	    m_freem(send);
	    if (header) m_free(header);
	    m_freem(m);
	    return ENOBUFS;
	}

	header->m_len = sizeof(BlockEDL);
	block = mtod(header, BlockEDL *);
	
	block->vci[0] = assoc->as_txadr.sa_data[6];	/* shost[0] */
	block->vci[1] = assoc->as_txadr.sa_data[7];	/* shost[1] */
	block->count = thisone;
	
	psar = &block->sar[0];
	(psar++)->all = sar.all;
	sar.bits.startbit = 0;
	sar.bits.sequence--;

	for (i=1; i<thisone; i++)
	{
	    (psar++)->all = sar.all;
	    sar.bits.sequence--;
	}
	
	m_cat(header, send);
	
	assoc->as_ifp->if_output(assoc->as_ifp, header, &assoc->as_txadr);
    }

    /* By here the whole mbuf chain has been given away */
    return 0;
}

/* Prepare the assoc for transmission. If this function was ever to
 * "object" then assoc_operation would need changed. This is also called
 * to "prepare" the promiscuous association when doing a meta signalling
 * transmit.
 */

int edl_tx_prepare(assoc, ptr, ifp)
    ASSOC		*assoc;
    struct sockaddr	*ptr;
    struct ifnet	*ifp;
{
    struct ether_header	*eth = (struct ether_header *)assoc->as_txadr.sa_data;
    
    assoc->as_txadr.sa_family = AF_UNSPEC;
    bcopy(ptr->sa_data, eth->ether_dhost, 6);
    bcopy(&ptr->sa_data[6], eth->ether_shost, 2);	/* hide vci here */
    eth->ether_type = ETHERTYPE_MSDL;

    return 0;
}

/*
 * ----------------------------------------------------------------------
 */

void msnl_data_init()
{
    edl_queue.ifq_head = edl_queue.ifq_tail = (MBUFP)0;
    edl_queue.ifq_len = edl_queue.ifq_drops = 0;
    edl_queue.ifq_maxlen = IFQ_MAXLEN;
    
    fdl_queue.ifq_head = fdl_queue.ifq_tail = (MBUFP)0;
    fdl_queue.ifq_len = fdl_queue.ifq_drops = 0;
    fdl_queue.ifq_maxlen = IFQ_MAXLEN;
}

