/*
 *	yes.c
 *	-----
 *
 * $Id: yes.c,v 1.13 1993/11/22 15:54:34 rjb17 Exp $
 *
 * Copyright (c) 1993 Cambridge University Computer Laboratory.
 *
 */

/*
 * Much of this driver - in particular the DMA code and the stability
 * on reception errors is due to the carefull thought and experimentation
 * that IAP put into his Wanda driver. Thanks Ian.
 *
 */

#include "../h/param.h"
#include "../h/errno.h"
#include "../h/systm.h"			/* min */
#include "../h/mbuf.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../h/vmmac.h"			/* svtophy */
#include "../net/net/if.h"
#include "../net/net/netisr.h"

#include "../h/buf.h"
#include "../h/uio.h"
#include "../io/tc/tc.h"
#include "../io/uba/ubavar.h"

#include "../net/netmsnl/msnl_sar.h"
#include "../net/netmsnl/msnl_assoc.h"
#include "../net/netmsnl/msnl_ioctl.h"

#include "yes.h"

#define YESMTU		(255 * 48)

/*
 * ------------------------------------------------------------------------
 */

/* #define YES_TRACE */
/* #define YES_TRACE_META_CELLS */
    
#define TEST_NUM	(2)

#ifdef YES_TRACE
#define TRC(x) x
#else
#define TRC(x)
#endif

#define TIMEPERTICK	(5)

/*
 * ------------------------------------------------------------------------
 */

#define KM_YES		(KM_TEMP | KM_CONTIG)

/*
 * ------------------------------------------------------------------------
 */

/* Board status flags */

#define BROKEN	1
#define WORKING 2

/* Bits in the status register */

#define RXDATAINFIFO	0x2000
#define RXVIOLATION	0x0400

#define RXOVER		0x0100
#define RXUNDER		0x0200
#define	RXHECBAD	0x0800
#define TXOVER		0x1000

#define TXAVAILABLE	0x0080
#define RXFIFO		0x0040
#define EOM		0x0020
#define DMADONE		0x0010
#define TXBLOCKED	0x0008
#define RXIDLE		0x0004
#define RXCONFUSED	0x0002
#define TXIDLE		0x0001

/* Bits in the control register */

#define TXINTEN		0x80
#define RXINTEN		0x40
#define EOMINTEN	0x20
#define DMAINTEN	0x10
#define TXFLOWEN	0x08
#define TXHEC		0x04
#define BIGENDIAN	0x02
#define RXFAULT		0x01

/*
 * ------------------------------------------------------------------------
 */

/* Offset of option ROM text from register set in slot space */

#define YES_OFFSET	(0x200000)
#define ROM_VERS_LEN    (6)

#define TC_PAGE_SIZE 0x800	/* TURBOCHANNEL PAGE SIZE in bytes */

#define MAX_CELLS_IN_FIFO  ( ( 4096/52 ) + 1 )

typedef volatile struct { 
    u_int p0[1];	/* 0x0000 Pad				*/
    u_int *dmatx;	/* 0x0004 Start DMA tx			*/
    u_int *dmarx;	/* 0x0008 Start DMA rx			*/
    u_int p1[1];	/* 0x000C Pad				*/
    u_int length;	/* 0x0010 Length counter		*/
    u_int hreset;	/* 0x0014 Write to reset board		*/
    u_int xreset;	/* 0x0018 Reset board and taxis		*/
    u_int status;	/* 0x001C Status register		*/
    u_int p2[4];	/* 0x0020 Pad				*/
    u_int rxfifo;	/* 0x0030 Receive fifo			*/
    u_int p3[58];	/* 0x0034 Pad				*/
    u_int control;	/* 0x011C Control register		*/
    u_int p4[4];	/* 0x0120 Pad				*/
    u_int txfifo;	/* 0x0130 Transmit fifo			*/
} yes_t;

/*
 * ------------------------------------------------------------------------
 */

typedef struct
{
    u_int	txok;		/* grand total of OK Tx'd blocks */
    u_int	txbad; 		/* total of binned Tx blocks */
    u_int	resets;		/* total number of h_resets */
    u_int	rxover;		/* Rx overruns */
    u_int	rxunder;	/* Rx underruns */
    u_int	rxviol;		/* Rx Violations */
    u_int	hecbad;		/* RX bad HEC */
    u_int	rxbad;		/* Bad VCI otherwise OK */
    u_int	rxbroken;	/* Yes 2.3 Rx Jammed */
    u_int	rxcells;	/* Number of cells without RX bad bits */
    u_int	rxints;		/* number of rx ints */
    u_int	rxupcalls;	/* number of blocks passed up */
} yesstats;

typedef struct
{
    yes_t		*regs;
    unsigned char	notidles;
    unsigned char	dcbflags;
    unsigned char       rev_major;
    unsigned char       rev_minor;
    signed char		vcount;
    unsigned char	pad[3];
    unsigned int	status;
    yesstats		s;
    int			segment_init;
    int			segment;
} YESDCB;

static YESDCB		yesdcbs[NYES]; /* index by slot number */

static struct ifnet	*yesifs[NYES];

/*
 * ------------------------------------------------------------------------
 */

#define MAXFDLBYTES 768
#define BYTEPERMDLCELL 32

#define CELLS_IN_TX_BLOCK	16		/* # cells to send w/o checking
						   fifo avaliability.
						   Could be fifosize/2,
						   but what the hell */

#define TX_MAX_TRY		75		/* Number of times to try
						   retransmitting for
						   before giving up. 
						   For 16 cells, this wants to
						   be > 71us */

/* Divide by 48 look-up table */

u_char divby3table[MAXFDLBYTES];


/*****************************************************************************/
/*    Reset the yes board.						     */
/*****************************************************************************/

/* N.B.: This assumes it is being called at splimp (or whatever it is
 * that the hardware is actually running at.
 */

static void yes_gen_reset(int dev, int severity);

#define yes_hreset(dev) yes_gen_reset(dev, 0)
#define yes_xreset(dev) yes_gen_reset(dev, 1)

static void
yes_gen_reset(int dev, int severity)
{
    yes_t *yes;
    u_int dstat;
    u_int i;
    u_int j;

    yes = yesdcbs[dev].regs;

    /*
     * Store any value in the `reset' field to initialise
     */
    if (! severity) {
	yes->hreset = 1;
    } else {
	
	TRC(printf("yes_%creset: status after reset %04X\n", 
		   severity ? 'x' : 'h', yes->status & 0xFFFF));
	
	/* xreset - full fifo and taxi reset.
	   This should be used very sparingly as it is V. slow,
	   and has never been proved to be necessary... 

	   We hit xreset, wait then read status register.
	   if we get a couple of successful reads, everything 
	   is back up, otherwise try re-locking taxi's 5 times
	   before giving up and admitting that the line is 
	   disconnected.
	   */


	for (i = 0; i < 5; i++)
	{
	    yes->xreset = 1;
	
	    for( j=0;j<2;j++ )
	    {
	        microdelay(500);

		if( ( dstat = yes->status) & RXVIOLATION ) break;
	    }

	    if( dstat & RXVIOLATION == 0 )
		break;
	}

    }

    yesdcbs[dev].s.resets ++;
  
    dstat = yes->status;
    yesdcbs[dev].notidles = 0;

    if( dstat & ( RXVIOLATION | RXOVER | RXUNDER | TXOVER ) )
    {
	/* we have failed to clear some serious condition by this
	   reset. Either the line is done, or Yes board dead beaf */

	yes->control = 0; /* disable yes board */
	yesdcbs[dev].dcbflags = BROKEN;

	printf("yes_%creset: status BROKEN (%04X)\n", 
	       severity ? 'x' : 'h', dstat & 0xFFFF);
    }
    else
    {
	yesdcbs[dev].dcbflags = WORKING;

	/* enable per cell interupts, HEC, NO flow control */

	if (yesdcbs[dev].rev_minor == 3)
	    yes->control = ( RXINTEN );	
	else 
	    yes->control = ( RXINTEN | TXHEC );	

	TRC(printf("yes_%creset: status ENABLE (%04X)\n", 
		   severity ? 'x' : 'h', dstat & 0xFFFF));
    }
}

static int yes_start(int dev)
{   
    /* try to reset the board  */
    yes_xreset(dev);
    /* Is it OK now ? */
    return (yesdcbs[dev].dcbflags == WORKING);
}


static int yes_decode_version(int unit, char *vbaddr)
{
    char			version[ROM_VERS_LEN];
    unsigned int		*p, i;

    p = (unsigned int *)(vbaddr + (TC_VERSION_OFFSET));
    for (i = 0; i < ROM_VERS_LEN; i++) {
        version[i] = p[i];
    }
    if (strncmp(version, "V2.002", ROM_VERS_LEN) == 0) {
        yesdcbs[unit].rev_major = 2;
        yesdcbs[unit].rev_minor = 2;
        printf("yes_decode_version: YES V2.2\n");
        return 1;
    }
    if (strncmp(version, "V2.003", ROM_VERS_LEN) == 0) {
        yesdcbs[unit].rev_major = 2;
        yesdcbs[unit].rev_minor = 3;
        return 1;
    }
    printf("yes_decode_version: failed\n");
    return 0;
}

/*
 ******************************************************************************
 * Routines to move cells between memory and the fifo on the yes board.
 ******************************************************************************
 */

#ifdef __mips
#define inline
#endif

static inline void
yes_fifo_to_memory_no_dma(yes_t *yes_ptr, u_int *m_ptr)
{
    m_ptr[0] = yes_ptr->rxfifo;
    m_ptr[1] = yes_ptr->rxfifo;
    m_ptr[2] = yes_ptr->rxfifo;
    m_ptr[3] = yes_ptr->rxfifo;

    m_ptr[4] = yes_ptr->rxfifo;
    m_ptr[5] = yes_ptr->rxfifo;
    m_ptr[6] = yes_ptr->rxfifo;
    m_ptr[7] = yes_ptr->rxfifo;

    m_ptr[8] = yes_ptr->rxfifo;
    m_ptr[9] = yes_ptr->rxfifo;
    m_ptr[10] = yes_ptr->rxfifo;
    m_ptr[11] = yes_ptr->rxfifo;
}

static inline void
yes_memory_to_fifo_no_dma(yes_t *yes_ptr, u_int *m_ptr, u_int hdr)
{
    yes_ptr->txfifo = hdr;

    yes_ptr->txfifo = m_ptr[0];
    yes_ptr->txfifo = m_ptr[1];
    yes_ptr->txfifo = m_ptr[2];
    yes_ptr->txfifo = m_ptr[3];

    yes_ptr->txfifo = m_ptr[4];
    yes_ptr->txfifo = m_ptr[5];
    yes_ptr->txfifo = m_ptr[6];
    yes_ptr->txfifo = m_ptr[7];

    yes_ptr->txfifo = m_ptr[8];
    yes_ptr->txfifo = m_ptr[9];
    yes_ptr->txfifo = m_ptr[10];
    yes_ptr->txfifo = m_ptr[11];
}

/*
 * ------------------------------------------------------------------------
 *
 * Use the DMAC on the yes board to copy a cell
 * from the yes receive fifo into memory.
 *
 * This code assumes that the "dma page" is smaller than the "mmu page"
 * or - explicitly that that the test for DMA crossing can be done
 * on the virtual address, and only if that shows OK do we bother
 * converting to the physical address.
 */

static inline void
yes_fifo_to_memory(yes_t *yes_ptr, u_int *vir_ptr, u_int *phys_ptr)
{
    register	u_int	da;

    /* Need to check if the physical address of the buffer crosses a
     * Turbochannel page boundry. If so we have to do programmed IO
     * to the virtual address. Otherwise we can dma to the physical
     * address. (we could do seperate small dmas but thats tedious)
     */

    if ( ( (u_int)(phys_ptr) & (TC_PAGE_SIZE-1) ) >= TC_PAGE_SIZE-48 )
    {
	yes_fifo_to_memory_no_dma( yes_ptr, vir_ptr);
    }
    else
    {
	/* A purist might also consider physical addresses that have some
	 * of the top bits set.
	 */

	da = (((u_int)(phys_ptr)) << 3);
	
	TRC(printf("yes_fifo_to_memory: ma=%08x pa=%08x da=%08x\n",
		   (u_int)memory_ptr, pa, da));
	
	/* A purist mught flush the write buffer at this point before
	 * the DMA, but I assert that any pending writes to the area
	 * of RAM we are about to DMA in to will occur before the
	 * writes to start the DMA occur. (No-one has any business to
	 * be wrting into a free pool IOBuf anyway... 
	 */
	
	yes_ptr->length = 0xF6;			/* 12 word transfer */
	yes_ptr->dmarx = (u_int *)da;		/* phys addr */
	
	/* we don't need to interlock with the DMA finishing except
	 * perhaps with the last cell in a block. The time taken to
	 * up call is certain to be longer than for the DMA to
	 * complete, so it's not a problem. 
	 */
    }
}

/*
 * Use the DMAC on the yes board to copy a cell
 * from memory to the yes transmit fifo.
 */
static inline void
    yes_memory_to_fifo_dma(yes_t *yes_ptr, u_int *phys_ptr, u_int hdr)
{
    register	u_int	da;
    
    da = ((u_int)(phys_ptr) << 3);
    
    TRC(printf("yes_memory_to_fifo: da=%08x\n", da));
    
    /* the write buffer shouldnt need flushing here as no
       re-ordering of writes can occur. */
    
    /*
     * There is a problem with first DMA word; it never makes
     * it out onto the wire, so transmit one more than required.
     * DMA terminates when counter hits 0x03(!??!), so 0xF7 is
     * the appropriate initial value to DMA 12 words.
     */
    
    yes_ptr->txfifo = hdr;
    yes_ptr->length = 0xF7;
    yes_ptr->dmatx = (u_int *)da;
    
    /* purists might wait for the DMA to finish before advancing and 
     * doing the free IOBuf. I assert that the DMA will run to completion
     * stalling anyone trying to modify the data queued to send
     */
}

/*****************************************************************************/
/*    Keep checking YES sanity						     */
/*****************************************************************************/

/*
 * This routine gets called for us regularly by the system.
 * Isnt it nice to us. It also passes the "unit" number.
 */

static void switch_on_yes(int dev)
{
    YESDCB		*dcb = &yesdcbs[dev];
    yes_t		*yes = dcb->regs;
    
    printf("yes %d: Activated\n",dev );
    
    if (yesdcbs[dev].rev_minor == 3)
      yes->control = ( RXINTEN );	
    else 
      yes->control = ( RXINTEN | TXHEC );	
    dcb->dcbflags = WORKING;
}

static void switch_off_yes(int dev)
{
    YESDCB		*dcb = &yesdcbs[dev];
    yes_t		*yes = dcb->regs;
    
    printf("yes %d: Disabled\n",dev );
    
    yes->control = 0;
    dcb->dcbflags = BROKEN;
}

#define SKEPTIC_ON		(2)
#define SKEPTIC_MAX		(6)

static int yes_watch(int dev)
{
    YESDCB		*dcb = &yesdcbs[dev];
    yes_t		*yes = dcb->regs;
    u_int		dstat;
    int			s = splimp();
    
    /* Ensure we get called in another timeout interval */

    yesifs[dev]->if_timer = TIMEPERTICK;

    /*
     *	 in V2.0 Yes board must read status after `reset' to clear
     *	 Rx violation flag
     */

#ifdef KEEP_STATS
    if (dcb->dcbflags == WORKING)
    {
	printf("yes %d : rxblocks %d, txblocks %d, rxints %d,\n", 
	       dev, dcb->s.rxok, dcb->s.txok, dcb->s.rxints );
	printf("yes %d : rxbad %d, rxunx %d, rxniob %d, txbad %d, resets %d\n",
	       dev, dcb->s.rxbad, dcb->s.rxunexpected, dcb->s.rxnoiob, 
	       dcb->s.txbad, dcb->s.resets);
    }
    else
    {
	printf("yes %d : Line held in violation\n",dev);
    }
#endif

    dcb->status = dstat = yes->status;

    if (dcb->dcbflags == BROKEN)
    {
	dstat = yes->status;		/* give chance to have recovered */
	if (!(dstat & RXVIOLATION))
	{
	    dcb->vcount--;
	    if (dcb->vcount <= 0)
		switch_on_yes(dev);
	}
	else if (dcb->vcount < (SKEPTIC_MAX - 1))
	    dcb->vcount += 2;
    }
    else
    {
	if (dstat & RXVIOLATION)
	{
	    if (dcb->vcount < SKEPTIC_ON)
		dcb->vcount++;
	    if (dcb->vcount == SKEPTIC_ON)
	    {
		dcb->vcount = SKEPTIC_MAX;
		switch_off_yes(dev);
		yes->xreset = 1;		/* hammer fifos, dont pause */
	    }
	}
	else if ((dstat & (RXDATAINFIFO | RXFIFO | RXIDLE)) 
		 == (RXDATAINFIFO | RXIDLE))
	{
	    yes_hreset(dev);
	    dcb->s.rxbroken ++;
	}
    }

    if (dcb->notidles > 2) 
    {
	/* Yes board appears to be wedged */
	
	printf("yes_watch: yes %d not idle, doing reset\n", dev);
	
	dcb->notidles = 0;	    
	
	yes_hreset(dev);
    }
    else
    {
	if (dcb->dcbflags == WORKING) 
	{
	    if ((yes->status & TXIDLE) == 0)
	    {
		dcb->notidles++;
	    }
	}
    }
    splx(s);
    return 0;
}

/*****************************************************************************/
/*    We are called to transmit a packet on the YES			     */
/*****************************************************************************/

/* static */ void
yes_tx_frame(int unit, u_short vci, int assemble, struct mbuf *m, int len)
{
    yes_t	*yes = yesdcbs[unit].regs;
    FDLHDR	hdr;
    int		s;
    u_int	*from_vir, *from_phys;
    YESDCB	*dcb = &yesdcbs[unit];

    /* don't need to flush cache as its only a write through cache. */

    hdr.all = 0;
    hdr.parts.vci = vci;

    hdr.bits.sequence = divby3table[((len+47)>>4)];
    hdr.bits.assemble = assemble;
    hdr.bits.framestart = 1;
    
    s = splimp();
    
    len = m->m_len;
    from_vir = mtod(m, u_int *);
    from_phys = (u_int *)svtophy(((u_int)from_vir));

    while (hdr.bits.sequence)
    {
	int		j;

	/* Invariant: from and len set for current mbuf */

	if( yes->status & TXAVAILABLE )
	{
	    for( j = CELLS_IN_TX_BLOCK; j>0 && hdr.bits.sequence; j-- )
	    {
		if (!len)
		{
		    m = m->m_next;
		    len = m->m_len;
		    from_vir = mtod(m, u_int *);
		    from_phys = (u_int *)svtophy(((u_int)from_vir));
		}
		
		if (hdr.bits.sequence == 1) {
		    hdr.bits.frameend = 1;
		}
		
		/* Send it and then patch up the invariant again */

		if (((u_int)(from_vir) & (TC_PAGE_SIZE-1) ) >= TC_PAGE_SIZE-48)
		{
		    yes_memory_to_fifo_no_dma(yes, from_vir, hdr.all);
		    from_vir += 12;
		    len -= 48;
		    if (len)
			from_phys = (u_int *)svtophy(((u_int)from_vir));
		}
		else
		{
		    yes_memory_to_fifo_dma(yes, from_phys, hdr.all);
		    from_vir += 12;
		    from_phys += 12;
		    len -= 48;
		}
		
		hdr.bits.framestart = 0;
		hdr.bits.sequence--;
	    }	/* end of send block of cells */
	   	    
	}      	/* if TX AVAIL */
	else
	{
	    /* TX was not avaliable, we cant wait on a semaphore or 
	     * TXAVAILABLE interrupt as ints may already be off.
	     * We have to just chew cpu for a while. We should
	     * have a max of 16 cells past the h/way point.	
	     * Assuming no XOFF, max wait is 71 micro secs.
	     */

	    int k;

	    for( k=0; k< TX_MAX_TRY ; k++)
	    {
		if( yes->status & TXAVAILABLE )
		    break;	/* OK to continue */
	    }
		   
	    if( k == TX_MAX_TRY )
	    {
		dcb->s.txbad ++;
		break;	/* junk rest of frame ! */
	    }
	    
	}

    }		/*end of while */

    if( hdr.bits.sequence == 0 )
	dcb->s.txok ++;
	
    splx(s);
}

/*
 * ------------------------------------------------------------------------
 */

extern struct mbuf *msnl_saneify_mbufs(struct mbuf *n, int *plen);

/*
 * ------------------------------------------------------------------------
 */

int yes_tx(ASSOC *assoc, struct mbuf *m)
{
    int			len;
    
    m = msnl_saneify_mbufs(m, &len);
    if (!m)
	return ENOBUFS;
    
    if ((!len) || (len > YESMTU))
    {
	m_freem(m);
	return EMSGSIZE;
    }

    yes_tx_frame(
		 assoc->as_ifp->if_unit,
		 *(unsigned short *)(&assoc->as_txadr.sa_data[2]),
		 assoc->as_txsar.fdlsar.bits.assemble,
		 m, len);
    m_freem(m);
    assoc->as_txsar.fdlsar.bits.assemble++;
    if ((assoc->as_txsar.fdlsar.bits.assemble & 0x0c) == 0x0c)
	    assoc->as_txsar.fdlsar.bits.assemble += 4;
    return 0;
}

/*****************************************************************************/
/*    Handle YES interrupts						     */
/*****************************************************************************/

/*
 * This structure is a sort of mirror of the association structure which
 * this pod keeps and which can be manuipilated at splimp.
 *
 * We always receive blocks (I know this isn't strictly a valid thing
 * to do) and let the splnet code worry about bad vcis.
 */

typedef struct {
    u_int		*lo_rxptr_vir;
    u_int		*lo_rxptr_phy;
    u_int		lo_rxlen;
    struct mbuf		*lo_head;
    struct mbuf		*lo_current;
    FDLHDR		lo_header;
} LOCAL;

LOCAL locals[NYES][ASSOC_MAX+1];

/* We stick the FDLqueue structure in the head of the lead block to avoid
 * yet another mbuf allocation. All blocks have a FDLqueue offset thus.
 * 4080 + sizeof(FDLQUEUE) is still just less than 4K
 */

#define SLABDATASIZE	(4080)
#define SLABOFFSET	(sizeof(FDLQUEUE))
#define SLABSIZE	(SLABDATASIZE + sizeof(FDLQUEUE))

#if M_CLUSTERSZ != 4096
You_loose;
#endif

/*
 * Interrupts.
 *
 * This code must run at splimp (what the hardware is) but the rest of the
 * networking code must run at splnet, so we queue it and push a software
 * interrupt. Note however that the higher level code must re-check the
 * validity of the association id because things could have changed in the
 * mean time.
 */

static void freeblock(u_int *addr)
{
    KM_FREE(addr, KM_YES);
}

static u_int dustbin[12];

#define junk_cell(yes)	yes_fifo_to_memory_no_dma(yes, dustbin)

/*
 * General Schema:
 *
 * Since links are point-to-point, we can expect
 * a block to be transmitted as a contiguous sequence
 * of cells.
 *
 * Once we get something we lock onto that association for the available
 * data block. We snap out of it if things go wrong or we get a different
 * vci or we come to the end of a block or the end of our buffer space.
 * note its only the errors that are volatile, the fifo unemptyness is not
 * so only return with error if bad things happen.
 */

unsigned int yes_rx_over = 0;
unsigned int yes_rx_under = 0;
unsigned int yes_rx_viol = 0;
unsigned int yes_rx_hecbad = 0;

#ifdef DOG
#define FRC
#endif

#ifdef FRC
unsigned int yes_frc_mbuf = 0;
unsigned int yes_frc_mbuf_n = 0;
unsigned int yes_frc_clbuf = 0;
unsigned int yes_frc_clbuf_n = 0;
unsigned int yes_frc_cache = 0;
unsigned int yes_frc_cache_n = 0;
unsigned int yes_frc_upcall = 0;
unsigned int yes_frc_upcall_n = 0;
unsigned int yes_frc_cell = 0;
unsigned int yes_frc_cell_n = 0;

#define FRC_VAL		*((unsigned int *)(0xaca00000))
#define FRC_START	frc_val = FRC_VAL
#define FRC_END(x)	x += FRC_VAL - frc_val; x/**/_n ++

#else

#define FRC_START
#define FRC_END(x)

#endif

void yesintr(int unit)
{
    register yes_t	*yes = yesdcbs[unit].regs;
    register LOCAL	*local = 0;
    int			oldpri;
    u_int		vci = 0;
    FDLHDR		inhdr;
    FDLHDR		expected;
    FDLSAR		fas;
    u_int		*to;
    int			to_len;
    struct mbuf		*n, *m;
    int			keep;
    int 		last_err = 0;
    YESDCB 		*dcb = &yesdcbs[unit];
#ifdef FRC
    unsigned int	frc_val, frc_val2;
#endif

    /* don't need to flush cache as its only a write through cache. */

    /*
     * Freeze the outside world. Im not too happy with this
     * as you probably want to take TX AVALIABLE ints even in the RX loop,
     * so you can get those 'stopsending me this 100Mb/s crap' messages out.
     */

    dcb->notidles = 0;	/* received data, so clear not idle flag */

    inhdr.all = 0;
    expected.all = 0;

    dcb->s.rxints ++;
  
    /*
     * Loop over all cells in the FIFO.    
     */

    while ((keep = yes->status) & RXFIFO)
    {
#ifdef FRC
	frc_val2 = FRC_VAL;
#endif
	if( keep & ( RXOVER | RXUNDER | RXVIOLATION | RXHECBAD ) )
	{
	    /* First we increment the statistics on what sort of badness
	     * there was
	     */
	    if (keep & RXOVER)		dcb->s.rxover++;
	    if (keep & RXUNDER)		dcb->s.rxunder++;
	    if (keep & RXVIOLATION)	dcb->s.rxviol++;
	    if (keep & RXHECBAD)	dcb->s.hecbad++;

	    /* Need to note that the the board has reported something
	     * dodgy so that if we see somthing like a bad VCI this is 
	     * probably because the FIFOs are out of sync and need 
	     * h_reseting.
	     *
	     * Delaying this hreset gives us more chance of the burst
	     * ending and thus making the reset successful. Whatever way
	     * we do it, there is a real chance of there being a short cell
	     * passed up. If you rely on ATM being reliable, well....
	     */
	    last_err = MAX_CELLS_IN_FIFO + 5; /* arbitary... */
	}
	else
	{
	    if( last_err > 0 ) last_err--;
	}

	dcb->s.rxcells++;

	inhdr.all = yes->rxfifo;

	if ((inhdr.parts.vci < 1) || (inhdr.parts.vci > ASSOC_MAX))
	{
	    /* We have "local" entries for everything, with bad ones
	     * being passed up to be dealt with above us, so this must
	     * be a dubious one (conversely if we dont get here there
	     * is a local structure to be used.
	     */
	    junk_cell(yes);

	    if (last_err > 0)
	    {
		yes_hreset(unit);
		break;
	    }
	    else
	    {
		last_err =  MAX_CELLS_IN_FIFO + 5;
		dcb->s.rxbad ++;
	    }
#ifdef FRC
	    yes_frc_cell += FRC_VAL - frc_val2;
	    yes_frc_cell_n ++;
#endif
	    continue;
	}
	
	/* Here then can use "local" structure */

	local = &locals[unit][inhdr.parts.vci];
	
	if (inhdr.all != local->lo_header.all)
	{
	    /* Some cell(s) have got lost. Free current stuff. */
	    if (local->lo_head)
	    {
		m_freem(local->lo_head);
		local->lo_head = local->lo_current = 0;
		local->lo_rxptr_phy = local->lo_rxptr_vir = 0;
		local->lo_rxlen = 0;
	    }

	    if (!inhdr.bits.framestart)
	    {
		junk_cell(yes);
		continue;
	    }
	    local->lo_header.all = inhdr.all;
	}
	
	/* Here then we are prepared to recieve this cell into "local"
	 * we may need to allocate storage.
	 */

	if (!local->lo_rxlen)
	{
	    u_int		*data;
	    struct mbuf		*n;
	    struct mbuf		*m1;

	    FRC_START;
	    n = m_get(M_DONTWAIT, MT_DATA);
	    FRC_END(yes_frc_mbuf);
	    if (!n)
	    {
		junk_cell(yes);
		continue;
	    }
	    
	    /* this macro is broken it thinks it takes an mbuf pointer */
	    FRC_START;
	    MCLGET(n, m1);
	    FRC_END(yes_frc_clbuf);
	    data = (u_int *)m1;
	    
	    if (!data)
	    {
		m_free(n);
		junk_cell(yes);
		/*
		 * This indicates receiver overrun in the sense of
		 * too much data rather than the fifos themselves.
		 * Something ought to be done about this but unix
		 * is broken in this respect (like everything else)
		 */
		continue;
	    }
	    n->m_off += SLABOFFSET;
	    
	    if (local->lo_head)
	    {
		m1 = local->lo_current;
		
		m1->m_len = ((u_char *)local->lo_rxptr_vir)
		    - mtod(m1, unsigned char *);

		m1->m_next = n;
	    }
	    else
		local->lo_head = n;

	    local->lo_current = n;
	    local->lo_rxptr_vir = (u_int *)(((u_char *)data) + SLABOFFSET);
	    local->lo_rxptr_phy = (u_int *)
		svtophy((u_int)local->lo_rxptr_vir);
	    local->lo_rxlen = SLABDATASIZE;

	    FRC_START;
	    clean_dcache( PHYS_TO_K0(local->lo_rxptr_phy), SLABDATASIZE);
	    FRC_END(yes_frc_cache);
	}
	
	yes_fifo_to_memory(yes, local->lo_rxptr_vir, local->lo_rxptr_phy);
	local->lo_rxptr_vir += 12;
	local->lo_rxptr_phy += 12;
	local->lo_rxlen -= 48;
	
	if (local->lo_header.bits.frameend)
	{
	    /* Pass up */
	    struct mbuf		*m1 = local->lo_current;
	    
	    m1->m_len = ((u_char *)local->lo_rxptr_vir)
		- mtod(m1, unsigned char *);

	    FRC_START;
	    dcb->s.rxupcalls++;
	    fdl_ifinput(local->lo_head, yesifs[unit], inhdr.parts.vci);
	    FRC_END(yes_frc_upcall);
	    local->lo_head = 0;
	    local->lo_rxlen = 0;
	    local->lo_header.all = 0;
	}
	else
	{
	    local->lo_header.bits.framestart = 0;

	    local->lo_header.bits.sequence--;
	    if (local->lo_header.bits.sequence==1)
		local->lo_header.bits.frameend = 1;
	}
#ifdef FRC
	yes_frc_cell += FRC_VAL - frc_val2;
	yes_frc_cell_n ++;
#endif
    }
}

/*****************************************************************************/
/*    Initialisation of interfaces					     */
/*****************************************************************************/

int yes_ioctl(ifp, cmd, data)
    struct ifnet	*ifp;
    int			cmd;
    caddr_t		data;
{
    YESDCB 		*dcb = &yesdcbs[ifp->if_unit];

    /* This interface is no way comming up ever so there */
    ifp->if_flags &= ~(IFF_UP | IFF_RUNNING);

    switch (cmd)
    {
#ifdef DOG
    case YESIFMAP:
	if (!(dcb->segment_init))
	{
	    /* No segment yet */
	    dcb->segment = vm_system_smget( dcb->regs, CLBYTES, 0666);
	    if (dcb->segment != -1)
		dcb->segment_init = 1;
	}
	*(u_int *)data = dcb->segment;
	return 0;
#endif
	
    default:
	return EADDRNOTAVAIL;
    }
}

int yesprobe(char *vbaddr, struct uba_device *uba)
{
    int		dev = uba->ui_unit;
    yes_t	*yes;

    bzero(&yesdcbs[dev], sizeof(YESDCB));
    bzero(&locals[dev][0], (ASSOC_MAX+1) * sizeof(LOCAL));
    
    yesdcbs[dev].regs = yes = (yes_t *)(vbaddr + YES_OFFSET);
    if (yes_decode_version(dev, vbaddr) == 0) 
	return 0;
    yes_start(dev);

    return 1;
}

int yesattach(struct uba_device *uba)
{
    struct ifnet	*ifp;
    int			j;
    
    for (j = 47; j <= 12287; j += 16) {
        divby3table[j>>4] = (j>>4) / 3;
    }

    KM_ALLOC(ifp, struct ifnet *, sizeof(struct ifnet), KM_DEVBUF, KM_NOWAIT);
    
    if (!ifp)
	panic("yesattach");
    
    bzero(ifp, sizeof(struct ifnet));
    
    ifp->if_name = "yes";
    ifp->if_unit = uba->ui_unit;
    
    ifp->if_mtu = YESMTU;
    /* not running - stops other protocols trying to use it */
    ifp->if_flags = IFF_NOTRAILERS | IFF_NOARP;

    ifp->if_timer = TIMEPERTICK;
    ifp->if_watchdog = yes_watch;

    ifp->if_type = IFT_OTHER;
    
    ifp->if_start = yes_start;

    ifp->if_output = yes_tx;
    ifp->if_ioctl = yes_ioctl;

    if_attach(ifp);

    yesifs[uba->ui_unit] = ifp;

    return 0;
}

/*****************************************************************************/
/*    Statics and Configuration						     */
/*****************************************************************************/

/*
 * This structure gets linked into the system via the ioconf.c file
 * which is magically created by the config program based on the 
 * configuaration line that looks something like
 * 	device		yes0	at ibus?	vector yesintr
 * and the line in tc_option_data.c that looks something like
 *    {   "YES-CBG ",     "yes",    0,      1,    'D',    0},
 */

u_short yesmagic1[NYES] = { 0, };
struct uba_device *yesmagic2[NYES] = { 0, };

struct uba_driver yesdriver =
{
    yesprobe, 0, yesattach, 0, yesmagic1, "yes", yesmagic2
};

/*
 * ------------------------------------------------------------------------
 */
