(* ===================================================================== *)
(* FILE          : Thm.sml                                               *)
(* DESCRIPTION   : The abstract data type of theorems.                   *)
(*                                                                       *)
(* AUTHORS       : (c) Mike Gordon, University of Cambridge              *)
(*                     Konrad Slind, University of Calgary               *)
(*                                   University of Cambridge             *)
(*                     Bruno Barras, University of Cambridge and INRIA   *)
(* DATE          : September 10, 1991, Konrad Slind                      *)
(* Modified      : September 23, 1997, Ken Larsen                        *)
(*               : 1999, Bruno Barras                                    *)
(*               : July 2000, Konrad Slind                               *)
(* ===================================================================== *)

structure Thm : RawThm =
struct

open Feedback Lib Term KernelTypes

val --> = Type.-->; 
infixr 3 -->; 
infix 5 |-> ; 

(*---------------------------------------------------------------------------
       Exception handling
 ---------------------------------------------------------------------------*)

val thm_err = mk_HOL_ERR "Thm"
fun ERR f m = raise thm_err f m
fun Assert b s1 s2 = if b then () else ERR s1 s2


(*---------------------------------------------------------------------------*
 * Miscellaneous syntax routines.                                            *
 *---------------------------------------------------------------------------*)

val bool      = Type.bool
val alpha     = Type.alpha
fun dom ty    = fst (Type.dom_rng ty)
fun rng ty    = snd (Type.dom_rng ty);
val EQ        = Portable.pointer_eq
val empty_tag = Tag.empty_tag


(*---------------------------------------------------------------------------
    The following are here because I didn't want to Thm to be dependent 
    on Dsyntax, which is slated for extinction. Since we are inside an
    abstract type, we can take some shortcuts with these routines (if 
    we are careful).
 ---------------------------------------------------------------------------*)
 
val F = Susp.delay (fn () => mk_thy_const{Name="F", Thy="bool", Ty=bool});

val mk_neg = Susp.delay (fn () => 
  let val notc = mk_thy_const{Name="~",Thy="bool",Ty=bool --> bool}
  in fn M => mk_comb(notc,M)
  end);

val mk_forall = Susp.delay (fn () =>
 let val forallc = prim_mk_const{Name="!", Thy="bool"}
 in fn v => fn tm =>
      mk_comb(inst[alpha |-> type_of v] forallc, mk_abs(v,tm))
 end);

val mk_conj = Susp.delay (fn () =>
 let val conjc = prim_mk_const{Name="/\\", Thy="bool"}
 in fn (M,N) => list_mk_comb(conjc,[M,N]) 
 end);

val mk_disj = Susp.delay (fn () =>
 let val disjc = prim_mk_const{Name="\\/", Thy="bool"}
 in fn (M,N) => list_mk_comb(disjc,[M,N]) 
 end);


local val DEST_IMP_ERR = thm_err "dest_imp" ""
in
fun dest_imp M =
 let val (Rator,conseq) = with_exn dest_comb M DEST_IMP_ERR
 in if is_comb Rator
    then let val (Rator,ant) = dest_comb Rator
         in if Rator=Term.imp then (ant,conseq) else raise DEST_IMP_ERR 
         end
    else case with_exn dest_thy_const Rator DEST_IMP_ERR
          of {Name="~", Thy="bool",...} => (conseq,Susp.force F)
           | otherwise => raise DEST_IMP_ERR
 end
end;

local val err = thm_err "dest_exists" ""
in
fun dest_exists tm = 
 let val (Rator,Rand) = with_exn dest_comb tm err
 in case with_exn dest_thy_const Rator err
     of {Name="?", Thy="bool",...} => with_exn dest_abs Rand err
      | otherwise => raise err
 end
end;


(*---------------------------------------------------------------------------*
 * The type of theorems and some basic operations on it.                     *
 *---------------------------------------------------------------------------*)

(* datatype thm = datatype KernelTypes.thm *)

fun tag  (THM(tg,_,_))  = tg
fun hyp  (THM(_,asl,_)) = asl
fun concl(THM(_,_,c))   = c

fun dest_thm(THM(_,asl,w)) = (asl,w)   (* Compatible with old code. *)
fun make_thm R seq = (Count.inc_count R; THM seq);   (* internal only *)

fun thm_frees (THM(_,asl,c)) = free_varsl (c::asl);

fun add_hyp h asl = if exists (aconv h) asl then asl else h::asl;
fun union_hyp [] l2  = l2
  | union_hyp l1 []  = l1
  | union_hyp l1 l2  = if EQ(l1,l2) then l1 else itlist add_hyp l1 l2;

fun tag_hyp_union thm_list =
  (itlist (Tag.merge o tag) (tl thm_list) (tag (hd thm_list)),
   itlist (union_hyp o hyp) thm_list []);

fun var_occursl v l = exists (var_occurs v) l;

fun is_bool tm = (type_of tm = bool);

(*---------------------------------------------------------------------------
 *                THE PRIMITIVE RULES OF INFERENCE
 *---------------------------------------------------------------------------*)


fun WEAK_ASSUME (tm,tmlist) =
   (Assert (all is_bool (tm::tmlist)) "WEAK_ASSUME" "not a proposition";
    make_thm Count.Assume (empty_tag,tm::tmlist,tm));

(*---------------------------------------------------------------------------*
 *                                                                           *
 *       ---------  ASSUME M                           [M is boolean]        *
 *         M |- M                                                            *
 *---------------------------------------------------------------------------*)

fun ASSUME tm = WEAK_ASSUME(tm,[]);
fun ASSUME tm =
   (Assert (is_bool tm) "ASSUME" "not a proposition";
    make_thm Count.Assume (empty_tag,[tm],tm));


(*---------------------------------------------------------------------------*
 *                                                                           *
 *         ---------  REFL M                                                 *
 *         |- M = M                                                          *
 *---------------------------------------------------------------------------*)

val mk_eq_nocheck = Term.prim_mk_eq;

fun refl_nocheck ty tm =
  make_thm Count.Refl (empty_tag, [], mk_eq_nocheck ty tm tm);

fun REFL tm = refl_nocheck (type_of tm) tm;


(*---------------------------------------------------------------------------*
 *                                                                           *
 *         ------------------------  BETA_CONV "(\v.M) N"                    *
 *         |- (\v.M) N = M [v|->N]                                           *
 *---------------------------------------------------------------------------*)

fun BETA_CONV tm = 
   make_thm Count.Beta 
      (empty_tag, [], mk_eq_nocheck (type_of tm) tm (beta_conv tm))
   handle HOL_ERR _ => ERR "BETA_CONV" "not a beta-redex"


(*---------------------------------------------------------------------------
 * ltheta is a substitution mapping from the template to the concl of
 * the given theorem. It checks that the template is an OK abstraction of
 * the given theorem. rtheta maps the template to another theorem, in which
 * the lhs in the first theorem have been replaced by the rhs in the second
 * theorem. The replacements provide the lhs and rhs.
 *---------------------------------------------------------------------------*)

fun SUBST replacements template (th as THM(O,asl,c)) =
 let val (ltheta, rtheta, hyps, oracles) =
         itlist (fn {redex, residue = THM(ocls,h,c)} =>
                 fn (ltheta,rtheta,hyps,Ocls) =>
                      let val _ = Lib.assert Term.is_var redex
                          val (lhs,rhs,_) = Term.dest_eq_ty c
                      in ((redex |-> lhs)::ltheta, 
                          (redex |-> rhs)::rtheta,
                          union_hyp h hyps,
                          Tag.merge ocls Ocls)
                      end)
                replacements ([],[],asl,O)
 in
   if aconv (subst ltheta template) c
    then make_thm Count.Subst (oracles,hyps,subst rtheta template)
      else th
 end;

(*---------------------------------------------------------------------------*
 *        A |- t1 = t2                                                       *
 *   ------------------------  ABS x            [Where x is not free in A]   *
 *    A |- (\x.t1) = (\x.t2)                                                 *
 *---------------------------------------------------------------------------*)

fun ABS v (THM(ocl,asl,c)) =
 let val (lhs,rhs,ty) = Term.dest_eq_ty c handle HOL_ERR _ 
                      => ERR "ABS" "not an equality"
     val vty = snd(dest_var v) handle HOL_ERR _ 
                      => ERR "ABS" "first argument is not a variable"
 in if var_occursl v asl
     then ERR "ABS" "The variable is free in the assumptions"
     else make_thm Count.Abs (ocl,asl,
           mk_eq_nocheck (vty --> ty) (mk_abs(v,lhs)) (mk_abs(v,rhs)))
 end;


(*---------------------------------------------------------------------------
 *     A |- M
 *  ---------------- INST_TYPE theta   [Where theta would not move A]
 *     A |- theta(M)
 *
 * The proviso means that no type variable in the domain of theta occurs
 * in A. It would be more sensible to just apply theta to A.
 *---------------------------------------------------------------------------*)

local open Type
      fun compare (v1,v2) = String.compare(dest_vartype v1,dest_vartype v2)
      val init_varset = Binaryset.empty compare
in
fun INST_TYPE [] th = th
  | INST_TYPE theta (THM(ocl,asl,c)) =
    let val domvars = rev_itlist(fn {redex,...} => fn S => 
                         Binaryset.add(S,redex)) theta init_varset
        fun in_dom v = Binaryset.member(domvars,v)
    in
      Assert (Lib.all (not o existsTYV in_dom) asl) "INST_TYPE"
         "type variable(s) in assumptions would be instantiated in concl";
      make_thm Count.InstType (ocl, asl, inst theta c)
    end
end;

(*---------------------------------------------------------------------------
 *          A |- M
 *  ---------------------- DISCH tm
 *     A - tm |- tm ==> M
 *
 * The term tm need not occur in A. All terms in A that are 
 * alpha-convertible to tm are removed.
 *---------------------------------------------------------------------------*)

fun DISCH w (THM(ocl,asl,c)) =
  (Assert (is_bool w) "DISCH" "not a proposition";
   make_thm Count.Disch
      (ocl, gather (not o aconv w) asl, Term.prim_mk_imp w c));


(*---------------------------------------------------------------------------
 *          A |- M ==> N ,  B |- M'
 *  ----------------------------------  MP
 *              A u B |- N
 *
 * M and M' must be alpha-convertible
 *---------------------------------------------------------------------------*)

fun MP (THM(o1,asl1,c1)) (THM(o2,asl2,c2)) =
 let val (ant,conseq) = dest_imp c1 
 in Assert (aconv ant c2) "MP" 
      "antecedent of first thm not alpha-convertible to concl. of second";
    make_thm Count.Mp (Tag.merge o1 o2, union_hyp asl1 asl2, conseq)
 end;


(*---------------------------------------------------------------------------*
 * Derived Rules -- these are here for efficiency purposes, and so that all  *
 * invocations of mk_thm are in Thm. The following derived rules do not use  *
 * any axioms of boolTheory:                                                 *
 *                                                                           *
 *   ALPHA, SYM, TRANS, MK_COMB, AP_TERM, AP_THM, EQ_MP, EQ_IMP_RULE,        *
 *   Beta, Mk_comb, Mk_abs                                                   *
 *                                                                           *
 *---------------------------------------------------------------------------*)

(*---------------------------------------------------------------------------
 * Equivalence of alpha-convertible terms
 *
 *     t1, t2 alpha-convertable
 *     -------------------------
 *            |- t1 = t2
 *
 * fun ALPHA t1 t2 = TRANS (REFL t1) (REFL t2)
 *---------------------------------------------------------------------------*)

fun ALPHA t1 t2 =
   if aconv t1 t2
   then make_thm Count.Alpha (empty_tag, [], mk_eq_nocheck (type_of t1) t1 t2)
   else ERR "ALPHA" "";


(*---------------------------------------------------------------------------
 * Simple version of alpha-conversion (needed for deriving ETA_CONV)
 *
 *       "\x1. t x1"   "\x2. t x2"   --->   |- "(\x1.t x1)=(\x2.t x2)"
 *
 * fun SIMPLE_ALPHA(t1,t2) =
 *   let val (x1,body1) = dest_abs t1
 *       and (x2,body2) = dest_abs t2
 *       val th1 = BETA_CONV `^t1 (x:^(type_of x1))`
 *       (* th1 = |- (\x1. t x1)x = t x *)
 *       and th2 = BETA_CONV `^t2 (x:^(type_of x2))`
 *       (* th2 = |- (\x2. t x2)x = t x *)
 *       and th3 = SPEC t1 (INST_TYPE [(type_of x1, `:'a`),
 *                                     (type_of body1, `:'b`)] ETA_AX)
 *       (* th3 = |- (\x. (\x1. t x1)x) = (\x1. t x1) *)
 *       and th4 = SPEC t2 (INST_TYPE [(type_of x2, `:'a`),
 *                                     (type_of body2, `:'b`)] ETA_AX)
 *       (* th4 = |- (\x. (\x2. t x2)x) = (\x2. t x2) *)
 *   in
 *   TRANS (TRANS (SYM th3)
 *                (ABS `x:^(type_of x1)` (TRANS th1 (SYM th2))))
 *         th4
 *   end
 *   handle _ => ERR{function = "SIMPLE_ALPHA",message = ""};
 *
 *
 * Eta-conversion
 *
 * 	"(\x.t x)"   --->    |- (\x.t x) = t  (if x not free in t)
 *
 * fun ETA_CONV (tm as Abs(Var(vty,_), cmb as Comb(t,_))) =
 *      (let val body_ty = type_of cmb
 *           val th = SPEC t (INST_TYPE [(vty,`:'a`), (body_ty, `:'b`)] ETA_AX)
 *           (* th = |- (\x. t x) = t *)
 *       in
 *       TRANS (SIMPLE_ALPHA(tm,lhs(concl th))) th
 *       end
 *       handle _ => ERR{function = "ETA_CONV",message = ""})
 *  | ETA_CONV _ = ERR{function = "ETA_CONV",message = ""};
 *
 *---------------------------------------------------------------------------*)

fun ETA_CONV tm =
   make_thm Count.EtaConv
      (empty_tag, [], mk_eq_nocheck (type_of tm) tm (eta_conv tm))
   handle HOL_ERR _ => ERR"ETA_CONV" "";


(*---------------------------------------------------------------------------*
 *  Symmetry of =
 *
 *       A |- t1 = t2
 *     ----------------
 *       A |- t2 = t1
 *
 * fun SYM th =
 *   let val (t1,t2) = dest_eq(concl th)
 *       val v = genvar(type_of t1)
 *   in
 *   SUBST [(th,v)] (mk_eq(v,t1)) (REFL t1)
 *   end
 *   handle _ =>  ERR "SYM" "";
 *---------------------------------------------------------------------------*)

fun SYM th =
 let val (lhs,rhs,ty) = Term.dest_eq_ty (concl th) 
  in make_thm Count.Sym
        (tag th, hyp th, mk_eq_nocheck ty rhs lhs)
 end
 handle HOL_ERR _ => ERR "SYM" "";

(*---------------------------------------------------------------------------
 * Transitivity of =
 *
 *   A1 |- t1 = t2  ,  A2 |- t2 = t3
 *  ---------------------------------
 *        A1 u A2 |- t1=t3
 *
 * fun TRANS th1 th2 =
 *   let val (t1,t2) = dest_eq(concl th1)
 *       and (t2',t3) = dest_eq(concl th2)
 *       val v = genvar(type_of t1)
 *   in
 *   SUBST [(th2,v)] (mk_eq(t1,v)) th1
 *   end
 *   handle _ =>  ERR{function = "TRANS",message = ""};
 *
 *---------------------------------------------------------------------------*)

fun TRANS th1 th2 =
   let val (lhs1,rhs1,ty) = Term.dest_eq_ty (concl th1)
       and (lhs2,rhs2,_)  = Term.dest_eq_ty (concl th2)
       val   _  = Assert (aconv rhs1 lhs2) "" ""
       val hyps = union_hyp (hyp th1) (hyp th2)
       val ocls = Tag.merge (tag th1) (tag th2)
   in
     make_thm Count.Trans (ocls, hyps, mk_eq_nocheck ty lhs1 rhs2)
   end
   handle HOL_ERR _ => ERR "TRANS" "";


(*---------------------------------------------------------------------------
 *     A1 |- f = g    A2 |- x = y
 *     ---------------------------
 *       A1 u A2 |- f x = g y
 *
 * fun MK_COMB (funth,argth) =
 *   let val f = lhs (concl funth)
 *       and x = lhs (concl argth)
 *   in
 *   SUBS_OCCS [([2], funth), ([2], argth)] (REFL (Comb(f,x))))
 *   ? failwith `MK_COMB`;
 *---------------------------------------------------------------------------*)

fun MK_COMB (funth,argth) =
   let val (f,g,ty) = Term.dest_eq_ty (concl funth)
       val (x,y,_)  = Term.dest_eq_ty (concl argth)
   in
     make_thm Count.MkComb
         (Tag.merge (tag funth) (tag argth),
          union_hyp (hyp funth) (hyp argth),
          mk_eq_nocheck (rng ty) (mk_comb(f,x)) (mk_comb(g,y)))
   end
   handle HOL_ERR _ => ERR "MK_COMB" "";

(*---------------------------------------------------------------------------
 * Application of a term to a theorem
 *
 *    A |- t1 = t2
 * ------------------
 *  A |- t t1 = t t2
 *
 * fun AP_TERM tm th =
 *   let val (t1,_) = dest_eq(concl th)
 *       val th1 = REFL (--`^tm ^t1`--)
 *       (* th1 = |- t t1 = t t1 *)
 *       and v  = genvar(type_of t1)
 *   in
 *   SUBST [(th,v)] (--`^tm ^t1 = ^tm ^v`--) th1
 *   end
 *   handle _ =>  ERR{function = "AP_TERM",message = ""};
 *---------------------------------------------------------------------------*)

fun AP_TERM f th =
 let val (lhs,rhs,_) = Term.dest_eq_ty (concl th)
 in make_thm Count.ApTerm
       (tag th, hyp th,
        mk_eq_nocheck (rng (type_of f)) (mk_comb(f,lhs)) (mk_comb(f,rhs)))
 end
 handle HOL_ERR _ => ERR "AP_TERM" "";


(*---------------------------------------------------------------------------
 * Application of a theorem to a term
 *
 *    A |- t1 = t2
 *   ----------------
 *   A |- t1 t = t2 t
 *
 * fun AP_THM th tm =
 *   let val (t1,_) = dest_eq(concl th)
 *       val th1 = REFL (--`^t1 ^tm`--)
 *       (* th1 = |- t1 t = t1 t *)
 *       and v   = genvar(type_of t1)
 *   in
 *   SUBST [(th,v)] (--`^t1 ^tm = ^v ^tm`--) th1
 *   end
 *   handle _ =>  ERR{function = "AP_THM",message = ""};
 *---------------------------------------------------------------------------*)

fun AP_THM th tm =
 let val (lhs,rhs,ty) = Term.dest_eq_ty (concl th)
 in make_thm Count.ApThm
       (tag th, hyp th,
        mk_eq_nocheck (rng ty) (mk_comb(lhs,tm)) (mk_comb(rhs,tm)))
 end
 handle HOL_ERR _ => ERR "AP_THM" "";

(*---------------------------------------------------------------------------
 *  Modus Ponens for =
 *
 *
 *   A1 |- t1 = t2  ,  A2 |- t1
 *  ----------------------------
 *        A1 u A2 |- t2
 *
 * fun EQ_MP th1 th2 =
 *   let val (t1,t2) = dest_eq(concl th1)
 *       val v = genvar(type_of t1)
 *   in  SUBST [(th1,v)] v th2
 *   end handle _ =>  ERR{function = "EQ_MP",message = ""};
 *---------------------------------------------------------------------------*)

fun EQ_MP th1 th2 =
   let val (lhs,rhs,_) = Term.dest_eq_ty (concl th1)
       val _ = Assert (aconv lhs (concl th2)) "" ""
   in
    make_thm Count.EqMp (Tag.merge (tag th1) (tag th2),
                         union_hyp (hyp th1) (hyp th2), rhs)
   end handle HOL_ERR _ => ERR "EQ_MP" "";

(*---------------------------------------------------------------------------
 *              A |- t1 = t2
 *    ------------------------------------
 *     A |- t1 ==> t2      A |- t2 ==> t1
 *
 * fun EQ_IMP_RULE th =
 *   let val (t1,t2) = dest_eq(concl th)
 *   in
 *   (DISCH t1 (EQ_MP th (ASSUME t1)), 
 *    DISCH t2 (EQ_MP (SYM th)(ASSUME t2)))
 *   end
 *   handle _ =>  ERR{function = "EQ_IMP_RULE",message = ""};
 *---------------------------------------------------------------------------*)

fun EQ_IMP_RULE th =
 let val (lhs,rhs,ty) = Term.dest_eq_ty (concl th)
     and A = hyp th
     and O = tag th
 in if ty = Type.bool
    then (make_thm Count.EqImpRule(O,A, Term.prim_mk_imp lhs rhs),
          make_thm Count.EqImpRule(O,A, Term.prim_mk_imp rhs lhs))
    else ERR "" ""
 end 
 handle HOL_ERR _ => ERR "EQ_IMP_RULE" "";

(*---------------------------------------------------------------------------
 * Specialization
 *
 *    A |- !(\x.u)
 *  --------------------   (where t is free for x in u)
 *    A |- u[t/x]
 *
 * fun SPEC t th =
 *   let val {Rator=F,Rand=body} = dest_comb(concl th)
 *   in
 *   if (not(#Name(dest_const F)="!"))
 *   then ERR{function = "SPEC",message = ""}
 *   else let val {Bvar=x,Body=u} = dest_abs body
 *        and v1 = genvar(type_of F)
 *        and v2 = genvar(type_of body)
 *        val th1 = SUBST[{var = v1,
 *                         thm = INST_TYPE[{redex   = (==`:'a`==),
 *                                          residue = type_of x}] FORALL_DEF}]
 *                       (--`^v1 ^body`--) th
 *        (* th1 = |- (\P. P = (\x. T))(\x. t1 x) *)
 *        val th2 = BETA_CONV(concl th1)
 *        (* th2 = |- (\P. P = (\x. T))(\x. t1 x) = ((\x. t1 x) = (\x. T)) *)
 *        val th3 = EQ_MP th2 th1
 *        (* th3 = |- (\x. t1 x) = (\x. T) *)
 *        val th4 = SUBST [{var= v2, thm=th3}] (--`^body ^t = ^v2 ^t`--)
 *                        (REFL (--`^body ^t`--))
 *        (* th4 = |- (\x. t1 x)t = (\x. T)t *)
 *        val {lhs=ls,rhs=rs} = dest_eq(concl th4)
 *        val th5 = TRANS(TRANS(SYM(BETA_CONV ls))th4)(BETA_CONV rs)
 *        (* th5 = |- t1 t = T *)
 *        in
 *        EQT_ELIM th5
 *        end
 *   end
 *   handle _ => ERR{function = "SPEC",message = ""};
 *
 *
 * pre-dB manner:
 * fun SPEC t th =
 *   let val {Bvar,Body} = dest_forall(concl th)
 *   in
 *   make_thm Count.(hyp th, subst[{redex = Bvar, residue = t}] Body)
 *   end
 *   handle _ => ERR{function = "SPEC",message = ""};
 *---------------------------------------------------------------------------*)

fun SPEC t th =
 let val (Rator,Rand) = dest_comb(concl th)
     val {Thy,Name,...} = dest_thy_const Rator
 in
   Assert (Name="!" andalso Thy="bool") "" "";
   make_thm Count.Spec 
       (tag th, hyp th, beta_conv(mk_comb(Rand, t)))
 end
 handle HOL_ERR _ => ERR "SPEC" "";


(*---------------------------------------------------------------------------
 * Generalization
 *
 *         A |- t
 *   -------------------   (where x not free in A)
 *       A |- !(\x.t)
 *
 * fun GEN x th =
 *   let val th1 = ABS x (EQT_INTRO th)
 *     (* th1 = |- (\x. t1 x) = (\x. T)  --ABS does not behave this way --KLS*)
 *      val abs = `\^x. ^(concl th)`
 *      and v1 = genvar `:(^(type_of x) -> bool) -> bool`
 *      and v2 = genvar `:bool`
 *      val th2 = SUBST [(INST_TYPE[(type_of x, `:'a`)]FORALL_DEF,v1)]
 *                      `($! ^abs) = (^v1 ^abs)`
 *                      (REFL `$! ^abs`)
 *      (* th2 = |- (!x. t1 x) = (\P. P = (\x. T))(\x. t1 x) *)
 *      val th3 = TRANS th2 (BETA_CONV(snd(dest_eq(concl th2))))
 *      (* th3 = |- (!x. t1 x) = ((\x. t1 x) = (\x. T)) *)
 *      in
 *      SUBST [(SYM th3, v2)] v2 th1
 *      end
 *      handle _ => ERR{function = "GEN",message = ""};
 *---------------------------------------------------------------------------*)


fun GEN x th =
  let val (asl,c) = dest_thm th
  in if var_occursl x asl
     then ERR  "GEN" "variable occurs free in hypotheses"
     else make_thm Count.Gen(tag th, asl, Susp.force mk_forall x c)
          handle HOL_ERR _ => ERR "GEN" ""
  end;


(*---------------------------------------------------------------------------
 * Existential introduction
 *
 *    A |- t[t']
 *  --------------  EXISTS ("?x.t[x]", "t'")  ( |- t[t'] )
 *   A |- ?x.t[x]
 *
 *  
 *
 * fun EXISTS (fm,tm) th =
 *   let val (x,t) = dest_exists fm
 *       val th1 = BETA_CONV `(\(^x). ^t) ^tm`
 *       (* th1 = |- (\x. t x)t' = t t' *)
 *       val th2 = EQ_MP (SYM th1) th
 *       (* th2 = |- (\x. t x)t' *)
 *       val th3 = SELECT_INTRO th2
 *       (* th3 = |- (\x. t x)(@x. t x) *)
 *       val th4 = AP_THM(INST_TYPE[(type_of x, `:'a`)]EXISTS_DEF) `\(^x).^t`
 *       (* th4 = |- (?x. t x) = (\P. P($@ P))(\x. t x) *)
 *       val th5 = TRANS th4 (BETA_CONV(snd(dest_eq(concl th4))))
 *       (* th5 = |- (?x. t x) = (\x. t x)(@x. t x) *)
 *   in
 *   EQ_MP (SYM th5) th3
 *   end
 *   handle _ => ERR{function = "EXISTS",message = ""};
 *
 *---------------------------------------------------------------------------*)

local val err = thm_err "EXISTS" ""
in
fun EXISTS (w,t) th =
 let val (ex,Rand) = with_exn dest_comb w err
     val {Name,Thy,...}  = with_exn dest_thy_const ex err
     val _ = Assert ("?"=Name andalso Thy="bool") "EXISTS" "not an existential"
     val _ = Assert (aconv (beta_conv(mk_comb(Rand,t))) (concl th)) 
                    "EXISTS" "incompatible structure"
   in make_thm Count.Exists (tag th, hyp th, w)
   end
end;

(*---------------------------------------------------------------------------
 * Existential elimination
 *
 *   A1 |- ?x.t[x]   ,   A2, "t[v]" |- t'
 *   ------------------------------------     (variable v occurs nowhere)
 *            A1 u A2 |- t'
 *
 * fun CHOOSE (v,th1) th2 =
 *   let val (x,body) = dest_exists(concl th1)
 *       and t'     = concl th2
 *       and v1     = genvar `:bool`
 *       val th3 = AP_THM(INST_TYPE[(type_of v, `:'a`)]EXISTS_DEF)`\(^x).^body`
 *       (* th3 = |- (?x. t x) = (\P. P($@ P))(\x. t x) *)
 *       val th4 = EQ_MP th3 th1
 *       (* th4 = |- (\P. P($@ P))(\x. t x) *)
 *       val th5 = EQ_MP (BETA_CONV(concl th4)) th4
 *       (* th5 = |- (\x. t x)(@x. t x) *)
 *       val th6 = BETA_CONV `(\(^x).^body)^v`
 *       (* th6 = |- (\x. t x)v = t v *)
 *       val Pa = snd(dest_eq(concl th6))
 *       val th7 = UNDISCH(SUBST [(SYM th6,v1)] `^v1 ==> ^t'` (DISCH Pa th2))
 *       (* th7 = |- t' *)
 *   in
 *   SELECT_ELIM th5 (v,th7)
 *   end
 *   handle _ => ERR{function = "CHOOSE",message = ""};
 *---------------------------------------------------------------------------*)

fun disch(w,wl) = Lib.gather (not o aconv w) wl;

fun CHOOSE (v,xth) bth =
  let val (x_asl, x_c) = dest_thm xth
      val (b_asl, b_c) = dest_thm bth
      val (Bvar,Body)  = dest_exists x_c
      val newhyps = union_hyp x_asl (disch (subst [Bvar |-> v] Body, b_asl))
      val occursv = var_occurs v
      val _ = Assert (not(occursv x_c) andalso 
                      not(occursv b_c) andalso 
                      not(exists occursv newhyps)) "" ""
  in make_thm Count.Choose 
       (Tag.merge (tag xth) (tag bth), newhyps,  b_c)
  end
  handle HOL_ERR _ => ERR "CHOOSE" "";


(*---------------------------------------------------------------------------
 * Conjunction introduction rule
 *
 *   A1 |- t1  ,  A2 |- t2
 *  -----------------------
 *    A1 u A2 |- t1 /\ t2
 *
 * fun CONJ th1 th2 = MP (MP (SPEC (concl th2) 
 *                             (SPEC (concl th1) AND_INTRO_THM))
 *                          th1)
 *                      th2;
 *---------------------------------------------------------------------------*)

fun CONJ th1 th2 =
   make_thm Count.Conj
        (Tag.merge (tag th1) (tag th2),
         union_hyp (hyp th1) (hyp th2),
         Susp.force mk_conj(concl th1, concl th2))
   handle HOL_ERR _ => ERR "CONJ" "";


(*---------------------------------------------------------------------------
 * Left conjunct extraction
 *
 *   A |- t1 /\ t2
 *   -------------
 *      A |- t1
 *
 * fun CONJUNCT1 th =
 *   let val (t1,t2) = dest_conj(concl th)
 *   in
 *   MP (SPEC t2 (SPEC t1 AND1_THM)) th
 *   end
 *   handle _ => ERR{function = "CONJUNCT1",message = ""};
 *
 *---------------------------------------------------------------------------*)

fun conj1 tm =
  let val (c,M) = dest_comb (rator tm)
      val {Name,Thy,...} = dest_thy_const c
  in 
    if Name="/\\" andalso Thy="bool" then M else ERR "" ""
  end


fun CONJUNCT1 th =
  make_thm Count.Conjunct1 (tag th, hyp th, conj1 (concl th))
  handle HOL_ERR _ => ERR "CONJUNCT1" "";


(*---------------------------------------------------------------------------
 *  Right conjunct extraction
 *
 *   A |- t1 /\ t2
 *   -------------
 *      A |- t2
 *
 * fun CONJUNCT2 th =
 *   let val (t1,t2) = dest_conj(concl th)
 *   in
 *   MP (SPEC t2 (SPEC t1 AND2_THM)) th
 *   end
 *   handle _ => ERR{function = "CONJUNCT2",message = ""};
 *---------------------------------------------------------------------------*)

fun conj2 tm =
  let val (Rator,M) = dest_comb tm
      val {Name,Thy,...} = dest_thy_const (rator Rator)
  in 
     if Name="/\\" andalso Thy="bool" then M else ERR "" ""
  end

fun CONJUNCT2 th =
 make_thm Count.Conjunct2 (tag th, hyp th, conj2 (concl th))
  handle HOL_ERR _ => ERR "CONJUNCT2" "";


(*---------------------------------------------------------------------------
 * Left disjunction introduction
 *
 *      A |- t1
 *  ---------------
 *   A |- t1 \/ t2
 *
 * fun DISJ1 th t2 = MP (SPEC t2 (SPEC (concl th) OR_INTRO_THM1)) th
 *           handle _ => ERR{function = "DISJ1",message = ""};
 *---------------------------------------------------------------------------*)

fun DISJ1 th w = make_thm Count.Disj1 
 (tag th, hyp th, Susp.force mk_disj (concl th, w))
 handle HOL_ERR _ => ERR "DISJ1" "";


(*---------------------------------------------------------------------------
 * Right disjunction introduction
 *
 *      A |- t2
 *  ---------------
 *   A |- t1 \/ t2
 *
 * fun DISJ2 t1 th = MP (SPEC (concl th) (SPEC t1 OR_INTRO_THM2)) th
 *          handle _ => ERR{function = "DISJ2",message = ""};
 *---------------------------------------------------------------------------*)

fun DISJ2 w th = make_thm Count.Disj2 
 (tag th, hyp th, Susp.force mk_disj(w,concl th))
 handle HOL_ERR _ => ERR "DISJ2" "";


(*---------------------------------------------------------------------------
 * Disjunction elimination
 *
 *   A |- t1 \/ t2   ,   A1,t1 |- t   ,   A2,t2 |- t
 *   -----------------------------------------------
 *               A u A1 u A2 |- t
 *
 * fun DISJ_CASES th1 th2 th3 =
 *   let val (t1,t2) = dest_disj(concl th1)
 *       and t = concl th2
 *       val th4 = SPEC t2 (SPEC t1 (SPEC t OR_ELIM_THM))
 *   in
 *   MP (MP (MP th4 th1) (DISCH t1 th2)) (DISCH t2 th3)
 *   end
 *   handle _ => ERR{function = "DISJ_CASES",message = ""};
 *---------------------------------------------------------------------------*)

fun dest_disj tm =
  let val (Rator,N) = dest_comb tm
      val (c,M)     = dest_comb Rator
      val {Name,Thy,...}   = dest_thy_const c
  in 
    if Name="\\/" andalso Thy="bool" then (M,N) else ERR "" ""
  end


fun DISJ_CASES dth ath bth =
  let val _ = Assert (aconv (concl ath) (concl bth)) "" ""
      val (disj1,disj2) = dest_disj (concl dth)
  in
   make_thm Count.DisjCases
       (itlist Tag.merge [tag dth, tag ath, tag bth] empty_tag,
        union_hyp (hyp dth) (union_hyp (disch(disj1, hyp ath))
                                       (disch(disj2, hyp bth))),
        concl ath)
  end
  handle HOL_ERR _ => ERR "DISJ_CASES" "";


(*---------------------------------------------------------------------------
 * NOT introduction
 *
 *     A |- t ==> F
 *     ------------
 *       A |- ~t
 *
 * fun NOT_INTRO th =
 *   let val (t,_) = dest_imp(concl th)
 *   in MP (SPEC t IMP_F) th
 *   end
 *   handle _ => ERR{function = "NOT_INTRO",message = ""};
 *---------------------------------------------------------------------------*)

fun NOT_INTRO th =
  let val (ant,c) = dest_imp(concl th)
  in Assert (c = Susp.force F) "" "";
     make_thm Count.NotIntro  (tag th, hyp th, Susp.force mk_neg ant)
  end
  handle HOL_ERR _ => ERR "NOT_INTRO" "";


(*---------------------------------------------------------------------------
 * Negation elimination
 *
 *       A |- ~ t
 *     --------------
 *      A |- t ==> F
 *
 * fun NOT_ELIM th =
 *   let val (_,t) = dest_comb(concl th)
 *   in MP (SPEC t F_IMP) th
 *   end
 *   handle _ => ERR{function = "NOT_ELIM",message = ""};
 *---------------------------------------------------------------------------*)

local fun dest M = 
        let val (Rator,Rand)  = dest_comb M
        in (dest_thy_const Rator, Rand)
        end
in
fun NOT_ELIM th =
  case with_exn dest (concl th) (thm_err "NOT_ELIM" "")
   of ({Name="~", Thy="bool",...},Rand)
       => make_thm Count.NotElim 
             (tag th, hyp th, Term.prim_mk_imp Rand (Susp.force F))
    | otherwise => ERR "NOT_ELIM" ""
end;
  

(*---------------------------------------------------------------------------*
 * Classical contradiction rule                                              *
 *                                                                           *
 *   A,"~t" |- F                                                             *
 *   --------------                                                          *
 *       A |- t                                                              *
 *                                                                           *
 * fun CCONTR t th =                                                         *
 * let val th1 = RIGHT_BETA(AP_THM NOT_DEF t)                                *
 *     and v   = genvar (--`:bool`--)                                        *
 *     val th2 = EQT_ELIM (ASSUME (--`^t = T`--))                            *
 *     val th3 = SUBST [(th1,v)] (--`^v ==> F`--) (DISCH (--`~ ^t`--) th)    *
 *     val th4 = SUBST[(ASSUME(--`^t = F`--),v)] (--`(^v ==>F)==>F`--)th3    *
 *     val th5 = MP th4 (EQT_ELIM (CONJUNCT2 IMP_CLAUSE4))                   *
 *     val th6 = EQ_MP (SYM(ASSUME (--`^t = F`--))) th5                      *
 * in                                                                        *
 * DISJ_CASES (SPEC t BOOL_CASES_AX) th2 th6                                 *
 * end handle _ => ERR{function = "CCONTR",message = ""}                     *
 *---------------------------------------------------------------------------*)

fun CCONTR w fth =
  (Assert (concl fth = Susp.force F) "CCONTR" "";
   make_thm Count.Ccontr 
       (tag fth, disch(Susp.force mk_neg w, hyp fth), w)
     handle HOL_ERR _ => ERR "CCONTR" "");


(*---------------------------------------------------------------------------*
 * Instantiate free variables in a theorem                                   *
 *---------------------------------------------------------------------------*)

local fun var_compare ((Name,Ty), (s,ty)) = 
          (case String.compare(Name,s)
            of EQUAL => Type.compare (Ty,ty)
             | x => x)
      val init_varset = Binaryset.empty var_compare
in
fun INST [] th = th
  | INST theta th =
     let val asl = hyp th
         val domvars = rev_itlist (fn {redex,...} => fn S => 
                         Binaryset.add(S,dest_var redex)) theta init_varset
                  handle _ => ERR "INST" "non-variable in domain of subst"
         fun in_dom v = Binaryset.member(domvars,v)
     in
      Assert (Lib.all (not o existsFV in_dom) asl)
         "INST" "attempt to substitute for a variable that is free in hyp";
      make_thm Count.Inst (tag th, asl, 
         subst theta (concl th) handle HOL_ERR _ => ERR "INST" "")
     end
end;
     
(*---------------------------------------------------------------------------*
 * Now some derived rules optimized for computations, avoiding most          *
 * of useless type-checking, using pointer equality and delayed              *
 * substitutions. See computeLib for further details.                        *
 *---------------------------------------------------------------------------*)

(*---------------------------------------------------------------------------*
 *    A |- t = (\x.m) n                                                      *
 *  --------------------- Beta                                               *
 *     A |- t = m{x\n}                                                       *
 *                                                                           *
 * Other implementation (less efficient not using explicit subst.):          *
 *   val Beta = Drule.RIGHT_BETA                                             *
 *---------------------------------------------------------------------------*)

fun Beta th =
   let val (lhs, rhs, ty) = Term.dest_eq_ty (concl th)
   in make_thm Count.Beta 
        (empty_tag, hyp th, mk_eq_nocheck ty lhs (Term.lazy_beta_conv rhs))
   end
   handle HOL_ERR _ => ERR "Beta" "";

(*---------------------------------------------------------------------------*
 *    A |- t = (\x.f x)                                                      *
 *  --------------------- x not free in f                                    *
 *     A |- t = f                                                            *
 *                                                                           *
 * Other implementation                                                      *
 *   fun Eta thm = TRANS thm (ETA_CONV (rhs (concl thm)))                    *
 *---------------------------------------------------------------------------*)

fun Eta th =
  let val (lhs, rhs, ty) = Term.dest_eq_ty (concl th)
  in make_thm Count.EtaConv 
       (empty_tag, hyp th, mk_eq_nocheck ty lhs (eta_conv rhs))
  end
  handle HOL_ERR _ => ERR "Eta" "";


(*---------------------------------------------------------------------------*
 * This rule behaves like a tactic: given a goal (reducing the rhs of thm),  *
 * it returns two subgoals (reducing the rhs of th1 and th2), together       *
 * with a validation (mkthm), that builds the normal form of t from the      *
 * normal forms of u and v.                                                  *
 * NB: we do not have to typecheck the rator u, and we replaced the alpha    *
 * conversion test with pointer equality.                                    *
 *                                                                           *
 *                     |- u = u    (th1)        |- v = v    (th2)            *
 *       (thm)             ...                     ...                       *
 *    A |- t = u v    A' |- u = u' (th1')     A'' |- v = v' (th2')           *
 *  ----------------------------------------------------------------         *
 *                A u A' u A'' |- t = u' v'                                  *
 *                                                                           *
 * Could be implemented outside Thm as:                                      *
 *   fun Mk_comb th =                                                        *
 *     let val {Rator,Rand} = dest_comb(rhs (concl th))                      *
 *         fun mka th1 th2 = TRANS th (MK_COMB(th1,th2)) in                  *
 *     (REFL Rator, REFL Rand, mka)                                          *
 *     end                                                                   *
 *---------------------------------------------------------------------------*)

fun Mk_comb thm =
   let val (lhs, rhs, ty) = Term.dest_eq_ty (concl thm)
       val (Rator,Rand) = dest_comb rhs
       fun mkthm th1' th2' =
         let val (lhs1, rhs1, _) = Term.dest_eq_ty (concl th1')
             val _ = Assert (EQ(lhs1,Rator)) "" ""
             val (lhs2, rhs2, _) = Term.dest_eq_ty (concl th2')
             val _ = Assert (EQ(lhs2,Rand)) "" ""
             val (ocls,hyps) = tag_hyp_union [thm, th1', th2']
         in make_thm Count.MkComb
	   (ocls, hyps,mk_eq_nocheck ty lhs (mk_comb(rhs1,rhs2)))
         end
	 handle HOL_ERR _ => ERR "Mk_comb" "";
       val aty = type_of Rand    (* typing! *)
       val th1 = refl_nocheck (aty --> ty) Rator
       val th2 = refl_nocheck aty Rand
   in (th1,th2,mkthm)
   end
   handle HOL_ERR _ => ERR "Mk_comb" "";

(*---------------------------------------------------------------------------*
 *                      |- u = u    (th1)                                    *
 *       (thm)              ...                                              *
 *    A |- t = \x.u    A' |- u = u' (th1')                                   *
 *  ---------------------------------------- x not in FV(A')                 *
 *            A u A' |- t = \x.u'                                            *
 *                                                                           *
 * Could be implemented outside Thm as:                                      *
 *   fun Mk_abs th =                                                         *
 *     let val {Bvar,Body} = dest_abs(rhs (concl th)) in                     *
 *     (Bvar, REFL Body, (fn th1 => TRANS th (ABS Bvar th1)))                *
 *     end                                                                   *
 *---------------------------------------------------------------------------*)

fun Mk_abs thm =
   let val (lhs, rhs, ty) = Term.dest_eq_ty (concl thm)
       val (Bvar,Body) = dest_abs rhs
       fun mkthm th1' =
         let val (lhs1, rhs1, _) = Term.dest_eq_ty (concl th1')
             val _ = Assert (EQ(lhs1,Body)) "" ""
             val _ = Assert (not (var_occursl Bvar (hyp th1'))) "" ""
             val (ocls,hyps) = tag_hyp_union [thm, th1']
         in make_thm Count.Abs
	   (ocls, hyps, mk_eq_nocheck ty lhs (mk_abs(Bvar, rhs1)))
         end
	 handle HOL_ERR _ => ERR "Mk_abs" ""
       val th1 = refl_nocheck (rng ty) Body
   in (Bvar,th1,mkthm)
   end
   handle HOL_ERR _ => ERR "Mk_abs" "";

(*---------------------------------------------------------------------------*
 * Same as SPEC, but without propagating the substitution.  Spec = SPEC.     *
 *---------------------------------------------------------------------------*)

fun Spec t th =
   let val (Rator,Rand) = dest_comb(concl th)
       val {Name,Thy,...} = dest_thy_const Rator
   in
     Assert (Thy="bool" andalso Name="!") "" "";
     make_thm Count.Spec 
        (tag th, hyp th, Term.lazy_beta_conv(mk_comb(Rand,t)))
   end
   handle HOL_ERR _ => ERR "Spec" "";

(*---------------------------------------------------------------------------*
 * Construct a theorem directly and attach the given tag to it.              *
 *---------------------------------------------------------------------------*)

fun mk_oracle_thm tg (asl,c) =
  (Assert (Lib.all is_bool (c::asl)) "mk_oracle_thm"  "not a proposition"
   ; make_thm Count.Oracle (tg,asl,c));


val mk_thm = mk_oracle_thm (Tag.read "MK_THM");


(*---------------------------------------------------------------------------*
 *    The following two are only used in Theory, and are not                 *
 *     externally available.                                                 *
 *---------------------------------------------------------------------------*)

fun mk_axiom_thm (r,c) =
   (Assert (type_of c = bool) "mk_axiom_thm"  "Not a proposition!";
    make_thm Count.Axiom (Tag.ax_tag r, [], c))

fun mk_defn_thm (witness_tag, c) =
   (Assert (type_of c = bool) "mk_defn_thm"  "Not a proposition!";
    make_thm Count.Definition (witness_tag,[],c))



(*---------------------------------------------------------------------------*
 * Fetching theorems from disk. The following parses the "raw" term          *
 * representation found in exported theory files.                            *
 *---------------------------------------------------------------------------*)

datatype lexeme 
   = dot 
   | lamb 
   | lparen 
   | rparen
   | ident of int
   | bvar  of int;

local val numeric = Char.contains "0123456789"
in
fun take_numb ss0 =
  let val (ns, ss1) = Substring.splitl numeric ss0
  in case Int.fromString (Substring.string ns)
      of SOME i => (i,ss1)
       | NONE   => ERR "take_numb" ""
  end
end;

fun lexer ss1 =
  case Substring.getc (Substring.dropl Char.isSpace ss1)
   of NONE         => NONE
    | SOME (c,ss2) =>
       case c
        of #"."  => SOME(dot,   ss2)
         | #"\\" => SOME(lamb,  ss2)
         | #"("  => SOME(lparen,ss2)
         | #")"  => SOME(rparen,ss2)
         | #"%"  => let val (n,ss3) = take_numb ss2 in SOME(ident n, ss3) end
         | #"$"  => let val (n,ss3) = take_numb ss2 in SOME(bvar n,  ss3) end
         |   _   => ERR "raw lexer" "bad character";

fun eat_rparen ss =
  case lexer ss
   of SOME (rparen, ss') => ss'
    |   _ => ERR "eat_rparen" "expected right parenthesis";

fun eat_dot ss =
  case lexer ss
   of SOME (dot, ss') => ss'
    |   _ => ERR "eat_dot" "expected a \".\"";

fun parse_raw table =
 let fun index i = Vector.sub(table,i)
     fun parse (stk,ss) =
      case lexer ss
       of SOME (bvar n,  rst) => (Bv n::stk,rst)
        | SOME (ident n, rst) => (index n::stk,rst)
        | SOME (lparen,  rst) =>
           (case lexer rst
             of SOME (lamb, rst') => parse (glamb (stk,rst'))
              |    _              => parse (parsel (parse (stk,rst))))
        |  _ => (stk,ss)
     and
     parsel (stk,ss) =
        case parse (stk,ss)
         of (h1::h2::t, ss') => (Comb(h2,h1)::t, eat_rparen ss')
          |   _              => ERR "raw.parsel" "impossible"
     and
     glamb(stk,ss) =
      case lexer ss
       of SOME (ident n, rst) =>
            (case parse (stk, eat_dot rst)
              of (h::t,rst1) => (Abs(index n,h)::t, eat_rparen rst1)
               |   _         => ERR "glamb" "impossible")
        | _ => ERR "glamb" "expected an identifier"
 in
  fn [QUOTE s] =>
       (case parse ([], Substring.all s)
         of ([v], _)  => v
          | otherwise => ERR "raw term parser" "parse failed")
   | otherwise => ERR "raw term parser" "expected a quotation"
 end;

val mk_disk_thm  = make_thm Count.Disk;

fun disk_thm vect =
  let val rtp = parse_raw vect
  in
    fn (s,asl,w) => mk_disk_thm(Tag.read_disk_tag s, map rtp asl, rtp w)
  end;

end; (* Thm *)
