(* makebase -- create the signature database from the Moscow ML 
 * library signatures.  PS 1995-11-19, 2000-06-29
 *)

(* The version number inserted in generated files: *)
val version = "<A HREF=\"http://www.cl.cam.ac.uk/Research/HVG/FTP/\">\
       \HOL Kananaskis 0";

(* HOL distribution directory: *)
val HOLpath = "/local/scratch/kxs/utah";

(* Default directory containing the signature files: *)
val libdirDef = Path.concat(HOLpath,"sigobj")

(* Default filename for the resulting help database: *)
val helpfileDef = Path.concat(Path.concat(HOLpath, "help"),"HOL.Help")

(* Default filename for the HOL reference page: *)
val HOLpageDef = Path.concat(Path.concat(HOLpath, "help"),"HOLindex.html")

(* Default filename for the ASCII format database: *)
val txtIndexDef = "index.txt"

(* Default filename for the LaTeX format database: *)
val texIndexDef = "index.tex"

(* Default directory for signatures in HTML format: *)
val htmlDirDef = "htmlsigs"

(* Default filename for the HTML format database: *)
val htmlIndexDef = htmlDirDef ^ "/idIndex.html"

(* Default filename for the LaTeX signatures: *)
val texSigs = "texsigsigs.tex"

(* Signatures not to be included in the help database: *)
val stoplist = [];

(* The background colour in generated HTML files (HTML colour code): *)
val bgcolor = "#fbf2e7";

(* To make the database, sort entries on normalized (lower case) name,
 * lump together equal normalized names in entry lists, and sort these by
 * structure name (except that a Str entry always precedes the others): 
 *)

fun mkbase (entries : Database.entry list) =
    let open Database
	fun compname (e1, e2) = keycompare (getname e1, getname e2)
	val entries = Listsort.sort compname entries
	infix THEN
	fun (comp1 THEN comp2) arg =
	    case comp1 arg of
		EQUAL => comp2 arg
	      | res   => res
	fun kindof Str     = "structure"
	  | kindof (Val _) = "val"
	  | kindof (Typ _) = "type"
	  | kindof (Exc _) = "exn"
	  | kindof (Con _) = "con"
	  | kindof (Term (_, NONE)) = ""
	  | kindof (Term (_, SOME kind)) = kind
	fun kindCompare (e1 as {comp=c1, ...}, e2 as {comp=c2, ...}) = 
	    String.compare(kindof c1, kindof c2)
	fun nameCompare (e1, e2) = 
	    String.compare(getname e1, getname e2)
	fun fileCompare (e1, e2) = String.compare(#file e1, #file e2)
	val entryCompare = kindCompare THEN nameCompare THEN fileCompare
	(* Share strings if result=argument: *)
	fun toLower s = 
	    let val s' = String.map Char.toLower s
	    in if s=s' then s else s' end
	(* Lump together names that differ only in capitalization;
           then sort each lump using entryCompare                   *)
	fun lump [] = []
	  | lump (x1 :: xr) = 
	    let fun mkLump lumpname lump = (toLower (getname lumpname), 
					    Listsort.sort entryCompare lump)
		fun h lump1name lump1 []       = [mkLump lump1name lump1]
		  | h lump1name lump1 (x1::xr) = 
		    if compname(x1, lump1name) = EQUAL then 
			h lump1name (x1 :: lump1) xr
		    else
			mkLump lump1name lump1 :: h x1 [x1] xr
	    in h x1 [x1] xr end
	val lumps = lump entries : (string * entry list) list
	fun mkOrderedTree xs = 
	    let fun h 0 xs = (Empty, xs)
		  | h n xs =
		    let val m = n div 2
			val (t1, (key, value) :: yr) = h m xs
			val (t2, zs)                 = h (n-m-1) yr
		    in (Node(key, value, t1, t2), zs) end
	    in #1 (h (length xs) xs) end
    in
	mkOrderedTree lumps
    end

(*---------------------------------------------------------------------------*)
(* Additional stuff specific to HOL.                                         *)
(*---------------------------------------------------------------------------*)

fun mk_HOLdocfile_entry (dir,s) = 
 let val content = 
      let val (pfx,sfx) = Substring.position "-lc.doc" (Substring.all s)
      in if Substring.size sfx = 0 orelse Substring.size sfx <> 7
         then String.substring(s,0,size s - 4)
         else String.substring(s,0,size s - 7)
      end
 in
    {comp=Database.Term(content, SOME"HOL"), 
     file=Path.concat(dir,s), line=0}
 end

local fun is_adocfile s =
         String.size s>5 andalso 
         String.extract(s, String.size s - 5, NONE) = ".adoc"
      fun is_docfile s =
         String.size s>4 andalso 
         String.extract(s, String.size s - 4, NONE) = ".doc"
in
fun docdir_to_entries path (endpath, entries) =
  let val dir = Path.concat (path, endpath)
      val L0 = Mosml.listDir dir
      val L1 = List.filter is_docfile L0
  in
    List.foldl (fn (s,l) => mk_HOLdocfile_entry (endpath,s)::l) entries L1
  end
end;

val docdirs = 
 let val instr = TextIO.openIn 
                   (Path.concat(HOLpath,"tools/documentation-directories"))
        handle _ => (print "Couldn't open documentation directories file";
                    raise Fail "File not found")
     val wholefile = TextIO.inputAll instr
     val _ = TextIO.closeIn instr
 in
   String.tokens Char.isSpace wholefile
 end

fun dirToBase (sigdir, docdirs, filename) =
    let val doc_entryl = List.foldl (docdir_to_entries HOLpath) [] docdirs
        val res = List.foldl (Parsspec.processfile stoplist sigdir) 
	                     doc_entryl Keepers.keepers
	val _ = print ("\nProcessed " ^ Int.toString (length res) 
		       ^ " entries in total.\n");
	val _ = print ("Building database...\n");
	val db = mkbase res
	val _ = print ("Writing database to file " ^ filename ^ "\n");
    in 
	Database.writebase(filename, db)
    end
    handle exn as OS.SysErr (str, _) => (print(str ^ "\n\n"); raise exn)

fun process (libdir, helpfile, txtIndex, texIndex, htmldir, htmlIndex, HOLpage)
 =
 (print ("Reading signatures in directory " ^ libdir ^ 
        "\nand writing help database in file " ^ helpfile ^ "\n")
 ; dirToBase (libdir, docdirs, helpfile)
 ; print ("\nWriting ASCII signature index in file " ^ txtIndex ^ "\n")
 ; Printbase.printASCIIBase(helpfile, txtIndex)
 ; print ("\nWriting Latex signature index in file " ^ texIndex ^ "\n")
 ; Printbase.printLatexBase(helpfile, texIndex)
 ; print ("\nCreating HTML versions of signature files\n")
 ; Htmlsigs.sigsToHtml 
     version bgcolor stoplist helpfile HOLpath (libdir, htmldir)
 ; print ("\nWriting HTML signature index in file " ^ htmlIndex ^ "\n")
 ; Htmlsigs.printHTMLBase version bgcolor HOLpath (helpfile, htmlIndex)
 ; HOLPage.printHOLPage version bgcolor HOLpath (helpfile, HOLpage)
     (*  ;
     print ("\nWriting LaTeX signature bodies in file " ^ texSigs ^ "\n");
     Texsigs.sigsToLatex stoplist libdir helpfile texSigs
     *)
 )    

val _ = 
    case CommandLine.arguments () of
	[]       => 
	    process (libdirDef, helpfileDef, 
		     txtIndexDef, texIndexDef, 
                     htmlDirDef, htmlIndexDef, HOLpageDef)
      | [libdir] => 
	    process (libdir, helpfileDef, 
		     txtIndexDef, texIndexDef, 
                     htmlDirDef, htmlIndexDef, HOLpageDef)
      | _ => print "Usage: makebase\n"
