(*---------------------------------------------------------------------------
 * Theorems about various wellfounded relations relating to datatypes.
 *---------------------------------------------------------------------------*)
structure WFScript =
struct

open HolKernel basicHol90Lib QLib Parse
     Num_induct pairTools boolTools;

infix ORELSE ORELSEC THEN THENC THENL |->;

val _ = new_theory "WF";

val WF_DEF = primWFTheory.WF_DEF;

(*---------------------------------------------------------------------------
 * Some number lemmas.
 *---------------------------------------------------------------------------*)

val NOT_LESS_0 = prim_recTheory.NOT_LESS_0;
val LESS_THM = prim_recTheory.LESS_THM;
val LESS_SUC_REFL = prim_recTheory.LESS_SUC_REFL;
val INV_SUC_EQ = prim_recTheory.INV_SUC_EQ;
val NOT_SUC = numTheory.NOT_SUC;

(*---------------------------------------------------------------------------
 * No infinite descending chains in 'a. Conceptually simpler (to some)
 * than the original definition, which is solely in terms of
 * predicates (and therefore logically simpler).
 *---------------------------------------------------------------------------*)
val wellfounded_def = 
Q.new_definition
  ("wellfounded_def",
   `wellfounded (R:'a->'a->bool) = ~?f. !n. R (f (n+1)) (f n)`);


(*---------------------------------------------------------------------------
 * First half of showing that the two definitions of wellfounded agree.
 *---------------------------------------------------------------------------*)
val WF_IMP_WELLFOUNDED = Q.prove
`!(R:'a->'a->bool). WF R ==> wellfounded R`
(GEN_TAC THEN CONV_TAC CONTRAPOS_CONV 
 THEN REWRITE_TAC[wellfounded_def,WF_DEF]
 THEN STRIP_TAC THEN NNF_TAC
 THEN Q.EXISTS_TAC`\p:'a. ?n:num. p = f n`
 THEN BETA_TAC THEN CONJ_TAC THENL
  [MAP_EVERY Q.EXISTS_TAC [`(f:num->'a) n`,  `n`] THEN REFL_TAC,
   GEN_TAC THEN DISCH_THEN (CHOOSE_THEN SUBST1_TAC)
    THEN Q.EXISTS_TAC`f(n+1)` THEN ASM_REWRITE_TAC[]
    THEN Q.EXISTS_TAC`n+1` THEN REFL_TAC]);

(*---------------------------------------------------------------------------
 * Second half.
 *---------------------------------------------------------------------------*)
local val RW_TAC     = Rewrite.REWRITE_TAC
      val ASM_RW_TAC = Rewrite.ASM_REWRITE_TAC
      val NUM_CONV   = Num_conv.num_CONV o Term
in
val WELLFOUNDED_IMP_WF = Q.prove
`!(R:'a->'a->bool). wellfounded R ==> WF R`
(RW_TAC[wellfounded_def,WF_DEF,NUM_CONV`1`,arithmeticTheory.ADD_CLAUSES]
  THEN GEN_TAC THEN CONV_TAC CONTRAPOS_CONV THEN RW_TAC[] 
  THEN NNF_TAC THEN REPEAT STRIP_TAC
  THEN Q.EXISTS_TAC`SIMP_REC w (\x. @q. R q x /\ B q)` THEN GEN_TAC
  THEN Q.SUBGOAL_THEN `!n. B(SIMP_REC w (\x. @q. R q x /\ B q) n)`
                      (ASSUME_TAC o SPEC_ALL) 
  THENL [INDUCT_TAC,ALL_TAC] 
  THEN ASM_RW_TAC[prim_recTheory.SIMP_REC_THM]
  THEN BETA_TAC THEN RES_TAC 
  THEN IMP_RES_TAC(BETA_RULE
     (Q.SPEC `\q. R q (SIMP_REC w (\x. @q. R q x /\ B q) n) /\ B q` 
              boolTheory.SELECT_AX)))
end;


val WF_IFF_WELLFOUNDED = Q.store_thm("WF_IFF_WELLFOUNDED",
`!(R:'a->'a->bool). WF R = wellfounded R`,
GEN_TAC THEN EQ_TAC THEN STRIP_TAC
  THENL [IMP_RES_TAC WF_IMP_WELLFOUNDED,
         IMP_RES_TAC WELLFOUNDED_IMP_WF]);



(*---------------------------------------------------------------------------
 * Predecessor and "<" for "num" are wellfounded relations.
 *---------------------------------------------------------------------------*)

val WF_PRED = 
Q.store_thm
("WF_PRED",
  `WF \x y. y = SUC x`,
 REWRITE_TAC[WF_DEF] THEN BETA_TAC THEN GEN_TAC 
  THEN CONV_TAC(CONTRAPOS_CONV THENC NNF_CONV) THEN DISCH_TAC 
  THEN INDUCT_TAC THEN CCONTR_TAC THEN RULE_ASSUM_TAC (REWRITE_RULE[]) 
  THEN RES_TAC THEN RULE_ASSUM_TAC(REWRITE_RULE[INV_SUC_EQ, GSYM NOT_SUC])
  THENL (map FIRST_ASSUM [ACCEPT_TAC, MATCH_MP_TAC]) 
  THEN FILTER_ASM_REWRITE_TAC is_eq [] THEN ASM_REWRITE_TAC[]);


(*----------------------------------------------------------------------------
 * This theorem would be a lot nicer if < was defined as the transitive
 * closure of predecessor.
 *---------------------------------------------------------------------------*)
val WF_LESS = Q.store_thm("WF_LESS", `WF $<`,
REWRITE_TAC[WF_DEF]
 THEN GEN_TAC THEN CONV_TAC CONTRAPOS_CONV
 THEN DISCH_THEN (fn th1 => 
       SUBGOAL_THEN (--`^(concl th1) ==> !i j. j<i ==> ~B j`--)
                    (fn th => MP_TAC (MP th th1)))
 THEN CONV_TAC (DEPTH_CONV NOT_EXISTS_CONV) THEN DISCH_TAC THENL
  [INDUCT_TAC THEN GEN_TAC THEN 
    REWRITE_TAC[NOT_LESS_0,LESS_THM]
    THEN DISCH_THEN (DISJ_CASES_THENL[SUBST1_TAC, ASSUME_TAC])
    THEN STRIP_TAC THEN RES_TAC,
   GEN_TAC THEN FIRST_ASSUM MATCH_MP_TAC 
    THEN Q.EXISTS_TAC`SUC w`
    THEN MATCH_ACCEPT_TAC LESS_SUC_REFL]);


(*---------------------------------------------------------------------------
 * Measure functions are definable as inverse image into (<). Every relation
 * arising from a measure function is wellfounded, which is really great!
 *---------------------------------------------------------------------------*)
val measure_def = 
Q.new_definition
("measure_def",
  `measure:('a->num)->'a->'a->bool = inv_image $<`);


val WF_measure = 
Q.store_thm
("WF_measure",
  `!m:'a->num. WF (measure m)`,
REWRITE_TAC[measure_def] 
 THEN MATCH_MP_TAC primWFTheory.WF_inv_image 
 THEN ACCEPT_TAC WF_LESS);


(*---------------------------------------------------------------------------
 * Wellfoundedness for lists
 *---------------------------------------------------------------------------*)

val LIST_INDUCT_TAC = INDUCT_THEN listTheory.list_INDUCT ASSUME_TAC;

val WF_LIST_PRED = Q.store_thm("WF_LIST_PRED",
`WF \L1 L2. ?h:'a. L2 = CONS h L1`,
REWRITE_TAC[WF_DEF] THEN BETA_TAC THEN GEN_TAC 
  THEN CONV_TAC(CONTRAPOS_CONV THENC NNF_CONV) THEN DISCH_TAC THEN 
  LIST_INDUCT_TAC THENL [ALL_TAC,GEN_TAC] THEN CCONTR_TAC THEN
  RULE_ASSUM_TAC (REWRITE_RULE[])  THEN RES_TAC
  THEN RULE_ASSUM_TAC(REWRITE_RULE[listTheory.NOT_NIL_CONS, 
                                   listTheory.CONS_11])
  THENL (map FIRST_ASSUM [ACCEPT_TAC, MATCH_MP_TAC]) THEN
  FILTER_ASM_REWRITE_TAC is_conj [] THEN ASM_REWRITE_TAC[]);

val _ = export_theory();

end;
