/*************************************************************************
  $Header: /home/jl/phd/rcbdd/RCS/reorder.c,v 1.4 1998/03/02 19:53:27 jl Exp jl $
  FILE:  reorder.c
  DESCR: BDD reordering functions
  AUTH:  Jorn Lind
  DATE:  (C) january 1998
*************************************************************************/
#include <stdlib.h>
#include <time.h>
#include "kernel.h"

typedef struct s_VarBlock
{
   int var;
   int size;
   int blk;
} VarBlock;

static VarBlock *varblocks;
static int       varblocknum;
static int      *block2level;

static int *extroots;  /* FIXME: better idea must exist? */
static int extrootsize;

static int *levellookup;
static int levellookupsize;

extern int bddfreepos;
extern int bddfreenum;
extern int bddproduced;

static int rehashcou;

static void blockdown(int);
static void blockup(int);
static void reorder_win2(void);
static void reorder_win2ite(void);
static void reorder_sift(void);
static void sift_bestpos(int);
static void addref_rec(int);
static void decref_remove(int);
static void decref_rec(int, int);
static int  mark_roots(void);
static void reorder_gbc(int);
static void local_gbc(int);
static int  reorder_makenode(int, int, int);
static void reorder_moveup(int);
static void reorder_movedown(int);
static void reorder_swap(int);
static int  reorder_varup(int);
static int  reorder_vardown(int);
static int  reorder_init(void);
static void reorder_done(void);

#define NODEHASH(lvl,l,h) ((PAIR((l),(h))%levellookupsize) + levellookup[lvl])

/*************************************************************************
  Reordering heuristics
*************************************************************************/

/*=== Reorder using a sliding window of size 2 =========================*/

static void reorder_win2(void)
{
   int lvl;

   for (lvl=0 ; lvl<varblocknum-1 ; lvl++)
   {
      int best = bdd_getnodenum();
      blockdown(varblocks[lvl].blk);
      
      if (best < bdd_getnodenum())
	 blockdown(varblocks[lvl].blk);
   }
}


static void reorder_win2ite(void)
{
   int lvl, lastsize;
   int c=0;
   
   do
   {
      lastsize = bdd_getnodenum();
      printf("WIN2ITE: %d", c);
      
      for (lvl=0 ; lvl<varblocknum-1 ; lvl++)
      {
	 int best = bdd_getnodenum();
	 blockdown(varblocks[lvl].blk);

	 
	 if (best < bdd_getnodenum())
	    blockdown(varblocks[lvl].blk);
      }

      printf(" -> %d\n", bdd_getnodenum());
   }
   while ((bdd_getnodenum()*100)/lastsize < 95  &&  c++<12);

   printf("BEO: ");
   for (c=0 ; c<varblocknum ; c++)
      printf("%d ", varblocks[c].blk);
   printf("\n");
}


/*=== Reorder by sifting =============================================*/

static void reorder_sift(void)
{
   int blk;

   for (blk=0 ; blk<varblocknum ; blk++)
      sift_bestpos(blk);

   printf("BEO: ");
   for (blk=0 ; blk<varblocknum ; blk++)
      printf("%d ", varblocks[blk].blk);
   printf("\n");
}


static void sift_bestpos(int blk)
{
   int best = bdd_getnodenum();
   int bestlevel = block2level[blk];
   int prc = rehashcou;
   
   printf("BSIFT %d/%d\n", blk, bestlevel);
   
      /* Move blk up to top */
   while (block2level[blk]>0 && prc+1>=rehashcou) /* FIXME: not really okay */
   {
      printf("U");
      blockup(blk);
      
      if (bdd_getnodenum() < best)
      {
	 best = bdd_getnodenum();
	 bestlevel = block2level[blk];
	 printf(" -> BSIFT-U %d/%d\n", blk, bestlevel);
      }
   }
   
      /* Move blk down to bottom */
   while (block2level[blk] < varblocknum-1  &&  prc+1>=rehashcou)
   {
      printf("D");
      blockdown(blk);

      if (bdd_getnodenum() < best)
      {
	 best = bdd_getnodenum();
	 bestlevel = block2level[blk];
	 printf(" -> BSIFT-D %d/%d\n", blk, bestlevel);
      }
   }

      /* Move to best pos */
   while (block2level[blk] < bestlevel)
      blockdown(blk);
   while (block2level[blk] > bestlevel)
      blockup(blk);

   printf("-> BSIFT-BEST %d/%d   %d nodes\n", blk, block2level[blk],
	  bdd_getnodenum());
}


/*=== Random reordering (mostly for debugging and test ) =============*/

static void reorder_random(void)
{
   int n;
   
   srand(time(NULL));
   
   for (n=0 ; n<2*varblocknum ; n++)
   {
      int blk = rand() % varblocknum;
      blockdown(blk);
   }

   printf("BEO: ");
   for (n=0 ; n<varblocknum ; n++)
      printf("%d ", varblocks[n].blk);
   printf("\n");
}


/*************************************************************************
  Managing variable blocks
*************************************************************************/

void bdd_clrvarblocks(void)
{
   free(varblocks);
   varblocknum = 0;
}


int bdd_addvarblock(BDD b)
{
   int *v, size, n, insertpos, var;

   if ((n=bdd_scanset(b, &v, &size)) < 0)
      return n;

   for (n=1 ; n<size ; n++)
      if (bddvar2level[v[n-1]]+1 != bddvar2level[v[n]])
      {
	 free(v);
	 return bdd_error(BDD_VARSET);
      }

   var = v[0];
   free(v);
   
   if (varblocknum == 0)
   {
      varblocks = (VarBlock*)malloc(sizeof(VarBlock));
      block2level = (int*)malloc(sizeof(int));
   }
   else
   {
      varblocks=(VarBlock*)realloc(varblocks,sizeof(VarBlock)*(varblocknum+1));
      block2level = (int*)realloc(block2level,sizeof(int)*(varblocknum+1));
   }
   if (varblocks == NULL  ||  block2level == NULL)
      return bdd_error(BDD_MEMORY);

   insertpos = -1;
   for (n=0 ; n<varblocknum && insertpos==-1 ; n++)
   {
      if (bddvar2level[var] < bddvar2level[varblocks[n].var])
      {
	 if (bddvar2level[var]+size-1 >= bddvar2level[varblocks[n].var])
	    return bdd_error(BDD_OVERLAP);

	 insertpos = n;
      }
      else
      if (bddvar2level[var] <= bddvar2level[varblocks[n].var]+
	                       varblocks[n].size-1)
	 return bdd_error(BDD_OVERLAP);
   }

   if (insertpos == -1)
      insertpos = varblocknum;
   else
      for (n=varblocknum ; n>insertpos ; n--)
      {
	 varblocks[n] = varblocks[n-1];
	 block2level[varblocks[n].blk] = n;
      }

   block2level[varblocknum] = insertpos;
   varblocks[insertpos].var = var;
   varblocks[insertpos].size = size;
   varblocks[insertpos].blk = varblocknum;
   varblocknum++;

   return 0;
}


/*=== Pushing one block down or up =====================================*/

/* FIXME: works only if the two blocks are exactly adjacent */

static void blockdown(int blk)
{
   VarBlock tmp;
   int lvl;
   int n;

   if (blk < 0  ||  blk > varblocknum-1)
      return;

   lvl = block2level[blk];
   if (lvl >= varblocknum-1)
      return;
#if 0
   printf("1-BD%d  : ", lvl);
   for (n=0 ; n<varblocks[lvl].size ; n++)
      printf("%d/%d ", bddvar2level[varblocks[lvl].var+n], varblocks[lvl].var+n);
   printf("\n");
   printf("1-BD%d: ", lvl+1);
   for (n=0 ; n<varblocks[lvl+1].size ; n++)
      printf("%d/%d ", bddvar2level[varblocks[lvl+1].var+n], varblocks[lvl+1].var+n);
   printf("\n");
#endif
   
   while (bddvar2level[varblocks[lvl].var] <
             bddvar2level[varblocks[lvl+1].var + varblocks[lvl+1].size-1])
   {
      for (n=0 ; n<varblocks[lvl].size ; n++)
      {
	 if (bddvar2level[varblocks[lvl].var+n] >=
	        bddvar2level[varblocks[lvl+1].var]-1  &&
	     bddvar2level[varblocks[lvl].var+n] <
	        bddvar2level[varblocks[lvl+1].var+varblocks[lvl+1].size-1])
	 {
	    reorder_vardown(varblocks[lvl].var+n);
	 }
      }
   }

   tmp = varblocks[lvl];
   varblocks[lvl] = varblocks[lvl+1];
   varblocks[lvl+1] = tmp;

#if 0
   printf("2-BD%d  : ", lvl);
   for (n=0 ; n<varblocks[lvl].size ; n++)
      printf("%d/%d ", bddvar2level[varblocks[lvl].var+n], varblocks[lvl].var+n);
   printf("\n");
   printf("2-BD%d: ", lvl+1);
   for (n=0 ; n<varblocks[lvl+1].size ; n++)
      printf("%d/%d ", bddvar2level[varblocks[lvl+1].var+n], varblocks[lvl+1].var+n);
   printf("\n");
#endif
   
   block2level[ varblocks[lvl].blk ] = lvl;
   block2level[ varblocks[lvl+1].blk ] = lvl+1;
}


static void blockup(int blk)
{
   int lvl;
   
   if (blk < 0  ||  blk > varblocknum-1)
      return;

   lvl = block2level[blk];
   if (lvl == 0)
      return;
   blockdown(varblocks[lvl-1].blk);
}


/*************************************************************************
  Kernel reordering routines
*************************************************************************/

/*=== Garbage collection for reordering ================================*/

static void decref_remove(int r)
{
   BddNode *node = &bddnodes[r];
   unsigned int hash = NODEHASH(node->level & MARKHIDE, node->high, node->low);
   int n = bddnodes[hash].hash;
   int l = -1;
   int f=0;
   
   while (n != 0)
   {
      if (n == r)
      {
	 if (l == -1)
	    bddnodes[hash].hash = node->next;
	 else
	    bddnodes[l].next = node->next;
	 n = 0;

	 printf("X%d ", r); /* FIXME */
	 node->low = -1;
	 node->next = bddfreepos;
	 bddfreepos = r;
	 f=1;
      }
      else
      {
	 l = n;
	 n = bddnodes[n].next;
      }
   }
}

static void decref_rec(int r, int remove)
{
   if (r < 2)
      return;

   DECREF(r);
   if (bddnodes[r].refcou == 0)
   {
      bddfreenum++;

      if (remove)
	 decref_remove(r);
      
      decref_rec(bddnodes[r].low, remove);
      decref_rec(bddnodes[r].high, remove);
   }
}


static void addref_rec(int r)
{
   if (r < 2)
      return;
   
   if (bddnodes[r].refcou == 0)
   {
      bddfreenum--;
      addref_rec(bddnodes[r].low);
      addref_rec(bddnodes[r].high);
   }
   INCREF(r);
}


static int mark_roots(void)
{
   int n;

   for (n=2,extrootsize=0 ; n<bddnodesize ; n++)
      if (bddnodes[n].refcou > 0)
      {
	 SETMARK(&bddnodes[n]);
	 extrootsize++;
      }

   if ((extroots=(int*)(malloc(sizeof(int)*extrootsize))) == NULL)
      return bdd_error(BDD_MEMORY);

   for (n=2,extrootsize=0 ; n<bddnodesize ; n++)
   {
      if (MARKED(&bddnodes[n]))
      {
	 UNMARK(&bddnodes[n]);
	 
	 extroots[extrootsize++] = n;
	 addref_rec(bddnodes[n].low);
	 addref_rec(bddnodes[n].high);
      }
   }

   bddnodes[0].hash = 0;
   bddnodes[1].hash = 0;
   
   return 0;
}


static void reorder_gbc(int dozero)
{
   int n;

   bddfreepos = 0;
   bddfreenum = 0;

   if (dozero)
      for (n=2 ; n<bddnodesize ; n++)
	 bddnodes[n].hash = 0;
   
   for (n=bddnodesize-1 ; n>2 ; n--)
   {
      register BddNode *node = &bddnodes[n];

      if (node->refcou > 0)
      {
	 register unsigned int hash;

	 hash = NODEHASH(node->level & MARKHIDE, node->low, node->high);
	 node->next = bddnodes[hash].hash;
	 bddnodes[hash].hash = n;
      }
      else
      {
	 node->low = -1;
	 node->next = bddfreepos;
	 bddfreepos = n;
	 bddfreenum++;
      }
   }

   rehashcou++;
}


static void local_gbc(int level)
{
   int n;
   int ll0 = levellookup[level];
   
   for (n=0 ; n<levellookupsize ; n++)
   {
      register unsigned int hash = n+ll0;
      register int r = bddnodes[hash].hash;
      bddnodes[hash].hash = 0;

      while (r != 0)
      {
	 register BddNode *node = &bddnodes[r];
	 register int next = node->next;

	 UNMARK(node);
	 
	 if (node->refcou > 0)
	 {
	    node->next = bddnodes[hash].hash;
	    bddnodes[hash].hash = r;
	 }
	 else
	 {
	    node->low = -1;
	    node->next = bddfreepos;
	    bddfreepos = r;
	 }

	 r = next;
      }
   }
}


/*=== Unique table handling for reordering =============================*/

static void rehash_all(void)
{
   int n;
   
   levellookupsize = bddnodesize / bddvarnum;
   for (n=0 ; n<bddvarnum ; n++)
      levellookup[n] = n*levellookupsize;

   reorder_gbc(0);
}


static int reorder_makenode(int level, int low, int high)
{
   register BddNode *node;
   register unsigned int hash;
   register int res;

      /* check whether childs are equal */
   if (low == high)
      return low;

      /* Try to find an existing node of this kind */
   hash = NODEHASH(level, low, high);
   res = bddnodes[hash].hash;
      
   while(res != 0)
   {
      if (/*bddnodes[res].level == level && FIXME */
	  bddnodes[res].low == low  &&
	  bddnodes[res].high == high)
	 return res;

      res = bddnodes[res].next;
   }
   
      /* No existing node -> build one */

      /* Any free nodes to use ? */
   if (bddfreepos == 0)
   {
      if (bdderrorcond)
	 return 0;
      
         /* Try to allocate more nodes */
      bdd_noderesize(rehash_all);
      hash = NODEHASH(level, low, high);
      
         /* Panic if that is not possible */
      if (bddfreepos == 0)
      {
	 bdderrorcond = abs(BDD_NODENUM);
	 return 0;
      }
   }

      /* Build new node */
   res = bddfreepos;
   bddfreepos = bddnodes[bddfreepos].next;
   bddproduced++;
   
   node = &bddnodes[res];
   node->level = level;
   node->low = low;
   node->high = high;

      /* Insert node */
   node->next = bddnodes[hash].hash;
   bddnodes[hash].hash = res;

   return res;
}


/*=== Swapping two adjacent variables ==================================*/

static void reorder_moveup(int level)
{
   register int n;
   int ll0 = levellookup[level-1];
   int ll1 = levellookup[level];
   
   for (n=0 ; n<levellookupsize ; n++)
   {
      unsigned int hash = n + ll0;
      register int r = bddnodes[n + ll1].hash;
      bddnodes[n + ll1].hash = 0;

      while (r != 0)
      {
	 register BddNode *node = &bddnodes[r];
	 register int next = node->next;

	 node->level--;
	 SETMARK(node);
	 
	 node->next = bddnodes[hash].hash;
	 bddnodes[hash].hash = r;

	 r = next;
      }
   }
}


/* Move live nodes at level to level+1 (in some cases).
*/
static void reorder_movedown(int level)
{
   register int n;
   int level1 = level+1;
   int ll0 = levellookup[level];
   int ll1 = levellookup[level+1];
   
   for (n=0 ; n<levellookupsize ; n++)
   {
      register int r;

      r = bddnodes[n + ll0].hash;
      bddnodes[n + ll0].hash = 0;

      while (r != 0)
      {
	 register BddNode *node = &bddnodes[r];
	 register unsigned int hash = n +ll0;
	 register int next=node->next;
	 
	 if (!MARKED(node))
	 {
	    if ((bddnodes[node->low].level & MARKHIDE) > level1  &&
		(bddnodes[node->high].level & MARKHIDE) > level1)
	    {
	       node->level++;
	       hash = n + ll1;
	    }
	 }

	 node->next = bddnodes[hash].hash;
	 bddnodes[hash].hash = r;

	 r = next;
      }
   }
}


static void reorder_swap(int level)
{
   int prc;
   
   do
   {
      int n, ll0=levellookup[level];
      prc = rehashcou;
   
      for (n=0 ; n<levellookupsize && rehashcou==prc ; n++)
      {
	 register unsigned int hash;
	 register int r = bddnodes[n + ll0].hash;
	 bddnodes[n + ll0].hash = 0;
	 
	 while (r != 0  &&  rehashcou==prc)
	 {
	    register BddNode *node = &bddnodes[r];
	    register int next=node->next;
	    
	    if (node->refcou > 0  && !MARKED(node))
	    {
	       register f0 = node->low;
	       register f1 = node->high;
	       register f00, f01, f10, f11;
	       
	       if ((bddnodes[f0].level & MARKHIDE) == level)
	       {
		  f00 = bddnodes[f0].low;
		  f01 = bddnodes[f0].high;
	       }
	       else
		  f00 = f01 = f0;
	       
	       if ((bddnodes[f1].level & MARKHIDE) == level)
	       {
		  f10 = bddnodes[f1].low;
		  f11 = bddnodes[f1].high;
	       }
	       else
		  f10 = f11 = f1;
	       
	       addref_rec(f0 = reorder_makenode(level+1, f00, f10));
	       addref_rec(f1 = reorder_makenode(level+1, f01, f11));
	       node = &bddnodes[r];  /* Might change in makenode */
	       
	       decref_rec(node->low,
			  (bddnodes[node->low].level & MARKHIDE)!=level);
	       decref_rec(node->high,
			  (bddnodes[node->high].level & MARKHIDE)!=level);
	       
	       SETMARK(node);
	       node->low = f0;
	       node->high = f1;
	    }

	    if (prc == rehashcou)
	    {
	       hash = NODEHASH(node->level & MARKHIDE, node->low, node->high);
	       node->next = bddnodes[hash].hash;
	       bddnodes[hash].hash = r;
	    }
	    
	    r = next;
	 }
      }
   }
   while (prc != rehashcou);
}


static int reorder_varup(int var)
{
   if (var < 0  ||  var >= bddvarnum)
      return bdd_error(BDD_VAR);
   if (bddvar2level[var] == 0)
      return 0;
   return reorder_vardown( bddlevel2var[bddvar2level[var]-1]);
}


#if 0
static void sanity(void)
{
   int l,n;
   int c=0;
   
   for (n=0 ; n<bddnodesize ; n++)
   {
      if (MARKED(&bddnodes[n]))
	 printf("MARK\n");
   }

   for (n=2 ; n<bddnodesize ; n++)
   {
      if (bddnodes[n].low != -1)
      {
	 if (bddnodes[n].level >= bddnodes[bddnodes[n].low].level)
	    printf("LOW\n");
	 if (bddnodes[n].level >= bddnodes[bddnodes[n].high].level)
	    printf("HIGH\n");
      }
      else
	 c++;
   }

   if (c != bddfreenum)
      printf("FREE C%d F%d\n", c, bddfreenum);
   
   for (l=0 ; l<bddvarnum ; l++)
   {
      for (n=0 ; n<levellookupsize ; n++)
      {
	 int r = bddnodes[n+levellookup[l]].hash;

	 while (r != 0)
	 {
	    if (bddnodes[r].level != l)
	       printf("LEVEL\n");
	    r = bddnodes[r].next;
	 }
      }
   }
}
#endif


static int reorder_vardown(int var)
{
   int level;
   int n;

   if (var < 0  ||  var >= bddvarnum)
      return bdd_error(BDD_VAR);
   if (bddvar2level[var] >= bddvarnum-1)
      return 0;

   level = bddvar2level[var];
   
      /* Update those in 'level+1' */
   reorder_moveup(level+1);
   
      /* Update those in 'level' with both childs.level>level+1 */
   reorder_movedown(level);
   
      /* Update the rest in 'level' */
   reorder_swap(level);
   
      /* Garbage collect and remove marks on this level */
   local_gbc(level);
   
      /* Swap the var<->level tables */
   n = bddlevel2var[level];
   bddlevel2var[level] = bddlevel2var[level+1];
   bddlevel2var[level+1] = n;

   n = bddvar2level[var];
   bddvar2level[var] = bddvar2level[ bddlevel2var[level] ];
   bddvar2level[ bddlevel2var[level] ] = n;

      /* Update all rename pairs */
   bdd_pairs_vardown(level);

   return 0;
}


/*************************************************************************
  User reordering interface
*************************************************************************/

static int reorder_init(void)
{
   int n;

   if ((levellookup=(int*)(malloc(sizeof(int)*bddvarnum))) == NULL)
      return -1;
   levellookupsize = bddnodesize / bddvarnum;
   for (n=0 ; n<bddvarnum ; n++)
      levellookup[n] = n*levellookupsize;
   
   if (mark_roots() < 0)
      return -1;
   reorder_gbc(1);

   return 0;
}


static void reorder_done(void)
{
   int n;
   
   for (n=0 ; n<extrootsize ; n++)
      SETMARK(&bddnodes[extroots[n]]);
   for (n=2 ; n<bddnodesize ; n++)
   {
      if (MARKED(&bddnodes[n]))
	 UNMARK(&bddnodes[n]);
      else
	 bddnodes[n].refcou = 0;
   }
   
   free(extroots);
   free(levellookup);
   bdderrorcond = 0;
   bdd_gbc();
}


/*
NAME    {* bdd\_reorder *}
SECTION {* kernel *}
SHORT   {* start dynamic reordering *}
PROTO   {* void bdd_reorder(int method) *}
DESCR   {* This function initiates dynamic reordering using the heuristic
           defined by {\tt method}, which may be one of the following
	   \begin{description}
	     \item {\tt BDD\_REORDER\_WIN2}\\
	       Reordering using a sliding window of size 2. This algorithm
	       swaps two adjacent variable blocks and if this results in
	       more nodes then the two blocks are swapped back again.
	       Otherwise the result is kept in the variable order. This is
	       then repeated for all variable blocks. Usually a fast and
	       efficient method.
	     \item {\tt BDD\_REORDER\_WIN2ITE}\\
	       The same as above but the process is repeated until no further
	       progress is done.
	     \item {\tt BDD\_REORDER\_SIFT}\\
	       Reordering where each block is moved through all possible
	       positions. The best of these is then used as the new position.
	       Potentially a very slow but good method.
	     \item {\tt BDD\_REORDER\_SIFTITE}\\
	       The same as above but the process is repeated until no further
	       progress is done. Can be extremely slow.
	     \item {\tt BDD\_REORDER\_RANDOM}\\
	       Mostly used for debugging purpose, but maybe usefull for others.
	       Selects a random position for each variable.
	   \end{description}
	   *}
ALSO    {* bdd\_addvarblock, bdd\_clrvarblocks *}
*/
void bdd_reorder(int method)
{
   if (reorder_init() < 0)
      return;

   switch(method)
   {
   case BDD_REORDER_WIN2:
      reorder_win2();
      break;
   case BDD_REORDER_WIN2ITE:
      reorder_win2ite();
      break;
   case BDD_REORDER_SIFT:
      reorder_sift();
      break;
   case BDD_REORDER_SIFTITE:
      /*reorder_siftite(); FIXME */
      break;
   case BDD_REORDER_RANDOM:
      reorder_random();
      break;
   }

   reorder_done();
}


/* EOF */
