(*---------------------------------------------------------------------------*
 * This example defines the abstract syntax for a simple ML-like language,   *
 * and a simple mutually-recursive function for computing the variables      *
 * in a program.  It exercises Elsa Gunter's type definition package for     *
 * mutually recursive concrete types.                                        *
 *                                                                           *
 * The example is also a demonstration of how Holmake works. Just invoke     *
 *                                                                           *
 *     Holmake                                                               *
 *                                                                           *
 * and wait. When Holmake is done, a compiled theory corresponding to this   *
 * file is found in MLSyntaxTheory.u{i,o}. It can be loaded into an          *
 * interactive session by                                                    *
 *                                                                           *
 *    load"MLSyntaxTheory";                                                  *
 *                                                                           *
 * Loading the theory can take a little while - about 5 seconds on my        *
 * machine.                                                                  *
 *                                                                           *
 * If you are working interactively, i.e., you don't want to pay any         *
 * attention to this Holmake stuff, do the following:                        *
 *                                                                           *
    app load ["Holdatatype","mutrecLib", "stringTheory", "setTheory"];       *
 *                                                                           *
 * and then proceed with cut-and-paste.                                      *  
 *---------------------------------------------------------------------------*)

open Datatype Theory;

(*---------------------------------------------------------------------------*
 * The following is a sneaky way to bring the two theories into the current  *
 * signature (when in batch mode that is: when working interactively, the    *
 * above invocation ("app load ....") is sufficient.                         *
 *---------------------------------------------------------------------------*)
local open stringTheory setTheory in end;


val _ = new_theory"MLSyntax";

(*---------------------------------------------------------------------------*
 * First, we define a type of variables.                                     *
 *---------------------------------------------------------------------------*)

val _ = Hol_datatype`var = VAR of string`;

val var_ty = Parse.Type`:var`;


(*--------------------------GRAMMAR------------------------------------------*
 *                                                                           *
 *      atexp ::= <var>                                                      *
 *              | "let" <dec> "in" <exp> "end"                               *
 *                                                                           *
 *      exp   ::= <atexp>                                                    *
 *              | <exp> <atexp>                                              *
 *              | "fn" <match>                                               *
 *                                                                           *
 *      match ::= <rule>                                                     *
 *              | <rule> "|" <match>                                         *
 *                                                                           *
 *      rule  ::= <pat> "=>" <exp>                                           *
 *                                                                           *
 *      dec   ::= "val" <valbind>                                            *
 *              | "local" <dec> "in" <dec> "end"                             *
 *              | <dec> ";" <dec>                                            *
 *                                                                           *
 *    valbind ::= <pat> "=" <exp>                                            *
 *              | <pat> "=" <exp> "and" <valbind>                            *
 *              | "valrec" <valbind>                                         *
 *                                                                           * 
 *      pat   ::= "_" (* wildcard *)                                         *
 *              | <var>                                                      *
 *                                                                           *
 *---------------------------------------------------------------------------*)

(*---------------------------------------------------------------------------*
 * Eventually, we'd like to give input that looked something like the above. *
 * In the meantime, we have to make do with the following description.       *
 * Sigh.                                                                     *
 *---------------------------------------------------------------------------*)
local open TypeInfo
in
val syntax_spec =
[{type_name = "atexp",
  constructors =
      [{name = "var_exp", arg_info = [existing var_ty]},
       {name = "let_exp", arg_info = [being_defined "dec",
                                      being_defined "exp"]}]},
 {type_name = "exp",
  constructors =
      [{name = "aexp", arg_info = [being_defined "atexp"]},
       {name = "app_exp", arg_info = [being_defined "exp",
				     being_defined "atexp"]},
       {name = "fn_exp", arg_info = [being_defined "match"]}]},
 {type_name = "match",
  constructors =
      [{name = "match", arg_info = [being_defined "rule"]},
       {name = "match_list", arg_info = [being_defined "rule",
                                         being_defined "match"]}]},
 {type_name = "rule",
  constructors =
      [{name = "rule", arg_info = [being_defined "pat",
                                   being_defined "exp"]}]},
 {type_name = "dec",
  constructors = 
      [{name = "val_dec", arg_info = [being_defined "valbind"]},
       {name = "local_dec", arg_info = [being_defined "dec",
                                        being_defined "dec"]},
       {name = "seq_dec", arg_info = [being_defined "dec",
                                      being_defined "dec"]}]},
 {type_name = "valbind",
  constructors =
      [{name = "bind", arg_info = [being_defined "pat",
                                   being_defined "exp"]},
       {name = "bind_list", arg_info = [being_defined "pat",
                                        being_defined "exp",
                                        being_defined "valbind"]},
       {name = "rec_bind", arg_info = [being_defined "valbind"]}]},
 {type_name = "pat",
  constructors =
      [{name = "wild_pat", arg_info = []},
       {name = "var_pat", arg_info = [existing var_ty]}]}]
end;


(*---------------------------------------------------------------------------*
 * Now define the type.                                                      *
 *---------------------------------------------------------------------------*)
val {Cases = syntax_cases,
     Constructors_Distinct = syntax_constructors_distinct, 
     Constructors_One_One = syntax_constructors_one_one,
     New_Ty_Existence = syntax_existence_thm, 
     New_Ty_Uniqueness = syntax_uniqueness_thm,
     New_Ty_Induct = syntax_induction_thm,
     Argument_Extraction_Defs} 
 = 
   Count.apply mutrecLib.define_type syntax_spec;


(*---------------------------------------------------------------------------*
 * A simple function for finding the set of variables in an expression of    *
 * the defined syntax.                                                       *
 *---------------------------------------------------------------------------*)

val vars_thm = mutrecLib.define_mutual_functions
{name = "vars_thm", rec_axiom = syntax_existence_thm, fixities = NONE,
 def = Parse.Term
 `(atexpV (var_exp v)       = {v})                                      /\
  (atexpV (let_exp d e)     = (decV d) UNION (expV e))                  /\
  (expV (aexp a)            = atexpV a)                                 /\
  (expV (app_exp e a)       = (expV e) UNION (atexpV a))                /\
  (expV (fn_exp m)          = matchV m)                                 /\
  (matchV (match r)         = ruleV r)                                  /\
  (matchV (match_list r ms) = (ruleV r) UNION (matchV ms))              /\
  (ruleV (rule p e)         = (patV p) UNION (expV e))                  /\
  (decV (val_dec b)         = vbindV b)                                 /\
  (decV (local_dec d1 d2)   = (decV d1) UNION (decV d2))                /\
  (decV (seq_dec d1 d2)     = (decV d1) UNION (decV d2))                /\
  (vbindV (bind p e)        = (patV p) UNION (expV e))                  /\
  (vbindV (bind_list p e B) = (patV p) UNION (expV e) UNION (vbindV B)) /\
  (vbindV (rec_bind vb)     = vbindV vb)                                /\
  (patV wild_pat            = {})                                       /\
  (patV (var_pat v)         = {v})`};


val _ = export_theory();
