(* ===================================================================== *)
(* FILE          : induct_then.sml                                       *)
(* DESCRIPTION   : General induction tactic for recursive types.         *)
(*                 Translated from hol88.                                *)
(*                                                                       *)
(* AUTHOR        : (c) Tom Melham, University of Cambridge               *)
(* DATE          : 87.08.23                                              *)
(* REVISED       : 90.06.02                                              *)
(* TRANSLATOR    : Konrad Slind, University of Calgary                   *)
(* DATE          : September 11, 1991                                    *)
(* ===================================================================== *)


structure Induct_then : Induct_then_sig =
struct
open Base_logic Term_io.Parse;
open Thm Term Dsyntax
open Conv;
  infix THENC;
  infix ORELSEC;
open Drule;
open Tactical;
  infix THEN;
  infix THENL;
  infix ORELSE;
open Thm_cont;
  infix THEN_TCL;
  infix ORELSE_TCL;

fun INDUCT_THEN_ERR{function,message} =
      HOL_ERR{origin_structure = "Induct_then",
	      origin_function = function,
	      message = message}

(* ---------------------------------------------------------------------*)
(* Internal function: 							*)
(*									*)
(* BETAS "f" tm : returns a conversion that, when applied to a term with*)
(*		 the same structure as the input term tm, will do a	*)
(*		 beta reduction at all top-level subterms of tm which	*)
(*		 are of the form "f <arg>", for some argument <arg>.	*)
(*									*)
(* ---------------------------------------------------------------------*)

fun BETAS fnn body = 
   if ((is_var body) orelse (is_const body)) 
   then REFL 
   else if (is_abs body)
        then ABS_CONV (BETAS fnn (#Body(dest_abs body))) 
        else let val {Rator,Rand} = dest_comb body 
             in 
             if (Rator = fnn) 
             then BETA_CONV 
             else let val cnv1 = BETAS fnn Rator
                      and cnv2 = BETAS fnn Rand
                      fun f {Rator,Rand} = (cnv1 Rator, cnv2 Rand)
                  in
	          (MK_COMB o (f o dest_comb))
                  end
             end;

(* ---------------------------------------------------------------------*)
(* Internal function: GTAC						*)
(*									*)
(*   !x. tm[x]  							*)
(*  ------------  GTAC "y"   (primes the "y" if necessary).		*)
(*     tm[y]								*)
(*									*)
(* NB: the x is always a genvar, so optimized for this case.		*)
(* ---------------------------------------------------------------------*)

fun GTAC y (A,g) = 
   let val {Bvar,Body} = dest_forall g 
       and y' = variant (free_varsl (g::A)) y
   in
   ([(A,subst[{redex = Bvar, residue = y'}]Body)],
    fn [th] => GEN Bvar (INST [{redex = y', residue = Bvar}] th))
   end;

(* ---------------------------------------------------------------------*)
(* Internal function: TACF						*)
(*									*)
(* TACF is used to generate the subgoals for each case in an inductive 	*)
(* proof.  The argument tm is formula which states one generalized	*)
(* case in the induction. For example, the induction theorem for num is:*)
(*									*)
(*   |- !P. P 0 /\ (!n. P n ==> P(SUC n)) ==> !n. P n			*)
(*									*)
(* In this case, the argument tm will be one of:			*)
(*									*)
(*   1:  "P 0"   or   2: !n. P n ==> P(SUC n)				*)
(*   									*)
(* TACF applied to each these terms to construct a parameterized tactic *)
(* which will be used to further break these terms into subgoals.  The  *)
(* resulting tactic takes a variable name x and a user supplied theorem *)
(* continuation ttac.  For a base case, like case 1 above, the resulting*)
(* tactic just throws these parameters away and passes the goal on 	*)
(* unchanged (i.e. \x ttac. ALL_TAC).  For a step case, like case 2, the*)
(* tactic applies GTAC x as many times as required.  It then strips off *)
(* the induction hypotheses and applies ttac to each one.  For example, *)
(* if tac is the tactic generated by:					*)
(*									*)
(*    TACF "!n. P n ==> P(SUC n)" "x:num" ASSUME_TAC			*)
(*									*)
(* then applying tac to the goal A,"!n. P[n] ==> P[SUC n] has the same 	*)
(* effect as applying:							*)
(*									*)
(*    GTAC "x:num" THEN DISCH_THEN ASSUME_TAC				*)
(*									*)
(* TACF is a strictly local function, used only to define TACS, below.	*)
(* ---------------------------------------------------------------------*)
local
fun ctacs tm = 
   if (is_conj tm) 
   then let val tac2 = ctacs (#conj2(dest_conj tm)) 
        in
        fn ttac => CONJUNCTS_THEN2 ttac (tac2 ttac) 
        end
   else fn ttac => ttac
in
fun TACF tm =
   let val (vs,body) = strip_forall tm 
   in
   if (is_imp body) 
   then let val TTAC = ctacs (#ant(dest_imp body)) 
        in
        fn x => fn ttac => 
           MAP_EVERY (GTAC o (K x)) vs THEN 
           DISCH_THEN (TTAC ttac) 
        end
   else
   fn x => fn ttac => Tactical.ALL_TAC
   end
end;

(* ---------------------------------------------------------------------*)
(* Internal function: TACS						*)
(*									*)
(* TACS uses TACF to generate a parameterized list of tactics, one for  *)
(* each conjunct in the hypothesis of an induction theorem.		*)
(*									*)
(* For example, if tm is the hypothesis of the induction thoerem for the*)
(* natural numbers---i.e. if:						*)
(*									*)
(*   tm = "P 0 /\ (!n. P n ==> P(SUC n))"				*)
(*									*)
(* then TACS tm yields the paremterized list of tactics:		*)
(*									*)
(*   \x ttac. [TACF "P 0" x ttac; TACF "!n. P n ==> P(SUC n)" x ttac]   *)
(*									*)
(* TACS is a strictly local function, used only in INDUCT_THEN.		*)
(* ---------------------------------------------------------------------*)

fun f {conj1,conj2} = (TACF conj1, TACS conj2)
and
    TACS tm = 
      let val (cf,csf) = f(dest_conj tm) 
                         handle _ => (TACF tm, K(K[]))
      in
      fn x => fn ttac => (cf x ttac)::(csf x ttac)
      end;

(* ---------------------------------------------------------------------*)
(* Internal function: GOALS						*)
(*									*)
(* GOALS generates the subgoals (and proof functions) for all the cases *)
(* in an induction. The argument A is the common assumption list for all*)
(* the goals, and tacs is a list of tactics used to generate subgoals 	*)
(* from these goals.							*)
(*									*)
(* GOALS is a strictly local function, used only in INDUCT_THEN.	*)
(* ---------------------------------------------------------------------*)
fun GOALS A [] tm = raise INDUCT_THEN_ERR{function = "GOALS",
					  message = "empty lsit"}
  | GOALS A [t] tm = 
      let val (sg,pf) = t (A,tm)
      in 
      ([sg],[pf])
      end
  | GOALS A (h::t) tm = 
      let val {conj1,conj2} = dest_conj tm
          val (sgs,pfs) = GOALS A t conj2 
          val (sg,pf) = h (A,conj1)
      in
      ((sg::sgs),(pf::pfs))
      end;

(* --------------------------------------------------------------------- *)
(* Internal function: GALPH						*)
(* 									*)
(* GALPH "!x1 ... xn. A ==> B":   alpha-converts the x's to genvars.	*)
(* --------------------------------------------------------------------- *)
local
fun rule v = 
   let val gv = genvar(type_of v) 
   in
   fn eq => let val th = FORALL_EQ v eq 
            in 
            TRANS th (GEN_ALPHA_CONV gv (rhs(concl th)))
            end
   end
in
fun GALPH tm =
   let val (vs,hy) = strip_forall tm 
   in
   if (is_imp hy)
   then itlist rule vs (REFL hy)
   else REFL tm
   end
end;

(* --------------------------------------------------------------------- *)
(* Internal function: GALPHA						*)
(* 									*)
(* Applies the conversion GALPH to each conjunct in a sequence.		*)
(* --------------------------------------------------------------------- *)

local 
val AND = --`$/\`--
in
fun f {conj1,conj2} = (GALPH conj1, GALPHA conj2)
and GALPHA tm = 
   let val (c,cs) = f(dest_conj tm) 
   in
   MK_COMB(AP_TERM AND c, cs)
   end
   handle _ => GALPH tm
end;

(* --------------------------------------------------------------------- *)
(* Internal function: mapshape						*)
(* 									*)
(* Applies the functions in fl to argument lists obtained by splitting   *)
(* the list l into sublists of lengths given by nl.			*)
(* --------------------------------------------------------------------- *)

fun mapshape [] _ _ =  [] |
    mapshape (n1::nums) (f1::funcs) args = 
       let val (f1_args,args') = split_after n1 args
       in 
       (f1 f1_args)::(mapshape nums funcs args')
       end;
           
(* --------------------------------------------------------------------- *)
(* INDUCT_THEN : general induction tactic for concrete recursive types.	*)
(* --------------------------------------------------------------------- *)
local
val bool = genvar (==`:bool`==)
in
fun INDUCT_THEN th = 
   let val {Bvar,Body} = dest_forall(concl th)
       val {ant = hy, ...} = dest_imp Body
       val bconv = BETAS Bvar hy 
       and tacsf = TACS hy
       val v = genvar (type_of Bvar) 
       val eta_th = CONV_RULE(RAND_CONV ETA_CONV) (UNDISCH(SPEC v th))
       val ([asm],con) = dest_thm eta_th
       val dis = DISCH asm eta_th
       val ind = GEN v (SUBST [{var = bool, thm = GALPHA asm}]
                              (mk_imp{ant = bool, conseq = con}) dis)
   in
   fn ttac => fn (A,t) =>
      let val lam = #Rand(dest_comb t)
          val spec =  SPEC lam (INST_TYPE (snd(Match.match_term v lam)) ind)
          val {ant,conseq} = dest_imp(concl spec)
          val beta = SUBST [{var = bool, thm = bconv ant}]
                           (mk_imp{ant = bool,conseq = conseq})
                           spec
          val tacs = tacsf (#Bvar(dest_abs lam)) ttac
          val (gll,pl) = GOALS A tacs (#ant(dest_imp(concl beta)))
          val pf = ((MP beta) o LIST_CONJ) o mapshape(map length gll)pl
      in
      (flatten gll, pf)
      end
      handle _ => raise INDUCT_THEN_ERR{function = "INDUCT_THEN",
                                        message = "tactic application error"}
   end
   handle (e as HOL_ERR{origin_structure = "Induct_then",
			origin_function = "INDUCT_THEN",...}) => raise e
        | _ => raise INDUCT_THEN_ERR{function = "INDUCT_THEN",
                                     message = "ill-formed induction theorem"}
end;

end; (* Induct_then *)
