(* -*- Emacs Mode: sml -*- *)

(*-------------------------------------------------------------------------*)
(*
   File:         mk_state_logic.ml

   Description:  This file defines the state abstracted logical operators used
		 in unity and some theorems valid for the combination of these
		 operators.

   Author:       (c) Copyright by Flemming Andersen
   Date:         October 23, 1989
*)
(*-------------------------------------------------------------------------*)

use "aux_definitions.sml";

Globals.tilde_symbols := ("~*" :: !Globals.tilde_symbols);

val path = 
   (!Globals.HOLdir)^"library/unity/theories/"^SysParams.theory_file_type^"/"
val _ = theory_path := path::(!theory_path);

delete_theory (path^"state_logic");
new_theory"state_logic";

val num_Axiom = theorem"prim_rec""num_Axiom";

val False = new_definition
  ("False",
   (--`(False:'a->bool) = \s:'a.F`--));

val True = new_definition
  ("True",
   (--`(True:'a->bool) = \s:'a.T`--));

val ~* = new_definition
  ("~*",
   (--`~* (p:'a->bool) = \s. ~p s`--));

val /\* = new_infix_definition
  ("/\\*", (--`/\* (p:'a->bool) (q:'a->bool) = \s. (p s) /\ (q s)`--), 650);

val \/* = new_infix_definition
  ("\\/*", (--`\/* (p:'a->bool) (q:'a->bool) = \s. (p s) \/ (q s)`--), 640);

(* State dependent universal quantification *)
(*  !*  *)
val !* = new_binder_definition
  ("!*", (--`$!* (P:'b->('a->bool)) = (\s. (!x. ((P x)s)))`--));

(* State dependent existential quantification *)
(*  ?*  *)
val ?* = new_binder_definition
  ("?*", (--`$?* (P:'b->('a->bool)) = (\s. (?x. ((P x)s)))`--));

(* ==>* *)
val ==>* = new_infix_definition
  ("==>*", (--`$==>* (p:'a->bool) (q:'a->bool) = \s. (p s) ==> (q s)`--), 630);

(* <* *)
val <* = new_infix_definition
  ("<*", (--`$<* (p:'a->num) (q:'a->num) = \s. (p s) < (q s)`--), 660);

(* >* *)
val >* = new_infix_definition
  (">*", (--`$>* (p:'a->num) (q:'a->num) = \s. (p s) > (q s)`--), 660);

(* <=* *)
val <=* = new_infix_definition
  ("<=*", (--`$<=* (p:'a->num) (q:'a->num) = \s. (p s) <= (q s)`--), 660);

(* >=* *)
val >=* = new_infix_definition
  (">=*", (--`$>=* (p:'a->num) (q:'a->num) = \s. (p s) >= (q s)`--), 660);

(* =* *)
val =* = new_infix_definition
  ("=*", (--`$=* (p:'a->'b) (q:'a->'b) = \s. (p s) = (q s)`--), 620);

(* State dependent conditional *)
(* =>* *)
val =>* = new_infix_definition
  ("=>*",
   (--`$=>* (p:'a->bool) (r1:'a->'b) (r2:'a->'b) =
       \s. (p s) => r1 s | r2 s`--),
   610);

(* +* *)
val +* = new_infix_definition
  ("+*", (--`$+* (p:'a->num) (q:'a->num) = \s. (p s) + (q s)`--), 680);

(* -* *)
val -* = new_infix_definition
  ("-*", (--`$-* (p:'a->num) (q:'a->num) = \s. (p s) - (q s)`--), 680);

(* ** *)
val ** = new_infix_definition
  ("**", (--`$** (p:'a->num) (q:'a->num) = \s. (p s) * (q s)`--), 690);

val Suc = new_definition
  ("Suc", (--`Suc (p:'a->num) = \s. SUC (p s)`--));

val Pre = new_definition
  ("Pre", (--`Pre (p:'a->num) = \s. PRE (p s)`--));

(* %* *)
val %* = new_infix_definition
  ("%*", (--`$%* (p:'a->num) (q:'a->num) = \s. (p s) MOD (q s)`--), 690);

(* /* *)
val /* = new_infix_definition
  ("/*", (--`$/* (p:'a->num) (q:'a->num) = \s. (p s) DIV (q s)`--), 690);

(* *** *)
val *** = new_infix_definition
  ("***", (--`$*** (p:'a->num) (q:'a->num) = \s. (p s) EXP (q s)`--), 710);

(* State dependent index *)
val Ind = new_infix_definition
  ("Ind", (--`$Ind (a:'a->('b->'c)) (i:'a->'b) = \s. (a s) (i s)`--), 720);

(* More State dependent operators to be defined ??? *)

(*
  Be aware that (!i :: i <= m. P i) = (!i. i <= m ==> P i)
*)
(*  !<=*  *)
val !<=* = new_definition
  ("!<=*",
   (--`!<=* (P:num->('a->bool)) m = (\s:'a. (!i. i <= m ==> ((P i)s)))`--));

(*
  Be aware that ?i :: i <= m. P i == ?i. i <= m /\ P i
*)
(*  ?<=*  *)
val ?<=* = new_definition
  ("?<=*",
   (--`?<=* (P:num->('a->bool)) m = (\s:'a. (?i. i <= m /\ ((P i)s)))`--));

(*  ?<*  *)
val ?<* = new_definition
  ("?<*",
   (--`?<* (P:num->('a->bool)) m = (\s:'a. (?i. i < m /\ ((P i)s)))`--));

(*   /<=\*   *)
val /<=\* = new_recursive_definition
    {name      = "/<=\\*",
     def       = --`(!P. /<=\* P 0       = (P:num->('a->bool)) 0) /\
                    (!P. /<=\* P (SUC i) = ((/<=\* P i) /\* (P (SUC i))))`--,
     fixity    = Prefix,
     rec_axiom = num_Axiom};

(*   \<=/*   *)
val \<=/* = new_recursive_definition
    {name      = "\\<=/*",
     def       = --`(!P. \<=/* P 0       = (P:num->('a->bool)) 0) /\
                    (!P. \<=/* P (SUC i) = ((\<=/* P i) \/* (P (SUC i))))`--,
     fixity    = Prefix,
     rec_axiom = num_Axiom};

(*   /<\*   *)
val /<\* = new_recursive_definition
    {name      = "/<\\*",
     def       = --`(!P. /<\* P 0       = (False:'a->bool)) /\
                    (!P. /<\* P (SUC i) = ((/<\* P i) /\* (P i)))`--,
     fixity    = Prefix,
     rec_axiom = num_Axiom};

(*   \</*   *)
val \</* = new_recursive_definition
    {name      = "\\</*",
     def       = --`(!P. \</* P 0       = (False:'a->bool)) /\
                    (!P. \</* P (SUC i) = ((\</* P i) \/* (P i)))`--,
     fixity    = Prefix,
     rec_axiom = num_Axiom};

(*-------------------------------------------------------------------------*)
(* Theorems valid in this theory                                           *)
(*-------------------------------------------------------------------------*)

val s  = --`s:'a`--
val p  = --`p:'a->bool`--
val q  = --`q:'a->bool`--
val r  = --`r:'a->bool`--
val i  = --`i:num`--
val P  = --`P:num->('a->bool)`--

val IMPLY_WEAK_lemma1 = prove_thm
   ("IMPLY_WEAK_lemma1",
    --`!p q p' q' (s:'a).
         (((p /\* q') \/* (p' /\* q)) \/* (q /\* q'))s
        ==>
	 (q \/* q')s`--,
    REPEAT(GEN_TAC) THEN
    REWRITE_TAC [/\*,\/*] THEN
    CONV_TAC (DEPTH_CONV BETA_CONV) THEN
    REWRITE_TAC [(SYM (SPEC_ALL DISJ_ASSOC))] THEN
    REPEAT STRIP_TAC THENL
      [ASM_REWRITE_TAC [],
       ASM_REWRITE_TAC [],
       ASM_REWRITE_TAC []]);

val IMPLY_WEAK_lemma2 = prove_thm
   ("IMPLY_WEAK_lemma2",
    --`!p q p' q' (s:'a).
         ((((~* p) /\* q') \/* ((~* p') /\* q)) \/* (q /\* q'))s
       ==>
         (q \/* q')s`--,
    REPEAT GEN_TAC THEN
    REWRITE_TAC [~*,/\*,\/*] THEN
    BETA_TAC THEN
    REWRITE_TAC [GEN_ALL (SYM (SPEC_ALL CONJ_ASSOC)),
                 SYM (SPEC_ALL DISJ_ASSOC),
		 NOT_CLAUSES,DE_MORGAN_THM] THEN
    REPEAT STRIP_TAC THENL
      [ASM_REWRITE_TAC [],
       ASM_REWRITE_TAC [],
       ASM_REWRITE_TAC []]);

val IMPLY_WEAK_lemma3 = prove_thm
   ("IMPLY_WEAK_lemma3",
    --`!p q r (s:'a).
         ((((~* p) /\* r) \/* ((~* q) /\* q)) \/* (q /\* r))s
       ==>
         r s`--,
    REPEAT GEN_TAC THEN
    REWRITE_TAC [~*,/\*,\/*] THEN
    CONV_TAC (DEPTH_CONV BETA_CONV) THEN
    REWRITE_TAC [(SYM (SPEC_ALL DISJ_ASSOC))] THEN
    REPEAT STRIP_TAC THEN
    RES_TAC);

val IMPLY_WEAK_lemma4 = prove_thm
  ("IMPLY_WEAK_lemma4",
   --`!p q p' q' r r' (s:'a).
        ((((~*(p \/* p')) /\* (p \/* r')) \/*
          ((~*(q \/* q')) /\* (q \/* r))) \/*
         ((q \/* r) /\* (p \/* r')))s
      ==>
        ((p /\* q) \/* r \/* r')s`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [SYM (SPEC_ALL DISJ_ASSOC),
                GEN_ALL (SYM (SPEC_ALL CONJ_ASSOC)),
                NOT_CLAUSES,DE_MORGAN_THM] THEN
   REPEAT STRIP_TAC THENL
   [RES_TAC
   ,ASM_REWRITE_TAC []
   ,RES_TAC
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ]);

val IMPLY_WEAK_lemma5 = prove_thm
  ("IMPLY_WEAK_lemma5",
   --`!p q r (s:'a).
        ((p /\* r) \/* (((p \/* q) /\* (q \/* r)) \/* r)) s
      ==>
        (q \/* r) s`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THENL
   [ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ,ASM_REWRITE_TAC []
   ]);

val IMPLY_WEAK_lemma6 = prove_thm
  ("IMPLY_WEAK_lemma6",
   --`!p q b r (s:'a).
        ((r /\* q) \/* (p /\* b) \/* (b /\* q)) s
      ==>
        ((q /\* r) \/* b) s`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val IMPLY_WEAK_lemma7 = prove_thm
  ("IMPLY_WEAK_lemma7",
   --`!p q b r (s:'a).
        (((r /\* q) \/* ((r /\* p) /\* b)) \/* (b /\* q)) s
      ==>
        ((q /\* r) \/* b) s`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_COMM_DISJ_lemma_a = TAC_PROOF
    (([],
      --`!p q r (s:'a).
           (r s /\ q s) \/ p s
         ==>
	   (q s /\ r s) \/ p s`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_COMM_DISJ_lemma_b = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (q s /\ r s) \/ p s
       ==>
         (r s /\ q s) \/ p s`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_COMM_DISJ_lemma = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (r s /\ q s) \/ p s
       = (q s /\ r s) \/ p s`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL CONJ_COMM_DISJ_lemma_a)
                    (SPEC_ALL CONJ_COMM_DISJ_lemma_b)));

val AND_COMM_OR_lemma = prove_thm
  ("AND_COMM_OR_lemma",
   --`!(p:'a->bool) q r.
        (r /\* q) \/* p
      = (q /\* r) \/* p`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] CONJ_COMM_DISJ_lemma)));

val CONJ_DISJ_COMM_lemma_a = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (p s /\ (r s \/ q s))
       ==>
         (p s /\ (q s \/ r s))`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_DISJ_COMM_lemma_b = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (p s /\ (q s \/ r s))
        ==>
         (p s /\ (r s \/ q s))`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_DISJ_COMM_lemma = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (p s /\ (r s \/ q s))
       = (p s /\ (q s \/ r s))`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL CONJ_DISJ_COMM_lemma_a)
                    (SPEC_ALL CONJ_DISJ_COMM_lemma_b)));

val AND_OR_COMM_lemma = prove_thm
  ("AND_OR_COMM_lemma",
   --`!(p:'a->bool) q r.
        p /\* (r \/* q)
      = p /\* (q \/* r)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] CONJ_DISJ_COMM_lemma)));

val DISJ_COMM_CONJ_lemma_a = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (r s \/ q s) /\ p s
       ==>
         (q s \/ r s) /\ p s`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_COMM_CONJ_lemma_b = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (q s \/ r s) /\ p s
       ==>
         (r s \/ q s) /\ p s`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_COMM_CONJ_lemma = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (r s \/ q s) /\ p s
       = (q s \/ r s) /\ p s`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL DISJ_COMM_CONJ_lemma_a)
                    (SPEC_ALL DISJ_COMM_CONJ_lemma_b)));

val OR_COMM_AND_lemma = prove_thm
  ("OR_COMM_AND_lemma",
   --`!(p:'a->bool) q r.
        (r \/* q) /\* p
      = (q \/* r) /\* p`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] DISJ_COMM_CONJ_lemma)));

val DISJ_COMM_DISJ_lemma_a = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (r s \/ q s) \/ p s
       ==>
         (q s \/ r s) \/ p s`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_COMM_DISJ_lemma_b = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (q s \/ r s) \/ p s
       ==>
         (r s \/ q s) \/ p s`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_COMM_DISJ_lemma = TAC_PROOF
  (([],
    --`!(p:'a->bool) q r s. (r s \/ q s) \/ p s = (q s \/ r s) \/ p s`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL DISJ_COMM_DISJ_lemma_a)
                    (SPEC_ALL DISJ_COMM_DISJ_lemma_b)));

val OR_COMM_OR_lemma = prove_thm
  ("OR_COMM_OR_lemma",
   --`!(p:'a->bool) q r. (r \/* q) \/* p = (q \/* r) \/* p`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] DISJ_COMM_DISJ_lemma)));

val DISJ_DISJ_COMM_lemma_a = TAC_PROOF
  (([], --`!p q r (s:'a). p s \/ (r s \/ q s) ==> p s \/ (q s \/ r s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_DISJ_COMM_lemma_b = TAC_PROOF
  (([], --`!p q r (s:'a). p s \/ (q s \/ r s) ==> p s \/ (r s \/ q s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_DISJ_COMM_lemma = TAC_PROOF
  (([], --`!p q r (s:'a). p s \/ (r s \/ q s)  = p s \/ (q s \/ r s) `--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL DISJ_DISJ_COMM_lemma_a)
                    (SPEC_ALL DISJ_DISJ_COMM_lemma_b)));

val OR_OR_COMM_lemma = prove_thm
  ("OR_OR_COMM_lemma",
   (--`!(p:'a->bool) q r. p \/* (r \/* q) = p \/* (q \/* r)`--),
   REPEAT GEN_TAC THEN
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] DISJ_DISJ_COMM_lemma)));

val CONJ_COMM_CONJ_lemma_a = TAC_PROOF
  (([], --`!p q r (s:'a). (r s /\ q s) /\ p s ==> (q s /\ r s) /\ p s`--),
   REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val CONJ_COMM_CONJ_lemma_b = TAC_PROOF
  (([], --`!p q r (s:'a). (q s /\ r s) /\ p s ==> (r s /\ q s) /\ p s`--),
   REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val CONJ_COMM_CONJ_lemma = TAC_PROOF
  (([], --`!p q r (s:'a). (r s /\ q s) /\ p s = (q s /\ r s) /\ p s`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL CONJ_COMM_CONJ_lemma_a)
                    (SPEC_ALL CONJ_COMM_CONJ_lemma_b)));

val AND_COMM_AND_lemma = prove_thm
  ("AND_COMM_AND_lemma",
   --`!(p:'a->bool) q r. (r /\* q) /\* p = (q /\* r) /\* p`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] CONJ_COMM_CONJ_lemma)));

val CONJ_CONJ_COMM_lemma_a = TAC_PROOF
  (([], --`!p q r (s:'a). p s /\ (r s /\ q s) ==> p s /\ (q s /\ r s)`--),
   REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val CONJ_CONJ_COMM_lemma_b = TAC_PROOF
  (([], --`!p q r (s:'a). p s /\ (q s /\ r s) ==> p s /\ (r s /\ q s)`--),
   REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val CONJ_CONJ_COMM_lemma = TAC_PROOF
  (([], --`!p q r (s:'a). p s /\ (r s /\ q s)  = p s /\ (q s /\ r s) `--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL CONJ_CONJ_COMM_lemma_a)
                    (SPEC_ALL CONJ_CONJ_COMM_lemma_b)));

val AND_AND_COMM_lemma = prove_thm
  ("AND_AND_COMM_lemma",
   --`!(p:'a->bool) q r. p /\* (r /\* q) = p /\* (q /\* r)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] CONJ_CONJ_COMM_lemma)));

val DISJ_CONJ_COMM_lemma_a = TAC_PROOF
  (([], --`!p q r (s:'a). p s \/ (r s /\ q s) ==> p s \/ (q s /\ r s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_CONJ_COMM_lemma_b = TAC_PROOF
  (([], --`!p q r (s:'a). p s \/ (q s /\ r s) ==> p s \/ (r s /\ q s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_CONJ_COMM_lemma = TAC_PROOF
  (([], --`!p q r (s:'a). p s \/ (r s /\ q s) = p s \/ (q s /\ r s)`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (IMP_ANTISYM_RULE
                    (SPEC_ALL DISJ_CONJ_COMM_lemma_a)
                    (SPEC_ALL DISJ_CONJ_COMM_lemma_b)));

val OR_AND_COMM_lemma = prove_thm
  ("OR_AND_COMM_lemma",
   --`!(p:'a->bool) q r. p \/* (r /\* q) = p \/* (q /\* r)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] DISJ_CONJ_COMM_lemma)));

val NOT_NOT_lemma = prove_thm
   ("NOT_NOT_lemma",
    --`!(p:'a->bool). (~* (~* p)) = p`--,
    REWRITE_TAC [~*] THEN
    CONV_TAC (DEPTH_CONV BETA_CONV) THEN
    REWRITE_TAC [NOT_CLAUSES,ETA_AX]);

val DISJ_COMM_lemma = TAC_PROOF
   (([], --`!p q (s:'a). p s \/ q s = q s \/ p s`--),
    REPEAT STRIP_TAC THEN
    STRIP_ASSUME_TAC (SPECL [--`^p s`--,--`^q s`--] DISJ_SYM));

val OR_COMM_lemma = prove_thm
   ("OR_COMM_lemma",
    --`!(p:'a->bool) q. (p \/* q) = (q \/* p)`--,
    REPEAT STRIP_TAC THEN
    REWRITE_TAC [\/*] THEN
    ASSUME_TAC DISJ_COMM_lemma THEN
    STRIP_ASSUME_TAC
        (MK_ABS (SPECL [p,q] 
                (ASSUME (--`!(p:'a->bool) q s. p s \/ q s = q s \/ p s`--)))));

val OR_OR_lemma = prove_thm
   ("OR_OR_lemma",
    --`!p:'a->bool. p \/* p = p`--,
    GEN_TAC THEN REWRITE_TAC [\/*,ETA_AX]);

val DISJ_ASSOC_lemma = TAC_PROOF
   (([], --`!p q  r (s:'a). ((p s \/ q s) \/ r s) = (p s \/ (q s \/ r s))`--),
    REWRITE_TAC [(SYM (SPEC_ALL DISJ_ASSOC))]);

val OR_ASSOC_lemma = prove_thm
   ("OR_ASSOC_lemma",
    (--`!(p:'a->bool) q r. (p \/* q) \/* r = p \/* (q \/* r)`--),
    REPEAT STRIP_TAC THEN REWRITE_TAC [\/*]  THEN
    CONV_TAC (DEPTH_CONV BETA_CONV) THEN
    ASSUME_TAC DISJ_ASSOC_lemma THEN
    STRIP_ASSUME_TAC
   (MK_ABS (SPECL [p,q,r]
      (ASSUME (--`!p q r (s:'a).
                    ((p s \/ q s) \/ r s) = (p s \/ (q s \/ r s))`--)))));

val CONJ_WEAK_lemma = TAC_PROOF
  (([], --`!p q (s:'a). p s /\ q s ==> q s`--),
    REPEAT STRIP_TAC THEN RES_TAC);

val AND_IMPLY_WEAK_lemma = prove_thm
  ("AND_IMPLY_WEAK_lemma",
    --`!p q (s:'a). (p /\* q) s ==> q s`--,
    REWRITE_TAC [/\*] THEN
    CONV_TAC (DEPTH_CONV BETA_CONV) THEN
    REWRITE_TAC [CONJ_WEAK_lemma]);

val SYM_CONJ_WEAK_lemma = TAC_PROOF
  (([], --`!p q (s:'a). p s /\ q s ==> p s`--),
    REPEAT STRIP_TAC THEN RES_TAC);

val SYM_AND_IMPLY_WEAK_lemma = prove_thm
  ("SYM_AND_IMPLY_WEAK_lemma",
    --`!p q (s:'a). (p /\* q) s ==> p s`--,
    REWRITE_TAC [/\*] THEN
    CONV_TAC (DEPTH_CONV BETA_CONV) THEN
    REWRITE_TAC [SYM_CONJ_WEAK_lemma]);

val OR_IMPLY_WEAK_lemma = prove_thm
  ("OR_IMPLY_WEAK_lemma",
   --`!p q (s:'a). p s ==> (p \/* q) s`--,
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THEN
   ASM_REWRITE_TAC []);

val SYM_OR_IMPLY_WEAK_lemma = prove_thm
  ("SYM_OR_IMPLY_WEAK_lemma",
   --`!p q (s:'a). p s ==> (q \/* p) s`--,
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THEN
   ASM_REWRITE_TAC []);

val IMPLY_WEAK_AND_lemma = prove_thm
  ("IMPLY_WEAK_AND_lemma",
   --`!(p:'a->bool) q r.
        (!s. p s ==> q s)
      ==>
        (!s. (p /\* r) s ==> (q /\* r) s)`--,
   REWRITE_TAC [/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THENL
     [RES_TAC,
      RES_TAC THEN
      ASM_REWRITE_TAC []]);

val IMPLY_WEAK_OR_lemma = prove_thm
  ("IMPLY_WEAK_OR_lemma",
   --`!(p:'a->bool) q r.
        (!s. p s ==> q s)
      ==>
        (!s. (p \/* r) s ==> (q \/* r) s)`--,
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THENL
     [RES_TAC THEN
      ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC []]);

val AND_AND_lemma = prove_thm
  ("AND_AND_lemma",
   --`!p:'a->bool. p /\* p = p`--,
   REWRITE_TAC [/\*,ETA_AX]);

val CONJ_COMM_lemma = TAC_PROOF
  (([],
    --`!p q (s:'a). (p s /\ q s) = (q s /\ p s)`--),
   REPEAT GEN_TAC THEN
   STRIP_ASSUME_TAC (SPECL [--`^p s`--,--`^q s`--] CONJ_SYM));

val AND_COMM_lemma = prove_thm
   ("AND_COMM_lemma",
   (--`!(p:'a->bool) q. (p /\* q) = (q /\* p)`--),
   REWRITE_TAC [/\*] THEN
   REPEAT GEN_TAC THEN
   ASSUME_TAC CONJ_COMM_lemma THEN
   STRIP_ASSUME_TAC
        (MK_ABS (SPECL [p,q] 
                (ASSUME (--`!p q (s:'a). p s /\ q s = q s /\ p s`--)))));

val CONJ_ASSOC_lemma = TAC_PROOF
  (([],
    --`!p q r (s:'a). ((p s /\ q s) /\ r s) = (p s /\ (q s /\ r s))`--),
    REWRITE_TAC [GEN_ALL (SYM (SPEC_ALL CONJ_ASSOC))]);

val AND_ASSOC_lemma = prove_thm
   ("AND_ASSOC_lemma",
    --`!(p:'a->bool) q r. (p /\* q) /\* r = p /\* (q /\* r)`--,
   REPEAT GEN_TAC THEN REWRITE_TAC [/\*]  THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   ASSUME_TAC CONJ_ASSOC_lemma THEN
   STRIP_ASSUME_TAC
   (MK_ABS (SPECL [p,q,r]
      (ASSUME (--`!p q  r (s:'a).
                 ((p s /\ q s) /\ r s) = (p s /\ (q s /\ r s))`--)))));

val NOT_True_lemma = prove_thm
   ("NOT_True_lemma",
    --`~* (True:'a->bool) = False`--,
   REWRITE_TAC [~*,True,False,ETA_AX]);

val NOT_False_lemma = prove_thm
   ("NOT_False_lemma",
    --`~* (False:'a->bool) = True`--,
   REWRITE_TAC [~*,True,False,ETA_AX]);

val AND_True_lemma = prove_thm
   ("AND_True_lemma",
    --`!p:'a->bool. p /\* True = p`--,
   REWRITE_TAC [/\*,True,ETA_AX]);

val OR_True_lemma = prove_thm
   ("OR_True_lemma",
    --`!p:'a->bool. p \/* True = True`--,
   REWRITE_TAC [\/*,True,ETA_AX]);

val AND_False_lemma = prove_thm
   ("AND_False_lemma",
    --`!p:'a->bool. p /\* False = False`--,
   REWRITE_TAC [/\*,False,ETA_AX]);

val OR_False_lemma = prove_thm
   ("OR_False_lemma",
    --`!p:'a->bool. p \/* False = p`--,
   REWRITE_TAC [\/*,False,ETA_AX]);

val P_OR_NOT_P_lemma = prove_thm
   ("P_OR_NOT_P_lemma",
    --`!p:'a->bool. p \/* (~* p) = True`--,
   REWRITE_TAC [\/*,~*,True] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [EXCLUDED_MIDDLE,OR_CLAUSES,NOT_CLAUSES,ETA_AX]);

val P_AND_NOT_P_lemma = prove_thm
   ("P_AND_NOT_P_lemma",
    --`!p:'a->bool. p /\* (~* p) = False`--,
   REWRITE_TAC [/\*,~*,False] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [NOT_AND,AND_CLAUSES,NOT_CLAUSES,ETA_AX]);

val CONJ_COMPL_DISJ_lemma1 = TAC_PROOF
  (([],
    --`!p q. p /\ ~q \/ p /\ q ==> p`--),
   REPEAT STRIP_TAC);

val CONJ_COMPL_DISJ_lemma2 = TAC_PROOF
  (([],
    --`!p q. p ==> p /\ ~q \/ p /\ q`--),
   REPEAT STRIP_TAC THEN
   ASM_REWRITE_TAC [] THEN
   PURE_ONCE_REWRITE_TAC [DISJ_SYM] THEN
   REWRITE_TAC [EXCLUDED_MIDDLE]);

val CONJ_COMPL_DISJ_lemma = TAC_PROOF
  (([],
    --`!p q. p /\ ~q \/ p /\ q = p`--),
   REWRITE_TAC [IMP_ANTISYM_RULE
                  (SPEC_ALL CONJ_COMPL_DISJ_lemma1)
                  (SPEC_ALL CONJ_COMPL_DISJ_lemma2)]);

val AND_COMPL_OR_lemma = prove_thm
   ("AND_COMPL_OR_lemma",
    --`!(p:'a->bool) q. ((p /\* (~* q)) \/* (p /\* q)) = p`--,
   REWRITE_TAC [~*,/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [CONJ_COMPL_DISJ_lemma,ETA_AX]);

val DISJ_NOT_CONJ_lemma1 = TAC_PROOF
  (([],
    --`!p q. (p \/ q) /\ ~q ==> p /\ ~q`--),
    REPEAT STRIP_TAC THENL
      [ASM_REWRITE_TAC [],RES_TAC,RES_TAC,RES_TAC]);

val DISJ_NOT_CONJ_lemma2 = TAC_PROOF
  (([],
    --`!p q. p /\ ~q ==> (p \/ q) /\ ~q`--),
    REPEAT STRIP_TAC THENL
      [ASM_REWRITE_TAC [],RES_TAC]);

val DISJ_NOT_CONJ_lemma = TAC_PROOF
  (([], --`!p q. (p \/ q) /\ ~q = p /\ ~q`--),
   REWRITE_TAC [IMP_ANTISYM_RULE
                  (SPEC_ALL DISJ_NOT_CONJ_lemma1)
                  (SPEC_ALL DISJ_NOT_CONJ_lemma2)]);

val OR_NOT_AND_lemma = prove_thm
  ("OR_NOT_AND_lemma",
   --`!(p:'a->bool) q. ((p \/* q) /\* (~* q)) = p /\* (~* q)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*,/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [DISJ_NOT_CONJ_lemma]);

val P_CONJ_Q_DISJ_Q_lemma1 = TAC_PROOF
  (([], --`!(p:'a->bool) q s. (p s /\ q s) \/ q s ==> q s`--),
   REPEAT STRIP_TAC);

val P_CONJ_Q_DISJ_Q_lemma2 = TAC_PROOF
  (([], --`!(p:'a->bool) q s. q s ==> (p s /\ q s) \/ q s`--),
   REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val P_CONJ_Q_DISJ_Q_lemma = TAC_PROOF
  (([], --`!(p:'a->bool) q s. (p s /\ q s) \/ q s = q s`--),
   ASM_REWRITE_TAC [IMP_ANTISYM_RULE
                         (SPEC_ALL P_CONJ_Q_DISJ_Q_lemma1)
                         (SPEC_ALL P_CONJ_Q_DISJ_Q_lemma2)]);

val P_AND_Q_OR_Q_lemma = prove_thm
  ("P_AND_Q_OR_Q_lemma",
   --`!(p:'a->bool) q. (p /\* q) \/* q = q`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [GEN_ALL (MK_ABS (SPECL [p,q] P_CONJ_Q_DISJ_Q_lemma)),ETA_AX]);

val P_DISJ_Q_CONJ_Q_lemma1 = TAC_PROOF
  (([], --`!(p:'a->bool) q s. (p s \/ q s) /\ q s ==> q s`--),
   REPEAT STRIP_TAC);

val P_DISJ_Q_CONJ_Q_lemma2 = TAC_PROOF
  (([], --`!(p:'a->bool) q s. q s ==> (p s \/ q s) /\ q s`--),
   REPEAT STRIP_TAC THEN ASM_REWRITE_TAC []);

val P_DISJ_Q_CONJ_Q_lemma = TAC_PROOF
  (([], --`!(p:'a->bool) q s. (p s \/ q s) /\ q s = q s`--),
   ASM_REWRITE_TAC [IMP_ANTISYM_RULE
                         (SPEC_ALL P_DISJ_Q_CONJ_Q_lemma1)
                         (SPEC_ALL P_DISJ_Q_CONJ_Q_lemma2)]);

val P_OR_Q_AND_Q_lemma = prove_thm
  ("P_OR_Q_AND_Q_lemma",
   --`!(p:'a->bool) q. (p \/* q) /\* q = q`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [GEN_ALL (MK_ABS (SPECL [p,q] P_DISJ_Q_CONJ_Q_lemma)),ETA_AX]);

val NOT_OR_AND_NOT_lemma = prove_thm
  ("NOT_OR_AND_NOT_lemma",
   --`!(p:'a->bool) q. ~* (p \/* q) = (~* p) /\* (~* q)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*,/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [NOT_CLAUSES,DE_MORGAN_THM]);

val NOT_AND_OR_NOT_lemma = prove_thm
  ("NOT_AND_OR_NOT_lemma",
   --`!(p:'a->bool) q. ~* (p /\* q) = (~* p) \/* (~* q)`--,
      REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*,/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [NOT_CLAUSES,DE_MORGAN_THM]);

val NOT_IMPLY_OR_lemma = prove_thm
  ("NOT_IMPLY_OR_lemma",
   --`!(p:'a->bool) q.
        (!s. (~* p)s ==> q s)
      = (!s. (p \/* q)s)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [IMP_DISJ_THM]);

val IMPLY_OR_lemma = prove_thm
  ("IMPLY_OR_lemma",
   --`!(p:'a->bool) q. (!s. p s ==> q s) = (!s. ((~* p) \/* q)s)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [IMP_DISJ_THM]);

val OR_IMPLY_lemma = prove_thm
  ("OR_IMPLY_lemma",
   --`!(p:'a->bool) q. (!s. (p \/* q)s) = (!s. (~* p)s ==> q s)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [IMP_DISJ_THM,NOT_CLAUSES]);

val NOT_OR_IMPLY_lemma = prove_thm
  ("NOT_OR_IMPLY_lemma",
   --`!(p:'a->bool) q. (!s. ((~* p) \/* q)s) = (!s. p s ==> q s)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [~*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [IMP_DISJ_THM,NOT_CLAUSES]);

val DISJ_CONJ_lemma1 = TAC_PROOF
  (([],
    --`!p q r (s:'a).
         (p s \/ q s /\ r s)
       ==>
         ((p s \/ q s) /\ (p s \/ r s))`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_CONJ_lemma2 = TAC_PROOF
  (([], --`!(p:'a->bool) q r s. 
             ((p s \/ q s) /\ (p s \/ r s)) ==> (p s \/ q s /\ r s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val DISJ_CONJ_lemma = TAC_PROOF
  (([], --`!(p:'a->bool) q r s. 
            (p s \/ q s /\ r s) = ((p s \/ q s) /\ (p s \/ r s))`--),
   REWRITE_TAC [IMP_ANTISYM_RULE
                      (SPEC_ALL DISJ_CONJ_lemma1)
                      (SPEC_ALL DISJ_CONJ_lemma2)]);

val OR_AND_DISTR_lemma = prove_thm
  ("OR_AND_DISTR_lemma",
   --`!(p:'a->bool) q r. p \/* (q /\* r) = (p \/* q) /\* (p \/* r)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] DISJ_CONJ_lemma)));

val CONJ_DISJ_lemma1 = TAC_PROOF
  (([], --`!(p:'a->bool) q r s.
            (p s /\ (q s \/ r s)) ==> (p s /\ q s \/ p s /\ r s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_DISJ_lemma2 = TAC_PROOF
  (([], --`!(p:'a->bool) q r s.
            (p s /\ q s \/ p s /\ r s) ==> (p s /\ (q s \/ r s))`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC [],
      ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_DISJ_lemma = TAC_PROOF
  (([], --`!(p:'a->bool) q r s.
            (p s /\ (q s \/ r s)) = (p s /\ q s \/ p s /\ r s)`--),
   REWRITE_TAC [IMP_ANTISYM_RULE
                      (SPEC_ALL CONJ_DISJ_lemma1)
                      (SPEC_ALL CONJ_DISJ_lemma2)]);

val AND_OR_DISTR_lemma = prove_thm
  ("AND_OR_DISTR_lemma",
   --`!(p:'a->bool) q r. p /\* (q \/* r) = (p /\* q) \/* (p /\* r)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [p,q,r] CONJ_DISJ_lemma)));

val NOT_IMPLIES_False_lemma = prove_thm
  ("NOT_IMPLIES_False_lemma",
   --`!(p:'a->bool). (!s. (~* p)s) ==> (!s. p s = False s)`--,
   REWRITE_TAC [False,~*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC []);

val NOT_P_IMPLIES_P_EQ_False_lemma = prove_thm
  ("NOT_P_IMPLIES_P_EQ_False_lemma",
   --`!(p:'a->bool). (!s. (~* p)s) ==> (p = False)`--,
   REPEAT STRIP_TAC THEN
   ASSUME_TAC (MK_ABS (UNDISCH_ALL (SPEC_ALL NOT_IMPLIES_False_lemma))) THEN
   UNDISCH_TAC (--`(\s:'a. p s) = (\s. False s)`--) THEN
   REWRITE_TAC [ETA_AX]);

val NOT_AND_IMPLIES_lemma = prove_thm
  ("NOT_AND_IMPLIES_lemma",
   --`!(p:'a->bool) q. (!s. (~* (p /\* q))s) = (!s. p s ==> ~* q s)`--,
   REWRITE_TAC [~*,/\*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [DE_MORGAN_THM,NOT_CLAUSES,IMP_DISJ_THM]);

val NOT_AND_IMPLIES_lemma1 = prove_thm
  ("NOT_AND_IMPLIES_lemma1",
   --`!(p:'a->bool) q. (!s. (~* (p /\* q))s) ==> (!s. p s ==> ~* q s)`--,
   REWRITE_TAC [NOT_AND_IMPLIES_lemma]);

val NOT_AND_IMPLIES_lemma2 = prove_thm
  ("NOT_AND_IMPLIES_lemma2",
   --`!(p:'a->bool) q. (!s. (~* (p /\* q))s) ==> (!s. q s ==> ~* p s)`--,
   REWRITE_TAC [NOT_AND_IMPLIES_lemma,~*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REPEAT STRIP_TAC THEN
   RES_TAC);

val CONJ_DISJ_IMPLY_lemma1 = TAC_PROOF
   (([], --`!(p:'a->bool) q s. p s /\ (p s \/ q s) ==> p s`--),
   REPEAT STRIP_TAC);

val CONJ_DISJ_IMPLY_lemma2 = TAC_PROOF
   (([], --`!(p:'a->bool) q s. p s ==> p s /\ (p s \/ q s)`--),
   REPEAT STRIP_TAC THENL
     [ASM_REWRITE_TAC [],ASM_REWRITE_TAC []]);

val CONJ_DISJ_IMPLY_lemma = TAC_PROOF
  (([], --`!(p:'a->bool) q s. p s /\ (p s \/ q s) = p s`--),
   REWRITE_TAC [IMP_ANTISYM_RULE
                  (SPEC_ALL CONJ_DISJ_IMPLY_lemma1)
                  (SPEC_ALL CONJ_DISJ_IMPLY_lemma2)]);

val CONJ_DISJ_ABS_IMPLY_lemma = TAC_PROOF
  (([], --`!(p:'a->bool) q.  (\s. p s /\ (p s \/ q s)) = p`--),
   REPEAT GEN_TAC THEN
   REWRITE_TAC [CONJ_DISJ_IMPLY_lemma,ETA_AX]);

val AND_OR_EQ_lemma = prove_thm
  ("AND_OR_EQ_lemma",
   --`!(p:'a->bool) q. p /\* (p \/* q) = p`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [/\*,\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   REWRITE_TAC [CONJ_DISJ_ABS_IMPLY_lemma]);

val AND_OR_EQ_AND_COMM_OR_lemma = prove_thm
  ("AND_OR_EQ_AND_COMM_OR_lemma",
   --`!(p:'a->bool) q. p /\* (q \/* p) = p /\* (p \/* q)`--,
   REPEAT GEN_TAC THEN
   REWRITE_TAC [AND_OR_EQ_lemma] THEN
   ONCE_REWRITE_TAC [OR_COMM_lemma] THEN
   REWRITE_TAC [AND_OR_EQ_lemma]);

val IMPLY_WEAK_lemma = prove_thm
  ("IMPLY_WEAK_lemma",
   --`!(p:'a->bool) q. (!s. p s) ==> (!s. (p \/* q) s)`--,
   REPEAT STRIP_TAC THEN
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   ASM_REWRITE_TAC []);

val IMPLY_WEAK_lemma_b = prove_thm
  ("IMPLY_WEAK_lemma_b",
   --`!(p:'a->bool) q s. p s ==> (p \/* q) s`--,
   REPEAT STRIP_TAC THEN
   REWRITE_TAC [\/*] THEN
   CONV_TAC (DEPTH_CONV BETA_CONV) THEN
   ASM_REWRITE_TAC []);

val ALL_AND_lemma1 = TAC_PROOF
  (([],
   --`!(P:num->('a->bool)) i s. (!i. P i s) = (P i s /\ (!i. P i s))`--),
   REPEAT STRIP_TAC THEN
   EQ_TAC THENL
     [
      REPEAT STRIP_TAC THENL
        [
         ASM_REWRITE_TAC []
        ,
         ASM_REWRITE_TAC []
        ],
      REPEAT STRIP_TAC THEN
      ASM_REWRITE_TAC []]);

val ALL_OR_lemma1 = TAC_PROOF
  (([],
   --`!(P:num->('a->bool)) i s. (?i. P i s) = (P i s \/ (?i. P i s))`--),
   REPEAT GEN_TAC THEN
   EQ_TAC THENL
     [REPEAT STRIP_TAC THEN
      DISJ2_TAC THEN
      EXISTS_TAC (--`i':num`--) THEN
      ASM_REWRITE_TAC []
     ,REPEAT STRIP_TAC THENL
        [EXISTS_TAC (--`i:num`--) THEN
         ASM_REWRITE_TAC []
        ,EXISTS_TAC (--`i':num`--) THEN
         ASM_REWRITE_TAC []]]);

val ALL_OR_lemma = prove_thm
  ("ALL_OR_lemma",
   --`!(P:num->('a->bool)) i. (($?* P) = ((P i) \/* ($?* P)))`--,
   GEN_TAC THEN GEN_TAC THEN
   REWRITE_TAC [?*,\/*] THEN
   BETA_TAC THEN
   STRIP_ASSUME_TAC (MK_ABS (SPECL [P,i] ALL_OR_lemma1)));

val ALL_i_OR_lemma1 = TAC_PROOF
  (([],
    --`!P (s:'a). (?i. \<=/* P i s) = (?i. P i s)`--),
   REPEAT STRIP_TAC THEN
   EQ_TAC THENL
     [
      STRIP_TAC THEN
      UNDISCH_TAC (--`\<=/* ^P i s`--) THEN
      SPEC_TAC (i,i) THEN
      INDUCT_TAC THENL
        [
         REWRITE_TAC [\<=/* ] THEN
         DISCH_TAC THEN
         EXISTS_TAC (--`0`--) THEN
         ASM_REWRITE_TAC []
        ,
         REWRITE_TAC [\<=/*,\/*] THEN
         BETA_TAC THEN
         REPEAT STRIP_TAC THENL
           [
            RES_TAC THEN
            EXISTS_TAC (--`i':num`--) THEN
            ASM_REWRITE_TAC []
           ,
            EXISTS_TAC (--`SUC i`--) THEN
            ASM_REWRITE_TAC []
           ]
        ],
      STRIP_TAC THEN
      UNDISCH_TAC (--`^P i s`--) THEN
      SPEC_TAC (i,i) THEN
      INDUCT_TAC THENL
        [
         DISCH_TAC THEN
         EXISTS_TAC (--`0`--) THEN
         ASM_REWRITE_TAC [\<=/* ]
        ,
         DISCH_TAC THEN
         EXISTS_TAC (--`SUC i`--) THEN
         REWRITE_TAC [\<=/*,\/*] THEN
         BETA_TAC THEN
         ASM_REWRITE_TAC []
        ]
     ]);

val ALL_i_OR_lemma = prove_thm
  ("ALL_i_OR_lemma",
   (--`!P. ((\s:'a. ?i. \<=/* P i s) = ($?* P))`--),
   REWRITE_TAC [?*] THEN
   GEN_TAC THEN
   STRIP_ASSUME_TAC (MK_ABS (SPEC P ALL_i_OR_lemma1)));

close_theory();
export_theory();

(* Emacs editor information
|  Local variables:
|  mode:sml
|  sml-prog-name:"hol90"
|  End:
*)
