(* ===================================================================== 
 * FILE          : traverse.sml 
 * DESCRIPTION   : A programmable term traversal engine for hol90
 *                                                                       
 * AUTHOR        : Donald Syme
 *                 Based loosely on original HOL rewriting by 
 *                 Larry Paulson et al.
 * ===================================================================== *)

(* =====================================================================
 * The Traversal/Reduction Engine.  Its pretty simple really
 *   - start with an intial context
 *   - collect context that arises from congruence rules
 *   - apply reducers where possible, "rewriters" at high priority and
 *     "dprocs" at low.
 * =====================================================================*)


structure Traverse : Traverse_sig = 
struct

type term = CoreHol.Term.term
type thm = CoreHol.Thm.thm;
type conv = Abbrev.conv;

structure Travrules = Travrules;
	
open liteLib;
open Lib CoreHol;
open Term Dsyntax Thm Drule Conv Psyntax LiteLib Parse
     Trace Equal Travrules Opening;

infix THENQC THENCQC ORELSEC IFCQC
fun WRAP x = STRUCT_WRAP "Traverse" x;
fun ERR x = STRUCT_ERR "Traverse" x;

val _ = say "Adding the traversal engine...";
  
val equality = (--`$= :'a -> 'a ->bool`--);
  
(* ---------------------------------------------------------------------
 * reducers and contexts
 * ---------------------------------------------------------------------*)
  
type context = exn   (* well known SML hack to allow any kind of data *)
  
datatype reducer = 
  REDUCER of {initial: context,
              addcontext : context * Thm.thm list -> context,
              apply: {solver:term list -> term -> thm, context: context,
                      stack: term list} -> conv
              };
fun dest_reducer (REDUCER x) = x 
  

(* ---------------------------------------------------------------------
 * Traversal states
 *    - stores the state of the reducers during the simpl. process
 * ---------------------------------------------------------------------*)
  
datatype trav_state = 
  TSTATE of {relation : preorder,
             contexts1 : context list,
             contexts2 : context list,
             freevars : term list}; 
  
fun initial_context {rewriters:reducer list,
                     dprocs:reducer list,
                     travrules=TRAVRULES tsdata,
                     relation} = 
  TSTATE{contexts1=map (#initial o dest_reducer) rewriters,
         contexts2=map (#initial o dest_reducer) dprocs,
         freevars=[],
         relation = find_relation relation (#relations tsdata)};
  
(* ---------------------------------------------------------------------
 * add_context
 *
 * ---------------------------------------------------------------------*)
       
fun add_context rewriters dprocs =
  let val rewrite_collectors' = map (#addcontext o dest_reducer) rewriters
      val dproc_collectors' = map (#addcontext o dest_reducer) dprocs
  in fn (context as 
         TSTATE {contexts1,contexts2,freevars,relation}, thms) =>
    if (null thms) then context else
    let val more_freevars = free_varsl (flatten (map hyp thms))
        val _ = map (fn thm => trace(2,MORE_CONTEXT thm)) thms
        fun mk_privcontext maker privcontext = maker (privcontext,thms)
        val newcontexts1 = 
          if (null thms) then contexts1
          else map2 mk_privcontext rewrite_collectors' contexts1
        val newcontexts2 = 
          if (null thms) then contexts2
          else map2 mk_privcontext dproc_collectors' contexts2
        val newfreevars = 
          if (null more_freevars) then freevars
          else more_freevars@freevars
    in TSTATE{contexts1=newcontexts1,
              contexts2=newcontexts2,
              freevars=newfreevars,
              relation=relation}
    end
  end;


(* ---------------------------------------------------------------------
 * change_relation
 *
 * ---------------------------------------------------------------------*)

fun change_relation 
  (TRAVRULES{relations,...})
  (context as 
   TSTATE {contexts1,contexts2,freevars,  
               relation as PREORDER(oldrelname,_,_)},
   rel) =
  if (name_of_const rel = oldrelname) then context
  else TSTATE{contexts1=contexts1,
              contexts2=contexts2,
              freevars=freevars,
              relation=find_relation rel relations};
    
(* ---------------------------------------------------------------------
 * Quick, General conversion routines.  These work for any preorder,
 * not just equality.
 * ---------------------------------------------------------------------*)

fun GEN_THENQC (PREORDER(_,TRANS,_)) (conv1,conv2) tm =
  let val th1 = conv1 tm
  in TRANS th1 (conv2 (rand (concl th1)))
     handle Interrupt => raise Interrupt | _ => th1
  end
  handle Interrupt => raise Interrupt
       | _ => conv2 tm
  
fun GEN_THENCQC  (PREORDER(_,TRANS,_)) (conv1,conv2) tm =
  let val th1 = conv1 tm 
  in let val th2 = conv2(rand(concl th1)) 
     in TRANS th1 th2
     end
     handle Interrupt => raise Interrupt | _ => th1
  end;;

(* perform continuation with argumnt indicating whether a change occurred *)
fun GEN_IFCQC  (PREORDER(_,TRANS,_)) (conv1,conv2) tm =
  let val th1 = conv1 tm
  in let val th2 = conv2 true (rand(concl th1)) 
     in TRANS th1 th2
     end
     handle Interrupt => raise Interrupt | _ => th1
  end
  handle Interrupt => raise Interrupt | _ => conv2 false tm;;


fun GEN_REPEATQC rel =
   let val op THENCQC = GEN_THENCQC rel
       fun REPEATQC conv tm =
	   (conv THENCQC (REPEATQC conv)) tm
   in REPEATQC
   end;

fun mapfilter2 f (h1::t1) (h2::t2) =
   (f h1 h2::mapfilter2 f t1 t2 
    handle Interrupt => raise Interrupt | _ => mapfilter2 f t1 t2)
  | mapfilter2 f _ _ = [];

(* ---------------------------------------------------------------------
 * TRAVERSE_IN_CONTEXT
 *
 * NOTES
 *   - Rewriters/Dprocs should always fail if unchanged (but not with
 *     UNCHANGED).
 *   - Congruence rule procs should only fail if they do not match.
 *     They can fail with UNCHANGED if nothing changes.
 *   - The depther should only fail with UNCHANGED.
 * ---------------------------------------------------------------------*)


fun TRAVERSE_IN_CONTEXT rewriters dprocs
     (travrules as (TRAVRULES {relations,congprocs,weakenprocs,...}))
    =
let val add_context' = add_context rewriters dprocs
    val change_relation' = change_relation travrules

    fun trav stack (context as TSTATE {contexts1,contexts2,
                  freevars,relation as (PREORDER (relname,_,_))}) =
      let 
       fun ctxt_solver stack = 
         EQT_ELIM o trav stack (change_relation' (context,equality))
       fun apply_reducer (REDUCER rdata) context =
         (#apply rdata) {solver=ctxt_solver,context=context,
                         stack=stack} 
       val high_priority = 
         FIRST_CONV (mapfilter2 apply_reducer rewriters contexts1)
       val low_priority = 
         FIRST_CONV (mapfilter2 apply_reducer dprocs contexts2)
       fun depther (thms,relation) = 
         trav stack (change_relation' (add_context' (context,thms), relation))
       val congproc_args = 
         {relation=relname,
          solver=(fn tm => ctxt_solver stack tm), (* do not eta-convert! *)
          depther=depther,
          freevars=freevars};
       fun apply_congproc congproc = congproc congproc_args;
       val descend = FIRST_CONV (mapfilter apply_congproc congprocs);
       val weaken = FIRST_CONV (mapfilter apply_congproc weakenprocs);
       val op IFCQC = GEN_IFCQC relation
       val op THENCQC = GEN_THENCQC relation
       val op THENQC = GEN_THENQC relation
       val REPEATQC = GEN_REPEATQC relation

       fun loop tm =
         let val conv =
             REPEATQC high_priority THENQC
             (descend IFCQC 
             (fn change => 
              ((if change then high_priority else NO_CONV) 
              ORELSEC low_priority ORELSEC weaken) THENCQC 
             loop));
        in (trace(4,REDUCE ("Reducing",tm)); conv tm)
        end;
      in loop
      end;
in trav

end;

(* ---------------------------------------------------------------------
 * TRAVERSE
 *
 * ---------------------------------------------------------------------*)

fun TRAVERSE (data as {dprocs,rewriters,travrules,relation}) thms = 
   let val context' = add_context rewriters dprocs (initial_context data,thms)
   in TRAVERSE_IN_CONTEXT rewriters dprocs travrules [] context'
   end;

val _ = say "done!\n";

end (* struct *)






