(* ========================================================================= *)
(* Simplification and rewriting.                                             *)
(* ========================================================================= *)

type gconv == int * conv;;

(* ------------------------------------------------------------------------- *)
(* Primitive rewriting conversions: unconditional and conditional equations. *)
(* ------------------------------------------------------------------------- *)

let REWR_CONV = PART_MATCH lhs;;

let IMP_REWR_CONV = PART_MATCH (lhs o snd o dest_imp);;

(* ------------------------------------------------------------------------- *)
(* Versions with ordered rewriting. We must have l' > r' for the rewrite     *)
(* |- l = r (or |- c ==> (l = r)) to apply.                                  *)
(* ------------------------------------------------------------------------- *)

let ORDERED_REWR_CONV ord th =
  let basic_conv = REWR_CONV th in
  fun tm ->
    let thm = basic_conv tm in
    let l,r = dest_eq(concl thm) in
    if ord l r then thm
    else failwith "ORDERED_REWR_CONV: wrong orientation";;

let ORDERED_IMP_REWR_CONV ord th =
  let basic_conv = IMP_REWR_CONV th in
  fun tm ->
    let thm = basic_conv tm in
    let l,r = dest_eq(rand(concl thm)) in
    if ord l r then thm
    else failwith "ORDERED_IMP_REWR_CONV: wrong orientation";;

(* ------------------------------------------------------------------------- *)
(* Standard AC-compatible term ordering: a "dynamic" lexicographic ordering. *)
(*                                                                           *)
(* This is a slight hack to make AC normalization work. However I *think*    *)
(* it's properly AC compatible, i.e. monotonic and total, WF on ground terms *)
(* (over necessarily finite signature) and with the properties for any       *)
(* binary operator +:                                                        *)
(*                                                                           *)
(*         (x + y) + z > x + (y + z)                                         *)
(*         x + y > y + x                   iff x > y                         *)
(*         x + (y + z) > y + (x + z)       iff x > y                         *)
(*                                                                           *)
(* The idea is that when invoking lex ordering with identical head operator  *)
(* "f", one sticks "f" at the head of an otherwise arbitrary ordering on     *)
(* subterms (the built-in CAML one). This avoids the potentially inefficient *)
(* calculation of term size in the standard orderings.                       *)
(* ------------------------------------------------------------------------- *)

let term_order =
  let rec lexify ord l1 l2 =
    if l1 = [] then false
    else if l2 = [] then true else
    let h1 = hd l1 and h2 = hd l2 in
    ord h1 h2 or (h1 = h2 & lexify ord (tl l1) (tl l2)) in
  let rec dyn_order top tm1 tm2 =
    let f1,args1 = strip_comb tm1
    and f2,args2 = strip_comb tm2 in
    if f1 = f2 then
      lexify (dyn_order f1) args1 args2
    else
      if f2 = top then false
      else if f1 = top then true
      else f1 > f2 in
  dyn_order `T`;;

(* ------------------------------------------------------------------------- *)
(* Create a gconv net for a theorem as a (cond) rewrite. The "rep" flag      *)
(* will cause any trivially looping rewrites to be ignored, and any that are *)
(* permutative to be ordered w.r.t. the standard order. The idea is that     *)
(* this flag will be set iff the conversion is going to get repeated.        *)
(* ------------------------------------------------------------------------- *)

let net_of_thm rep th =
  let t = concl th in
  let lconsts = freesl (hyp th) in
  let matchable = can o term_match lconsts in
  try let l,r = dest_eq t in
      if rep & free_in l r then I
      else if rep & matchable l r & matchable r l then
        enter lconsts (l,(1,ORDERED_REWR_CONV term_order th))
      else enter lconsts (l,(1,REWR_CONV th))
  with Failure _ ->
      let l,r = dest_eq (rand t) in
      if rep & free_in l r then I
      else if rep &  matchable l r & matchable r l then
        enter lconsts (l,(3,ORDERED_IMP_REWR_CONV term_order th))
      else enter lconsts(l,(3,IMP_REWR_CONV th));;

(* ------------------------------------------------------------------------- *)
(* Create a gconv net for a conversion with a term index.                    *)
(* ------------------------------------------------------------------------- *)

let net_of_conv tm conv sofar =
  enter [] (tm,(2,conv)) sofar;;

(* ------------------------------------------------------------------------- *)
(* Create a gconv net for a congruence rule (in canonical form!)             *)
(* ------------------------------------------------------------------------- *)

let net_of_cong th sofar =
  let conc,n = repeat (fun (tm,m) -> snd(dest_imp tm),m+1) (concl th,0) in
  if n = 0 then failwith "net_of_cong: Non-implicational congruence" else
  let pat = lhs conc in
  let conv = PART_MATCH (lhand o funpow n rand) th in
  enter [] (pat,(4,conv)) sofar;;

(* ------------------------------------------------------------------------- *)
(* Rewrite maker for ordinary and conditional rewrites (via "cf" flag).      *)
(*                                                                           *)
(* We follow Don in going from ~(s = t) to (s = t) = F *and* (t = s) = F.    *)
(* Well, why not? However, we don't abandon s = t where FV(t) is not a       *)
(* subset of FV(s) in favour of (s = t) = T, as he does.                     *)
(* Note: looping rewrites are not discarded here, only when netted.          *)
(* ------------------------------------------------------------------------- *)

let mk_rewrites =
  let IMP_CONJ_CONV = REWR_CONV(ITAUT `p ==> q ==> r = p /\ q ==> r`)
  and IMP_EXISTS_RULE =
    let cnv = REWR_CONV(ITAUT `(!x. P x ==> Q) = (?x. P x) ==> Q`) in
    fun v th -> CONV_RULE cnv (GEN v th) in
  let collect_condition oldhyps th =
    let conds = subtract (hyp th) oldhyps in
    if conds = [] then th else
    let jth = itlist DISCH conds th in
    let kth = CONV_RULE (REPEATC IMP_CONJ_CONV) jth in
    let cond,eqn = dest_imp(concl kth) in
    let fvs = subtract (subtract (frees cond) (frees eqn)) (freesl oldhyps) in
    itlist IMP_EXISTS_RULE fvs kth in
  let rec split_rewrites oldhyps cf th sofar =
    let tm = concl th in
    if is_forall tm then
      split_rewrites oldhyps cf (SPEC_ALL th) sofar
    else if is_conj tm then
      split_rewrites oldhyps cf (CONJUNCT1 th)
        (split_rewrites oldhyps cf (CONJUNCT2 th) sofar)
    else if is_imp tm & cf then
      split_rewrites oldhyps cf (UNDISCH th) sofar
    else if is_eq tm then
      (if cf then collect_condition oldhyps th else th)::sofar
    else if is_neg tm then
      let ths = split_rewrites oldhyps cf (EQF_INTRO th) sofar in
      if is_eq (rand tm)
      then split_rewrites oldhyps cf (EQF_INTRO (GSYM th)) ths
      else ths
    else
      split_rewrites oldhyps cf (EQT_INTRO th) sofar in
  fun cf th sofar -> split_rewrites (hyp th) cf th sofar;;

(* ------------------------------------------------------------------------- *)
(* Rewriting (and application of other conversions) based on a convnet.      *)
(* ------------------------------------------------------------------------- *)

let REWRITES_CONV net tm =
  let pconvs = lookup tm net in
  try tryfind (fun (_,cnv) -> cnv tm) pconvs
  with Failure _ -> failwith "REWRITES_CONV";;

(* ------------------------------------------------------------------------- *)
(* Decision procedures may accumulate their state in different ways (e.g.    *)
(* term nets and predicate-indexed lists of Horn clauses). To allow mixing   *)
(* of arbitrary types for state storage, we use a trick due to RJB via DRS.  *)
(* ------------------------------------------------------------------------- *)

type prover = Prover of conv * (thm list -> prover);;

let mk_prover applicator augmentor =
  let rec mk_prover state =
    let apply = applicator state
    and augment thms = mk_prover (augmentor state thms) in
    Prover(apply,augment) in
  mk_prover;;

let augment(Prover(_,aug)) thms = aug thms;;

let apply(Prover(conv,_)) tm = conv tm;;

(* ------------------------------------------------------------------------- *)
(* Example, not actually used.                                               *)
(* ------------------------------------------------------------------------- *)

let mk_rewrite_prover =
  let add_thms_to_net net thms =
    itlist (net_of_thm true) (itlist (mk_rewrites false) thms []) net in
  mk_prover REWRITES_CONV add_thms_to_net empty_net;;

(* ------------------------------------------------------------------------- *)
(* Set tableaux up as a prover: state is a goalstate with conclusion "F",    *)
(* and the augmentor does a CHOOSE_TAC with Skolemized input theorems.       *)
(* ------------------------------------------------------------------------- *)

(****

let CTAUT_PROVER =
  let CANON = CONV_RULE (NNF_CONV THENC SKOLEM_CONV) in
  let refn = by o REPEAT_TCL CHOOSE_THEN ASSUME_TAC o CANON in
  let DENEG = CONV_RULE(REWR_CONV(TAUT `(~p ==> F) = p`)) in
  let applicator gs tm =
    let tm' = mk_neg tm in
    let th = ASSUME tm' in
    let _,[],just = by CTAUT_TAC (refn th gs) in
    DENEG (DISCH tm' (just null_inst [])) in
  let augmentor gs thl =
    itlist refn thl gs in
  mk_prover applicator augmentor (mk_goalstate([],`F`));;

*****)

(* ------------------------------------------------------------------------- *)
(* Type of simpsets. We have a convnet containing rewrites (implicational    *)
(* and otherwise), other term-indexed context-free conversions like          *)
(* BETA_CONV, and congruence rules. Then there is a list of provers that     *)
(* have their own way of storing and using context, and finally a rewrite    *)
(* maker function, to allow customization.                                   *)
(*                                                                           *)
(* We also have a type of (traversal) strategy, following Konrad.            *)
(* ------------------------------------------------------------------------- *)

type simpset =
  Simpset of gconv net                          (* Rewrites & congruences *)
           * (strategy -> strategy)             (* Prover for conditions  *)
           * prover list                        (* Subprovers for prover  *)
           * (thm -> thm list -> thm list)      (* Rewrite maker          *)

and strategy == simpset -> int -> term -> thm;;

(* ------------------------------------------------------------------------- *)
(* Very simple prover: recursively simplify, try provers, or ASSUME it.      *)
(* ------------------------------------------------------------------------- *)

let basic_prover strat (Simpset(net,prover,provers,rewmaker) as ss) lev tm =
  let sth = try strat ss lev tm with Failure _ -> REFL tm in
  try EQT_ELIM sth
  with Failure _ -> try
    let tth = tryfind (fun pr -> apply pr (rand(concl sth))) provers in
    EQ_MP (SYM sth) tth
  with Failure _ -> ASSUME tm;;

(* ------------------------------------------------------------------------- *)
(* Functions for changing or augmenting components of simpsets.              *)
(* ------------------------------------------------------------------------- *)

let ss_of_thms thms (Simpset(net,prover,provers,rewmaker)) =
  let cthms = itlist rewmaker thms [] in
  let net' = itlist (net_of_thm true) cthms net in
  Simpset(net',prover,provers,rewmaker);;

let ss_of_conv keytm conv (Simpset(net,prover,provers,rewmaker)) =
  let net' = net_of_conv keytm conv net in
  Simpset(net',prover,provers,rewmaker);;

let ss_of_congs thms (Simpset(net,prover,provers,rewmaker)) =
  let net' = itlist net_of_cong thms net in
  Simpset(net',prover,provers,rewmaker);;

let ss_of_prover newprover (Simpset(net,_,provers,rewmaker)) =
  Simpset(net,newprover,provers,rewmaker);;

let ss_of_provers newprovers (Simpset(net,prover,provers,rewmaker)) =
  Simpset(net,prover,newprovers@provers,rewmaker);;

let ss_of_maker newmaker (Simpset(net,prover,provers,_)) =
  Simpset(net,prover,provers,newmaker);;

(* ------------------------------------------------------------------------- *)
(* Perform a context-augmentation operation on a simpset.                    *)
(* ------------------------------------------------------------------------- *)

let AUGMENT_SIMPSET cth (Simpset(net,prover,provers,rewmaker)) =
  let provers' = map (C augment [cth]) provers in
  let cthms = rewmaker cth [] in
  let net' = itlist (net_of_thm true) cthms net in
  Simpset(net',prover,provers',rewmaker);;

(* ------------------------------------------------------------------------- *)
(* Try to find a valid rewrite from a gconv list; if necessary use a         *)
(* conditional rewrite if the condition can be solved.                       *)
(* ------------------------------------------------------------------------- *)

let IMP_REWRITES_CONV strat (Simpset(net,prover,provers,rewmaker) as ss) lev
                      pconvs tm =
  tryfind (fun (n,cnv) ->
    if n >= 4 then fail() else
    let th = cnv tm in
    let etm = concl th in
    if is_eq etm then th else
    if lev <= 0 then failwith "IMP_REWRITES_CONV: Too deep" else
    let cth = prover strat ss (lev-1) (lhand etm) in
    MP th cth) pconvs;;

(* ------------------------------------------------------------------------- *)
(* Run through the arms of a congruence theorem using (contextual            *)
(* augmentations of) the simpset "ss" and strategy "strat".                  *)
(* ------------------------------------------------------------------------- *)

let rec RUN_SUB_CONV strat ss lev triv th =
  let tm = concl th in
  if is_imp tm then
    let subtm = lhand tm in
    let avs,bod = strip_forall subtm in
    let (t,t'),ss',mk_fun =
      try dest_eq bod,ss,I with Failure _ ->
      let cxt,deq = dest_imp bod in
      dest_eq deq,AUGMENT_SIMPSET (ASSUME cxt) ss,DISCH cxt in
    let eth,triv' = try strat ss' lev t,false with Failure _ -> REFL t,triv in
    let th' = MP (INST [rand(concl eth),t'] th) (GENL avs (mk_fun eth)) in
    RUN_SUB_CONV strat ss lev triv' th'
  else if triv then fail() else th;;

(* ------------------------------------------------------------------------- *)
(* Try performing a SUB_CONV of strategy "strat" using (contextual           *)
(* augmentations of) the simpset "ss".                                       *)
(* ------------------------------------------------------------------------- *)

let GEN_SUB_CONV strat ss lev pconvs tm =
  try tryfind (fun (n,cnv) ->
        if n < 4 then fail() else
        let th = cnv tm in
        RUN_SUB_CONV strat ss lev true th) pconvs
  with Failure _ ->
      if is_comb tm then
        let l,r = dest_comb tm in
        try let th1 = strat ss lev l in
            try let th2 = strat ss lev r in MK_COMB(th1,th2)
            with Failure _ -> AP_THM th1 r
        with Failure _ -> AP_TERM l (strat ss lev r)
      else if is_abs tm then
        let v,bod = dest_abs tm in
        let th = strat ss lev bod in
        try ABS v th with Failure _ ->
        let gv = genvar(type_of v) in
        let gbod = vsubst[gv,v] bod in
        let gth = ABS gv (strat ss lev gbod) in
        let gtm = concl gth in
        let l,r = dest_eq gtm in
        let v' = variant (frees gtm) v in
        let l' = alpha v' l and r' = alpha v' r in
        EQ_MP (ALPHA gtm (mk_eq(l',r'))) gth
      else failwith "GEN_SUB_CONV";;

(* ------------------------------------------------------------------------- *)
(* Depth conversions.                                                        *)
(* ------------------------------------------------------------------------- *)

let rec ONCE_DEPTH_SQCONV (Simpset(net,prover,provers,rewmaker) as ss) lev tm =
  let pconvs = lookup tm net in
  try IMP_REWRITES_CONV ONCE_DEPTH_SQCONV ss lev pconvs tm
  with Failure _ ->
      GEN_SUB_CONV ONCE_DEPTH_SQCONV ss lev pconvs tm;;

let rec DEPTH_SQCONV (Simpset(net,prover,provers,rewmaker) as ss) lev tm =
  let pconvs = lookup tm net in
  try let th1 = GEN_SUB_CONV DEPTH_SQCONV ss lev pconvs tm in
      let tm1 = rand(concl th1) in
      let pconvs1 = lookup tm1 net in
      try TRANS th1 (IMP_REWRITES_CONV DEPTH_SQCONV ss lev pconvs1 tm1)
      with Failure _ -> th1
  with Failure _ ->
      IMP_REWRITES_CONV DEPTH_SQCONV ss lev pconvs tm;;

let rec REDEPTH_SQCONV (Simpset(net,prover,provers,rewmaker) as ss) lev tm =
  let pconvs = lookup tm net in
  let th =
    try let th1 = GEN_SUB_CONV REDEPTH_SQCONV ss lev pconvs tm in
        let tm1 = rand(concl th1) in
        let pconvs1 = lookup tm1 net in
        try TRANS th1 (IMP_REWRITES_CONV REDEPTH_SQCONV ss lev pconvs1 tm1)
        with Failure _ -> th1
    with Failure _ ->
        IMP_REWRITES_CONV REDEPTH_SQCONV ss lev pconvs tm in
  try let th' = REDEPTH_SQCONV ss lev (rand(concl th)) in
      TRANS th th'
  with Failure _ -> th;;

let rec TOP_DEPTH_SQCONV (Simpset(net,prover,provers,rewmaker) as ss) lev tm =
  let pconvs = lookup tm net in
  let th1 =
    try IMP_REWRITES_CONV TOP_DEPTH_SQCONV ss lev pconvs tm
    with Failure _ -> GEN_SUB_CONV TOP_DEPTH_SQCONV ss lev pconvs tm in
  try let th2 = TOP_DEPTH_SQCONV ss lev (rand(concl th1)) in
          TRANS th1 th2
  with Failure _ -> th1;;

let rec TOP_SWEEP_SQCONV (Simpset(net,prover,provers,rewmaker) as ss) lev tm =
  let pconvs = lookup tm net in
  try let th1 = IMP_REWRITES_CONV TOP_SWEEP_SQCONV ss lev pconvs tm in
      try let th2 = TOP_SWEEP_SQCONV ss lev (rand(concl th1)) in
          TRANS th1 th2
      with Failure _ -> th1
  with Failure _ -> GEN_SUB_CONV TOP_SWEEP_SQCONV ss lev pconvs tm;;

(* ------------------------------------------------------------------------- *)
(* Simplification functions.                                                 *)
(* ------------------------------------------------------------------------- *)

let GEN_SIMP_CONV (strat:strategy) ss lev thl =
  let ss' = itlist AUGMENT_SIMPSET thl ss in
  TRY_CONV (strat ss' lev);;

let ONCE_SIMP_CONV ss = GEN_SIMP_CONV ONCE_DEPTH_SQCONV ss 1;;

let SIMP_CONV ss = GEN_SIMP_CONV TOP_DEPTH_SQCONV ss 3;;

let GEN_SIMP_RULE strat ss lev thl =
  CONV_RULE(GEN_SIMP_CONV strat ss lev thl);;

let ONCE_SIMP_RULE ss thl = CONV_RULE(ONCE_SIMP_CONV ss thl);;

let SIMP_RULE ss thl = CONV_RULE(SIMP_CONV ss thl);;

let GEN_SIMP_TAC strat ss lev thl = CONV_TAC(GEN_SIMP_CONV strat ss lev thl);;

let ONCE_SIMP_TAC ss thl = CONV_TAC(ONCE_SIMP_CONV ss thl);;

let SIMP_TAC ss thl = CONV_TAC(SIMP_CONV ss thl);;

let GEN_ASM_SIMP_TAC strat ss lev = ASM (GEN_SIMP_TAC strat ss lev);;

let ONCE_ASM_SIMP_TAC ss = ASM (ONCE_SIMP_TAC ss);;

let ASM_SIMP_TAC ss = ASM (SIMP_TAC ss);;

(* ------------------------------------------------------------------------- *)
(* Maintenence of basic rewrites and conv nets for rewriting.                *)
(* ------------------------------------------------------------------------- *)

let set_basic_rewrites,extend_basic_rewrites,basic_rewrites,basic_net =
  let rewrites = ref ([]:thm list)
  and conv_net = ref (empty_net: gconv net) in
  let set_basic_rewrites thl =
    let canon_thl = itlist (mk_rewrites false) thl [] in
    (rewrites := canon_thl;
     conv_net := itlist (net_of_thm true) canon_thl empty_net) in
  let extend_basic_rewrites thl =
    let canon_thl = itlist (mk_rewrites false) thl [] in
     (rewrites := canon_thl @ !rewrites;
      conv_net := itlist (net_of_thm true) canon_thl (!conv_net)) in
  let basic_rewrites() = !rewrites in
  let basic_net() = !conv_net in
  set_basic_rewrites,extend_basic_rewrites,basic_rewrites,basic_net;;

(* ------------------------------------------------------------------------- *)
(* Main rewriting conversions.                                               *)
(* ------------------------------------------------------------------------- *)

let GENERAL_REWRITE_CONV rep (cnvl:conv->conv) (builtin_net:gconv net) thl =
  let thl_canon = itlist (mk_rewrites false) thl [] in
  let final_net = itlist (net_of_thm rep) thl_canon builtin_net in
  cnvl (REWRITES_CONV final_net);;

let GEN_REWRITE_CONV (cnvl:conv->conv) thl =
  GENERAL_REWRITE_CONV false cnvl empty_net thl;;

let PURE_REWRITE_CONV thl =
  GENERAL_REWRITE_CONV true TOP_DEPTH_CONV empty_net thl;;

let REWRITE_CONV thl =
  GENERAL_REWRITE_CONV true TOP_DEPTH_CONV (basic_net()) thl;;

let PURE_ONCE_REWRITE_CONV thl =
  GENERAL_REWRITE_CONV false ONCE_DEPTH_CONV empty_net thl;;

let ONCE_REWRITE_CONV thl =
  GENERAL_REWRITE_CONV false ONCE_DEPTH_CONV (basic_net()) thl;;

(* ------------------------------------------------------------------------- *)
(* The following is occasionally useful as a hack.                           *)
(* ------------------------------------------------------------------------- *)

let LIMITED_REWRITE_CONV =
  let LIMITED_ONCE_REWRITE_CONV ths =
    GEN_REWRITE_CONV ONCE_DEPTH_CONV ths THENC
    GENERAL_REWRITE_CONV true TOP_DEPTH_CONV (basic_net()) [] in
  fun n ths tm ->
        funpow n (CONV_RULE(RAND_CONV(LIMITED_ONCE_REWRITE_CONV ths)))
                 (REFL tm);;

(* ------------------------------------------------------------------------- *)
(* Rewriting rules and tactics.                                              *)
(* ------------------------------------------------------------------------- *)

let GEN_REWRITE_RULE cnvl thl = CONV_RULE(GEN_REWRITE_CONV cnvl thl);;

let PURE_REWRITE_RULE thl = CONV_RULE(PURE_REWRITE_CONV thl);;

let REWRITE_RULE thl = CONV_RULE(REWRITE_CONV thl);;

let PURE_ONCE_REWRITE_RULE thl = CONV_RULE(PURE_ONCE_REWRITE_CONV thl);;

let ONCE_REWRITE_RULE thl = CONV_RULE(ONCE_REWRITE_CONV thl);;

let PURE_ASM_REWRITE_RULE thl th =
    PURE_REWRITE_RULE ((map ASSUME (hyp th)) @ thl) th;;

let ASM_REWRITE_RULE thl th =
    REWRITE_RULE ((map ASSUME (hyp th)) @ thl) th;;

let PURE_ONCE_ASM_REWRITE_RULE thl th =
    PURE_ONCE_REWRITE_RULE ((map ASSUME (hyp th)) @ thl) th;;

let ONCE_ASM_REWRITE_RULE thl th =
    ONCE_REWRITE_RULE ((map ASSUME (hyp th)) @ thl) th;;

let GEN_REWRITE_TAC cnvl thl = CONV_TAC(GEN_REWRITE_CONV cnvl thl);;

let PURE_REWRITE_TAC thl = CONV_TAC(PURE_REWRITE_CONV thl);;

let REWRITE_TAC thl = CONV_TAC(REWRITE_CONV thl);;

let PURE_ONCE_REWRITE_TAC thl = CONV_TAC(PURE_ONCE_REWRITE_CONV thl);;

let ONCE_REWRITE_TAC thl = CONV_TAC(ONCE_REWRITE_CONV thl);;

let (PURE_ASM_REWRITE_TAC: thm list -> tactic) =
  ASM PURE_REWRITE_TAC;;

let (ASM_REWRITE_TAC: thm list -> tactic) =
  ASM REWRITE_TAC;;

let (PURE_ONCE_ASM_REWRITE_TAC: thm list -> tactic) =
  ASM PURE_ONCE_REWRITE_TAC;;

let (ONCE_ASM_REWRITE_TAC: thm list -> tactic) =
  ASM ONCE_REWRITE_TAC;;

(* ------------------------------------------------------------------------- *)
(* Some basic simpsets.                                                      *)
(* ------------------------------------------------------------------------- *)

let empty_ss = Simpset(empty_net,basic_prover,[],mk_rewrites false);;

(******

set_interface_map real_interface_map;;

let COND_CONG =
  TAUT `(g = g') ==>
        (g' ==> (t = t')) ==>
        (~g' ==> (e = e')) ==>
        ((g => t | e) = (g' => t' | e'))`;;

let IMP_CONG =
  TAUT `(p = p') ==>
        (p' ==> (q = q')) ==>
        (p ==> q = p' ==> q')`;;

let hol_ss =
  ss_of_thms [REFL_CLAUSE; EQ_CLAUSES; NOT_CLAUSES_WEAK;
              AND_CLAUSES; OR_CLAUSES; IMP_CLAUSES;
              FORALL_SIMP; EXISTS_SIMP; ABS_SIMP;
              SELECT_REFL; NOT_CLAUSES; COND_CLAUSES;
              FST; SND]
  empty_ss;;

let holc_ss =
  (ss_of_congs [IMP_CONG; COND_CONG] o
   ss_of_maker (mk_rewrites true))
  hol_ss;;

let num_ss =
  ss_of_thms [ARITH;
              ADD_CLAUSES; MULT_CLAUSES;
              LE_REFL; LT_REFL]
  hol_ss;;

let real_ss =
  ss_of_thms  [REAL_ADD_LID; REAL_ADD_RID;
               REAL_MUL_LID; REAL_MUL_RID;
               REAL_MUL_LZERO; REAL_MUL_RZERO;
               REAL_NEG_0; REAL_MUL_LINV; REAL_MUL_RINV]
  holc_ss;;

let real_canon_ss =
 ss_of_thms [REAL_ADD_LDISTRIB; REAL_ADD_RDISTRIB;
             REAL_SUB_LDISTRIB; REAL_SUB_RDISTRIB;
             REAL_MUL_LNEG; REAL_MUL_RNEG; REAL_NEG_NEG;
             REAL_ARITH `--a + b = b - a`;
             REAL_ARITH `a + --b = a - b`;
             REAL_ARITH `a - --b = a + b`;
             REAL_ARITH `--a - b = --(a + b)`;
             REAL_ARITH `a + (b - c) = (a + b) - c`;
             REAL_ARITH `a - (b - c) = a + (c - b)`;
             REAL_ARITH `(a - b) + c = (a + c) - b`;
             REAL_ARITH `(a - b) - c = a - (b + c)`;
             REAL_ARITH `(a - b = c) = (a = b + c)`;
             REAL_ARITH `(a = b - c) = (a + c = b)`;
             REAL_ARITH `(--a = b)   = (a + b = &0)`;
             REAL_ARITH `(a = --b)   = (a + b = &0)`;
             REAL_ARITH `a - b <= c  = a <= b + c`;
             REAL_ARITH `--a <= b    = &0 <= a + b`;
             REAL_ARITH `a <= --b    = a + b <= &0`;
             REAL_ARITH `a <= b - c  = a + c <= b`;
             REAL_ARITH `a - b < c   = a < b + c`;
             REAL_ARITH `--a < b     = &0 < a + b`;
             REAL_ARITH `a < --b     = a + b < &0`;
             REAL_ARITH `a < b - c  = a + c < b`;
             REAL_ADD_AC; REAL_MUL_AC]
  real_ss;;

let example_ss =
  (ss_of_congs [COND_CONG; IMP_CONG] o
   ss_of_conv `(\x. y) z` BETA_CONV o
   ss_of_thms [AND_CLAUSES; OR_CLAUSES; IMP_CLAUSES; EQ_CLAUSES;
               NOT_CLAUSES; PAIR; FST; SND] o
   ss_of_maker (mk_rewrites false))
  empty_ss;;


time (SIMP_CONV example_ss []) `P x /\ x num_lt 1 ==> P x \/ Q x`;;

time (SIMP_CONV real_ss []) `~(&0 * &34 = x * &1) ==>
                      (y * x * inv x * inv x * x = y)`;;

time (SIMP_CONV real_ss []) `~(&0 * &34 = x * &1) ==>
                             (z = y * inv x * x) ==> (y * x * inv x = z)`;;

time (SIMP_CONV real_ss []) `~(&0 * &34 = x * &1) ==>
                             ~(y * inv x * x = &0) ==> (z * inv(y) * y = z)`;;

******)
