(* ========================================================================= *)
(* Convenient library functions used a lot in HOL88.                         *)
(* ========================================================================= *)

let fail() = failwith "";;

exception Interrupt;;

unix__signal unix__SIGINT (unix__Signal_handle (fun () -> raise Interrupt));;

(* ------------------------------------------------------------------------- *)
(* Combinators.                                                              *)
(* ------------------------------------------------------------------------- *)

let curry f x y = f(x,y);;

let uncurry f(x,y) = f x y;;

let I x = x;;

let K x y = x;;

let C f x y = f y x;;

let W f x = f x x;;

let o f g x = f(g x);; #infix "o";;

let F_F f g (x,y) = (f x,g y);; #infix "F_F";;

(* ------------------------------------------------------------------------- *)
(* Attempting function or predicate applications.                            *)
(* ------------------------------------------------------------------------- *)

let can f x = try (f x; true) with Failure _ -> false;;

let assert p x = if p x then x else failwith "assert";;

(* ------------------------------------------------------------------------- *)
(* Repetition of a function.                                                 *)
(* ------------------------------------------------------------------------- *)

let rec funpow n f x =
  if n < 1 then x else funpow (n-1) f (f x);;

let rec repeat f x =
  try let y = f x in repeat f y with Failure _ -> x;;

(* ------------------------------------------------------------------------- *)
(* To avoid consing in various situations, we propagate this exception.      *)
(* ------------------------------------------------------------------------- *)

exception Unchanged;;

let qcomb2 con fn1 fn2 (l,r) =
  try let l' = fn1 l in
      try let r' = fn2 r in con(l',r')
      with Unchanged -> con(l',r)
  with Unchanged ->
      let r' = fn2 r in con(l,r');;

let qcomb con fn (l,r) =
  try let l' = fn l in
      try let r' = fn r in con(l',r')
      with Unchanged -> con(l',r)
  with Unchanged ->
      let r' = fn r in con(l,r');;

let rec qmap fn l =
  if l = [] then raise Unchanged else
  let h = hd l and t = tl l in
  try let t' = qmap fn t in
      let h' = try fn h with Unchanged -> h in
      h'::t'
  with Unchanged ->
    let h' = fn h in h'::t;;

let qtry fn arg =
  try fn arg with Unchanged -> arg;;

(* ------------------------------------------------------------------------- *)
(* We have "lazy" objects to delay calculation and avoid recalculation.      *)
(* ------------------------------------------------------------------------- *)

type ('a,'b)Lazysum = Unrealized of (('a ->'b) * 'a) | Realized of ('b);;

let lazy f x = ref(Unrealized(f,x));;

let eager y = ref(Realized(y));;

let eval r =
  match !r with
    Realized(y) -> y
  | Unrealized(f,x) -> let y = f(x) in (r := Realized(y); y);;

(* ------------------------------------------------------------------------- *)
(* The most basic list operations.                                           *)
(* ------------------------------------------------------------------------- *)

let hd =
  fun [] -> failwith "hd"
    | (h::_) -> h;;

let tl =
  fun [] -> failwith "tl"
    | (_::t) -> t;;

let map f =
  let rec mapf l =
    match l with
      [] -> []
    | (x::t) -> let y = f x in y::(mapf t) in
  mapf;;

let rec last =
  fun [x] -> x
    | (h::t) -> last t
    | [] -> failwith "last";;

let rec butlast =
  fun [_] -> []
    | (h::t) -> h::(butlast t)
    | [] -> failwith "butlast";;

let rec el =
  fun 0 (h::_) -> h
    | n (_::t) -> el (n - 1) t
    | _ _ -> failwith "el";;

(* ------------------------------------------------------------------------- *)
(* Various versions of list iteration.                                       *)
(* ------------------------------------------------------------------------- *)

let rec itlist f =
    fun [] b -> b
      | (h::t) b -> f h (itlist f t b);;

let rec rev_itlist f =
    fun [] b -> b
      | (h::t) b -> rev_itlist f t (f h b);;

let rec end_itlist f =
    fun []     -> failwith "end_itlist"
      | [x]    -> x
      | (h::t) -> f h (end_itlist f t);;

let rec itlist2 f =
    fun [] [] b -> b
      | (h1::t1) (h2::t2) b -> f h1 h2 (itlist2 f t1 t2 b)
      | _ _ _ -> failwith "itlist2";;

let rec rev_itlist2 f =
    fun [] [] b -> b
      | (h1::t1) (h2::t2) b -> rev_itlist2 f t1 t2 (f h1 h2 b)
      | _ _ _ -> failwith "rev_itlist2";;

(* ------------------------------------------------------------------------- *)
(* Iterative splitting (list) and stripping (tree) via destructor.           *)
(* ------------------------------------------------------------------------- *)

let rec splitlist dest x =
  try let l,r = dest x in
      let ls,res = splitlist dest r in
      (l::ls,res)
  with Failure _ -> ([],x);;

let rev_splitlist dest =
  let rec rsplist ls x =
    try let l,r = dest x in
        rsplist (r::ls) l
    with Failure _ -> (x,ls) in
  fun x -> rsplist [] x;;

let striplist dest =
  let rec strip x acc =
    try let l,r = dest x in
        strip l (strip r acc)
    with Failure _ -> x::acc in
  fun x -> strip x [];;

(* ------------------------------------------------------------------------- *)
(* Apply a destructor as many times as elements in list.                     *)
(* ------------------------------------------------------------------------- *)

let rec nsplit dest clist x =
  if clist = [] then [],x else
  let l,r = dest x in
  let ll,y = nsplit dest (tl clist) r in
  l::ll,y;;

(* ------------------------------------------------------------------------- *)
(* Replication and sequences.                                                *)
(* ------------------------------------------------------------------------- *)

let rec replicate x n =
    if n < 1 then []
    else x::(replicate x (n - 1));;

let upto n =
  let rec down l n = if n < 0 then l else down (n::l) (n - 1) in
  down [] n;;

(* ------------------------------------------------------------------------- *)
(* Various useful list operations.                                           *)
(* ------------------------------------------------------------------------- *)

let forall p l = itlist (fun h a -> p(h) & a) l true;;

let forall2 p l1 l2 = itlist2 (fun h1 h2 a -> p h1 h2 & a) l1 l2 true;;

let exists p l = itlist (fun h a -> p(h) or a) l false;;

let length =
  let rec len k l =
    if l = [] then k else len (k + 1) (tl l) in
  fun l -> len 0 l;;

let filter p =
  let rec ffilter =
    fun [] -> raise Unchanged
      | (h::t) -> if p h then h::(ffilter t) else qtry ffilter t in
  fun l -> qtry ffilter l;;

let rec mapfilter f l =
  match l with
    [] -> []
  | (h::t) -> let rest = mapfilter f t in
              try (f h)::rest with Failure _ -> rest;;

let rec find p =
  fun [] -> failwith "find"
    | (h::t) -> if p(h) then h else find p t;;

let rec tryfind f =
  fun [] -> failwith "tryfind"
    | (h::t) -> try f h with Failure _ -> tryfind f t;;

let flat l = itlist (prefix @) l [];;

let partition p l =
    itlist (fun a (yes,no) -> if p a then a::yes,no else yes,a::no) l ([],[]);;

let gather p l = itlist (fun x y -> if (p x) then x::y else y) l [];;

let rec remove p l =
  match l with
    [] -> failwith "remove"
  | (h::t) -> if p(h) then h,t else
              let y,n = remove p t in y,h::n;;

let rec chop_list n l =
  if n = 0 then [],l else
  try let m,l' = chop_list (n-1) (tl l) in (hd l)::m,l'
  with Failure _ -> failwith "chop_list";;

let index x =
  let rec ind n =
    fun [] -> failwith "index"
      | (h::t) -> if x = h then n else ind (n + 1) t in
  ind 0;;

(* ------------------------------------------------------------------------- *)
(* "Set" operations on lists.                                                *)
(* ------------------------------------------------------------------------- *)

let rec mem x =
  fun [] -> false
    | (h::t) -> x = h or mem x t;;

let insert x l =
  if mem x l then l else x::l;;

let union l1 l2 = itlist insert l1 l2;;

let Union l = itlist union l [];;

let intersect l1 l2 = filter (fun x -> mem x l2) l1;;

let subtract l1 l2 = filter (fun x -> not mem x l2) l1;;

let subset l1 l2 = forall (fun t -> mem t l2) l1;;

let set_eq l1 l2 = subset l1 l2 & subset l2 l1;;

(* ------------------------------------------------------------------------- *)
(* Association lists.                                                        *)
(* ------------------------------------------------------------------------- *)

let assoc x l = snd(find (fun p -> fst p = x) l);;

let rev_assoc x l = fst(find (fun p -> snd p = x) l);;

let rec assoc2 x (l,r) =
  if x = hd l then hd r else assoc2 x (tl l,tl r);;

(* ------------------------------------------------------------------------- *)
(* Zipping, unzipping etc.                                                   *)
(* ------------------------------------------------------------------------- *)

let rec zip =
    fun [] [] -> []
      | (h1::t1) (h2::t2) -> (h1,h2)::(zip t1 t2)
      | _ _ -> failwith "zip";;

let rec unzip =
  fun [] -> [],[]
    | ((a,b)::rest) -> let alist,blist = unzip rest in
                       (a::alist,b::blist);;

(* ------------------------------------------------------------------------- *)
(* Sorting.                                                                  *)
(* ------------------------------------------------------------------------- *)

let rec sort cmp lis =
  match lis with
    [] -> []
  | piv::rest ->
      let r,l = partition (curry cmp piv) rest in
      (sort cmp l) @ (piv::(sort cmp r));;

(* ------------------------------------------------------------------------- *)
(* Removing adjacent (NB!) equal elements from list. May raise Unchanged.    *)
(* ------------------------------------------------------------------------- *)

let uniq =
  let rec uniq x l =
    match l with
      [] -> raise Unchanged
    | (h::t) -> if x = h then
                  try uniq x t
                  with Unchanged -> l
                else x::(uniq h t) in
  fun l -> if l = [] then [] else uniq (hd l) (tl l);;

(* ------------------------------------------------------------------------- *)
(* Convert list into set by eliminating duplicates.                          *)
(* ------------------------------------------------------------------------- *)

let setify s =
  let s' = sort (uncurry prefix<=) s in
  try uniq s' with Unchanged -> s;;

(* ------------------------------------------------------------------------- *)
(* Insertion in a set, preserving the (assumed) sortedness over "<".         *)
(* ------------------------------------------------------------------------- *)

let set_insert =
  let canon_eq x y = try x = y with Invalid_argument _ -> false
  and canon_lt x y = try x < y with Invalid_argument _ -> false in
  let rec sinsert x l =
    if l = [] then [x] else
    let h = hd l in
    if canon_eq h x then failwith "sinsert" else
    if canon_lt x h then x::l else
    h::(sinsert x (tl l)) in
  fun x l -> try sinsert x l with Failure "sinsert" -> l;;

(* ------------------------------------------------------------------------- *)
(* Union of two sets, preserving the (assumed) sortedness over "<".          *)
(* ------------------------------------------------------------------------- *)

let rec set_merge l1 l2 =
  let canon_eq x y = try x = y with Invalid_argument _ -> false
  and canon_lt x y = try x < y with Invalid_argument _ -> false in
  if l1 = [] then l2
  else if l2 = [] then l1 else
  let h1 = hd l1 and t1 = tl l1
  and h2 = hd l2 and t2 = tl l2 in
  if canon_eq h1 h2 then h1::(set_merge t1 t2)
  else if canon_lt h1 h2 then h1::(set_merge t1 l2)
  else h2::(set_merge l1 t2);;

(* ------------------------------------------------------------------------- *)
(* String operations (surely there is a better way...)                       *)
(* ------------------------------------------------------------------------- *)

let implode l = itlist (prefix ^) l "";;

let explode s =
  let rec exap n l =
      if n < 0 then l else
      exap (n - 1) ((sub_string s n 1)::l) in
  exap (string_length s - 1) [];;

(* ------------------------------------------------------------------------- *)
(* Multiset operations.                                                      *)
(* ------------------------------------------------------------------------- *)

let rec munion l1 l2 =
  if l1 = [] then l2
  else if l2 = [] then l1 else
  let h1 = hd l1 in
  try let _,l2' = remove (prefix= h1) l2 in
      h1::(munion (tl l1) l2')
  with Failure _ ->
      h1::(munion (tl l1) l2);;

let rec msubtract l1 l2 =
  if l2 = [] then l1
  else if l1 = [] then l1 else
  let h2 = hd l2 in
  try let _,l1' = remove (prefix= h2) l1 in
      msubtract l1' (tl l2)
  with Failure _ -> msubtract l1 (tl l2);;

(* ------------------------------------------------------------------------- *)
(* Setting of flags.                                                         *)
(* ------------------------------------------------------------------------- *)

let flags,get_flag_value,new_flag,set_flag =
  let flag_store = ref ([]: (string * bool ref) list) in
  let flags () = map (fun p -> fst p) (!flag_store)
  and get_flag_value s = !(assoc s (!flag_store)) in
  let new_flag (s,b) = if can get_flag_value s then failwith "new_flag"
                       else flag_store := (s,ref b)::(!flag_store)
  and set_flag (s,b) =
    find (fun (t,c) -> if s = t then (c := b; true) else false)
         (!flag_store) in
  flags,get_flag_value,new_flag,set_flag;;

(* ------------------------------------------------------------------------- *)
(* GCD and LCM.                                                              *)
(* ------------------------------------------------------------------------- *)

let abs x = if x < 0 then -x else x;;

let sgn(x) = x >= 0;;

let gcd =
  let rec gxd x y =
    if y = 0 then x else gxd y (x mod y) in
  fun x y -> if x < y then gxd y x else gxd x y;;

let lcm x y = (x * y) / gcd x y;;

(* ------------------------------------------------------------------------- *)
(* All pairs arising from applying a function over two lists.                *)
(* ------------------------------------------------------------------------- *)

let allpairs f l1 l2 =
  itlist (union o C map l2 o f) l1 [];;

(* ------------------------------------------------------------------------- *)
(* Issue a report with a newline.                                            *)
(* ------------------------------------------------------------------------- *)

let report s =
  print_string s; print_newline();;

(* ------------------------------------------------------------------------- *)
(* Convenient function for issuing a warning.                                *)
(* ------------------------------------------------------------------------- *)

let warn cond s =
  if cond then report ("Warning: "^s) else ();;

(* ------------------------------------------------------------------------- *)
(* Flags to switch on verbose mode.                                          *)
(* ------------------------------------------------------------------------- *)

let verbose = ref true;;
let report_timing = ref true;;

(* ------------------------------------------------------------------------- *)
(* Switchable version of "report".                                           *)
(* ------------------------------------------------------------------------- *)

let remark s =
  if !verbose then report s else ();;

(* ------------------------------------------------------------------------- *)
(* Time a function.                                                          *)
(* ------------------------------------------------------------------------- *)

let time f x =
  let {unix__tms_utime=start_time} = unix__times() in
  let result = f x in
  let {unix__tms_utime=finish_time} = unix__times() in
  if !report_timing then
    report ("CPU time (user): "^(string_of_float(finish_time -. start_time)))
  else ();
  result;;

(* ------------------------------------------------------------------------- *)
(* Prolog exception.                                                         *)
(* ------------------------------------------------------------------------- *)

exception Cut;;
