(* ========================================================================= *)
(* More syntax constructors, and prelogical utilities like matching.         *)
(* ========================================================================= *)

let genvar =
  let gcounter = ref 0 in
  fun ty -> let count = !gcounter in
             (gcounter := count + 1;
              mk_var("_"^(string_of_int count),ty));;

(* ------------------------------------------------------------------------- *)
(* Similar to variant, but even avoids constants, and ignores types.         *)
(* ------------------------------------------------------------------------- *)

let mk_primed_var =
  let rec svariant avoid s =
    if mem s avoid or can get_const_type s then
      svariant avoid (s^"'")
    else s in
  fun avoid v ->
    let s,ty = dest_var v in
    let s' = svariant (mapfilter (fst o dest_var) avoid) s in
    mk_var(s',ty);;

(* ------------------------------------------------------------------------- *)
(* Is a variable free in a term? A more general but less efficient version   *)
(* for two arbitrary terms comes later in this file.                         *)
(* ------------------------------------------------------------------------- *)

let rec vfree_in v tm =
  if is_var tm then tm = v
  else if is_const tm then false
  else if is_comb tm then
    let l,r = dest_comb tm in
    vfree_in v l or vfree_in v r
  else
    let bv,bod = dest_abs tm in
    if bv = v then false else vfree_in v bod;;

(* ------------------------------------------------------------------------- *)
(* General substitution (for any free expression).                           *)
(* ------------------------------------------------------------------------- *)

let subst =
  let mk_qcomb = qcomb mk_comb in
  let rec ssubst ilist tm =
    if ilist = [] then raise Unchanged else
    try fst (find (aconv tm o snd) ilist) with Failure _ ->
    if is_comb tm then
      mk_qcomb (ssubst ilist) (dest_comb tm)
    else if is_abs tm then
      let v,bod = dest_abs tm in
      mk_abs(v,ssubst (filter(prefix not o vfree_in v o snd) ilist) bod)
    else raise Unchanged in
  fun ilist ->
    let ts,xs = split ilist in
    let gs = map (genvar o type_of) xs in
    fun tm -> try vsubst (zip ts gs) (ssubst (zip gs xs) tm)
              with Unchanged -> tm;;

(* ------------------------------------------------------------------------- *)
(* Produces a sequence of variants, considering previous inventions.         *)
(* ------------------------------------------------------------------------- *)

let rec variants av vs =
  if vs = [] then [] else
  let vh = variant av (hd vs) in vh::(variants (vh::av) (tl vs));;

(* ------------------------------------------------------------------------- *)
(* Alpha conversion term operation.                                          *)
(* ------------------------------------------------------------------------- *)

let alpha v tm =
  let v0,bod = try dest_abs tm
               with Failure _ -> failwith "alpha: Not an abstraction"in
  if v = v0 then tm else
  if type_of v = type_of v0 & not vfree_in v bod then
    mk_abs(v,vsubst[v,v0]bod)
  else failwith "alpha: Invalid new variable";;

(* ------------------------------------------------------------------------- *)
(* Type matching.                                                            *)
(* ------------------------------------------------------------------------- *)

let rec type_match vty cty sofar =
  if is_vartype vty then
     if vty = cty
     or try rev_assoc vty sofar = cty with Failure _ -> false
     then sofar else (cty,vty)::sofar
  else
     let vop,vargs = dest_type vty and cop,cargs = dest_type cty in
     if vop = cop then itlist2 type_match vargs cargs sofar
     else failwith "type_match";;

(* ------------------------------------------------------------------------- *)
(* Conventional matching version of mk_const (but with a sanity test).       *)
(* ------------------------------------------------------------------------- *)

let mk_mconst(c,ty) =
  try let uty = get_const_type c in
      let mat = type_match uty ty [] in
      let con = mk_const(c,mat) in
      if type_of con = ty then con else fail()
  with Failure _ -> failwith "mk_const: generic type cannot be instantiated";;

(* ------------------------------------------------------------------------- *)
(* Like mk_comb, but instantiates type variables in rator if necessary.      *)
(* ------------------------------------------------------------------------- *)

let mk_icomb(tm1,tm2) =
  let "fun",[ty;_] = dest_type (type_of tm1) in
  let tyins = type_match ty (type_of tm2) [] in
  mk_comb(inst tyins tm1,tm2);;

(* ------------------------------------------------------------------------- *)
(* Instantiates types for constant c and iteratively makes combination.      *)
(* ------------------------------------------------------------------------- *)

let list_mk_icomb cname =
  let cnst = mk_const(cname,[]) in
  fun args -> rev_itlist (C (curry mk_icomb)) args cnst;;

(* ------------------------------------------------------------------------- *)
(* Free variables in assumption list and conclusion of a theorem.            *)
(* ------------------------------------------------------------------------- *)

let thm_frees th =
  let asl,c = dest_thm th in
  itlist (union o frees) asl (frees c);;

(* ------------------------------------------------------------------------- *)
(* Is one term free in another?                                              *)
(* ------------------------------------------------------------------------- *)

let rec free_in tm1 tm2 =
  if aconv tm1 tm2 then true
  else if is_comb tm2 then
    let l,r = dest_comb tm2 in free_in tm1 l or free_in tm1 r
  else if is_abs tm2 then
    let bv,bod = dest_abs tm2 in
    not vfree_in bv tm1 & free_in tm1 bod
  else false;;

(* ------------------------------------------------------------------------- *)
(* Searching for terms.                                                      *)
(* ------------------------------------------------------------------------- *)

let rec find_term p tm =
  if p tm then tm else
  if is_abs tm then find_term p (body tm) else
  if is_comb tm then
    let l,r = dest_comb tm in
    try find_term p l with Failure _ -> find_term p r
  else failwith "find_term";;

let find_terms =
  let rec accum tl p tm =
    let tl' = if p tm then insert tm tl else tl in
    if is_abs tm then
       accum tl' p (body tm)
    else if is_comb tm then
       accum (accum tl' p (rator tm)) p (rand tm)
    else tl' in
  accum [];;

(* ------------------------------------------------------------------------- *)
(* Syntax to destroy either variable or constant (handy for benign defs).    *)
(* ------------------------------------------------------------------------- *)

let dest_cvar tm =
  try dest_var tm with Failure _ -> dest_const tm;;

(* ------------------------------------------------------------------------- *)
(* General syntax for binders.                                               *)
(*                                                                           *)
(* NB! The "mk_binder" function expects polytype "A", which is the domain.   *)
(* ------------------------------------------------------------------------- *)

let is_binder s tm =
  try fst(dest_const(rator tm)) = s & is_abs(rand tm)
  with Failure _ -> false;;

let dest_binder s tm =
  try if fst(dest_const(rator tm)) = s
      then dest_abs(rand tm) else fail()
  with Failure _ -> failwith "dest_binder";;

let mk_binder op (v,tm) =
  let cop = mk_const(op,[type_of v,aty]) in
  mk_comb(cop,mk_abs(v,tm));;

(* ------------------------------------------------------------------------- *)
(* Syntax for binary operators.                                              *)
(* ------------------------------------------------------------------------- *)

let is_binop op tm =
  try rator(rator tm) = op with Failure _ -> false;;

let mk_binop op tm1 =
  let f = mk_comb(op,tm1) in
  fun tm2 -> mk_comb(f,tm2);;

let dest_binop op tm =
  try let f,r = dest_comb tm in
      let xop,l = dest_comb f in
      if xop = op then l,r else fail()
  with Failure _ -> failwith "dest_binop";;

let list_mk_binop op = end_itlist (mk_binop op);;

let binops op = striplist (dest_binop op);;

(* ------------------------------------------------------------------------- *)
(* Syntax for let terms.                                                     *)
(* ------------------------------------------------------------------------- *)

let dest_let tm =
  try let l,aargs = strip_comb tm in
      let vars,lebod = strip_abs (hd aargs) in
      let eqs = zip vars (tl aargs) in
      let le,bod = dest_comb lebod in
      if fst(dest_const le) = "LET_END" then eqs,bod else fail()
  with Failure _ -> failwith "dest_let: not a let-term";;

let is_let = can dest_let;;

(* ------------------------------------------------------------------------- *)
(* Syntax for generalized abstractions.                                      *)
(*                                                                           *)
(* These are here because they are used by the preterm->term translator;     *)
(* preterms regard generalized abstractions as an atomic notion. This is     *)
(* slightly unclean --- for example we need locally some operations on       *)
(* universal quantifiers --- but probably simplest. It has to go somewhere!  *)
(* ------------------------------------------------------------------------- *)

let dest_gabs =
  let dest_forall = dest_binder "!" in
  let strip_forall = splitlist dest_forall in
  let dest_geq = dest_binary "GEQ" in
  fun tm ->
    try if is_abs tm then dest_abs tm else
        let l,r = dest_comb tm in
        if not fst(dest_const l) = "GABS" then fail() else
        let ltm,rtm = dest_geq(snd(strip_forall(body r))) in
        rand ltm,rtm
    with Failure _ -> failwith "dest_gabs: Not a generalized abstraction";;

let is_gabs = can dest_gabs;;

let mk_gabs =
  let mk_forall = mk_binder "!" in
  let list_mk_forall(vars,bod) = itlist (curry mk_forall) vars bod in
  let mk_geq(t1,t2) =
    let p = mk_const("GEQ",[type_of t1,aty]) in
    mk_comb(mk_comb(p,t1),t2) in
  fun (tm1,tm2) ->
    if is_var tm1 then mk_abs(tm1,tm2) else
    let fvs = frees tm1 in
    let fty = mk_fun_ty (type_of tm1) (type_of tm2) in
    let f = variant (frees tm1 @ frees tm1) (mk_var("f",fty)) in
    let bod = mk_abs(f,list_mk_forall(fvs,mk_geq(mk_comb(f,tm1),tm2))) in
    mk_comb(mk_const("GABS",[fty,aty]),bod);;

let list_mk_gabs(vs,bod) = itlist (curry mk_gabs) vs bod;;

let strip_gabs = splitlist dest_gabs;;

(* ------------------------------------------------------------------------- *)
(* Useful function to create stylized arguments using numbers.               *)
(* ------------------------------------------------------------------------- *)

let make_args =
  let rec margs n s avoid tys =
    if tys = [] then [] else
    let v = variant avoid (mk_var(s^(string_of_int n),hd tys)) in
    v::(margs (n + 1) s (v::avoid) (tl tys)) in
  fun s avoid tys ->
    if length tys = 1 then
      [variant avoid (mk_var(s,hd tys))]
    else
      margs 0 s avoid tys;;

(* ------------------------------------------------------------------------- *)
(* Compactor for terms; useful for things stored permanently.                *)
(* ------------------------------------------------------------------------- *)

let term_hashtable = hashtbl__new 1001
and type_hashtable = hashtbl__new 501;;

let compact =
  let enter_type ty =
    try hashtbl__find type_hashtable ty
    with Not_found -> (hashtbl__add type_hashtable ty ty; ty)
  and enter_term tm =
    try hashtbl__find term_hashtable tm
    with Not_found -> (hashtbl__add term_hashtable tm tm; tm) in
  let rec compact_type ty =
    if is_vartype ty then enter_type ty else
    let tycon,tyargs = dest_type ty in
    enter_type (mk_type(tycon,map compact_type tyargs)) in
  let rec compact_term tm =
    if is_var tm then
      let n,ty = dest_var tm in
      enter_term(mk_var(n,enter_type ty))
    else if is_const tm then
      enter_term tm
    else if is_abs tm then
      let v,bod = dest_abs tm in
      enter_term(mk_abs(compact_term v,compact_term bod))
    else
      let l,r = dest_comb tm in
      enter_term(mk_comb(compact_term l,compact_term r)) in
  compact_term;;

(* ------------------------------------------------------------------------- *)
(* Compactor for theorems -- this is, of course, just a derived rule.        *)
(* ------------------------------------------------------------------------- *)

let COMPACT th =
  let tm' = compact(concl th) in
  EQ_MP (REFL tm') th;;
