signature Socket =
sig
    type ('addressfam, 'sock) socket
    type 'addressfam sock_addr

    (* witness types for the socket parameter *)
    type dgram
    type 'a stream
    type passive	(* for passive streams *)
    type active		(* for active (connected) streams *)

    (* witness types for the addressfam parameter *)
    type pf_file
    type pf_inet

    (* address constructors *)
    val fileAddr : string -> pf_file sock_addr
    val inetAddr : string -> int -> pf_inet sock_addr

    (* socket constructors *)
    val fileStream : unit -> (pf_file, 'a stream) socket
    val fileDgram  : unit -> (pf_file, dgram) socket
    val inetStream : unit -> (pf_inet, 'a stream) socket
    val inetDgram  : unit -> (pf_inet, dgram) socket


    (* socket management *)
    val accept  : ('a, passive stream) socket
		    -> ('a, active stream) socket * 'a sock_addr
    val bind    : ('a, 'b) socket * 'a sock_addr -> unit
    val connect : ('a, 'b) socket * 'a sock_addr -> unit
    val listen  : ('a, passive stream) socket * int -> unit
    val close   : ('a, 'b) socket -> unit
    datatype shutdown_mode = NO_RECVS | NO_SENDS | NO_RECVS_OR_SENDS
    val shutdown : ('a, 'b stream) socket * shutdown_mode -> unit

    val select :
	('a, 'b) socket list -> ('a, 'b) socket list 
	                     -> ('a, 'b) socket list -> (int * int) option ->
	('a, 'b) socket list * ('a, 'b) socket list * ('a, 'b) socket list 


    (* Sock I/O option types *)
    type out_flags = {don't_route : bool, oob : bool}
    type in_flags = {peek : bool, oob : bool}

    type 'a buf = {buf : 'a, ofs : int, size : int option}
    

    (* Sock output operations *)
    val sendVec	   : ('a, active stream) socket * Word8Vector.vector -> int
    val sendVec'   : ('a, active stream) socket * Word8Vector.vector buf *
	             out_flags -> int
    val sendVecTo  : ('a, dgram) socket * 'a sock_addr * Word8Vector.vector 
	           -> int
    val sendVecTo' : ('a, dgram) socket * 'a sock_addr * 
                     Word8Vector.vector buf *  out_flags -> int

    (* Sock input operations *)
    val recvVec  : ('a, active stream) socket * int -> Word8Vector.vector
    val recvVec' 
        : ('a, active stream) socket * int * in_flags -> Word8Vector.vector
    val recvVecFrom 
        : ('a, dgram) socket * int -> Word8Vector.vector * 'a sock_addr
    val recvVecFrom'
        : ('a, dgram) socket * int * in_flags 
          -> Word8Vector.vector * 'a sock_addr

end

(* 

    [select rsocks wsocks esocks timeout] blocks the calling process
    until there some input/output operations become possible on some
    sockets. The three list arguments are, respectively, a set of
    descriptors to check for reading (first argument), for writing
    (second argument), or for exceptional conditions (third argument).
    If timeout is NONE then select blocks until some input/output
    operations become possible; otherwise if timeout is SOME(ts, tus)
    then select blocks the calling process at most ts seconds and tus
    microseconds.

    A server socket is considered ready for reading if there is a
    pending connection which can be accepted with `accept'.  A client
    socket is ready for writing when its connection is fully
    established.
*)
