 #########################################################################
 #                                                                       #
 # Copyright (C) 1993, 1994 by General Electric Company.		 #
 # All rights reserved. 						 #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	widget.tcl
#
# Description:
#	Tk library enhancements that are common to multiple sorts of
#	widgets.
#
#	This file includes procedures that allow for multiple bindings
#	on events for widgets, and for bindings on some new events.  The
#	binding scheme implemented here is used for the following
#	general types of events.
#
#c	Destroy
#		Execute an action when a widget is destroyed.  This
#		binding is dispatched through the `bind all <Destroy>'
#		in `init.tcl.'
#c	GainFocus
#		Report that a widget has gained focus from another widget
#		in the same application.  This is a more restrictive event
#		than <FocusIn>; for instance, having the window manager
#		direct focus to the application will not cause <GainFocus>
#		unless the application's focus changes as well.
#c	LoseFocus
#		Report that a widget has lost the focus to another widget in
#		the same application.  Like <GainFocus>, this event is more
#		restrictive than <FocusOut>.
#c	UpdateContent
#		Request that a widget that has a pending change post the
#		change.  This is used for entry widgets that allow the user
#		to edit their contents without posting to the text variable
#		until a change is committed.
#c	Validate
#		Request that a widget check that its content meets constraints,
#		for instance, that an entry's text value is numeric and within
#		its range.
#c	Unmap
#		Notification that a widget is no longer visible.
#
#	The file also implements support for composite widgets that are
#	treated as first-class widgets (by renaming the widget command so that
#	user code may intercept it).
#
# Global variables:
#c	widget:error$w
#		Action that the user has requested for an error message
#		resulting from a failure to validate the content of widget
#		$w.  THIS VARIABLE MAY NOT BE AN ARRAY ELEMENT.
#c	widget_priv(event,$event,$w)
#		Tcl command to execute in response to the event $event
#		occurring on the widget or widget class $w.

 # widget.tcl,v 1.1.1.1 1994/12/07 10:17:30 donald Exp
 # /homes/drs1004/repository/tkaux/widget.tcl,v
 # widget.tcl,v
# Revision 1.1.1.1  1994/12/07  10:17:30  donald
# First import after some small chanegs.
#
 # Revision 1.19  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.18  1994/10/27  18:27:39  kennykb
 # Updated legal notices prior to release.
 #
 # Revision 1.17  1994/09/15  14:02:47  kennykb
 # Fixed performance bug where cleaning up after destroyed widgets takes
 # O(n**4) time.  Should now be linear.
 #
 # Revision 1.16  1994/03/16  20:39:32  kennykb
 # Fixed typo in `widget:destroyApp' that was causing `application
 # destroyed' check to be bypassed.
 #
 # Revision 1.15  1993/11/12  20:04:28  kennykb
 # Removed all the stuff having to deal with composite widgets; a new
 # file, `composite.tcl', will be created that handles this material.
 #
 # Revision 1.14  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.13  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.12  1993/10/26  21:43:14  kennykb
 # Added an Unmap action to the list of actions recognized by widget_bind
 # and widget_event, in order to allow focus management to defocus a
 # window if it s unmapped.
 #
 # Revision 1.11  1993/10/25  16:16:30  kennykb
 # Changed code to make sure that BOTH class bindings AND widget bindings
 # are applied by widget_event.  This is needed, for example, in focus
 # management, where an entry that owns focus is destroyed.  Both the
 # focus Destroy procedure (bound at widget level) and the entry Destroy
 # procedure (bound at Class level) need to get the chance to clean up.
 #
 # Revision 1.10  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.9  1993/10/20  19:06:24  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.8  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.7  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.6  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.5  1993/10/14  13:54:59  kennykb
 # Added "widget_wiatVariable" and the associated procedures.  Changed
 # widget_propagateConfig and widget_propagateAction to be a trifle more
 # efficient.  THey're still FAR too slow.
 #
 # Revision 1.4  1993/07/21  19:44:36  kennykb
 # Finished cleaning up structured commentary.
 #
 # Revision 1.3  1993/07/20  19:17:12  kennykb
 # Improved structured comments.
 # Changed modules through `g' in the alphabet to follow `:' and `_' naming
 # conventions.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:34:10  kennykb
 # Initial revision
 #

# Procedure:	widget:check
#
# Synopsis:
#	Internal procedure to validate the content of a widget.
#
# Usage:
#c	widget:check pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:check' causes the Validate event to occur on a given single
#	widget.  (It uses widget_event (q.v.) to cause the event.)
#
#	If the validation succeeds, widget:check returns.  Otherwise, it
#	uses `widget:error' (q.v.) to display an error message, and resignals
#	the error.

proc widget:check w {
	set status [catch {widget_event $w Validate} message]
	if {$status != 0} {
		global errorInfo
		set info $errorInfo
		widget:error $w "[winlabel $w]: $message"
		error $message $info
	}
}

# Procedure:	widget:checkall
#
# Synopsis:
#	Internal procedure to validate the content of a widget tree.
#
# Usage:
#c	widget:checkall pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:checkall' is used to validate an entire portion of the
#	widget tree at once.  It runs `widget:check' on the specified widget
#	and on all of its subordinates.  It returns normally if all the
#	checks succeed, and otherwise throws an error (and displays a message)
#	detailing the check that failed.

proc widget:checkall w {
	widget:check $w
	foreach c [winfo children $w] {
		if {$c != [winfo toplevel $c]} {
			widget:checkall $c
		}
	}
}

# Procedure:	widget:error
#
# Synopsis:
#	Internal procedure to report a validation error on a widget.
#
# Usage:
#c	widget:error pathName message
#
# Parameters:
#c	pathName
#		Path name of a widget.
#c	message
#		Error message to display
#
# Return value:
#	None.
#
# Description:
#	widget:error makes a modal dialog box describing an error in
#	validating the contents of a widget, and waits for the dialog to be
#	dismissed by the user.

proc widget:error {w message} {
	global widget:error$w
	modalDialog transient choicebox $w.error \
		-text $message \
		-buttons OK \
		-icon rtfm \
		-textvariable widget:error$w
	widget_waitVariable widget:error$w
	unset widget:error$w
	modalDialog.end $w.error
}

# Procedure:	widget_bind
#
# Synopsis:
#	Establish an event handler on a widget.
#
# Usage:	widget_bind pathName event ?action?
#	-or-	widget_bind Class event ?action?
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	Class
#		Widget class
#c	event
#		Event to catch.  May be one of:
#c			Destroy GainFocus
#c			LoseFocus UpdateContent
#c			Validate Unmap
#c	action
#		Tcl command to execute when the specified event occurs
#		on the specified widget.  If the binding is for a class, the
#		widget name will be appended to `action'.
#		If `action' is the null string, any existing binding is
#		removed.  If `action' begins with a plus sign, the specified
#		action is appended to the set of bindings for the widget.
#
# Return value:
#	The Tcl command that will be executed when the specified event occurs.
#
# Description:
#	`widget_bind' establishes a new binding for a given event on a
#	specified widget.  It is used to manage events that do not correspond
#	to X events, or that must be multiply dispatched.
#
#	The events, and their meanings, are as follows.
#
#c	Destroy
#		Widget has been destroyed.
#c	GainFocus
#		Widget has gained the keyboard focus by
#		means of `focus_goTo.'  This is a more
#		restrictive event than <FocusIn>, which
#		seems to happen almost at random.
#	LoseFocus
#		Widget has lost the keyboard focus by
#		means of `focus_goTo.'  This is a more
#		restrictive event than <FocusOut>, which
#		happens for various reasons outside the
#		application's control.
#c	UpdateContent
#		Widget has been requested to update its
#		content because a command is to be executed.
#c	Validate
#		Widget has been requested to check that its
#		content meets constraints.
#c	Unmap
#		Widget has been unmapped from the screen.

proc widget_bind {w event {string ?}} {
	global widget_priv
	case $event in {
		{ Destroy GainFocus 
		  LoseFocus UpdateContent 
		  Validate Unmap } {
		}
		default {
			error "widget_bind: unknown event $event"
		}
	}
	if {$string == "?"} {
		if [info exists widget_priv(event,$event,$w)] {
			set string $widget_priv(event,$event,$w)
		} else {
			set string ""
		}
	} elseif {$string == ""} {
		catch {unset widget_priv(event,$event,$w)}
	} else {
		if {[string index $string 0] == "+"} {
			set string [string range $string 1 end]
			if [info exists widget_priv(event,$event,$w)] {
				append string \n
				append string $widget_priv(event,$event,$w)
			}
		}
		set widget_priv(event,$event,$w) $string
	}
	return $string
}

# Procedure:	widget_addBinding
#
# Synopsis:
#	Add a binding to the list of bindings for a given widget and event.
#
# Usage:
#c	widget_addBinding pathName event action
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	event
#		Event for which to watch (see widget_bind)
#c	action
#		Tcl command to execute when the specified event occurs.
#
# Return value:
#	Complete set of bindings for the specified widget and event.
#
# Description:
#	widget_addBinding adds an action to the list of actions for a
#	specified event on a given widget.  It differs from calling widget_bind
#	with an event beginning with a plus sign in that if the specified
#	action is already on the list of actions to perform, it will not be
#	added a second time.

proc widget_addBinding {w event string} {
	global widget_priv
	if [info exists widget_priv(event,$event,$w)] {
		set curBinding $widget_priv(event,$event,$w)
	} else {
		set curBinding ""
	}
	if {[string first $curBinding $string] >= 0} {
		return $curBinding
	} else {
		return [widget_bind $w $event +$string]
	}
}

# Procedure:	widget_event
#
# Synopsis:
#	Cause an event on a widget.
#
# Usage:
#c	widget_event pathName event
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	event
#		Event that has occurred (see widget_bind)
#
# Return value:
#	None.
#
# Description:
#	widget_event is used when one of the events monitored by widget_bind
#	occurs.  It executes the actions bound to the event for the widget,
#	and for its class.

proc widget_event {w event} {
	global widget_priv
	set action {}
	if [info exists widget_priv(event,$event,$w)] {
		set action $widget_priv(event,$event,$w)\n
	}
	if {[winfo exists $w] 
	    && [info exists widget_priv(event,$event,[winfo class $w])]} {
		append action "$widget_priv(event,$event,[winfo class $w]) $w\n"
	}
	if {$action != ""} {
		return [uplevel #0 $action]
	} else {
		return 0
	}
}

# Procedure:	widget:destroy
#
# Synopsis:
#	Internal procedure executed in response to all <Destroy> events
#	to clean up after widget destruction.
#
# Usage:
#c	widget:destroy pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:destroy' is called when a <Destroy> X event is received for
#	a given widget.  It executes all the `Destroy' bindings that have
#	been established using `widget_bind' for the given widget (and its
#	class).
#
#	Finally all `widget_priv' entries belonging to the given widget
#	are deleted.

proc widget:destroy w {
	global widget_priv
	global widget_type
	widget_event $w Destroy
	widget_unsetPriv widget_priv $w {
		event,Destroy event,GainFocus event,LoseFocus,
		event,Unmap event,UpdateContent event,Validate
	}
}

# Procedure:	widget:unmap
#
# Synopsis:
#	Internal procedure executed in response to all <Unmap> events
#	to clean up after widget disappearance.
#
# Usage:
#c	widget:unmap pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:unmap' is called when an <Unmap> X event is received for
#	a given widget.  It executes all the `Unmap' bindings that have
#	been established using `widget_bind' for the given widget (and its
#	class).

proc widget:unmap w {
	global widget_priv
	widget_event $w Unmap
}

# Procedure:	widget_waitVariable
#
# Synopsis:
#	Wait for a variable
#
# Usage:
#c	widget_waitVariable name
#
# Parameters:
#	name
#		Name of a variable
#
# Return value:
#	None specified.
#
# Description:
#	The `widget_waitVariable' is identical to the Tk command, `tkwait
#	variable', except that it records the name of the variable in an
#	array named `widget_waitVariables'.  The rationale is that a `tkwait
#	variable' hangs the process if the application is destroyed while
#	the `tkwait' is pending.

proc widget_waitVariable {vname} {
	global widget_waitVariables
	global widget_appDestroyed
	if {![info exists widget_waitVariables($vname)]} {
		set widget_waitVariables($vname) 1
	} else {
		incr widget_waitVariables($vname)
	}
	uplevel 1 tkwait variable [list $vname]
	set content [uplevel #0 set $vname]
	if {[incr widget_waitVariables($vname) -1] == 0} {
		unset widget_waitVariables($vname)
	}
	if {[info exists widget_appDestroyed]} {
		error \
"widget_waitVariable $vname terminated prematurely:
   application has been destroyed."
	}
}

# Procedure:	widget:destroyApp
#
# Synopsis:
#	Throw errors at widget_waitVariable when an application is destroyed.
#
# Usage:
#c	widget:destroyApp
#
# Parameters:
#	None
#
# Return value:
#	None specified.
#
# Description:
#	widget:destroyApp is called when the application root window is
#	destroyed.  It sets the values of all variables active in
#	widget_waitVariable, so that all the waits will terminate.

proc widget:destroyApp {} {
	global widget_waitVariables
	global widget_appDestroyed
	set widget_appDestroyed 1
	if [catch {array names widget_waitVariables} names] return
	foreach vname $names {
		widget:destroyApp2 $vname
	}
}
proc widget:destroyApp2 {vname} {
	upvar #0 $vname var
	set var "@APP@DESTROYED@"
}

# Procedure:	widget_checkAndDestroy
#
# Synopsis:
#	Destroy a widget, if the widget exists.
#
# Usage:
#c	widget_checkAndDestroy pathName
#
# Parameters:
#c	pathName
#		Path name of a widget to destroy.
#
# Return value:
#	None.
#
# Description:
#	widget_checkAndDestroy checks if a specified widget exists.  If it
#	exists, it is destroyed, otherwise, nothing happens.
#
# Notes:
#	The commonest use of `widget_checkAndDestroy' is so that destroying a
# 	widget may destroy its parent.  For example, destroying a widget
#	that has been packed into a `transient' frame should destroy the
#	top-level window as well.
#
# Example:
#c	widget_addBinding $w.child Destroy \
#
#c		"widget_checkAndDestroy $w"

proc widget_checkAndDestroy w {
	if [catch {winfo exists $w} exists] return
	if {$exists && ($w == "." || [info commands $w] == $w)} {
		destroy $w
	}
}

# Procedure:	widget_unsetPriv
#
# Synopsis:
#	Service procedure that cleans up working storage for a widget.
#
# Usage:
#c	widget_unsetPriv array widgetName prefices
#
# Parameters:
#c	array
#		The array (e.g., widget_priv, focus_priv) to be
#		cleaned up.
#c	widgetName
#		The widget whose entries are to be cleaned up
#c	prefices
#		The prefices to the widget name that form
#		keys for the array.
#
# Return value:
#	None.

proc widget_unsetPriv {aname w prefices} {
	upvar #0 $aname a
	foreach p $prefices {
		if [info exists a($p,$w)] {
			unset a($p,$w)
		}
	}
}
